// Canon BubbleJet BJ130e & compatibles models driver Class
//
// 2001, Philippe Houdoin
// Based on gsbj10e.c GhostScript driver
//
// Printers supported
//   bj10e, bj10ex, bj10sx, bj20, bj130e, bj300 and bj330
// 
// Todo list:
// - all! :-(

#define DEBUG	1

#include <Debug.h>

#include <stdio.h>
#include <string.h>			// for memset()

#include "BJ130e.h"


// Constructor & destructor
// ------------------------

BJ130e::BJ130e()
	: PrinterDriver()
{
	SetResolution(360);
	
	switch(YResolution())
		{
		case 180:
			m_head_nb_bytes = 3;
			m_graphic_mode = (XResolution() == 180 ? 11 : 12);
			break;
			
		case 360:
			m_head_nb_bytes = 6;
			m_graphic_mode = (XResolution() == 180 ? 14 : 16);
			break;
		};
		
	m_head_nb_nozzles 	= 8 * m_head_nb_bytes;
}


BJ130e::~BJ130e()
{
	if ( Transport() )
		{
		ExitPrinter();
		CloseTransport();
		};
}

// Methods
// -------

// --------------------------------------------------
status_t BJ130e::PrintPage
	(
	int32		page_number,
	int32		page_count
	)
{
	uint32 	vskip;
	uint32	w;
	void *	scanline;
	uint8 *	buffer;
	int		nozzle;
		
	// Fire renderer thread
	StartPageRenderer(B_GRAY1 );

	// BEWARE HERE: scanline info (ScanLineWidth(), etc)
	// aren't available yet, not until first GetScanLine() returns.

	if ( page_number == 1 )
		// Init printer before first page
		InitPrinter();

	BeginPage();
	
	buffer	= NULL;

	vskip 	= 0;
	w 		= 0;
	nozzle	= 0;

	while( GetScanLine(&scanline) == B_OK )
		{
#if 0
		if ( 	nozzle == 0 &&
				IsEmptyScanLine(scanline) )
			// empty line, just skip
			vskip++;
		else
#endif
			{
			if ( ! buffer )
				{
				// first time: allocate the temporary swipe buffer...
				w = ScanLineWidth();
				
				buffer = (uint8 *) malloc((w * m_head_nb_bytes) + 16); // +16 = margin
				if ( buffer == NULL )
					break;
					
				memset(buffer, 0, (w * m_head_nb_bytes));
				};

			// okay, we've at least one not empty raster line to print
			// but we need to send nb_nozzles raster line at once
			// so accumulate them
			
			AddToSwipe((uint8 *) scanline, w, nozzle, buffer);
			
			nozzle++;
			if ( nozzle >= m_head_nb_nozzles )
				{
//				if ( vskip )
//					AdvancePaper(vskip);
//				vskip = 0;

				// write swip
				// WriteSwipe(buffer, w);
#if 1
		// ESC [ g <count> <mode> <raw data>
		WriteCommand('g', (w * m_head_nb_bytes) +1);
		Transport()->Write(&m_graphic_mode, 1);
		Transport()->Write(buffer, w * m_head_nb_bytes);
		
		Transport()->Write("\r\x1b\x33\x18\n", 5);	// New swipe
		// Transport()->Write("\r", 1);
#endif				
				// start next swipe
				nozzle 	= 0;
				memset(buffer, 0, (w * m_head_nb_bytes));
				};
			};	// end else part
			
		NextScanLine();
		};	// next scan line
		
	if ( nozzle )
		{
		// write last swipe, if any
//		if ( vskip )
//			AdvancePaper(vskip);

		// WriteSwipe(buffer, w);
#if 1
		// ESC [ g <count> <mode> <raw data>
		WriteCommand('g', (w * m_head_nb_bytes) +1);
		Transport()->Write(&m_graphic_mode, 1);
		Transport()->Write(buffer, w * m_head_nb_bytes);
		
		Transport()->Write("\r\x1b\x33\x18\n", 5);	// New swipe
		// Transport()->Write("\r", 1);
#endif
		};

	if ( buffer )
		free(buffer);
		
	EndPage();
		
	StopPageRenderer();
	
	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::AddToSwipe
	(
	uint8 *	in,
	uint32	w,
	uint32	nozzle,
	uint8 *	swipe
	)
{
	int8 	bit;
	uint32	x;
	uint8 *	out;
	uint8	mask;
	
	// nozzle -> byte offset, bit offset
	// example: if nozzle = 4, byte offset = 0 (first byte of 6 bytes per nozzles column,
	// bit offset = 7 - (nozzle % 8) = 3
	// so for each bit in scanline we add to buffer[byte_offset] += bit_value << bit_offset;

	bit = 7;
	
	out = swipe;
	out += (nozzle / 8); 
	mask =  1 << (7 - (nozzle % 8));	// compute bit mask for this nozzle line
	
	for ( x = 0; x < w; x++ )
		{
		if ( *in & (1 << bit) )
			*out |= mask;
		out += m_head_nb_bytes;	// next swipe column
		
		bit--;
		if ( bit < 0 )
			{
			bit = 7;
			in++;	// next byte
			}; 
		}; // next column

	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::WriteSwipe
	(
	uint8 *	in,
	uint32	w
	)
{
	uint32	x;
	uint32	hskip;
	int		i;
	uint8 *	start;
	uint8 * stop;
	int 	count;
	uint8	byte;

	/*

	'in' point to swipe ready-to-be-sent data, and look like this:
	
	x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 x10 x11 x12 x13 .... xn
	(n = w * m_head_nb_bytes - 1)
	
	where x0-(m_head_nb_bytes-1) is the first column bytes of the m_head_nb_nozzles nozzles swipe.
	For m_head_nb_nozzles = 48 / m_head_nb_bytes = 6:
	
	-----> 
 	x0	x6	x12	x18	x24	...
	x1	x7	x13	x19	x25	...
	x2	x8 	x14	x20	x26	...
	x3	x9 	x15	x21	x27	...
	x4	x10	x16	x22	x28	...
	x5	x11	x17	x23	x29	...

	But here we try to optimize data output speed:
	instead of sending null column(s), 
	we send the quicker/smaller AdvanceHead() command(s)
	
	*/

	start = stop = NULL;
	hskip = 0;

	for ( x = 0; x < w; x++ )
		{
		// is it a null column?
		for ( i = 0; i < m_head_nb_bytes; i++ )
			{
			if (in[i])
				break;
			};

		if ( i < m_head_nb_bytes )
			{
			// not empty column!
			// okay, so we'll have at least this column to send...
			if ( ! start )
				start = in;

			stop = in + m_head_nb_bytes;
			}
		else
			{
			// emtpy column

			if ( start )
				{
				// send previous not empty columns
				if ( hskip )
					AdvanceHead(hskip);

				count =  stop - start;

				// ESC [ g <count> <mode> <raw data>
				WriteCommand('g', count + 1);
				Transport()->Write(&m_graphic_mode, 1);
				Transport()->Write(start, count);
			
				start = stop = NULL;
				hskip = 0;
				};

			hskip++;
			};

		in += m_head_nb_bytes;	// next column bytes
		};	
	
	if ( start )
		{
		// send last not empty columns data
		if ( hskip )
			AdvanceHead(hskip);

		count =  stop - start;

		// ESC [ g <count> <mode> <raw data>
		WriteCommand('g', count + 1);
		Transport()->Write(&m_graphic_mode, 1);
		Transport()->Write(start, count);
		};
		
	// New swipe
	Transport()->Write("\r\x1b", 2);
	Transport()->Write("3", 1);
	byte = m_head_nb_nozzles;
	Transport()->Write(&byte, 1);
	Transport()->Write("\n", 1);

	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::InitPrinter()
{
	// Reset to default setting...
	// ESC [ K <count lo> <count hi> <data0> <data1> ...
	WriteCommand('K', 2, "\000\044");

	// turn off auto carriage return ESC 5 0|1
	Transport()->Write("\0335\x00", 3);

	// Set spacing ESC [ \ 
	WriteCommand('\\', 4);
	WriteLoHi(0);
	WriteLoHi(YResolution());

	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::ExitPrinter()
{
	if (! Transport() )
		return B_ERROR;

	// Reset to default setting...
	// ESC [ K 02 00 00 0F
	WriteCommand('K', 1, "\x00");
		
	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::BeginPage()
{
	// Set Page Lenght 0x0b = 11 pouces de haut = Letter
	// 0x0c = 12 pouces de haut = A4
	char page_length[] = { 0x1b, 'C', 0, 0x0c };
	Transport()->Write(page_length, 4);

	// Set Top of Form ( ESC 4 )
	Transport()->Write("\x1b\x34", 2);

	// Set head at start
	Transport()->Write("\r", 1);

	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::EndPage()
{
	// Form Feed
	return Transport()->Write("\x0c", 1);
}


// --------------------------------------------------
status_t BJ130e::AdvancePaper
	(
	int		count
	)
{
	while ( count > 255 )
		{
		// ESC J <nb of 1/yres skip>
		Transport()->Write("\x1bJ\xFF", 3);	// skip 255 vertical space
		count -= 255;
		};
		
	if ( count )
		{
		uint8 b = count;
		Transport()->Write("\x1bJ", 2);
		Transport()->Write(&b, 1);// skip count vertical space
		};
		
	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::AdvanceHead
	(
	int		count
	)
{

	int 	nb;
	uint8	byte;

	// print head has only 120 dpi step precision

	nb	   = 120 * count / XResolution();
	count -= (XResolution() * nb / 120); 

	if ( nb )
		{
		// first, quick horizontal skip, at 120 dpi resolution
		// ESC d <nb of 1/120 dpi skip>
		Transport()->Write("\033d", 2);
		WriteLoHi(nb);
		};
	
	if ( count )
		{
		// second, print nothing up to x_resolution * count dots

		int i, j;

		// ESC [ g <count> <mode> <raw data>
		WriteCommand('g', (count * m_head_nb_bytes) + 1);		
		byte = m_graphic_mode;
		Transport()->Write(&byte, 1);
		
		for ( i = 0; i < count ; i++)
			{
			for ( j = 0; j < m_head_nb_bytes; j++ )
				Transport()->Write("\x0", 1);
			};
		};

	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::WriteCommand
	(
	uint8 			command,
	int				length = 0,
	const void *	data = NULL
	)
{
	uint8	tmp[3] = { 0x1b, '[', ' ' };
	
	tmp[2] = command;
	
	Transport()->Write(tmp, 3);
	WriteLoHi(length);
	
	if ( data != NULL && length > 0 )
		Transport()->Write(data, length);
		
	return B_OK;
}


// --------------------------------------------------
status_t BJ130e::WriteLoHi(int value)
{
	uint8 tmp[2];
	
	tmp[0] = (value & 0xff);
	tmp[1] = (value & 0xffff) >> 8;
	
	return Transport()->Write(tmp, 2);
}

	 
// --------------------------------------------------
status_t BJ130e::WriteHiLo(int value)
{
	uint8 tmp[2];
	
	tmp[0] = (value & 0xffff) >> 8;
	tmp[1] = (value & 0xff);
	
	return Transport()->Write(tmp, 2);
}
 



