/*
 * linux.c
 * Copyright (c) 1997 Be, Inc.	All Rights Reserved 
 */
#include "fakeout.h"
#include <stdarg.h>
#include <PCI.h>
#include <OS.h>
#include <Errors.h>
#include <stdlib.h>
#include <linux/timer.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/interrupt.h>
#include <linux/etherdevice.h>
#include <KernelExport.h>

#define ABS(x) (((x) < 0) ? -(x) : (x))

#define ETHER_BUF_SIZE  64		/* size of minimum ethernet buffer */
#define N_ETHER_BUFS    512		/* total number of these buffers */

#define TX_RETRIES 4

#define ETHER_LOW_MEM_THRESH  40	/* when to start worrying about low mem */

/*
 * For dealing with bitmaps (locating free buffers)
 */
#define CHUNKNO(x) ((x) / sizeof(ether_bitmap[0]))
#define CHUNKBIT(x) (1 << ((x) % sizeof(ether_bitmap[0])))

/*
 * lock/unlock a device (or anything with a ps & lock field)
 */
#define ALOCK_ACQUIRE(x) alock_acquire(&(x)->ps, &(x)->lock)
#define ALOCK_RELEASE(x) alock_release(&(x)->ps, &(x)->lock)


#undef outb
#undef outw
#undef outl
#undef inb
#undef inw
#undef inl


typedef struct timer_stuff {
	struct timer_list *timer;
	sem_id sem;
	struct timer_stuff *next;
} timer_stuff_t;

typedef struct intstuff {
	unsigned int irq;
	void (*handler)(int, void *, struct pt_regs *);
	void *dev_id;
	struct intstuff *next;
} intstuff_t;

typedef struct alock {
	cpu_status ps;
	spinlock lock;
} alock_t;

/* inserting some functions for the 3com509-driver */
/*	Außerdem wird eine Zeigerverwaltung benötigt. Definieren wir einfach die
	entsprechenden 16 Zeiger. Geändert am 9.5.98 (Slice/Michael)
*/
void *irq2dev_map[16];
/*	Die Funktion dev_tint wird angeblich nicht benötigt. Und tatsächlich habe
	ich keine Probleme erkannt, wenn die Funktion so definiert wird:
*/
void dev_tint(struct device *dev) { return; }
/*	Die Funktionalität "check_region" wird so definiert, daß es immer ein 
	"false" zurückgibt. Somit wird vorgegaugelt, daß keine Doppelvergabe	
	stattgefunden hat.
*/
int check_region(unsigned int from, unsigned int extent) { return false; }


static unsigned char *dev_alloc(size_t size);
static void dev_free(unsigned char *buf, size_t size);

static unsigned char ether_bufs[N_ETHER_BUFS * ETHER_BUF_SIZE];
static unsigned long ether_bitmap[N_ETHER_BUFS / sizeof(unsigned long)];
static int ether_nbufs = N_ETHER_BUFS;

static timer_stuff_t *timers;
static sem_id locksem = B_ERROR;
unsigned long bh_active;

static intstuff_t *ints;
static struct device *devices;

static alock_t ilock;
static long global_lock;
static alock_t devlock;


static void Lock(void)
{
	if (locksem == B_ERROR) {
		locksem = create_sem(1, "lock");
	}
	acquire_sem(locksem);
}

static void Unlock(void)
{
	release_sem(locksem);
}

status_t
timer_thread(void *arg)
{
	timer_stuff_t *stuff = (timer_stuff_t *)arg;
	status_t res;
	timer_stuff_t **p;
	bigtime_t expires;
	bigtime_t now;
	bigtime_t wait;
	
	expires = (stuff->timer->expires * 1000000) / 100;
	now = system_time();
	if (expires > now) {
		wait = expires - now;
	} else {
		wait = 0;
	}
	dprintf("Timer thread waiting for %d microseconds\n", wait);
	if (wait) {
		res = acquire_sem_etc(stuff->sem, 1, B_TIMEOUT, wait);
	} else {
		res = B_TIMED_OUT;
	}
	if (res == B_TIMED_OUT) {
		stuff->timer->function(stuff->timer->data);
	}
	dprintf("Timer thread done\n");
	Lock();
	delete_sem(stuff->sem);
	for (p = &timers; *p; p = &(*p)->next) {
		if (*p == stuff) {
			dev_free((unsigned char *)stuff, sizeof(*stuff));
			*p = (*p)->next;
			break;
		}
	}
	Unlock();
	dprintf("Timer thread exiting\n");
	return (0);
}


/*
 * This can be called with interrupts disabled
 * XXX: it is wrong to be using a semaphore for locking
 */
void add_timer(
			   struct timer_list *timer
			   )
{
	thread_id id;
	timer_stuff_t *ntimer;

	ntimer = (void *)dev_alloc(sizeof(*ntimer));
	ntimer->timer = timer;
	ntimer->sem = create_sem(0, "timer sem");
	Lock();
	ntimer->next = timers;
	timers = ntimer;
	Unlock();

	dprintf("Adding timer\n");
	id = spawn_kernel_thread(timer_thread, "timer", B_NORMAL_PRIORITY, 
							 (void *)ntimer);
	resume_thread(id);
}

int del_timer(struct timer_list *timer)
{
	timer_stuff_t *stuff;

	dprintf("Deleting timer\n");
	Lock();
	for (stuff = timers; stuff; stuff = stuff->next) {
		if (stuff->timer == timer) {
			release_sem(stuff->sem);
			Unlock();
			return (0); /* ??? */
		}
	}
	Unlock();
	return (-1); /* ??? */
}


int barrier() {
}

void outsl(int port, void *arg, int count)
{
	long *larg = (long *)arg;

	while (count-- > 0) {
		write_io_32(port, *larg++);
	}
}

void insl(int port, void *arg, int count)
{
	long *larg = (long *)arg;

	while (count-- > 0) {
		*larg++ = read_io_32(port);
	}
}


int fjiffies(void)
{
	return (100 * system_time() / 1000000);
}


void *kmalloc(unsigned int size, int type)
{
	return (malloc(size));
}

unsigned short  eth_type_trans(struct sk_buff *skb, struct device *dev)
{
	/* figure out the protocol field and return it */
	return (0);
}


static void
alock_acquire(cpu_status *psp, long *lockp)
{
	cpu_status ps;
	
	ps = disable_interrupts();
	acquire_spinlock(lockp);
	*psp = ps; /* only after we have the lock */
}

static void
alock_release(cpu_status *psp, long *lockp)
{
	cpu_status ps = *psp;
	
	release_spinlock(lockp);
	restore_interrupts(ps); /* use copy since psp could be changed */
}

static void
alock_init(cpu_status *psp, long *lockp)
{
	*psp = 0;
	*lockp = 0;
}

#if DEBUG
static void
print_bitmap(const char *a)
{
	int i;
	int count = 0;

	for (i = 0; i < sizeof(ether_bitmap); i++) {
		if (ether_bitmap[CHUNKNO(i)] & CHUNKBIT(i)) {
			count++;
		}
	}
	dprintf("%s: %d\n", a, count);
}
#endif

static unsigned char *dev_alloc(size_t size)
{
	int i;
	int j;
	int nbufs;
	int bitno;
	int good;
	cpu_status ps;

	ALOCK_ACQUIRE(&devlock);
	size = (size + ETHER_BUF_SIZE - 1) & ~(ETHER_BUF_SIZE - 1);
	nbufs = size / ETHER_BUF_SIZE;
	for (i = 0; i < (N_ETHER_BUFS - (nbufs - 1)); i++) {
		good = 1;
		for (j = 0; j < nbufs; j++) {
			bitno = i + j;
			if (ether_bitmap[CHUNKNO(bitno)] & CHUNKBIT(bitno)) {
				good = 0;
				break;
			}
		}
		if (good) {
			for (j = 0; j < nbufs; j++) {
				bitno = i + j;
				ether_bitmap[CHUNKNO(bitno)] |= CHUNKBIT(bitno);
				ether_nbufs--;
			}
			ALOCK_RELEASE(&devlock);
			return (&ether_bufs[i * ETHER_BUF_SIZE]);
		}
	}
	ALOCK_RELEASE(&devlock);
	return (NULL);
}

static void
dev_free(unsigned char *buf, size_t size)
{
	int nbufs;
	int i;
	int bitno;
	int bufno;
	cpu_status ps;

	ALOCK_ACQUIRE(&devlock);
	nbufs = (size + ETHER_BUF_SIZE - 1) / ETHER_BUF_SIZE;
	bufno = (buf - ether_bufs) / ETHER_BUF_SIZE;
	for (bitno = bufno; bitno < (bufno + nbufs); bitno++) {
		ether_bitmap[CHUNKNO(bitno)] &= ~CHUNKBIT(bitno);
		ether_nbufs++;
	}
	ALOCK_RELEASE(&devlock);
}


void skb_reserve(struct sk_buff *skb, int len)
{
	skb->data += len;
	skb->tail += len;
}

static struct sk_buff *alloc_skb(unsigned int size, int ignored)
{
	struct sk_buff *sk;
	unsigned char *ptr;
	int len = size;

	size = (size + 15) & ~15;
	ptr = dev_alloc(size + sizeof(struct sk_buff));
	if (ptr == NULL) {
		return (NULL);
	}
	sk = (struct sk_buff *)(ptr + size);
	sk->next = NULL;
	sk->truesize = size + sizeof(struct sk_buff);
	sk->head = ptr;
	sk->data = ptr;
	sk->tail = ptr;
	sk->end = ptr + len;
	sk->len = 0;
	return (sk);
}


struct sk_buff *dev_alloc_skb(unsigned int length)
{
	struct sk_buff *skb;

	skb = alloc_skb(length + 16, GFP_ATOMIC);
	if (skb) {
		skb_reserve(skb, 16);
	}
	return (skb);
}

/* free skbuff */
void dev_kfree_skb(struct sk_buff *sk, int mode)
{
	if (sk) {
		dev_free(sk->head, sk->truesize);
	}
}


int
register_netdev(struct device *dev)
{
	struct device **devpp;

	/*
	 * Find tail of list
	 */
	for (devpp = &devices; *devpp; devpp = &(*devpp)->next) {
	}

	/*
	 * Install new device at tail
	 */
	dev->next = NULL;
	*devpp = dev;
	return (0);
}

struct device *
init_etherdev(struct device *dev, int size)
{
	if (dev == NULL) {
		dev = malloc(sizeof(*dev));
		memset(dev, 0, sizeof(*dev));
	}
	dev->mc_count = 0;
	dev->mc_list = NULL;
	if (size > 0) {
		dev->priv = malloc(size);
	}
	return (dev);
}

void
ether_setup(struct device *dev)
{
	init_etherdev(dev, 0);
}
	

/* receive packet */
void netif_rx(struct sk_buff *sk) 
{
	struct sk_buff **skpp;
	struct sk_buff *tmp;
	struct device *dev = sk->dev;

	if (dev->sksem >= B_NO_ERROR) {
		/*
		 * If low on memory, start freeing bufs
		 */
		if (ether_nbufs < ETHER_LOW_MEM_THRESH) {
			tmp = dev->sklist;
			if (tmp) {
				dev->sklist = tmp->next;
				dev_kfree_skb(tmp, 0);
			}
		}

		/*
		 * Preserve the order when adding to this list
		 */
		for (skpp = &dev->sklist; *skpp; skpp = &(*skpp)->next) {
			/* find tail of list */
		}
		sk->next = NULL;
		*skpp = sk;
		release_sem_etc(dev->sksem, 1, B_DO_NOT_RESCHEDULE);
	} else {
		dprintf("packet dropped\n");
		dev_kfree_skb(sk, 0);
	}
}

static bool
interrupt_wrapper(void *vdata)
{
	intstuff_t *stuff = (intstuff_t *)vdata;

	acquire_spinlock(&ilock.lock);
	atomic_or(&global_lock, 1);

	stuff->handler(stuff->irq, stuff->dev_id, NULL);

	atomic_and(&global_lock, 0);
	release_spinlock(&ilock.lock);
	
	return (TRUE);
}

/* set interrupt routine */
int request_irq(unsigned int irq,
				void (*handler)(int, void *, struct pt_regs *),
				unsigned long flags,
				const char *device,
				void *dev_id
				)
{
	intstuff_t *stuff;

	stuff = malloc(sizeof(*stuff));
	stuff->irq = irq;
	stuff->handler = handler;
	stuff->dev_id = dev_id;
	Lock();
	stuff->next = ints;
	ints = stuff;
	Unlock();
	install_io_interrupt_handler(irq, interrupt_wrapper, (void *)stuff, 0);
	return (0);
}

void free_irq(unsigned int irq, void *dev_id)
{
	intstuff_t **p;
	intstuff_t *tmp;

	Lock();
	for (p = &ints; *p; p = &(*p)->next) {
		if ((*p)->dev_id == dev_id) {
			tmp = *p;
			remove_io_interrupt_handler(tmp->irq, interrupt_wrapper, (void *) tmp);
			*p = (*p)->next;
			free(tmp);
			break;
		}
	}
	Unlock();
}

  

void request_region(void)
{
	/* do nothing */
}

int pcibios_find_device(
						unsigned short vendor_id,
						unsigned short device_id,
						unsigned short pci_index,
						unsigned char *pci_bus,
						unsigned char *pci_device_fn
						)
{
	pci_info p;
	int i;

	for (i = 0; ; i++) {
		if (get_nth_pci_info(i, &p) != B_NO_ERROR) {
			break;
		}
		if (p.vendor_id != vendor_id) {
			continue;
		}
		if (p.device_id != device_id) {
			continue;
		}
		// if (pci_index == 0)
		*pci_bus = p.bus;
		*pci_device_fn = (p.device << 4) |  p.function;
		dprintf("Found dev bus: %02x, dev_fn: %02x\n",
				*pci_bus, *pci_device_fn);

		return (0); // SXSS
		// }
		// pci_index--; /* keep looking. */ //ig: here we overwrite correct pci_info
	}
	return (-1); // FAIL
}


int pcibios_present(void) {
	return (1);
}

int pcibios_read_config_byte(
							  unsigned char bus,
							  unsigned char device_fn,
							  unsigned char where,
							  unsigned char *value
							  )
{
	*value = read_pci_config(bus, device_fn >> 4, device_fn & 0xf, where,
							 sizeof(*value));
	return (0);
}

int pcibios_read_config_word(
							  unsigned char bus,
							  unsigned char device_fn,
							  unsigned char where,
							  unsigned short *value
							  )
{
	*value = read_pci_config(bus, device_fn >> 4, device_fn & 0xf, where,
							 sizeof(*value));
	return (0);
}

int pcibios_read_config_dword(
							  unsigned char bus,
							  unsigned char device_fn,
							  unsigned char where,
							  unsigned int *value
							  )
{
	*value = read_pci_config(bus, device_fn >> 4, device_fn & 0xf, where,
							 sizeof(*value));
	return (0);
}

int pcibios_write_config_word(
							   unsigned char bus,
							   unsigned char device_fn,
							   unsigned char where,
							   unsigned short value
							   )
{
	write_pci_config(bus, device_fn >> 4, device_fn & 0xf, where, 
					 sizeof(value), value);
}

int pcibios_write_config_byte(
							   unsigned char bus,
							   unsigned char device_fn,
							   unsigned char where,
							   unsigned char value
							   )
{
	write_pci_config(bus, device_fn >> 4, device_fn & 0xf, where, 
					 sizeof(value), value);
}



void __printk(const char *format, ...)
{
	va_list args;
	char buf[256];

	va_start(args, format);
	vsprintf(buf, format, args);
	dprintf(buf);
	va_end(args);
}

void __panic(const char *format, ...)
{
	va_list args;
	char buf[256];

	va_start(args, format);
	vsprintf(buf, format, args);
	dprintf(buf);
	va_end(args);
	dprintf("Spinning forever\n");
	for (;;);
	
}

int set_bit(int bit, void *addr)
{
	long *a = (long *)addr;
	long res;

	res = atomic_or(a, (1 << bit));
	return (((res >> bit) & 1) ? -1 : 0);
}


void mark_bh(int nr)
{
	set_bit(nr, &bh_active);
}

unsigned char *skb_put(struct sk_buff *skb, int len)
{
	unsigned char *tmp=skb->tail;
	skb->tail+=len;
	skb->len+=len;
	if(skb->tail>skb->end)
	{
		panic("skput:over: %d", len);
	}
	return tmp;
}


int
cli(void)
{
	int ret = 0;

	ALOCK_ACQUIRE(&ilock);
	if (atomic_or(&global_lock, 1) == 0) {
		ret++;
	}
	return (ret);
}

int
sti(void)
{
	int ret = 0;

	if (atomic_and(&global_lock, 0) == 1) {
		ret++;
	}
	ALOCK_RELEASE(&ilock);
	return (ret);
}

void
_save_flags(unsigned long *flags)
{
	if (cli()) {
		*flags = 0;
		sti();
	} else {
		*flags = 1;
	}
}

void
restore_flags(unsigned long flags)
{
	if (flags == 0) {
		sti();
	}
}

void
_linux_outw(unsigned short data, unsigned short port)
{
	write_io_16(port, data);
}

void
_linux_outl(unsigned long data, unsigned short port)
{
	write_io_32(port, data);
}

void
_linux_outb(unsigned char data, unsigned short port)
{
	write_io_8(port, data);
}


unsigned char _linux_inb(unsigned short port)
{
	unsigned char res;

	res = read_io_8(port);
	return (res);
}

unsigned short _linux_inw(unsigned short port)
{
	unsigned short res;

	res = read_io_16(port);
	return (res);
}

unsigned long _linux_inl(unsigned short port) 
{
	unsigned long res;

	res = read_io_32(port);
	return (res);
}

void *
linux_ether_probe(int *ncards, int (*init_module)(void))
{
	struct device *dev = NULL;
	int n;

	Lock();
	devices = NULL;
	if (init_module() == 0) {
		dev = devices;
		n = 0;
		while (devices) {
			n++;
			devices = devices->next;
		}
		*ncards = n;
	}
	Unlock();
	if (dev) {
		dev->sklist = NULL;
		dev->sksem = B_ERROR;
		alock_init(&dev->ps, &dev->lock);
		return ((void *)dev);
	}
	return (NULL);
}

int 
linux_ether_open(void *cookie, int cardno, void **session_cookie)
{
	struct device *dev = (struct device *)cookie;
	int n;
	int res;

	for (n = 0; n < cardno; n++) {
		if (dev == NULL) {
			return (EPERM);
		}
		dev = dev->next;
	}
	res = dev->init(dev);
	if (res != 0) {
		return (-ABS(res));
	}
	if (dev->open == NULL) {
		return (ENODEV);
	}
	res = dev->open(dev);
	if (res != 0) {
		return (-ABS(res));
	}
	dev->sksem = create_sem(0, "sk");
	*session_cookie = dev;
	return (0);
}

int
linux_ether_getaddr(void *cookie, char *buf)
{
	struct device *dev = (struct device *)cookie;

	memcpy(buf, dev->dev_addr, 6);
	return (B_NO_ERROR);
}

int
linux_ether_getframesize(void *cookie, char *buf)
{
	struct device *dev = (struct device *)cookie;
	const int hardcoded = 1514;

	memcpy(buf, &hardcoded, sizeof(hardcoded));
	return (B_NO_ERROR);
}

#define addreq(a, b) (memcmp(a, b, 6) == 0)

int
linux_ether_addmulti(void *cookie, char *buf)
{
	struct device *dev = (struct device *)cookie;
	struct dev_mc_list **lpp;
	struct dev_mc_list *tmp;

	for (lpp = &dev->mc_list; *lpp; lpp = &(*lpp)->next) {
		if (addreq((*lpp)->dmi_addr, buf)) {
			return (B_ERROR);
		}
	}
	tmp = malloc(sizeof(*tmp));
	memcpy(tmp->dmi_addr, buf, 6);
	(*lpp) = tmp;
	tmp->next = NULL;
	dev->mc_count++;
	dev->set_multicast_list(dev);
	return (B_NO_ERROR);
}

int
linux_ether_remmulti(void *cookie, char *buf)
{
	struct device *dev = (struct device *)cookie;
	struct dev_mc_list **lpp;
	struct dev_mc_list *tmp;

	for (lpp = &dev->mc_list; *lpp; lpp = &(*lpp)->next) {
		if (addreq((*lpp)->dmi_addr, buf)) {
			tmp = *lpp;
			*lpp = (*lpp)->next;
			free(tmp);
			dev->mc_count--;
			dev->set_multicast_list(dev);
			return (B_NO_ERROR);
		}
	}
	return (B_ERROR);
}

int
linux_ether_setpromisc(void *cookie, int promisc)
{
	struct device *dev = (struct device *)cookie;
	
	if (promisc) {
		dev->flags |= IFF_PROMISC;
	} else {
		dev->flags &= ~IFF_PROMISC;
	}
	dev->set_multicast_list(dev);
	return (B_NO_ERROR);
}

int
linux_ether_close(void *cookie)
{
	struct device *dev = (struct device *)cookie;
	int res;

	res = dev->stop(dev);
	if (res != 0) {
		return (-ABS(res));
	}
	delete_sem(dev->sksem);
	dev->sksem = B_ERROR;
	return (B_NO_ERROR);
}

int
linux_ether_write(void *cookie, char *buf, size_t size)
{
	struct sk_buff *sk;
	struct device *dev = (struct device *)cookie;
	size_t actsize;
	int i;
	
	actsize = max(size, 60);
	sk = dev_alloc_skb(actsize);
	if (sk == NULL) {
		dprintf("Couldn't allocate skb: %d\n", actsize);
		return (ENOBUFS);
	}
	memcpy(skb_put(sk, size), buf, size);
	if (actsize > size) {
		memset(sk->data + size, 0, actsize - size);
	}
	for (i = 0; i < TX_RETRIES; i++) {
		if (dev->hard_start_xmit(sk, dev) == 0) {
			return (size);
		}
		snooze(1000);
	}
	dprintf("ether xmit failed\n");
	dev_kfree_skb(sk, 0);
	return (B_ERROR);
}

int
linux_ether_read(void *cookie, char *buf, size_t size, int nonblocking)
{
	status_t res;
	sem_id sem;
	struct sk_buff *sk;
	struct device *dev = (struct device *)cookie;

	ALOCK_ACQUIRE(dev);
	if (nonblocking && dev->sklist == NULL) {
		ALOCK_RELEASE(dev);
		return (0);
	}
	ALOCK_RELEASE(dev);

	do {
		res = acquire_sem(dev->sksem);
		if (res < B_NO_ERROR) {
			dprintf("acquire sem failed\n");
			return (res);
		}

		ALOCK_ACQUIRE(dev);
		sk = dev->sklist;
		if (sk) {
			dev->sklist = dev->sklist->next;
		}
		ALOCK_RELEASE(dev);

		if (sk == NULL && nonblocking) {
			dprintf("This shouldn't happen\n");
			return (B_ERROR);
		}
	} while (sk == NULL);
	size = min(size, sk->len);
	memcpy(buf, sk->data, size);
	dev_kfree_skb(sk, 0);
	return (size);
}



void
linux_ether_dumpstats(void *cookie)
{
	struct device *dev = (struct device *)cookie;
	struct enet_statistics *stats;

	stats = dev->get_stats(dev);
#define DOIT(x) dprintf("%s: %d\n", #x, stats->x);
	DOIT(rx_packets);
	DOIT(tx_packets);
	DOIT(rx_errors);
	DOIT(tx_errors);
	DOIT(rx_dropped);
	DOIT(tx_dropped);
	DOIT(multicast);
	DOIT(collisions);
	DOIT(rx_length_errors);
	DOIT(rx_over_errors);
	DOIT(rx_crc_errors);
	DOIT(rx_frame_errors);
	DOIT(rx_fifo_errors);
	DOIT(rx_missed_errors);
	DOIT(tx_aborted_errors);
	DOIT(tx_carrier_errors);
	DOIT(tx_fifo_errors);
	DOIT(tx_heartbeat_errors);
	DOIT(tx_window_errors);
#undef DOIT
}
