//
//
//	LWOB objects parser
//
//										(C) JoOl 1998

#include "ParserLwob.h"
#include "LAPerrors.h"

#if __POWERPC__
#pragma export on
#endif
char			handlerName[]		= "Lightwave v2.0 (.lwob)";
char			handlerVersion[]	= "0.9.0";
char			handlerAuthor[]		= "Alain GAUTHEROT";
char			handlerEmail[]		= "gauthero@club-internet.fr";
#if __POWERPC__
#pragma export reset
#endif


ParserLwob*
instantiate_parser(int32 bs)
{
	return new ParserLwob(bs);
}


ParserLwob::ParserLwob(int32 blockSize)
{
	// input is a big-endian binary file
	SetLexer(new LAPbinLexer(max_c(1024, blockSize),
			/* LAP_BIG_ENDIAN or LAP_LITLE_ENDIAN */
			LAP_BIG_ENDIAN));

	// error output, tracer output, display
	SetTo(new LAPtracer(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_SCREEN_OUTPUT),
			new LAPtextBufferedOutput(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_NO_OUTPUT/*LAP_BUFFER_SCREEN_OUTPUT*/),
			new LAPdisplay(),
			LAP_OWN_ERROR_OUTPUT | LAP_OWN_TRACE_OUTPUT | LAP_OWN_DISPLAY);
}


ParserLwob::~ParserLwob()
{
	if (data)
		delete (pScene* )data;
}


void
ParserLwob::_Terminate()
{
}


status_t
ParserLwob::DoAbort()
{
	PierrotParser::Reset();
	return LAP_NO_ERROR;
}


status_t
ParserLwob::_Identify()
{
	ReadChunk;
	if (chunkId != CHUNK_FORM)
		return LAP_BAD_INPUT_TYPE;

	uint32			formType;
	ReadInt32(formType);
	if (formType != CHUNK_LWOB)
		return LAP_BAD_INPUT_TYPE;

	return LAP_NO_ERROR;
}


status_t
ParserLwob::_Parse()
{
	PierrotParser::Reset();

	// read file header
	ReadChunk;
	if (chunkId != CHUNK_FORM)
		return LAP_BAD_INPUT_TYPE;

	uint32			formType;
	ReadInt32(formType);
	if (formType != CHUNK_LWOB)
		return LAP_BAD_INPUT_TYPE;

	scene = new pScene((char* )GetLexer()->GetInputName());

	ReadMainChunk(chunkLen-sizeof(formType));

	pSphere				normSphere(pPoint3(0.0, 0.0, 0.0), 1.0);
	scene->Preprocess(&normSphere, appliInterface, 0.0, 1.0);

	scene->ambient.Set(0.7, 0.7, 0.7);
	if (scene->lights.CountItems() == 0)
	{
		pLight*			l1 = NewDirectionalLight("directional light 1", pVector3(-0.5, 0.5, 0.8));
		l1->ambient.Set(0.2, 0.2, 0.2);
		l1->diffuse.Set(1.0, 1.0, 1.0);
		l1->specular.Set(1.0, 1.0, 1.0);
	}

	Output(*traceOutput, scene, PRINT_LIGHT_ALL | PRINT_MATERIAL_ALL |
			PRINT_OBJECT_BOUNDS /*| PRINT_OBJECT_MATRIX | PRINT_OBJECT_POINTS*/);

	data = scene;
	scene = NULL;
	return LAP_NO_ERROR;
}


void
ParserLwob::ParsePoint(pPoint3& pt)
{
	ReadFloat(pt.x);
	ReadFloat(pt.y);
	ReadFloat(pt.z);
}


void
ParserLwob::ParseColor(pSpectra& col)
{
	char			bytes[4];
	ReadBytes((char* )bytes, 4);
	col.Set(float(bytes[0])/255.0,
			float(bytes[1])/255.0,
			float(bytes[2])/255.0);
}


void
ParserLwob::ParseIpercentage(float& p)
{
	int16			w;
	ReadInt16(w);
	p = float(w)/256.0;
}


char*
ParserLwob::ParseName()
{
	LAPtextBufferedOutput		buf(64, LAP_BUFFER_GROW_ON_OVERFLOW,
									LAP_BUFFER_NO_OUTPUT);
	char			currentCharacter;
	ReadBytes(&currentCharacter, 1);
	while (currentCharacter != 0)
	{
		buf << currentCharacter;
		ReadBytes(&currentCharacter, 1);
	}

	return buf.Detach();
}


uint32
ParserLwob::ParsePadName(char*& out)
{
	LAPtextBufferedOutput		buf(64, LAP_BUFFER_GROW_ON_OVERFLOW,
									LAP_BUFFER_NO_OUTPUT);
	char			currentCharacter;
	ReadBytes(&currentCharacter, 1);
	while (currentCharacter != 0)
	{
		buf << currentCharacter;
		ReadBytes(&currentCharacter, 1);
	}

	out = buf.Detach();
	uint32			len = strlen(out)+1;
	if (len & 0x01)
	{
		SkipBytes(1);
		len++;
	}
	return len;
}


void
ParserLwob::SkipChunk(uint32 chunkId, uint32 chunkLen)
{
	char		tmp[6];
	tmp[0] = (chunkId >> 24) & 0xff;
	tmp[1] = (chunkId >> 16) & 0xff;
	tmp[2] = (chunkId >>  8) & 0xff;
	tmp[3] = (chunkId      ) & 0xff;
	tmp[4] = 0;

	SkipBytes(chunkLen);
	*traceOutput << "skip chunk " << tmp << ' ' << int32(chunkLen) << " bytes" << eol;
}


void
ParserLwob::ReadMainChunk(uint32 len)
{
	*traceOutput << "ReadMainChunk: " << int32(len) << eol;
	*traceOutput << adv;

	pObject*			object = NewObject("main");

	uint32			currLen = 0;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == CHUNK_PNTS)
			ReadPointsChunk(chunkLen, object);

		else if (chunkId == CHUNK_SRFS)
			ReadSurfacesListChunk(chunkLen);

		else if (chunkId == CHUNK_POLS)
			ReadFacesListChunk(chunkLen, object);

		else if (chunkId == CHUNK_PCHS)
			ReadFacesListChunk(chunkLen, object);

		else if (chunkId == CHUNK_SURF)
			ReadSurfaceChunk(chunkLen);

		else
		{
			skipChunk;
		}
		currLen += chunkLen+8;
		PadChunk;
	}
}


void
ParserLwob::ReadPointsChunk(uint32 len, pObject* object)
{
	*traceOutput << "ReadPointsChunk: " << int32(len) << eol;
	*traceOutput << adv;

	int32			nbVertices = len/12;
	Assert("ReadPointsChunk", nbVertices*12);
	*traceOutput << nbVertices << " vertices" << eol << adv;

	object->vertices->SetAlloc(nbVertices, 0);

	for (int32 i = 0; i < nbVertices; i++)
	{
		pPoint3			pt;
		ParsePoint(pt);

		object->vertices->AddItem(pt);
//		Output(*traceOutput, pt);
//		*traceOutput << eol;
	}

	*traceOutput << ret << ret;
}


void
ParserLwob::ReadSurfacesListChunk(uint32 len)
{
	*traceOutput << "ReadSurfacesListChunk: " << int32(len) << eol;
	*traceOutput << adv;

	uint32			currLen = 0;
	while (currLen < len)
	{
		char*			surfName;
		currLen += ParsePadName(surfName);

		pMaterial*		mat = NewMaterial(surfName);

//		*traceOutput << scene->materials.CountItems() << " '" << surfName << '\'' << eol;
		delete[] surfName;
	}

	*traceOutput << ret;
}


void
ParserLwob::ReadFacesListChunk(uint32 len, pObject* object)
{
	*traceOutput << "ReadFacesListChunk: " << int32(len) << eol;
	*traceOutput << adv;

	object->faces.SetAlloc(POLY_FACES_ALLOC, POLY_FACES_BLOCK);

	uint32			currLen = 0;
	while (currLen < len)
	{
		uint16					nbVert;
		PierrotArray<int32 >	indices(16, 8);
		int16					surfIndex;

		ReadInt16(nbVert);
		for (uint16 i = 0; i < nbVert; i++)
		{
			uint16				index;
			ReadInt16(index);
			indices.AddItem(int32(index));
		}
		ReadInt16(surfIndex);

		if (surfIndex < 0)
		{
			currLen += ReadDetailPolygonChunk();
			surfIndex = -surfIndex;
		}
/*
		*traceOutput << int32(nbVert);
		for (uint16 i = 0; i < indices.CountItems(); i++)
			*traceOutput << ' ' << int32(indices[i]);
		*traceOutput << ", " << int32(surfIndex) << eol;
*/
		if (indices.CountItems() > 2)
		{
			object->faces.AddItem(new pFace(scene->materials[surfIndex-1],
									indices.CountItems(), indices.Items()));
		}

		currLen += (nbVert + 2) << 1;
	}

	*traceOutput << ret;
}


uint32
ParserLwob::ReadDetailPolygonChunk()
{
	*traceOutput << "ReadDetailPolygonChunk" << eol;
	*traceOutput << adv;

	uint16					nbVert;
	PierrotArray<uint16 >	indices(16, 8);
	int16					surfIndex;

	ReadInt16(nbVert);
	for (uint16 i = 0; i < nbVert; i++)
	{
		uint16				index;
		ReadInt16(index);
		indices.AddItem(index);
	}
	ReadInt16(surfIndex);
/*
	*traceOutput << nbVert;
	for (uint16 i = 0; i < indices.CountItems(); i++)
		*traceOutput << ' ' << indices[i];
	*traceOutput << ", " << surfIndex << eol;
	*traceOutput << ret;
*/
	return (nbVert+2) << 1;
}


void
ParserLwob::ReadSurfaceChunk(uint32 len)
{
	*traceOutput << "ReadSurfaceChunk: " << int32(len) << eol;
	*traceOutput << adv;

	pSpectra		color;
	uint16			flags;
	float			percentage;
	int16			glossiness;
	int16			reflectMode;
	char*			name;
	float			miscFloat;
	int16			misc16;
	int32			misc32;
	pPoint3			vec;

	uint32			currLen = ParsePadName(name);
	*traceOutput << "surface name: '" << name << '\'' << eol;

	int32			matIndex = scene->materials.Search(name);
	if (matIndex < 0)
	{
		SkipBytes(len-currLen);
		return ;
	}
	pMaterial*		material = scene->materials[matIndex];
	material->ambient.Set(0.2, 0.2, 0.2);
	delete[] name;

	while (currLen < len)
	{
		ReadSubchunk;

		if (chunkId == SURF_COLOR)
		{
//			ParseColor(color);
			ParseColor(material->diffuse);
			*traceOutput << "color: ";
			Output(*traceOutput, color);
			*traceOutput << eol;
		}

		else if (chunkId == SURF_FLAG)
		{
			ReadInt16(flags);
			*traceOutput << "flags: " << flags << eol;
		}

		else if (chunkId == SURF_LUMI)
		{
			ParseIpercentage(percentage);
			*traceOutput << "lumi: " << percentage << eol;
		}
		else if (chunkId == SURF_DIFF)
		{
			ParseIpercentage(percentage);
			*traceOutput << "diff: " << percentage << eol;
		}
		else if (chunkId == SURF_SPEC)
		{
			ParseIpercentage(percentage);
			material->specular.Set(percentage, percentage, percentage);
			*traceOutput << "spec: " << percentage << eol;
			chunkLen = 2;			// bug in some files
		}
		else if (chunkId == SURF_REFL)
		{
			ParseIpercentage(percentage);
			*traceOutput << "refl: " << percentage << eol;
			chunkLen = 2;			// bug in some files
		}
		else if (chunkId == SURF_TRAN)
		{
			ParseIpercentage(percentage);
			material->opacity = 1.0 - percentage;
			*traceOutput << "tran: " << percentage << eol;
		}

		else if (chunkId == SURF_VLUM)
		{
			ReadFloat(percentage);
			*traceOutput << "vlum: " << percentage << eol;
		}
		else if (chunkId == SURF_VDIF)
		{
			ReadFloat(percentage);
			*traceOutput << "vdif: " << percentage << eol;
		}
		else if (chunkId == SURF_VSPC)
		{
			ReadFloat(percentage);
			*traceOutput << "vspc: " << percentage << eol;
		}
		else if (chunkId == SURF_VRFL)
		{
			ReadFloat(percentage);
			material->specular.Set(percentage, percentage, percentage);
			*traceOutput << "vrfl: " << percentage << eol;
		}
		else if (chunkId == SURF_VTRN)
		{
			ReadFloat(percentage);
			material->opacity = 1.0 - percentage;
			*traceOutput << "vtrn: " << percentage << eol;
		}

		else if (chunkId == SURF_GLOS)
		{
			ReadInt16(glossiness);
			*traceOutput << "glosiness: " << glossiness << eol;
			// float(glossiness)/8.0;
			chunkLen = 2;			// bug in some files
		}

		else if (chunkId == SURF_RFLT)
		{
			ReadInt16(reflectMode);
			*traceOutput << "reflectMode: " << reflectMode << eol;
		}

		else if (chunkId == SURF_RIMG)
		{
			ParsePadName(name);
			*traceOutput << "reflect image: '" << name << '\'' << eol;
			delete[] name;
		}

		else if (chunkId == SURF_RSAN)
		{
			ReadFloat(miscFloat);
			*traceOutput << "seam angle: " << miscFloat << eol;
		}

		else if (chunkId == SURF_RIND)
		{
			ReadFloat(miscFloat);
			*traceOutput << "refractive index: " << miscFloat << eol;
		}

		else if (chunkId == SURF_EDGE)
		{
			ReadFloat(miscFloat);
			*traceOutput << "edge transparency threshold: " << miscFloat << eol;
		}

		else if (chunkId == SURF_SMAN)
		{
			ReadFloat(miscFloat);
			*traceOutput << "max smoothing angle: " << miscFloat << eol;
		}

		// texture definition chunks
		else if (chunkId == SURF_CTEX)
		{
			ParsePadName(name);
			*traceOutput << "ctex: '" << name << '\'' << eol;
			delete[] name;
		}
		else if (chunkId == SURF_DTEX)
		{
			ParsePadName(name);
			*traceOutput << "dtex: '" << name << '\'' << eol;
			delete[] name;
		}
		else if (chunkId == SURF_STEX)
		{
			ParsePadName(name);
			*traceOutput << "stex: '" << name << '\'' << eol;
			delete[] name;
		}
		else if (chunkId == SURF_RTEX)
		{
			ParsePadName(name);
			*traceOutput << "rtex: '" << name << '\'' << eol;
			delete[] name;
		}
		else if (chunkId == SURF_TTEX)
		{
			ParsePadName(name);
			*traceOutput << "ttex: '" << name << '\'' << eol;
			delete[] name;
		}
		else if (chunkId == SURF_LTEX)
		{
			ParsePadName(name);
			*traceOutput << "ltex: '" << name << '\'' << eol;
			delete[] name;
		}
		else if (chunkId == SURF_BTEX)
		{
			ParsePadName(name);
			*traceOutput << "btex: '" << name << '\'' << eol;
			delete[] name;
		}
		else if (chunkId == SURF_TFLG)
		{
			ReadInt16(flags);
			*traceOutput << "texture flags: " << flags << eol;
		}

		else if (chunkId == SURF_TSIZ)
		{
			ParsePoint(vec);
			*traceOutput << "texture size: "; Output(*traceOutput, vec);
			*traceOutput << eol;
		}
		else if (chunkId == SURF_TCTR)
		{
			ParsePoint(vec);
			*traceOutput << "texture center: "; Output(*traceOutput, vec);
			*traceOutput << eol;
		}
		else if (chunkId == SURF_TFAL)
		{
			ParsePoint(vec);
			*traceOutput << "texture falloff: "; Output(*traceOutput, vec);
			*traceOutput << eol;
		}
		else if (chunkId == SURF_TVEL)
		{
			ParsePoint(vec);
			*traceOutput << "texture velocity: "; Output(*traceOutput, vec);
			*traceOutput << eol;
		}

		else if (chunkId == SURF_TCLR)
		{
			ParseColor(color);
			*traceOutput << "texture color: "; Output(*traceOutput, color);
			*traceOutput << eol;
		}
		else if (chunkId == SURF_TVAL)
		{
			ParseIpercentage(percentage);
			*traceOutput << "texture value: " << percentage << eol;
		}
		else if (chunkId == SURF_TAMP)
		{
			ReadFloat(miscFloat);
			*traceOutput << "texture amplitude: " << miscFloat << eol;
		}

		else if ((chunkId & 0xffffff00) == SURF_TFPi)
		{
			ReadFloat(miscFloat);
			*traceOutput << "tfp " << int32(chunkId & 0xf) << ": " << miscFloat << eol;
		}

		else if ((chunkId & 0xffffff00) == SURF_TIPi)
		{
			ReadInt16(misc16);
			*traceOutput << "tip " << int32(chunkId & 0xf) << ": " << misc16 << eol;
		}

		else if ((chunkId & 0xffffff00) == SURF_TSPi)
		{
			ReadFloat(miscFloat);
			*traceOutput << "tsp " << int32(chunkId & 0xf) << ": " << miscFloat << eol;
		}

		else if ((chunkId & 0xffffff00) == SURF_TFRQ)
		{
			ReadInt16(misc16);
			*traceOutput << "freq: " << misc16 << eol;
		}

		else if (chunkId == SURF_TIMG)
		{
			ParsePadName(name);
			*traceOutput << "timg: '" << name << '\'' << eol;
			delete[] name;
		}

		else if (chunkId == SURF_TALP)
		{
			ParsePadName(name);
			*traceOutput << "talp: '" << name << '\'' << eol;
			delete[] name;
		}

		else if (chunkId == SURF_TWRP)
		{
			ReadInt16(misc16);
			*traceOutput << "width wrap: " << misc16 << ", height wrap: ";
			ReadInt16(misc16);
			*traceOutput << misc16 << eol;
		}

		else if (chunkId == SURF_TAAS)
		{
			ReadFloat(miscFloat);
			*traceOutput << "antialiasing strength: " << miscFloat << eol;
		}

		else if (chunkId == SURF_SHDR)
		{
			ParsePadName(name);
			*traceOutput << "shader: '" << name << '\'' << eol;
			delete[] name;
		}

		else if (chunkId == SURF_SDAT)
		{
			*traceOutput << "shader data: " << chunkLen << " bytes" << eol;
			SkipBytes(chunkLen);
		}

		else if (chunkId == SURF_IMSQ)
		{
			ReadInt16(misc16);
			*traceOutput << "seq options, offset: " << misc16 << ", flags: ";
			ReadInt16(misc16);
			*traceOutput << misc16 << ", loop length: ";
			ReadInt16(misc16);
			*traceOutput << misc16 << eol;
		}

		else if (chunkId == SURF_FLYR)
		{
			ReadInt32(misc32);
			*traceOutput << "flyer clip, begin: " << misc32 << ", end: ";
			ReadInt32(misc32);
			*traceOutput << misc32 << eol;
		}

		else if (chunkId == SURF_IMCC)
		{
			ReadInt16(misc16);
			*traceOutput << "color cycle options, sycle speed: " << misc16
						<< ", low index: ";
			ReadInt16(misc16);
			*traceOutput << misc16 << ", high index: ";
			ReadInt16(misc16);
			*traceOutput << misc16 << eol;
		}

		else
		{
			skipChunk;
		}
		currLen += chunkLen+6;
		PadChunk;
	}

	*traceOutput << ret;
}

