//
//
//	LAP text parser class
//
//										(C) JoOl 1998

#include "LAPimpexp.h"

#include "LAPtextParser.h"
#include "LAPerrors.h"
#include "LAPparserException.h"


LAPtextParser::LAPtextParser()
{
	lexer = NULL;
}


LAPtextParser::~LAPtextParser()
{
}


void
LAPtextParser::SetLexer(LAPtextLexer* l)
{
	LAPparser::SetLexer(l);
	lexer = l;
	lexer->SetCharCodes();
	lexer->SetLexemes();
}


status_t
LAPtextParser::SetInput(entry_ref ref, uint8 flags)
{
	if (!lexer)
		return LAP_BAD_INIT;
	if (!display)
		return LAP_BAD_INIT;
	if (!errorOutput)
		return LAP_BAD_INIT;

	abortRequested = false;
	peekFlag = false;

	entry_ref			tref = ref;
	BEntry				e(&tref, true);
	status_t			s = lexer->SetInput(tref);
	if (s != LAP_NO_ERROR)
		return s;

	if (Identify() != LAP_NO_ERROR)
		return LAP_BAD_INPUT_TYPE;

	peekFlag = false;
	s = lexer->SetInput(tref);
	if (s != LAP_NO_ERROR)
		return s;
	if ((flags & LAP_TRACE) && (traceOutput))
		SetTraceFile(ref, B_WRITE_ONLY | B_CREATE_FILE | B_ERASE_FILE);
	if (flags & LAP_ERRORS_TRACE)
		SetMessagesTraceFile(ref, B_WRITE_ONLY | B_CREATE_FILE | B_ERASE_FILE);
	return LAP_NO_ERROR;
}


void
LAPtextParser::NextToken()
{
	if (peekFlag)
	{
		peekFlag = false;
		return ;
	}
	currentToken = lexer->NextToken();
}


void
LAPtextParser::Match(LAPtoken t)
{
	NextToken();
	if (currentToken == t)
		return ;

	// generate error
	GenerateMismatchToken("LAPtextParser::Match", t, lexer->GetLexeme(t), currentToken, NULL);
}


void
LAPtextParser::MatchNumber(LAPliteralType ex)
{
	Match(LAP_TOKEN_NUMBER);
	LAPliteralType			ac = lexer->GetLiteralType();
	// ex = expected literal
	// ac = actual literal
	if (ex == ac)
		return ;

	GenerateMismatchLiteral("LAPtextParser::MatchNumber", ex, currentToken);
}


void
LAPtextParser::GenerateMismatchLiteral(const char* Location, LAPliteralType ExpectedType,
									LAPtoken ActualToken)
{
	int32					ExpectedToken;
	const char*				ExpectedLexeme = NULL;
	const char*				ActualLexeme = lexer->GetLexeme(ActualToken);
	if (ExpectedType == LAP_CHARACTER_LITERAL)
	{
		ExpectedLexeme = "<char>";
		ExpectedToken = LAP_CHARACTER_LITERAL;
	}
	else if (ExpectedType == LAP_INTEGER_LITERAL)
	{
		ExpectedLexeme = "<integer>";
		ExpectedToken = LAP_INTEGER_LITERAL;
	}
	else if (ExpectedType == LAP_FLOAT_LITERAL)
	{
		ExpectedLexeme = "<float>";
		ExpectedToken = LAP_FLOAT_LITERAL;
	}
	else if (ExpectedType == LAP_STRING_LITERAL)
	{
		ExpectedLexeme = "<string>";
		ExpectedToken = LAP_STRING_LITERAL;
	}
	else
	{
		ExpectedLexeme = "<no literal>";
		ExpectedToken = LAP_NO_LITERAL;
	}

	int32				ActualType = lexer->GetLiteralType();
	if (ActualType == LAP_CHARACTER_LITERAL)
	{
		throw LAPmismatchToken(Location, ExpectedToken, ExpectedLexeme,
									ActualToken, ActualLexeme, lexer->GetCharLiteral());
	}
	else if (ActualType == LAP_INTEGER_LITERAL)
	{
		int64		i = lexer->GetInt64Literal();
		throw LAPmismatchToken(Location, ExpectedToken, ExpectedLexeme,
									ActualToken, ActualLexeme, i);
	}
	else if (ActualType == LAP_FLOAT_LITERAL)
	{
		double		d = lexer->GetDoubleLiteral();
		throw LAPmismatchToken(Location, ExpectedToken, ExpectedLexeme,
									ActualToken, ActualLexeme, d);
	}
	else if (ActualType == LAP_STRING_LITERAL)
	{
		throw LAPmismatchToken(Location, ExpectedToken, ExpectedLexeme,
									ActualToken, ActualLexeme, lexer->GetStringLiteral());
	}
	else
	{
		throw LAPmismatchToken(Location, ExpectedToken, ExpectedLexeme,
									ActualToken, ActualLexeme);
	}
}



void
LAPtextParser::GenerateMismatchToken(const char* Location ,
									LAPtoken ExpectedToken, const char* ExpectedLexeme,
									LAPtoken ActualToken, const char* ActualLexeme)
{
	if (!ExpectedLexeme)
		ExpectedLexeme = lexer->GetLexeme(ExpectedToken);
	if (!ActualLexeme)
		ActualLexeme = lexer->GetLexeme(ActualToken);

	if (ActualToken == LAP_TOKEN_NUMBER)
	{
		if (lexer->GetLiteralType() == LAP_INTEGER_LITERAL)
		{
			int64			Val = lexer->GetInt64Literal();
			throw LAPmismatchToken(Location,
										ExpectedToken, ExpectedLexeme,
										ActualToken, ActualLexeme, Val);
		}
		else if (lexer->GetLiteralType() == LAP_FLOAT_LITERAL)
		{
			double			Val = lexer->GetDoubleLiteral();
			throw LAPmismatchToken(Location,
										ExpectedToken, ExpectedLexeme,
										ActualToken, ActualLexeme, Val);
		}
		else	// if (lexer->GetLiteralType() == LAP_STRING_LITERAL)
		{
			char*			ValStr = lexer->GetStringLiteral();
			throw LAPmismatchToken(Location,
										ExpectedToken, ExpectedLexeme,
										ActualToken, ActualLexeme, ValStr);
		}
	}
	else if (currentToken == LAP_TOKEN_IDENTIFIER)
	{
		char*			ValStr = lexer->GetStringLiteral();
		throw LAPmismatchToken(Location,
									ExpectedToken, ExpectedLexeme,
									ActualToken, ActualLexeme, ValStr);
	}
	else if (currentToken == LAP_TOKEN_ILLEGAL)
	{
		char			Val = lexer->GetCharLiteral();
		throw LAPmismatchToken(Location,
									ExpectedToken, ExpectedLexeme,
									ActualToken, ActualLexeme, Val);
	}
	else			// lexeme
	{
		throw LAPmismatchToken(Location,
									ExpectedToken, ExpectedLexeme,
									ActualToken, ActualLexeme);
	}
}


bool
LAPtextParser::Synchronize(LAPtokenSet& ts)
{
	while (!Eof()
			&& !ts.Contains(currentToken))
		NextToken();

	return Eof();
}


bool
LAPtextParser::Synchronize(LAPtokenSet& ts1, LAPtokenSet& ts2)
{
	while (!Eof()
			&& !ts1.Contains(currentToken)
			&& !ts2.Contains(currentToken))
		NextToken();

	return Eof();
}


bool
LAPtextParser::Synchronize(LAPtokenSet& ts1, LAPtokenSet& ts2, LAPtokenSet& ts3)
{
	while (!Eof()
			&& !ts1.Contains(currentToken)
			&& !ts2.Contains(currentToken)
			&& !ts3.Contains(currentToken))
		NextToken();

	return Eof();
}


bool
LAPtextParser::Peek(LAPtoken t)
{
	if (!peekFlag)
	{
		peekFlag = true;
		currentToken = lexer->NextToken();
	}
	return (currentToken == t);
}


void
LAPtextParser::SkipSpaces()
{
	lexer->SkipSpaces();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::SkipSpaces");
}


void
LAPtextParser::LexInt(int32& Out)
{
	MatchNumber(LAP_INTEGER_LITERAL);
	Out = lexer->GetInt32Literal();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::LexInt");
}


void
LAPtextParser::LexInt(int64& Out)
{
	MatchNumber(LAP_INTEGER_LITERAL);
	Out = lexer->GetInt64Literal();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::LexInt");
}


void
LAPtextParser::ParseInt(int32& Out)
{
	Match(LAP_TOKEN_NUMBER);
	LAPliteralType			ac = lexer->GetLiteralType();
	if (ac == LAP_FLOAT_LITERAL)
		Out = int32(lexer->GetDoubleLiteral());
	else
		Out = lexer->GetInt32Literal();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::ParseInt");
}


void
LAPtextParser::ParseInt(int64& Out)
{
	Match(LAP_TOKEN_NUMBER);
	LAPliteralType			ac = lexer->GetLiteralType();
	if (ac == LAP_FLOAT_LITERAL)
		Out = int64(lexer->GetDoubleLiteral());
	else
		Out = lexer->GetInt64Literal();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::ParseInt");
}


void
LAPtextParser::LexFloat(float& Out)
{
	MatchNumber(LAP_FLOAT_LITERAL);
	Out = lexer->GetFloatLiteral();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::LexFloat");
}


void
LAPtextParser::LexFloat(double& Out)
{
	MatchNumber(LAP_FLOAT_LITERAL);
	Out = lexer->GetDoubleLiteral();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::LexFloat");
}


void
LAPtextParser::ParseFloat(float& Out)
{
	Match(LAP_TOKEN_NUMBER);
	LAPliteralType			ac = lexer->GetLiteralType();
	if (ac == LAP_INTEGER_LITERAL)
		Out = float(lexer->GetInt64Literal());
	else
		Out = lexer->GetFloatLiteral();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::ParseFloat");
}


void
LAPtextParser::ParseFloat(double& Out)
{
	Match(LAP_TOKEN_NUMBER);
	LAPliteralType			ac = lexer->GetLiteralType();
	if (ac == LAP_INTEGER_LITERAL)
		Out = double(lexer->GetInt64Literal());
	else
		Out = lexer->GetDoubleLiteral();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::ParseFloat");
}


char*
LAPtextParser::LexWord()
{
	lexer->LexWord();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::LexWord");

	return lexer->DetachStringLiteral();
}


char*
LAPtextParser::LexIdentifier()
{
	Match(LAP_TOKEN_IDENTIFIER);
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::LexIdentifier");

	return lexer->DetachStringLiteral();
}


void
LAPtextParser::SkipLine()
{
	lexer->SkipLine();
	peekFlag = false;

	if ((abortRequested) && (Locker.CountLocks() == 0))
		throw LAPloadAborted("LAPtextParser::SkipLine");
}
