//
//
//	Amachi OpenGL view
//
//


#include "AmachiGLView.h"
#include "AmachiDefs.h"
#include "SceneItem.h"


int32
AmachiGLView::renderer(void* o)
{
	AmachiGLView*		v = (AmachiGLView* )o;

    while (acquire_sem_etc(v->quitSem, 1, B_TIMEOUT, 0) == B_NO_ERROR)
    {
		int32			semCount;

		if (acquire_sem(v->renderSem) == B_NO_ERROR)
		{
			if (!v->exitRenderThread)
			{
				v->Render();
			}
			get_sem_count(v->renderSem, &semCount);
			acquire_sem_etc(v->renderSem, semCount, 0, 0);

			if (!v->exitRenderThread)
			{
				if (!v->manualMotion)
					v->Spin();
				if (v->NeedsUpdate())
					release_sem(v->renderSem);
			}
		}

		release_sem(v->quitSem);
    }

	return B_NO_ERROR;
}



AmachiGLView::AmachiGLView(BRect frame, char* name, uint32 resize_mode,
					uint32 flags, uint32 options)
:BGLView(frame, name, resize_mode, flags, options),
 manualMotion(true),
 scene(NULL),
 current(NULL),
 configView(NULL)
{
	glGetIntegerv(GL_MAX_LIGHTS, &maxLightNumber);

	black.Set(0.0, 0.0, 0.0);
	bgColor = &black;

	hwRatio = frame.Height() / frame.Width();
	depthOfView = 30.0;

	float			tmp;
	tmp = depthOfView * 0.002;
	tStep.Set(tmp, tmp, tmp);
	tmp = M_PI / 32.0;
	rStep.Set(tmp, tmp, tmp);

	viewsys.SetDefault();
	oldViewsys.SetDefault();

//	current.SetDefault();
	previous.SetModes(-1);

	renderSem = create_sem(0, "render sem");
	quitSem = create_sem(1, "quit sem");

	EnableDirectMode(true);
}


AmachiGLView::~AmachiGLView()
{
	delete_sem(renderSem);
	delete_sem(quitSem);
}


void
AmachiGLView::SetTo(pScene*			new_scene,
					GLconfig*		new_config,
					AmachiGLConfig*	new_configView)
{
	if (scene)
	{
		// save viewsys
		scene->viewsys = viewsys;
	}

	scene = new_scene;
	current = new_config;
	EnforceGLstate();
	configView = new_configView;

	// set transparency item
	if (configView->Window()->Lock())
	{
		configView->SetEnabled(scene != NULL);
		configView->ReadFrom(*current);
		configView->SetTarget(this);
		if (scene)
			configView->EnableTransparency(scene->HasTransparentFaces());
		else
			configView->EnableTransparency(false);
		configView->Window()->Unlock();
	}

	if (scene)
	{
		viewsys = scene->viewsys;
		bgColor = &scene->background;
		DeclareLights();
	}

	// render
	release_sem(renderSem);
}


bool
AmachiGLView::NeedsUpdate()
{
	return EnforceGLstate() || (viewsys != oldViewsys);
}


void
AmachiGLView::SetProjection(float scale, int16 projType)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	if (projType == AMACHI_PERSPECTIVE_PROJECTION)
		gluPerspective(60.0, 1.0 / hwRatio, 0.15, 120.0);
	else
	{
		if (hwRatio < 1.0)
			glOrtho(-scale / hwRatio, scale / hwRatio, -scale, scale,
					-1.0, depthOfView * 4.0);
		else
			glOrtho(-scale, scale, -scale * hwRatio, scale * hwRatio,
					-1.0, depthOfView * 4.0);
	}
}

/*
void
AmachiGLView::ReadConfig()
{
	configView->ReadProjection(*current);
	configView->ReadShading(*current);
	configView->ReadTransparency(*current);
	configView->ReadZbuffer(*current);
	configView->ReadLighting(*current);
	configView->ReadFrontFace(*current);
	configView->ReadBackFace(*current);
	configView->ReadCulling(*current);
	configView->ReadFullscreen(*current);
	configView->ReadMotion(*current);
}


void
AmachiGLView::WriteConfig()
{
	configView->WriteProjection(*current);
	configView->WriteShading(*current);
	configView->WriteTransparency(*current);
	configView->WriteZbuffer(*current);
	configView->WriteLighting(*current);
	configView->WriteFrontFace(*current);
	configView->WriteBackFace(*current);
	configView->WriteCulling(*current);
	configView->WriteFullscreen(*current);
	configView->WriteMotion(*current);
}
*/

bool
AmachiGLView::EnforceGLstate()
{
	if (!current)
		return false;

	bool				changed = false;

	if (previous.GetProjection() != current->GetProjection())
	{
		SetProjection(1.0, current->GetProjection());
		changed = true;
	}

	if (previous.GetShading() != current->GetShading())
	{
		if (current->GetShading() == AMACHI_FLAT_SHADING)
			glShadeModel(GL_FLAT);
		else
			glShadeModel(GL_SMOOTH);
		changed = true;
	}

	if (previous.GetZbuffer() != current->GetZbuffer())
	{
		if (current->GetZbuffer() == AMACHI_NO_ZBUFFER)
			glDisable(GL_DEPTH_TEST);
		else
		{
			glEnable(GL_DEPTH_TEST);
			glDepthFunc(GL_LESS);
		}
		changed = true;
	}

	if (previous.GetFrontFace() != current->GetFrontFace())
	{
		if (current->GetFrontFace() == AMACHI_FILLED_RENDERING)
			glPolygonMode(GL_FRONT, GL_FILL);
		else
			glPolygonMode(GL_FRONT, GL_LINE);
		changed = true;
	}

	if (previous.GetBackFace() != current->GetBackFace())
	{
		if (current->GetBackFace() == AMACHI_FILLED_RENDERING)
			glPolygonMode(GL_BACK, GL_FILL);
		else
			glPolygonMode(GL_BACK, GL_LINE);
		changed = true;
	}

	if (previous.GetCulling() != current->GetCulling())
	{
		if (current->GetCulling() == AMACHI_NO_CULLING)
			glDisable(GL_CULL_FACE);
		else
		{
			glEnable(GL_CULL_FACE);
			glFrontFace(GL_CCW);
			if (current->GetCulling() == AMACHI_CW_CULLING)
				glCullFace(GL_BACK);
			else
				glCullFace(GL_FRONT);
		}
		changed = true;
	}

	if (previous.GetLighting() != current->GetLighting())
	{
		if (!(current->GetLighting() & AMACHI_USE_LIGHTING))
			glDisable(GL_LIGHTING);
		else
		{
			if (current->GetLighting() & AMACHI_LOCAL_VIEWER_LIGHTING)
				glLightModelf(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE);
			else
				glLightModelf(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);

			if (current->GetLighting() & AMACHI_TWO_SIDES_LIGHTING)
				glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
			else
				glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);

			if (scene)
			{
				int32			m = maxLightNumber;
				if (scene)
					m = min_c(scene->lights.CountItems(), maxLightNumber);
				for (int32 i = 0; i < m; i++)
					glEnable(GL_LIGHT0 + i);
			}
			glEnable(GL_LIGHTING);
		}
		changed = true;
	}

	if (previous.GetFullscreen() != current->GetFullscreen())
	{
		BDirectWindow*		win = (BDirectWindow* )Window();
		if ((current->GetFullscreen() == AMACHI_USE_FULLSCREEN)
			&& (!win->IsFullScreen()))
			win->SetFullScreen(true);
		else
			win->SetFullScreen(false);
		changed = true;
	}

	if (previous.GetMotion() != current->GetMotion())
	{
		manualMotion = (current->GetMotion() == AMACHI_MANUAL_MOTION);
		changed = true;
	}

	previous = *current;

	return changed;
}


void
AmachiGLView::DeclareLights()
{
	LockGL();

		// disable all lights
		for (int32 i = 0; i < maxLightNumber; i++)
			glDisable(GL_LIGHT0 + i);

		// set ambient light
		glLightModelfv(GL_LIGHT_MODEL_AMBIENT, (float* )&scene->ambient);

//		if (scene->lights->CountItems() > maxLightNumber)
//			issue warning;
		for (int32 i = 0; i < min_c(maxLightNumber, scene->lights.CountItems()); i++)
		{
			pLight*			currLight = scene->lights[i];
			float			pos[4];
			if (currLight->IsDirectional() && !currLight->IsPositional())
			{
				pDirectionalLight*		dl = (pDirectionalLight* )currLight;
				pos[0] = dl->direction.x;
				pos[1] = dl->direction.y;
				pos[2] = dl->direction.z;
				pos[3] = 0.0;
			}
			else if (!currLight->IsDirectional() && currLight->IsPositional())
			{
				pPositionalLight*		pl = (pPositionalLight* )currLight;
				pos[0] = pl->position.x;
				pos[1] = pl->position.y;
				pos[2] = pl->position.z;
				pos[3] = 1.0;
			}
			glLightfv(GL_LIGHT0 + i, GL_POSITION, pos);
			glLightfv(GL_LIGHT0 + i, GL_SPECULAR, (float* )&currLight->specular);
			glLightfv(GL_LIGHT0 + i, GL_DIFFUSE, (float* )&currLight->diffuse);
			glLightfv(GL_LIGHT0 + i, GL_AMBIENT, (float* )&currLight->ambient);
			glLightf(GL_LIGHT0 + i, GL_CONSTANT_ATTENUATION, currLight->constantAttenuation);
			glLightf(GL_LIGHT0 + i, GL_LINEAR_ATTENUATION, currLight->linearAttenuation);
			glLightf(GL_LIGHT0 + i, GL_QUADRATIC_ATTENUATION, currLight->quadraticAttenuation);

			if (currLight->IsDirectional() && currLight->IsPositional())
			{
				pSpotlight*		sl = (pSpotlight* )currLight;
				glLightfv(GL_LIGHT0 + i, GL_SPOT_DIRECTION, (float* )&sl->spotDirection);
				glLightf(GL_LIGHT0 + i, GL_SPOT_EXPONENT, sl->spotExponent);
				glLightf(GL_LIGHT0 + i, GL_SPOT_CUTOFF, sl->spotCutOff);
			}

//			if (currLight->IsOn())
				glEnable(GL_LIGHT0 + i);
		}

	UnlockGL();
}


void
AmachiGLView::AttachedToWindow()
{
	BRect			bounds = Bounds();
	BGLView::AttachedToWindow();

	LockGL();
		glViewport(0.0, 0.0, bounds.IntegerWidth() + 1.0, bounds.IntegerHeight() + 1.0);
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		glEnableClientState(GL_NORMAL_ARRAY);
		glEnableClientState(GL_VERTEX_ARRAY);

		EnforceGLstate();
		DoClear();
		SwapBuffers();

	UnlockGL();

	// launch rendering thread
	exitRenderThread = false;
	rendererThread = spawn_thread(renderer, "renderer", B_NORMAL_PRIORITY,
								(void* )this);
	resume_thread(rendererThread);
}


void
AmachiGLView::DetachedFromWindow()
{
	BGLView::DetachedFromWindow();

	// the rendering thread already owns quitSem, awaiting a renderSem event
	exitRenderThread = true;

	// ask to render, so that the thread checks the "exitRenderThread" flag
	release_sem(renderSem);

	// prevent the thread from acquiring quitSem
	acquire_sem(quitSem);

	// wait for the thread to quit
	int32				dummy;
	wait_for_thread(rendererThread, &dummy);

	// release quitSem so that it can be deleted
	release_sem(quitSem);
}



void
AmachiGLView::WindowActivated(bool active)
{
	MakeFocus(active);
}


void
AmachiGLView::FrameResized(float w, float h)
{
	LockGL();

		BGLView::FrameResized(w,h);

		BRect				b = Bounds();
		w = b.Width();
		h = b.Height();
		hwRatio = h / w;
		glViewport(0.0, 0.0, (GLint )(w + 1.0), (GLint )(h + 1.0));

		DoClear();

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		SetProjection(1.0,  (current) 	? current->GetProjection()
										: AMACHI_PERSPECTIVE_PROJECTION);
	UnlockGL();

	release_sem(renderSem);
}


void
AmachiGLView::Draw(BRect updateRect)
{
//	Render();
	release_sem(renderSem);
}


void
AmachiGLView::KeyDown(const char* bytes, int32 len)
{
	if (*bytes == '4')
		viewsys.RotateV(-rStep.y);
	else if (*bytes == '6')
		viewsys.RotateV(rStep.y);

	else if (*bytes == '8')
		viewsys.RotateU(-rStep.x);
	else if (*bytes == '2')
		viewsys.RotateU(rStep.x);

	else if (*bytes == '1')
		viewsys.RotateN(-rStep.z);
	else if (*bytes == '3')
		viewsys.RotateN(rStep.z);

	else if (*bytes == '7')
		viewsys.Zoom(tStep.z);
	else if (*bytes == '9')
		viewsys.Zoom(-tStep.z);

	else if (*bytes == B_ESCAPE)
	{
		if (!scene)
			viewsys.SetDefault();
		else
			viewsys.SetTo(scene->sbounds.radius * 1.5, scene->sbounds.center);
	}

	else if (*bytes == B_TAB)
	{
		if (configView->Window()->Lock())
		{
			if (current->GetFullscreen() == AMACHI_USE_FULLSCREEN)
				current->SetFullscreen(AMACHI_NO_FULLSCREEN);
			else
				current->SetFullscreen(AMACHI_USE_FULLSCREEN);
			configView->WriteFullscreen(*current);
			configView->Window()->Unlock();

			BMessage		msg(AMACHI_SET_GLPARAM);
			msg.AddInt32("mode", AMACHI_FULLSCREEN);
			Window()->PostMessage(&msg, this);
		}
	}

	else
		return ;

	if (*bytes != B_TAB)
		release_sem(renderSem);
}


void
AmachiGLView::DoClear()
{
	glClearColor(0.0, 0.0, 0.0, 1.0);
	glClearDepth(1.0);

	if (current)
		glClear(GL_COLOR_BUFFER_BIT
				| ((current->GetZbuffer() == AMACHI_USE_ZBUFFER) ? GL_DEPTH_BUFFER_BIT: 0));
	else
		glClear(GL_COLOR_BUFFER_BIT);
}


void
AmachiGLView::Clear()
{
	LockGL();
		DoClear();
		SwapBuffers();
	UnlockGL();
}


void
AmachiGLView::Render()
{
	if (exitRenderThread)
		return ;

	LockGL();

		oldViewsys = viewsys;
		if (!exitRenderThread)
			DoClear();
		if (!exitRenderThread)
			DoRender();
		if (!exitRenderThread)
			SwapBuffers();

	UnlockGL();
}


void
AmachiGLView::DoRender()
{
	if (!scene)
		return ;
	if (!scene->Lock())
		return ;

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();

//		glLoadIdentity();
		viewsys.BuildMatrix();
		glLoadMatrixf(viewsys.GetMatrix());

	if (current->GetFrontFace() == AMACHI_DOTS_RENDERING)
	{
		for (int32 i = 0; i < scene->objects.CountItems(); i++)
			DoRenderObjectDots(scene->objects[i]);
	}
	else
	{
		if (current->GetTransparency() == AMACHI_NO_TRANSPARENCY)
		{
			for (int32 i = 0; i < scene->objects.CountItems(); i++)
				DoRenderObject(scene->objects[i], AMACHI_RENDER_ALL_FACES_AS_OPAQUE);
		}
		else
		{
			for (int32 i = 0; i < scene->objects.CountItems(); i++)
				DoRenderObject(scene->objects[i], AMACHI_RENDER_ONLY_OPAQUE_FACES);
			for (int32 i = 0; i < scene->objects.CountItems(); i++)
				DoRenderObject(scene->objects[i], AMACHI_RENDER_ONLY_TRANSPARENT_FACES);
		}
	}

	glPopMatrix();

	scene->Unlock();
}


void
AmachiGLView::DoRenderObjectDots(pObject* object)
{
	glPushMatrix();
// multiply by "matrix" (useless ?)
//		glMultMatrixf((float* )&object->matrix);

		pPointsArray&			vrt = *object->vertices;
		pPointsArray&			dotsData = object->dlVertices;
		glColor4ub(255, 255, 255, 0);
		glBegin(GL_POINTS);

			for (int32 i = 0; i < object->dots.CountItems(); i++)
				glVertex3fv((float* )&dotsData[object->dots[i]]);
			for (int32 i = 0; i < vrt.CountItems(); i++)
				glVertex3fv((float* )&vrt[i]);
		glEnd();
	glPopMatrix();
}


void
AmachiGLView::DoRenderObject(pObject* object, uint8 rflags)
{
	pMaterial*			currentMaterial = NULL;
	float				alpha;
	int16				saveCullingMode;

	glPushMatrix();
// multiply by "matrix" (useless ?)
//		glMultMatrixf((float* )&object->matrix);

		// render dots
		pPointsArray&						dotsData = object->dlVertices;
		glBegin(GL_POINTS);
			for (int32 i = 0; i < object->dots.CountItems(); i++)
				glVertex3fv((float* )&dotsData[object->dots[i]]);
		glEnd();

		// render lines
		glBegin(GL_LINES);
			for (int32 i = 0; i < object->lines.CountItems(); i++)
			{
				glVertex3fv((float* )&dotsData[object->lines[i].start]);
				glVertex3fv((float* )&dotsData[object->lines[i].end]);
			}
		glEnd();

		glNormalPointer(GL_FLOAT, 0, (float* )object->normals->Items());
		glVertexPointer(3, GL_FLOAT, 0, (float* )object->vertices->Items());

		// render faces
		for (int32 i = 0; i < object->faces.CountItems(); i++)
		{
			pFace*				face = object->faces[i];
			pColor				color;

			// render opaque faces ?
			if ((face->material->opacity >= 1.0) && (rflags == AMACHI_RENDER_ONLY_TRANSPARENT_FACES))
				continue;
			// render transparent faces ?
			if (face->material->opacity < 1.0)
			{
				if (rflags == AMACHI_RENDER_ONLY_OPAQUE_FACES)
					continue ;
				if (rflags == AMACHI_RENDER_ONLY_TRANSPARENT_FACES)
				{
					glBlendFunc(GL_SRC_ALPHA, GL_ONE);
					glEnable(GL_BLEND);
					glDepthMask(GL_FALSE);
					glDisable(GL_CULL_FACE);
					saveCullingMode = current->GetCulling();
					current->SetCulling(AMACHI_NO_CULLING);
				}
			}

			if (face->material != currentMaterial)
			{
				currentMaterial = face->material;
				glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, (float* )&currentMaterial->ambient);
				glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, (float* )&currentMaterial->diffuse);
				glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, (float* )&currentMaterial->specular);
				glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, (float* )&currentMaterial->emission);
				glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, currentMaterial->shininess);
				color = face->GetColor();
				glColor3ub(color.r, color.g, color.b);
			}

			if (face->IsTriangle())
			{
				glBegin(GL_TRIANGLES);
					glArrayElement(face->index[0]);
					glArrayElement(face->index[1]);
					glArrayElement(face->index[2]);
				glEnd();
			}
			else if (face->IsQuad())
			{
				glBegin(GL_QUADS);
					glArrayElement(face->index[0]);
					glArrayElement(face->index[1]);
					glArrayElement(face->index[2]);
					glArrayElement(face->index[3]);
				glEnd();
			}
			else
			{
				glBegin(GL_POLYGON);
					for (int32 j = 0; j < face->vcount; j++)
						glArrayElement(face->index[j]);
				glEnd();
			}
			if ((face->material->opacity < 1.0)
				&& (rflags == AMACHI_RENDER_ONLY_TRANSPARENT_FACES))
			{
				glDisable(GL_BLEND);
				glDepthMask(GL_TRUE);

				// restore culling mode
				current->SetCulling(saveCullingMode);
				if (current->GetCulling() != AMACHI_NO_CULLING)
					glEnable(GL_CULL_FACE);
			}
		}
	glPopMatrix();
}


void
AmachiGLView::Spin()
{
	viewsys.RotateV(-rStep.y);
}


void
AmachiGLView::MessageReceived(BMessage* msg)
{
	if (msg->HasRef("refs"))
	{
		be_app->RefsReceived(msg);
		return;
	}

	if (msg->what != AMACHI_SET_GLPARAM)
	{
		BView::MessageReceived(msg);
		return ;
	}

	int32				mode;
	msg->FindInt32("mode", &mode);

	if (configView->Window()->Lock())
	{
		bool			change = true;

		switch (mode)
		{
			case AMACHI_PROJECTION:
				configView->ReadProjection(*current);
				break;

			case AMACHI_SHADING:
				configView->ReadShading(*current);
				break;

			case AMACHI_TRANSPARENCY:
				configView->ReadTransparency(*current);
				break;

			case AMACHI_ZBUFFER:
				configView->ReadZbuffer(*current);
				break;

			case AMACHI_LIGHTING:
				configView->ReadLighting(*current);
				configView->WriteLighting(*current);
				break;

			case AMACHI_FRONT_RENDERING:
				configView->ReadFrontFace(*current);
				break;

			case AMACHI_BACK_RENDERING:
				configView->ReadBackFace(*current);
				break;

			case AMACHI_CULLING:
				configView->ReadCulling(*current);
				break;

			case AMACHI_FULLSCREEN:
				configView->ReadFullscreen(*current);
				break;

			case AMACHI_MOTION:
				configView->ReadMotion(*current);
				break;

			default:
				change = false;
				break;
		}

		configView->Window()->Unlock();

		if (change)
			release_sem(renderSem);
	}
}

