//
//
//	LAP text Lexer class
//
//										(C) JoOl 1998


#include "LAPimpexp.h"

#include "LAPtextLexer.h"
#include "LAPlexerException.h"
#include "LAPerrors.h"
#include "LAPlexeme.h"



LAPtextLexer::LAPtextLexer(int32 bufSize, LAPlexemeBuffer* lb)
: LAPlexer(new LAPtextBufferedInput(bufSize))
{
	lexeme = lb;
	literalType = LAP_NO_LITERAL;
	token = LAP_TOKEN_EOF;

	lexemes.Insert(new LAPlexeme("<illegal>", LAP_TOKEN_ILLEGAL));
	lexemes.Insert(new LAPlexeme("<number>", LAP_TOKEN_NUMBER));
	lexemes.Insert(new LAPlexeme("<identifier>", LAP_TOKEN_IDENTIFIER));

	input = (LAPtextBufferedInput* )_input;
}


LAPtextLexer::LAPtextLexer(LAPtextBufferedInput* tbi, LAPlexemeBuffer* lb)
: LAPlexer(tbi)
{
	lexeme = lb;
	literalType = LAP_NO_LITERAL;
	token = LAP_TOKEN_EOF;

	lexemes.Insert(new LAPlexeme("<illegal>", LAP_TOKEN_ILLEGAL));
	lexemes.Insert(new LAPlexeme("<number>", LAP_TOKEN_NUMBER));
	lexemes.Insert(new LAPlexeme("<identifier>", LAP_TOKEN_IDENTIFIER));

	input = (LAPtextBufferedInput* )_input;
}


LAPtextLexer::~LAPtextLexer()
{
}


LAPtextBufferedInput*
LAPtextLexer::GetInput() const
{
	return input;
}


void
LAPtextLexer::SetCharCode(char Car, uint8 Kod)
{
	charCode[uint8(Car)] = Kod;
}


void
LAPtextLexer::SetCharCode(uchar Car1, uchar Car2, uint8 Kod)
{
	for (int16 i = int16(Car1); i <= int16(Car2); i++)
		charCode[i] = Kod;
}


void
LAPtextLexer::SetDefaultCharCodes()
{
	SetCharCode(0, 255, LAP_CHAR_ILLEGAL);

	SetCharCode('a', 'z', LAP_CHAR_ALPHA);
	SetCharCode('A', 'Z', LAP_CHAR_ALPHA);
	SetCharCode('_', LAP_CHAR_ALPHA);

	SetCharCode('0', '9', LAP_CHAR_DIGIT);

	SetCharCode(' ', LAP_CHAR_SPACE);
	SetCharCode('\t', LAP_CHAR_SPACE);
	SetCharCode('\n', LAP_CHAR_SPACE);
}


void
LAPtextLexer::SetCharCodes()
{
	SetDefaultCharCodes();
}


void
LAPtextLexer::SetLexemes()
{
}


const char*
LAPtextLexer::GetLexeme(LAPtoken t)
{
	for (int32 i = 0; i < lexemes.CountItems(); i++)
	{
		LAPlexeme*		l = (LAPlexeme* )lexemes.ItemAt(i);
		if (l->GetToken() == t)
			return l->GetName();
	}
	return NULL;
}


status_t
LAPtextLexer::_SetInput(entry_ref )
{
	lineNumber = 1;
	charNumber = 1;

	if (errorOutput && input)
		errorOutput->SetHeader(input->GetInputLeaf(), &lineNumber);

	return LAP_NO_ERROR;
}


int32
LAPtextLexer::GetLineNumber() const
{
	return lineNumber;
}


int32*
LAPtextLexer::GetLineNumberPointer()
{
	return &lineNumber;
}


int32
LAPtextLexer::GetCharNumber() const
{
	return charNumber;
}


int32*
LAPtextLexer::GetCharNumberPointer()
{
	return &charNumber;
}


void
LAPtextLexer::Skip()
{
	charNumber++;
	input->Skip();
}


void
LAPtextLexer::Fwd()
{
	*lexeme << input->Read();
	charNumber++;
}


LAPliteralType
LAPtextLexer::GetLiteralType() const
{
	return literalType;
}


char
LAPtextLexer::GetCharLiteral() const
{
	return literal.Char;
}


int32
LAPtextLexer::GetInt32Literal() const
{
	return ((int32* )(&literal.Int64))[1];
//	return literal.Int64;
}


int64
LAPtextLexer::GetInt64Literal() const
{
	return literal.Int64;
}


float
LAPtextLexer::GetFloatLiteral() const
{
	return float(literal.Double);
}


double
LAPtextLexer::GetDoubleLiteral() const
{
	return literal.Double;
}


char*
LAPtextLexer::GetStringLiteral() const
{
	return literal.String;
}


char*
LAPtextLexer::DetachStringLiteral()
{
	char*		Out = literal.String;
	literal.String = NULL;
	return Out;
}


void
LAPtextLexer::ProcessIllegal(char ill)
{
	literal.Char = ill;
	literalType = LAP_CHARACTER_LITERAL;
	token = LAP_TOKEN_ILLEGAL;
	throw LAPillegalCharacter("LAPtextLexer::ProcessIllegal", literal.Char);
}


void
LAPtextLexer::SkipSpaces()
{
	while (!input->Eof() && IsSpace(input->Peek()))
	{
		if (input->Peek() == '\n')
			lineNumber++;
		Skip();
	}
}


void
LAPtextLexer::SkipLine()
{
	while (!input->Eof() && (input->Peek() != '\n'))
	{
		Skip();
	}
	if (!input->Eof() && (input->Peek() == '\n'))
	{
		lineNumber++;
		charNumber = 1;
		Skip();
	}
}


void
LAPtextLexer::LexNumber()
{
	token = LAP_TOKEN_NUMBER;
	literalType = LAP_INTEGER_LITERAL;

	// [0-9]*
	while (!input->Eof() && IsDigit(input->Peek()))
	{ Fwd(); }

	if (!input->Eof())
	{
		if(input->Peek() == '.')									// Check for \.[0-9]*
		{
			literalType = LAP_FLOAT_LITERAL;
			Fwd();											// Skip '.'
			while (!input->Eof() && IsDigit(input->Peek()))
			{ Fwd(); }
		}
	}

	if (!input->Eof())											// Check for [eE][+\-][0-9]+
	{
		if ((input->Peek() == 'E') || (input->Peek() == 'e'))
		{
			literalType = LAP_FLOAT_LITERAL;

			Fwd();											// Skip 'E' or 'e'
			if ((input->Peek() == '-') || (input->Peek() == '+'))		// Skip [+\-]
			{ Fwd(); }
			while (!input->Eof() && IsDigit(input->Peek()))			// [0-9]*, should be [0-9]+
			{ Fwd(); }
		}
	}

	const char*		L = lexeme->GetBuffer();
	if (literalType == LAP_INTEGER_LITERAL)
		literal.Int64 = atol(L);
	else
		literal.Double = atof(L);
}


void
LAPtextLexer::LexWord()
{
	token = LAP_TOKEN_IDENTIFIER;
	literalType = LAP_STRING_LITERAL;

	while (!input->Eof() && !IsSpace(input->Peek()))
	{ Fwd(); }

	char*		L = lexeme->Detach();
	literal.String = L;

	// lexeme ?
	LAPlexeme*		lexm = lexemes.Search(L);
	if (lexm)
		token = lexm->GetToken();
}


void
LAPtextLexer::LexIdentifier()
{
	token = LAP_TOKEN_IDENTIFIER;
	literalType = LAP_STRING_LITERAL;

	while (!input->Eof())
	{
		if (!IsDigit(input->Peek()) && !IsAlpha(input->Peek()))
			break;
		Fwd();
	}

	char*		L = lexeme->Detach();
	literal.String = L;

	// lexeme ?
	LAPlexeme*		lexm = lexemes.Search(L);
	if (lexm)
		token = lexm->GetToken();
}


void
LAPtextLexer::LexTo(char end)
{
	token = LAP_TOKEN_IDENTIFIER;
	literalType = LAP_STRING_LITERAL;

	while ((!input->Eof()) && (input->Peek() != end))
	{ Fwd(); }

	char*		L = lexeme->Detach();
	literal.String = L;

	// lexeme ?
	LAPlexeme*		lexm = lexemes.Search(L);
	if (lexm)
		token = lexm->GetToken();
}


void
LAPtextLexer::ReadLine()
{
	LexTo('\n');
}


LAPtoken
LAPtextLexer::NextToken()
{
	if (token == LAP_STRING_LITERAL)
		delete[] literal.String;
	lexeme->Reset();

	return _NextToken();
}

