%{
/*
   +----------------------------------------------------------------------+
   | Zend Engine                                                          |
   +----------------------------------------------------------------------+
   | Copyright (c) 1998-2001 Zend Technologies Ltd. (http://www.zend.com) |
   +----------------------------------------------------------------------+
   | This source file is subject to version 0.92 of the Zend license,     |
   | that is bundled with this package in the file LICENSE, and is        | 
   | available at through the world-wide-web at                           |
   | http://www.zend.com/license/0_92.txt.                                |
   | If you did not receive a copy of the Zend license and are unable to  |
   | obtain it through the world-wide-web, please send a note to          |
   | license@zend.com so we can mail you a copy immediately.              |
   +----------------------------------------------------------------------+
   | Authors: Andi Gutmans <andi@zend.com>                                |
   |          Zeev Suraski <zeev@zend.com>                                |
   +----------------------------------------------------------------------+
*/

/* 
 * LALR shift/reduce conflicts and how they are resolved:
 *
 * - 2 shift/reduce conflicts due to the dangeling elseif/else ambiguity.  Solved by shift.
 * - 1 shift/reduce conflict due to arrays within encapsulated strings. Solved by shift. 
 * - 1 shift/reduce conflict due to objects within encapsulated strings.  Solved by shift.
 * 
 */


#include "zend_compile.h"
#include "zend.h"
#include "zend_list.h"
#include "zend_globals.h"
#include "zend_API.h"

#define YYERROR_VERBOSE
#define YYSTYPE znode
#ifdef ZTS
# define YYPARSE_PARAM compiler_globals
# define YYLEX_PARAM compiler_globals
#endif


%}

%pure_parser
%expect 4

%left T_INCLUDE T_INCLUDE_ONCE T_EVAL T_REQUIRE T_REQUIRE_ONCE
%left ','
%left T_LOGICAL_OR
%left T_LOGICAL_XOR
%left T_LOGICAL_AND
%right T_PRINT
%left '=' T_PLUS_EQUAL T_MINUS_EQUAL T_MUL_EQUAL T_DIV_EQUAL T_CONCAT_EQUAL T_MOD_EQUAL T_AND_EQUAL T_OR_EQUAL T_XOR_EQUAL T_SL_EQUAL T_SR_EQUAL
%left '?' ':'
%left T_BOOLEAN_OR
%left T_BOOLEAN_AND
%left '|'
%left '^'
%left '&'
%nonassoc T_IS_EQUAL T_IS_NOT_EQUAL T_IS_IDENTICAL T_IS_NOT_IDENTICAL
%nonassoc '<' T_IS_SMALLER_OR_EQUAL '>' T_IS_GREATER_OR_EQUAL
%left T_SL T_SR
%left '+' '-' '.'
%left '*' '/' '%'
%right '!' '~' T_INC T_DEC T_INT_CAST T_DOUBLE_CAST T_STRING_CAST T_ARRAY_CAST T_OBJECT_CAST T_BOOL_CAST T_UNSET_CAST '@'
%right '['
%nonassoc T_NEW
%token T_EXIT
%token T_IF
%left T_ELSEIF
%left T_ELSE
%left T_ENDIF
%token T_LNUMBER
%token T_DNUMBER
%token T_STRING
%token T_STRING_VARNAME
%token T_VARIABLE
%token T_NUM_STRING
%token T_INLINE_HTML
%token T_CHARACTER
%token T_BAD_CHARACTER
%token T_ENCAPSED_AND_WHITESPACE
%token T_CONSTANT_ENCAPSED_STRING
%token T_ECHO
%token T_DO
%token T_WHILE
%token T_ENDWHILE
%token T_FOR
%token T_ENDFOR
%token T_FOREACH
%token T_ENDFOREACH
%token T_DECLARE
%token T_ENDDECLARE
%token T_AS
%token T_SWITCH
%token T_ENDSWITCH
%token T_CASE
%token T_DEFAULT
%token T_BREAK
%token T_CONTINUE
%token T_OLD_FUNCTION
%token T_FUNCTION
%token T_CONST
%token T_RETURN
%token T_USE
%token T_GLOBAL
%token T_STATIC
%token T_VAR
%token T_UNSET
%token T_ISSET
%token T_EMPTY
%token T_CLASS
%token T_EXTENDS
%token T_OBJECT_OPERATOR
%token T_DOUBLE_ARROW
%token T_LIST
%token T_ARRAY
%token T_LINE
%token T_FILE
%token T_COMMENT
%token T_ML_COMMENT
%token T_OPEN_TAG
%token T_OPEN_TAG_WITH_ECHO
%token T_CLOSE_TAG
%token T_WHITESPACE
%token T_START_HEREDOC
%token T_END_HEREDOC
%token T_DOLLAR_OPEN_CURLY_BRACES
%token T_CURLY_OPEN
%token T_PAAMAYIM_NEKUDOTAYIM

%% /* Rules */

start:
	top_statement_list
;

top_statement_list:	
		top_statement_list  { zend_do_extended_info(CLS_C); } top_statement { ELS_FETCH(); HANDLE_INTERACTIVE(); }
	|	/* empty */
;


top_statement:
		statement
	|	declaration_statement	{ zend_do_early_binding(CLS_C); }
;


inner_statement_list:
		inner_statement_list  { zend_do_extended_info(CLS_C); } inner_statement { ELS_FETCH(); HANDLE_INTERACTIVE(); }
	|	/* empty */
;


inner_statement:
		statement
	|	declaration_statement
;


statement:
		unticked_statement { zend_do_ticks(CLS_C); }
;


unticked_statement:
		'{' inner_statement_list '}'
	|	T_IF '(' expr ')' { zend_do_if_cond(&$3, &$4 CLS_CC); } statement { zend_do_if_after_statement(&$4, 1 CLS_CC); } elseif_list else_single { zend_do_if_end(CLS_C); }
	|	T_IF '(' expr ')' ':' { zend_do_if_cond(&$3, &$4 CLS_CC); } inner_statement_list { zend_do_if_after_statement(&$4, 1 CLS_CC); } new_elseif_list new_else_single T_ENDIF ';' { zend_do_if_end(CLS_C); }
	|	T_WHILE '(' { $1.u.opline_num = get_next_op_number(CG(active_op_array));  } expr  ')' { zend_do_while_cond(&$4, &$5 CLS_CC); } while_statement { zend_do_while_end(&$1, &$5 CLS_CC); }
	|	T_DO { $1.u.opline_num = get_next_op_number(CG(active_op_array));  zend_do_do_while_begin(CLS_C); } statement T_WHILE '(' { $5.u.opline_num = get_next_op_number(CG(active_op_array)); } expr ')' ';' { zend_do_do_while_end(&$1, &$5, &$7 CLS_CC); }
	|	T_FOR 
			'('
				for_expr
			';' { zend_do_free(&$3 CLS_CC); $4.u.opline_num = get_next_op_number(CG(active_op_array)); }
				for_expr
			';' { zend_do_extended_info(CLS_C); zend_do_for_cond(&$6, &$7 CLS_CC); }
				for_expr
			')' { zend_do_free(&$9 CLS_CC); zend_do_for_before_statement(&$4, &$7 CLS_CC); }
			for_statement { zend_do_for_end(&$7 CLS_CC); }
	|	T_SWITCH '(' expr ')'	{ zend_do_switch_cond(&$3 CLS_CC); } switch_case_list { zend_do_switch_end(&$6 CLS_CC); }
	|	T_BREAK ';'				{ zend_do_brk_cont(ZEND_BRK, NULL CLS_CC); }
	|	T_BREAK expr ';'		{ zend_do_brk_cont(ZEND_BRK, &$2 CLS_CC); }
	|	T_CONTINUE ';'			{ zend_do_brk_cont(ZEND_CONT, NULL CLS_CC); }
	|	T_CONTINUE expr ';'		{ zend_do_brk_cont(ZEND_CONT, &$2 CLS_CC); }
	|	T_RETURN ';'						{ zend_do_return(NULL, 0 CLS_CC); }
	|	T_RETURN expr_without_variable ';'	{ zend_do_return(&$2, 0 CLS_CC); }
	|	T_RETURN cvar ';'					{ zend_do_return(&$2, 1 CLS_CC); }
	|	T_GLOBAL global_var_list ';'
	|	T_STATIC static_var_list ';'
	|	T_ECHO echo_expr_list ';'
	|	T_INLINE_HTML			{ zend_do_echo(&$1 CLS_CC); }
	|	expr ';'				{ zend_do_free(&$1 CLS_CC); }
	|	T_USE use_filename ';'		{ zend_error(E_COMPILE_ERROR,"use: Not yet supported. Please use include_once() or require_once()");  zval_dtor(&$2.u.constant); }
	|	T_UNSET '(' unset_variables ')' ';'
	|	T_FOREACH '(' w_cvar T_AS { zend_do_foreach_begin(&$1, &$3, &$2, &$4, 1 CLS_CC); } w_cvar foreach_optional_arg ')' { zend_do_foreach_cont(&$6, &$7, &$4 CLS_CC); } foreach_statement { zend_do_foreach_end(&$1, &$2 CLS_CC); }
	|	T_FOREACH '(' expr_without_variable T_AS { zend_do_foreach_begin(&$1, &$3, &$2, &$4, 0 CLS_CC); } w_cvar foreach_optional_arg ')' { zend_do_foreach_cont(&$6, &$7, &$4 CLS_CC); } foreach_statement { zend_do_foreach_end(&$1, &$2 CLS_CC); }
	|	T_DECLARE { zend_do_declare_begin(CLS_C); } '(' declare_list ')' declare_statement { zend_do_declare_end(CLS_C); }
	|	';'		/* empty statement */
;

unset_variables:
		unset_variable
	|	unset_variables ',' unset_variable
;

unset_variable:
		cvar	{ zend_do_end_variable_parse(BP_VAR_UNSET, 0 CLS_CC); zend_do_unset(&$1 CLS_CC); }
;

use_filename:
		T_CONSTANT_ENCAPSED_STRING			{ $$ = $1; }
	|	'(' T_CONSTANT_ENCAPSED_STRING ')'	{ $$ = $2; }
;


declaration_statement:
		unticked_declaration_statement	{ zend_do_ticks(CLS_C); }
;


unticked_declaration_statement:
		T_FUNCTION { $1.u.opline_num = CG(zend_lineno); } is_reference T_STRING { zend_do_begin_function_declaration(&$1, &$4, 0, $3.op_type CLS_CC); }
			'(' parameter_list ')' '{' inner_statement_list '}' { zend_do_end_function_declaration(&$1 CLS_CC); }
	|	T_OLD_FUNCTION { $1.u.opline_num = CG(zend_lineno); } is_reference T_STRING  { zend_do_begin_function_declaration(&$1, &$4, 0, $3.op_type CLS_CC); }
			parameter_list '(' inner_statement_list ')' ';' { zend_do_end_function_declaration(&$1 CLS_CC); }
	|	T_CLASS T_STRING { zend_do_begin_class_declaration(&$2, NULL CLS_CC); } '{' class_statement_list '}' { zend_do_end_class_declaration(CLS_C); }
	|	T_CLASS T_STRING T_EXTENDS T_STRING { zend_do_begin_class_declaration(&$2, &$4 CLS_CC); } '{' class_statement_list '}' { zend_do_end_class_declaration(CLS_C); }
;


foreach_optional_arg:
		/* empty */				{ $$.op_type = IS_UNUSED; }
	|	T_DOUBLE_ARROW w_cvar	{ $$ = $2; }
;


for_statement:
		statement
	|	':' inner_statement_list T_ENDFOR ';'
;


foreach_statement:
		statement
	|	':' inner_statement_list T_ENDFOREACH ';'
;


declare_statement:
		statement
	|	':' inner_statement_list T_ENDDECLARE ';'
;


declare_list:
		T_STRING '=' static_scalar					{ zend_do_declare_stmt(&$1, &$3 CLS_CC); }
	|	declare_list ',' T_STRING '=' static_scalar	{ zend_do_declare_stmt(&$3, &$5 CLS_CC); }
;


switch_case_list:
		'{' case_list '}'					{ $$ = $2; }
	|	'{' ';' case_list '}'				{ $$ = $3; }
	|	':' case_list T_ENDSWITCH ';'		{ $$ = $2; }
	|	':' ';' case_list T_ENDSWITCH ';'	{ $$ = $3; }
;


case_list:
		/* empty */	{ $$.op_type = IS_UNUSED; }
	|	case_list T_CASE expr case_separator { zend_do_extended_info(CLS_C);  zend_do_case_before_statement(&$1, &$2, &$3 CLS_CC); } inner_statement_list { zend_do_case_after_statement(&$$, &$2 CLS_CC); $$.op_type = IS_CONST }
	|	case_list T_DEFAULT case_separator { zend_do_extended_info(CLS_C);  zend_do_default_before_statement(&$1, &$2 CLS_CC); } inner_statement_list { zend_do_case_after_statement(&$$, &$2 CLS_CC); $$.op_type = IS_CONST; }
;


case_separator:
		':'
	|	';'
;


while_statement:
		statement
	|	':' inner_statement_list T_ENDWHILE ';'
;



elseif_list:
		/* empty */
	|	elseif_list T_ELSEIF '(' expr ')' { zend_do_if_cond(&$4, &$5 CLS_CC); } statement { zend_do_if_after_statement(&$5, 0 CLS_CC); }
;


new_elseif_list:
		/* empty */
	|	new_elseif_list T_ELSEIF '(' expr ')' ':' { zend_do_if_cond(&$4, &$5 CLS_CC); } inner_statement_list { zend_do_if_after_statement(&$5, 0 CLS_CC); }
;


else_single:
		/* empty */
	|	T_ELSE statement
;


new_else_single:
		/* empty */
	|	T_ELSE ':' inner_statement_list
;


parameter_list: 
		non_empty_parameter_list
	|	/* empty */
;


non_empty_parameter_list:
		T_VARIABLE				{ znode tmp;  fetch_simple_variable(&tmp, &$1, 0 CLS_CC); $$.op_type = IS_CONST; $$.u.constant.value.lval=1; $$.u.constant.type=IS_LONG; INIT_PZVAL(&$$.u.constant); zend_do_receive_arg(ZEND_RECV, &tmp, &$$, NULL, BYREF_NONE CLS_CC); }
	|	'&' T_VARIABLE			{ znode tmp;  fetch_simple_variable(&tmp, &$2, 0 CLS_CC); $$.op_type = IS_CONST; $$.u.constant.value.lval=1; $$.u.constant.type=IS_LONG; INIT_PZVAL(&$$.u.constant); zend_do_receive_arg(ZEND_RECV, &tmp, &$$, NULL, BYREF_FORCE CLS_CC); }
	|	T_CONST T_VARIABLE 		{ znode tmp;  fetch_simple_variable(&tmp, &$2, 0 CLS_CC); $$.op_type = IS_CONST; $$.u.constant.value.lval=1; $$.u.constant.type=IS_LONG; INIT_PZVAL(&$$.u.constant); zend_do_receive_arg(ZEND_RECV, &tmp, &$$, NULL, BYREF_NONE CLS_CC); }
	|	T_VARIABLE '=' static_scalar				{ znode tmp;  fetch_simple_variable(&tmp, &$1, 0 CLS_CC); $$.op_type = IS_CONST; $$.u.constant.value.lval=1; $$.u.constant.type=IS_LONG; INIT_PZVAL(&$$.u.constant); zend_do_receive_arg(ZEND_RECV_INIT, &tmp, &$$, &$3, BYREF_NONE CLS_CC); }
	|	non_empty_parameter_list ',' T_VARIABLE 	{ znode tmp;  fetch_simple_variable(&tmp, &$3, 0 CLS_CC); $$=$1; $$.u.constant.value.lval++; zend_do_receive_arg(ZEND_RECV, &tmp, &$$, NULL, BYREF_NONE CLS_CC); }
	|	non_empty_parameter_list ',' '&' T_VARIABLE	{ znode tmp;  fetch_simple_variable(&tmp, &$4, 0 CLS_CC); $$=$1; $$.u.constant.value.lval++; zend_do_receive_arg(ZEND_RECV, &tmp, &$$, NULL, BYREF_FORCE CLS_CC); }
	|	non_empty_parameter_list ',' T_CONST T_VARIABLE				{ znode tmp;  fetch_simple_variable(&tmp, &$4, 0 CLS_CC); $$=$1; $$.u.constant.value.lval++; zend_do_receive_arg(ZEND_RECV, &tmp, &$$, NULL, BYREF_NONE CLS_CC); }
	|	non_empty_parameter_list ',' T_VARIABLE '=' static_scalar 	{ znode tmp;  fetch_simple_variable(&tmp, &$3, 0 CLS_CC); $$=$1; $$.u.constant.value.lval++; zend_do_receive_arg(ZEND_RECV_INIT, &tmp, &$$, &$5, BYREF_NONE CLS_CC); }
;


function_call_parameter_list:
		non_empty_function_call_parameter_list	{ $$ = $1; }
	|	/* empty */				{ $$.u.constant.value.lval = 0; }
;


non_empty_function_call_parameter_list:
		expr_without_variable	{ $$.u.constant.value.lval = 1;  zend_do_pass_param(&$1, ZEND_SEND_VAL, $$.u.constant.value.lval CLS_CC); }
	|	cvar					{ $$.u.constant.value.lval = 1;  zend_do_pass_param(&$1, ZEND_SEND_VAR, $$.u.constant.value.lval CLS_CC); }
	|	'&' w_cvar 				{ $$.u.constant.value.lval = 1;  zend_do_pass_param(&$2, ZEND_SEND_REF, $$.u.constant.value.lval CLS_CC); }
	|	non_empty_function_call_parameter_list ',' expr_without_variable	{ $$.u.constant.value.lval=$1.u.constant.value.lval+1;  zend_do_pass_param(&$3, ZEND_SEND_VAL, $$.u.constant.value.lval CLS_CC); }
	|	non_empty_function_call_parameter_list ',' cvar						{ $$.u.constant.value.lval=$1.u.constant.value.lval+1;  zend_do_pass_param(&$3, ZEND_SEND_VAR, $$.u.constant.value.lval CLS_CC); }
	|	non_empty_function_call_parameter_list ',' '&' w_cvar				{ $$.u.constant.value.lval=$1.u.constant.value.lval+1;  zend_do_pass_param(&$4, ZEND_SEND_REF, $$.u.constant.value.lval CLS_CC); }
;

global_var_list:
		global_var_list ',' global_var	{ zend_do_fetch_global_or_static_variable(&$3, NULL, ZEND_FETCH_GLOBAL CLS_CC); }
	|	global_var						{ zend_do_fetch_global_or_static_variable(&$1, NULL, ZEND_FETCH_GLOBAL CLS_CC); }
;


global_var:
		T_VARIABLE			{ $$ = $1; }
	|	'$' r_cvar			{ $$ = $2; }
	|	'$' '{' expr '}'	{ $$ = $3; }
;


static_var_list:
		static_var_list ',' T_VARIABLE { zend_do_fetch_global_or_static_variable(&$3, NULL, ZEND_FETCH_STATIC CLS_CC); }
	|	static_var_list ',' T_VARIABLE '=' static_scalar { zend_do_fetch_global_or_static_variable(&$3, &$5, ZEND_FETCH_STATIC CLS_CC); }
	|	T_VARIABLE  { zend_do_fetch_global_or_static_variable(&$1, NULL, ZEND_FETCH_STATIC CLS_CC); }
	|	T_VARIABLE '=' static_scalar { zend_do_fetch_global_or_static_variable(&$1, &$3, ZEND_FETCH_STATIC CLS_CC); }

;


class_statement_list:
		class_statement_list class_statement
	|	/* empty */
;


class_statement:
		T_VAR class_variable_decleration ';'
	|	T_FUNCTION { $1.u.opline_num = CG(zend_lineno); } is_reference T_STRING { zend_do_begin_function_declaration(&$1, &$4, 1, $3.op_type CLS_CC); } '(' 
			parameter_list ')' '{' inner_statement_list '}' { zend_do_end_function_declaration(&$1 CLS_CC); }
	|	T_OLD_FUNCTION { $1.u.opline_num = CG(zend_lineno); } is_reference T_STRING { zend_do_begin_function_declaration(&$1, &$4, 1, $3.op_type CLS_CC); }
			parameter_list '(' inner_statement_list ')' ';' { zend_do_end_function_declaration(&$1 CLS_CC); }

;

is_reference:
		/* empty */	{ $$.op_type = ZEND_RETURN_VAL; }
	|	'&'			{ $$.op_type = ZEND_RETURN_REF; }

class_variable_decleration:
		class_variable_decleration ',' T_VARIABLE					{ zend_do_declare_property(&$3, NULL CLS_CC); }
	|	class_variable_decleration ',' T_VARIABLE '=' static_scalar	{ zend_do_declare_property(&$3, &$5 CLS_CC); }
	|	T_VARIABLE						{ zend_do_declare_property(&$1, NULL CLS_CC); }
	|	T_VARIABLE '=' static_scalar	{ zend_do_declare_property(&$1, &$3 CLS_CC); }
;


echo_expr_list:	
	|	echo_expr_list ',' expr { zend_do_echo(&$3 CLS_CC); }
	|	expr					{ zend_do_echo(&$1 CLS_CC); }
;


for_expr:
		/* empty */			{ $$.op_type = IS_CONST;  $$.u.constant.type = IS_BOOL;  $$.u.constant.value.lval = 1; }
	|	non_empty_for_expr	{ $$ = $1; }
;

non_empty_for_expr:
		non_empty_for_expr ','	{ zend_do_free(&$1 CLS_CC); } expr { $$ = $4; }
	|	expr					{ $$ = $1; }
;


expr_without_variable:	
		T_LIST '(' { zend_do_list_init(CLS_C); } assignment_list ')' '=' expr { zend_do_list_end(&$$, &$7 CLS_CC); }
	|	cvar '=' expr		{ zend_do_end_variable_parse(BP_VAR_W, 0 CLS_CC); zend_do_assign(&$$, &$1, &$3 CLS_CC); }
	|	cvar '=' '&' w_cvar	{ zend_do_end_variable_parse(BP_VAR_W, 0 CLS_CC); zend_do_assign_ref(&$$, &$1, &$4 CLS_CC); }
	|	cvar '=' '&' function_call { zend_do_end_variable_parse(BP_VAR_W, 0 CLS_CC); zend_do_assign_ref(&$$, &$1, &$4 CLS_CC); }
	|	cvar '=' '&' T_NEW class_name { zend_do_extended_fcall_begin(CLS_C); zend_do_begin_new_object(&$4, &$5 CLS_CC); } ctor_arguments { zend_do_end_new_object(&$3, &$5, &$4, &$7 CLS_CC); zend_do_extended_fcall_end(CLS_C); zend_do_end_variable_parse(BP_VAR_W, 0 CLS_CC); zend_do_assign_ref(&$$, &$1, &$3 CLS_CC); }
	|	T_NEW class_name { zend_do_extended_fcall_begin(CLS_C); zend_do_begin_new_object(&$1, &$2 CLS_CC); } ctor_arguments { zend_do_end_new_object(&$$, &$2, &$1, &$4 CLS_CC); zend_do_extended_fcall_end(CLS_C);}
	|	cvar T_PLUS_EQUAL expr 	{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_ADD, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_MINUS_EQUAL expr	{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_SUB, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_MUL_EQUAL expr		{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_MUL, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_DIV_EQUAL expr		{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_DIV, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_CONCAT_EQUAL expr	{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_CONCAT, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_MOD_EQUAL expr		{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_MOD, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_AND_EQUAL expr		{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_BW_AND, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_OR_EQUAL expr 		{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_BW_OR, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_XOR_EQUAL expr 		{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_BW_XOR, &$$, &$1, &$3 CLS_CC); }
	|	cvar T_SL_EQUAL expr	{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_SL, &$$, &$1, &$3 CLS_CC); } 
	|	cvar T_SR_EQUAL expr	{ zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); zend_do_binary_assign_op(ZEND_ASSIGN_SR, &$$, &$1, &$3 CLS_CC); } 
	|	rw_cvar T_INC { zend_do_post_incdec(&$$, &$1, ZEND_POST_INC CLS_CC); }
	|	T_INC rw_cvar { zend_do_pre_incdec(&$$, &$2, ZEND_PRE_INC CLS_CC); }
	|	rw_cvar T_DEC { zend_do_post_incdec(&$$, &$1, ZEND_POST_DEC CLS_CC); }
	|	T_DEC rw_cvar { zend_do_pre_incdec(&$$, &$2, ZEND_PRE_DEC CLS_CC); }
	|	expr T_BOOLEAN_OR { zend_do_boolean_or_begin(&$1, &$2 CLS_CC); } expr { zend_do_boolean_or_end(&$$, &$1, &$4, &$2 CLS_CC); }
	|	expr T_BOOLEAN_AND { zend_do_boolean_and_begin(&$1, &$2 CLS_CC); } expr { zend_do_boolean_and_end(&$$, &$1, &$4, &$2 CLS_CC); }  
	|	expr T_LOGICAL_OR { zend_do_boolean_or_begin(&$1, &$2 CLS_CC); } expr { zend_do_boolean_or_end(&$$, &$1, &$4, &$2 CLS_CC); }
	|	expr T_LOGICAL_AND { zend_do_boolean_and_begin(&$1, &$2 CLS_CC); } expr { zend_do_boolean_and_end(&$$, &$1, &$4, &$2 CLS_CC); }
	|	expr T_LOGICAL_XOR expr { zend_do_binary_op(ZEND_BOOL_XOR, &$$, &$1, &$3 CLS_CC); }
	|	expr '|' expr	{ zend_do_binary_op(ZEND_BW_OR, &$$, &$1, &$3 CLS_CC); }
	|	expr '&' expr	{ zend_do_binary_op(ZEND_BW_AND, &$$, &$1, &$3 CLS_CC); }
	|	expr '^' expr	{ zend_do_binary_op(ZEND_BW_XOR, &$$, &$1, &$3 CLS_CC); }
	|	expr '.' expr 	{ zend_do_binary_op(ZEND_CONCAT,&$$,&$1,&$3 CLS_CC); }
	|	expr '+' expr 	{ zend_do_binary_op(ZEND_ADD,&$$,&$1,&$3 CLS_CC); }
	|	expr '-' expr 	{ zend_do_binary_op(ZEND_SUB,&$$,&$1,&$3 CLS_CC); }
	|	expr '*' expr	{ zend_do_binary_op(ZEND_MUL,&$$,&$1,&$3 CLS_CC); }
	|	expr '/' expr	{ zend_do_binary_op(ZEND_DIV,&$$,&$1,&$3 CLS_CC); }
	|	expr '%' expr 	{ zend_do_binary_op(ZEND_MOD,&$$,&$1,&$3 CLS_CC); }
	| 	expr T_SL expr	{ zend_do_binary_op(ZEND_SL, &$$, &$1, &$3 CLS_CC); }
	|	expr T_SR expr	{ zend_do_binary_op(ZEND_SR, &$$, &$1, &$3 CLS_CC); }
	|	'+' expr { $1.u.constant.value.lval=0; $1.u.constant.type=IS_LONG; $1.op_type = IS_CONST; INIT_PZVAL(&$1.u.constant); zend_do_binary_op(ZEND_ADD, &$$, &$1, &$2 CLS_CC); }
	|	'-' expr { $1.u.constant.value.lval=0; $1.u.constant.type=IS_LONG; $1.op_type = IS_CONST; INIT_PZVAL(&$1.u.constant); zend_do_binary_op(ZEND_SUB, &$$, &$1, &$2 CLS_CC); }
	|	'!' expr { zend_do_unary_op(ZEND_BOOL_NOT, &$$, &$2 CLS_CC); }
	|	'~' expr { zend_do_unary_op(ZEND_BW_NOT, &$$, &$2 CLS_CC); }
	|	expr T_IS_IDENTICAL expr		{ zend_do_binary_op(ZEND_IS_IDENTICAL, &$$, &$1, &$3 CLS_CC); }
	|	expr T_IS_NOT_IDENTICAL expr	{ zend_do_binary_op(ZEND_IS_NOT_IDENTICAL, &$$, &$1, &$3 CLS_CC); }
	|	expr T_IS_EQUAL expr			{ zend_do_binary_op(ZEND_IS_EQUAL, &$$, &$1, &$3 CLS_CC); }
	|	expr T_IS_NOT_EQUAL expr 		{ zend_do_binary_op(ZEND_IS_NOT_EQUAL, &$$, &$1, &$3 CLS_CC); }
	|	expr '<' expr 					{ zend_do_binary_op(ZEND_IS_SMALLER, &$$, &$1, &$3 CLS_CC); }
	|	expr T_IS_SMALLER_OR_EQUAL expr { zend_do_binary_op(ZEND_IS_SMALLER_OR_EQUAL, &$$, &$1, &$3 CLS_CC); }
	|	expr '>' expr 					{ zend_do_binary_op(ZEND_IS_SMALLER, &$$, &$3, &$1 CLS_CC); }
	|	expr T_IS_GREATER_OR_EQUAL expr { zend_do_binary_op(ZEND_IS_SMALLER_OR_EQUAL, &$$, &$3, &$1 CLS_CC); }
	|	'(' expr ')' 	{ $$ = $2; }
	|	expr '?' { zend_do_begin_qm_op(&$1, &$2 CLS_CC); }
		expr ':' { zend_do_qm_true(&$4, &$2, &$5 CLS_CC); }
		expr	 { zend_do_qm_false(&$$, &$7, &$2, &$5 CLS_CC); }
	|	function_call { $$ = $1; }
	|	internal_functions_in_yacc { $$ = $1; }
	|	T_INT_CAST expr 	{ zend_do_cast(&$$, &$2, IS_LONG CLS_CC); }
	|	T_DOUBLE_CAST expr 	{ zend_do_cast(&$$, &$2, IS_DOUBLE CLS_CC); }
	|	T_STRING_CAST expr	{ zend_do_cast(&$$, &$2, IS_STRING CLS_CC); } 
	|	T_ARRAY_CAST expr 	{ zend_do_cast(&$$, &$2, IS_ARRAY CLS_CC); }
	|	T_OBJECT_CAST expr 	{ zend_do_cast(&$$, &$2, IS_OBJECT CLS_CC); }
	|	T_BOOL_CAST expr	{ zend_do_cast(&$$, &$2, IS_BOOL CLS_CC); }
	|	T_UNSET_CAST expr	{ zend_do_cast(&$$, &$2, IS_NULL CLS_CC); }
	|	T_EXIT exit_expr	{ zend_do_exit(&$$, &$2 CLS_CC); }
	|	'@' { zend_do_begin_silence(&$1 CLS_CC); } expr { zend_do_end_silence(&$1 CLS_CC); $$ = $3; }
	|	scalar				{ $$ = $1; }
	|	T_ARRAY '(' array_pair_list ')' { $$ = $3; }
	|	'`' encaps_list '`'		{ zend_do_shell_exec(&$$, &$2 CLS_CC); }
	|	T_PRINT expr  { zend_do_print(&$$, &$2 CLS_CC); }
;

function_call:
		T_STRING	'(' { $2.u.opline_num = zend_do_begin_function_call(&$1 CLS_CC); }
				function_call_parameter_list
				')' { zend_do_end_function_call(&$1, &$$, &$4, 0, $2.u.opline_num CLS_CC); zend_do_extended_fcall_end(CLS_C); }
	|	cvar '(' { zend_do_begin_dynamic_function_call(&$1 CLS_CC); } 
				function_call_parameter_list 
				')' { zend_do_end_function_call(&$1, &$$, &$4, 0, 1 CLS_CC); zend_do_extended_fcall_end(CLS_C);}
	|	T_STRING T_PAAMAYIM_NEKUDOTAYIM T_STRING '(' { zend_do_extended_fcall_begin(CLS_C); zend_do_begin_class_member_function_call(&$1, &$3 CLS_CC); } 
											function_call_parameter_list 
											')' { zend_do_end_function_call(&$3, &$$, &$6, 1, 1 CLS_CC); zend_do_extended_fcall_end(CLS_C);}
;


exit_expr:
		/* empty */	{ memset(&$$, 0, sizeof(znode)); $$.op_type = IS_UNUSED; }	
	|	'(' ')'		{ memset(&$$, 0, sizeof(znode)); $$.op_type = IS_UNUSED; }	
	|	'(' expr ')'	{ $$ = $2; }
;


ctor_arguments:
		/* empty */	{ $$.u.constant.value.lval=0; }
	|	'(' function_call_parameter_list ')'	{ $$ = $2; }
;


class_name:
		T_STRING	{ $$ = $1; }
	|	r_cvar		{ $$ = $1; }
;



common_scalar:
		T_LNUMBER 					{ $$ = $1; }
	|	T_DNUMBER 					{ $$ = $1; }
	|	T_CONSTANT_ENCAPSED_STRING	{ $$ = $1; }
	|	T_LINE 						{ $$ = $1; }
	|	T_FILE 						{ $$ = $1; }
;


static_scalar: /* compile-time evaluated scalars */
		common_scalar		{ $$ = $1; }
	|	T_STRING 		{ zend_do_fetch_constant(&$$, &$1, ZEND_CT CLS_CC); }
	|	'+' static_scalar	{ $$ = $1; }
	|	'-' static_scalar	{ zval minus_one;  minus_one.type = IS_LONG; minus_one.value.lval = -1;  mul_function(&$2.u.constant, &$2.u.constant, &minus_one);  $$ = $2; }
	|	T_ARRAY '(' static_array_pair_list ')' { $$ = $3; $$.u.constant.type = IS_CONSTANT_ARRAY; }
;


scalar:
		T_STRING 				{ zend_do_fetch_constant(&$$, &$1, ZEND_RT CLS_CC); }
	|	T_STRING_VARNAME		{ $$ = $1; }
	|	common_scalar			{ $$ = $1; }
	|	'"' encaps_list '"' 	{ $$ = $2; }
	|	'\'' encaps_list '\''	{ $$ = $2; }
	|	T_START_HEREDOC encaps_list T_END_HEREDOC { $$ = $2; zend_do_end_heredoc(CLS_C); }
;


static_array_pair_list:
		/* empty */ { $$.op_type = IS_CONST; INIT_PZVAL(&$$.u.constant); array_init(&$$.u.constant); }
	|	non_empty_static_array_pair_list possible_comma	{ $$ = $1; }
;

possible_comma:
		/* empty */
	|	','
;

non_empty_static_array_pair_list:
		non_empty_static_array_pair_list ',' static_scalar T_DOUBLE_ARROW static_scalar	{ zend_do_add_static_array_element(&$$, &$3, &$5); }
	|	non_empty_static_array_pair_list ',' static_scalar { zend_do_add_static_array_element(&$$, NULL, &$3); }
	|	static_scalar T_DOUBLE_ARROW static_scalar { $$.op_type = IS_CONST; INIT_PZVAL(&$$.u.constant); array_init(&$$.u.constant); zend_do_add_static_array_element(&$$, &$1, &$3); }
	|	static_scalar { $$.op_type = IS_CONST; INIT_PZVAL(&$$.u.constant); array_init(&$$.u.constant); zend_do_add_static_array_element(&$$, NULL, &$1); }
;

expr:
		r_cvar					{ $$ = $1; }
	|	expr_without_variable	{ $$ = $1; }
;


r_cvar:
	cvar { zend_do_end_variable_parse(BP_VAR_R, 0 CLS_CC); $$ = $1; }
;


w_cvar:
	cvar { zend_do_end_variable_parse(BP_VAR_W, 0 CLS_CC); $$ = $1; }
;


rw_cvar:
	cvar { zend_do_end_variable_parse(BP_VAR_RW, 0 CLS_CC); $$ = $1; }
;


cvar:
		cvar_without_objects { $$ = $1; }
	|	cvar_without_objects T_OBJECT_OPERATOR { zend_do_push_object(&$1 CLS_CC); } ref_list { $$ = $4; }
;


cvar_without_objects:
		reference_variable { $$ = $1; }
	|	simple_indirect_reference reference_variable { zend_do_indirect_references(&$$, &$1, &$2 CLS_CC); }
;


reference_variable:
		reference_variable '[' dim_offset ']'	{ fetch_array_dim(&$$, &$1, &$3 CLS_CC); }
	|	reference_variable '{' expr '}'		{ fetch_string_offset(&$$, &$1, &$3 CLS_CC); }
	|	compound_variable			{ zend_do_fetch_globals(&$1 CLS_CC); zend_do_begin_variable_parse(CLS_C); fetch_simple_variable(&$$, &$1, 1 CLS_CC); }
;
	

compound_variable:
		T_VARIABLE			{ $$ = $1; }
	|	'$' '{' expr '}'	{ $$ = $3; }
;

dim_offset:
		/* empty */		{ $$.op_type = IS_UNUSED; }
	|	expr			{ $$ = $1; }
;

ref_list:
		object_property  { $$ = $1; }
	|	ref_list T_OBJECT_OPERATOR { zend_do_push_object(&$1 CLS_CC); } object_property { $$ = $4; }
;

object_property:
		object_dim_list { $$ = $1; }
	|	cvar_without_objects { zend_do_end_variable_parse(BP_VAR_R, 0 CLS_CC); } { znode tmp_znode;  zend_do_pop_object(&tmp_znode CLS_CC);  zend_do_fetch_property(&$$, &tmp_znode, &$1 CLS_CC);}
;

object_dim_list:
		object_dim_list '[' dim_offset ']'	{ fetch_array_dim(&$$, &$1, &$3 CLS_CC); }
	|	object_dim_list '{' expr '}'		{ fetch_string_offset(&$$, &$1, &$3 CLS_CC); }
	|	variable_name { znode tmp_znode;  zend_do_pop_object(&tmp_znode CLS_CC);  zend_do_fetch_property(&$$, &tmp_znode, &$1 CLS_CC);}
;

variable_name:
		T_STRING		{ $$ = $1; }
	|	'{' expr '}'	{ $$ = $2; }
;


simple_indirect_reference:
		'$' { $$.u.constant.value.lval = 1; }
	|	simple_indirect_reference '$' { $$.u.constant.value.lval++; }
;

assignment_list:
		assignment_list ',' assignment_list_element
	|	assignment_list_element
;


assignment_list_element:
		cvar								{ zend_do_add_list_element(&$1 CLS_CC); }
	|	T_LIST '(' { zend_do_new_list_begin(CLS_C); } assignment_list ')'	{ zend_do_new_list_end(CLS_C); }
	|	/* empty */							{ zend_do_add_list_element(NULL CLS_CC); }
;


array_pair_list:
		/* empty */ { zend_do_init_array(&$$, NULL, NULL, 0 CLS_CC); }
	|	non_empty_array_pair_list possible_comma	{ $$ = $1; }
;

non_empty_array_pair_list:
		non_empty_array_pair_list ',' expr T_DOUBLE_ARROW expr	{ zend_do_add_array_element(&$$, &$5, &$3, 0 CLS_CC); }
	|	non_empty_array_pair_list ',' expr			{ zend_do_add_array_element(&$$, &$3, NULL, 0 CLS_CC); }
	|	expr T_DOUBLE_ARROW expr	{ zend_do_init_array(&$$, &$3, &$1, 0 CLS_CC); }
	|	expr 				{ zend_do_init_array(&$$, &$1, NULL, 0 CLS_CC); }
	|	non_empty_array_pair_list ',' expr T_DOUBLE_ARROW '&' w_cvar { zend_do_add_array_element(&$$, &$6, &$3, 1 CLS_CC); }
	|	non_empty_array_pair_list ',' '&' w_cvar { zend_do_add_array_element(&$$, &$4, NULL, 1 CLS_CC); }
	|	expr T_DOUBLE_ARROW '&' w_cvar	{ zend_do_init_array(&$$, &$4, &$1, 1 CLS_CC); }
	|	'&' w_cvar 			{ zend_do_init_array(&$$, &$2, NULL, 1 CLS_CC); }
;

encaps_list:
		encaps_list encaps_var { zend_do_end_variable_parse(BP_VAR_R, 0 CLS_CC);  zend_do_add_variable(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list T_STRING			{ zend_do_add_string(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list T_NUM_STRING		{ zend_do_add_string(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list T_ENCAPSED_AND_WHITESPACE	{ zend_do_add_string(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list T_CHARACTER 		{ zend_do_add_char(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list T_BAD_CHARACTER		{ zend_do_add_string(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list '['		{ $2.u.constant.value.lval = (long) '['; zend_do_add_char(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list ']'		{ $2.u.constant.value.lval = (long) ']'; zend_do_add_char(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list '{'		{ $2.u.constant.value.lval = (long) '{'; zend_do_add_char(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list '}'		{ $2.u.constant.value.lval = (long) '}'; zend_do_add_char(&$$, &$1, &$2 CLS_CC); }
	|	encaps_list T_OBJECT_OPERATOR  { znode tmp;  $2.u.constant.value.lval = (long) '-';  zend_do_add_char(&tmp, &$1, &$2 CLS_CC);  $2.u.constant.value.lval = (long) '>'; zend_do_add_char(&$$, &tmp, &$2 CLS_CC); }
	|	/* empty */			{ zend_do_init_string(&$$ CLS_CC); }

;



encaps_var:
		T_VARIABLE { zend_do_fetch_globals(&$1 CLS_CC); zend_do_begin_variable_parse(CLS_C); fetch_simple_variable(&$$, &$1, 1 CLS_CC); }
	|	T_VARIABLE '[' { zend_do_begin_variable_parse(CLS_C); } encaps_var_offset ']'	{ zend_do_fetch_globals(&$1 CLS_CC);  fetch_array_begin(&$$, &$1, &$4 CLS_CC); }
	|	T_VARIABLE T_OBJECT_OPERATOR T_STRING { zend_do_begin_variable_parse(CLS_C); fetch_simple_variable(&$2, &$1, 1 CLS_CC); zend_do_fetch_property(&$$, &$2, &$3 CLS_CC); }
	|	T_DOLLAR_OPEN_CURLY_BRACES expr '}' { zend_do_begin_variable_parse(CLS_C);  fetch_simple_variable(&$$, &$2, 1 CLS_CC); }
	|	T_DOLLAR_OPEN_CURLY_BRACES T_STRING_VARNAME '[' expr ']' '}' { zend_do_begin_variable_parse(CLS_C);  fetch_array_begin(&$$, &$2, &$4 CLS_CC); }
	|	T_CURLY_OPEN cvar '}' { $$ = $2; }
;


encaps_var_offset:
		T_STRING		{ $$ = $1; }
	|	T_NUM_STRING	{ $$ = $1; }
	|	T_VARIABLE		{ fetch_simple_variable(&$$, &$1, 1 CLS_CC); }
;


internal_functions_in_yacc:
		T_ISSET '(' isset_variables ')' { $$ = $3; }
	|	T_EMPTY '(' cvar ')'	{ zend_do_isset_or_isempty(ZEND_ISEMPTY, &$$, &$3 CLS_CC); }
	|	T_INCLUDE expr 			{ zend_do_include_or_eval(ZEND_INCLUDE, &$$, &$2 CLS_CC); }
	|	T_INCLUDE_ONCE expr 	{ zend_do_include_or_eval(ZEND_INCLUDE_ONCE, &$$, &$2 CLS_CC); }
	|	T_EVAL '(' expr ')' 	{ zend_do_include_or_eval(ZEND_EVAL, &$$, &$3 CLS_CC); }
	|	T_REQUIRE expr			{ zend_do_include_or_eval(ZEND_REQUIRE, &$$, &$2 CLS_CC); }
	|	T_REQUIRE_ONCE expr		{ zend_do_include_or_eval(ZEND_REQUIRE_ONCE, &$$, &$2 CLS_CC); }
;

isset_variables:
		cvar 				{ zend_do_isset_or_isempty(ZEND_ISSET, &$$, &$1 CLS_CC); }
	|	isset_variables ',' { zend_do_boolean_and_begin(&$1, &$2 CLS_CC); } cvar { znode tmp; zend_do_isset_or_isempty(ZEND_ISSET, &tmp, &$4 CLS_CC); zend_do_boolean_and_end(&$$, &$1, &tmp, &$2 CLS_CC); }
;	

%%

