// MultiLang.h
// 多言語化支援クラスの実装

#include "MultiLang.h"

using namespace std;

const int LINE_BUFFER_SIZE = 1024;

TMultiLang::TMultiLang()
	:m_lang_db(new LANG_DB),
	 m_selected_db(NULL),
	 m_language(DEFAULT_LANGUAGE)

{
	initLangDataBase();
	SetLang(DEFAULT_LANGUAGE);
}

TMultiLang::~TMultiLang()
{
//	delete m_lang_db;
}

// インスタンスの取得
TMultiLang* TMultiLang::GetMultiLang()
{
	static TMultiLang multi_lang;
	return &multi_lang;
}

// 言語変更
bool TMultiLang::SetLang(const char* _language,bool _avoid_error = true)
{
	LANG_DB::iterator ite;
	ite = m_lang_db->find(_language);
	if (ite != m_lang_db->end() )
	{
		m_selected_db = &ite->second;
//		m_selected_db = m_lang_db[_language];

		m_language = _language;
	
		// 登録されているobserverに言語変更の通知　
		NotifyObservers(_language);

		return true;
	}
	else // そんな言語はインストールされてないぞと
	{
		// とりあえず、エラー回避のため
		// インストールされている言語にする
		if (_avoid_error == true && IsEmpty() != true)
		{
			LANG_DB::iterator ite;
			ite = m_lang_db->begin();
			
			m_selected_db = &ite->second;
			m_language = ite->first;
			
			NotifyObservers(m_language.c_str());
			return true;
		}
		else
		{		
			return false;
		}
	}
}

// 母国語による言語名取得
const char* TMultiLang::GetLang()
{
	STRING_DB::iterator ite;
	ite = m_selected_db->find(m_language.c_str());
	if (ite != m_selected_db->end())
	{

		return ite->second.c_str();
	}
	else
	{
		return m_language.c_str();
	}
}

// 文字列の取得
const char* TMultiLang::GetString(const char* _key)
{
	STRING_DB::iterator ite;
	ite = m_selected_db->find(_key);
	
	// 普通に探してあったらそれを持ってくる
	if(ite != m_selected_db->end())
	{
		return ite->second.c_str();
	}
	else
	{
		// 選択されている言語に文字列が存在しなかったら
		// 英語から文字を引っ張ってくる
		STRING_DB& default_db = (*m_lang_db)[DEFAULT_LANGUAGE];
		ite = default_db.find(_key);
	
		if (ite != default_db.end())
		{
			return ite->second.c_str();
		}
		else // それでも存在しなかったら空文字列を返す
		{
			return NULL;
		}
	}	
	return NULL;	
}

// 現在選択されている言語とは違う別の言語の取得
const char* TMultiLang::GetStringFromOtherLang(const char* _lang,
									const char* _key)
{
	// 指定されている言語名が本当にあるかどうかを調べて
	// なかったらNULLを返す
	LANG_DB::iterator lang_ite;
	lang_ite = m_lang_db->find(_lang);
	
	if (lang_ite == m_lang_db->end() ) return NULL;	

	STRING_DB::iterator ite;
	STRING_DB& search_db = lang_ite->second; //検索するデータベースの設定
	
	ite = search_db.find(_key);
	// 普通に探してあったらそれを持ってくる
	if(ite != search_db.end())
	{
		return ite->second.c_str();
	}
	else
	{
		// 選択されている言語に文字列が存在しなかったら
		// 英語から文字を引っ張ってくる
		STRING_DB& default_db = (*m_lang_db)[DEFAULT_LANGUAGE];
		ite = default_db.find(_key);
	
		if (ite != default_db.end())
		{
			return ite->second.c_str();
		}
		else // それでも存在しなかったら空文字列を返す
		{
			return NULL;
		}
	}	
	return NULL;	
}


// データベースの初期化
void TMultiLang::initLangDataBase()
{
	status_t ret;
	app_info info;
	ret = be_app->GetAppInfo(&info);
	ASSERT(ret == B_OK);
	BEntry entry(&info.ref);

	BDirectory dir;
	ret = entry.GetParent(&dir);
	ASSERT(ret == B_OK);

	discoverLangFile(dir,0);
}

// .lngファイルを探しだして、データベースに登録
void TMultiLang::discoverLangFile(BDirectory& _dir,int _level)
{
	BEntry entry;
	while(_dir.GetNextEntry(&entry) == B_OK)
	{
		// ディレクトリだったら再起で呼び出して調べる
		// 4階層以上になったら調べない
		if (entry.IsDirectory() && _level < SEARCH_DEEP_LEVEL) 
		{
			BDirectory dir(&entry);
			discoverLangFile(dir,_level+1);
		}
		else
		{
			char buffer[B_FILE_NAME_LENGTH];
			entry.GetName(buffer);
			
			// 拡張子の部分を指すポインタ
			char* p_extension = buffer+strlen(buffer)-4;
			
			// 拡張子が".lng"かどうか調べる
			if (strcmp(p_extension,LANG_FILE_EXTENSION) == 0) 
			{
				addLangFile(entry);	 // あっていたらファイルの登録		
			}
		}	
	}
}

// .lngファイルの追加
// 引数はそのファイルを示すエントリ
void TMultiLang::addLangFile(BEntry& _entry)
{
	STRING_DB add_db;

	// 言語名を取り出す作業
	char filename[B_FILE_NAME_LENGTH];
	_entry.GetName(filename);

	string lang_name = filename;
	
	// 拡張子の前の部分を切り出して言語名とする
	lang_name = lang_name.substr(0,strlen(filename) - 4); 

	BFile lang_file(&_entry,B_READ_ONLY);
	off_t size;
	lang_file.GetSize(&size); // サイズを取得してバッファに全部ぶちこんでから解析
	
	char* filebuf = new char[size+1]; // ファイルの内容全部
	lang_file.Read(reinterpret_cast<void*>(filebuf),size);	

	istrstream lang_stream(filebuf);

	while (!(lang_stream.eof()) )
	{
		char separators[] = { LANG_FILE_SEPARATOR,'\0' };
		char line[LINE_BUFFER_SIZE];		

		lang_stream.getline(line,LINE_BUFFER_SIZE); // 一行読み込む
		char* line_end = line+strlen(line); // 文字列の終わり
		
		// 読み込んだ一行にセパレータが含まれているかどうか調べる
		if (find(line,line_end,LANG_FILE_SEPARATOR) != line_end) 
		{
			// キーと値に切り分け
			string key = strtok(line,separators);
			string value = strtok(NULL,separators);
		
			// 挿入
			add_db.insert(make_pair(key,value) );
		}
	}
	
	delete[] filebuf;
	
	// 言語データベースに追加
	m_lang_db->insert(make_pair(lang_name,add_db));
}

// インストールされている言語のリストを返す
// メモリは自動的に割り当てられるので、解放は呼び出し側の責任となる
// メモリの割り当てに失敗したらNULLを返す
string* TMultiLang::GetLangListFromFileName()
{
	try
	{
		int size = GetLangListSize();
		string* lang_list = new string[size];

		LANG_DB::iterator ite;
		int i;
		for (ite = m_lang_db->begin(),i=0;ite != m_lang_db->end();++ite,++i)
		{
			lang_list[i] = ite->first;
		}
		return lang_list;
	}
	catch (std::bad_alloc& error)
	{
		return NULL;
	}
}

// インストールされている言語のリストを返す
// これは現在の言語の設定による
// 例えば、日本語があらかじめSetLang等によって指定されていたら日本語名でのリストを返す
// メモリは自動的に割り当てられるので、解放は呼び出し側の責任となる
// メモリの割り当てに失敗したらNULLを返す
string* TMultiLang::GetLangList()
{
	try
	{
		string* lang_origin_list = GetLangListFromFileName(); // 言語のもともとの配列
		ASSERT(lang_origin_list != NULL);

		string* lang_list = NULL; // 返す配列

		size_t size = GetLangListSize();
		lang_list = new string[size];

		for (size_t i = 0;i < size;i++)
		{
			STRING_DB::iterator ite;

			// 言語ファイル名と同じ名前のレコードがあるかどうか調べる
			// あったら、それを配列にいれ、無かったらファイル名を配列にいれる
			// (例:レコードがある場合:日本語,英語
			//	  レコードがない場合:japanese,english)
			ite = m_selected_db->find(lang_origin_list[i]);
			
			if (ite != m_selected_db->end()) // 見つかった
			{
				lang_list[i] = ite->second;	
			}
			else
			{
				lang_list[i] = lang_origin_list[i];
			}
		}
		
		delete[] lang_origin_list;
		return lang_list;			
	}
	catch(std::bad_alloc& error)
	{
		return NULL;
	}
}

// 他の言語名でのリストを返す
string* TMultiLang::GetLangListFromOtherLang(const char* _lang)
{
	try
	{
		string* lang_origin_list = GetLangListFromFileName(); // 言語のもともとの配列
		ASSERT(lang_origin_list != NULL);

		// 渡された言語名がリストの一覧に無かったらNULLを返す
		LANG_DB::iterator lang_ite;
		lang_ite = m_lang_db->find(_lang);
		if (lang_ite == m_lang_db->end() ) return NULL;

		STRING_DB& search_db = lang_ite->second; // 探すデータベースの設定

		string* lang_list = NULL; // 返す配列
		size_t size = GetLangListSize();
		lang_list = new string[size];

		for (size_t i = 0;i < size;i++)
		{
			STRING_DB::iterator ite;

			// 言語ファイル名と同じ名前のレコードがあるかどうか調べる
			// あったら、それを配列にいれ、無かったらファイル名を配列にいれる
			// (例:レコードがある場合:日本語,英語
			//	  レコードがない場合:japanese,english)
			ite = search_db.find(lang_origin_list[i]);
			
			if (ite != search_db.end()) // 見つかった
			{
				lang_list[i] = ite->second;	
			}
			else
			{
				lang_list[i] = lang_origin_list[i];
			}
		}
		
		delete[] lang_origin_list;
		return lang_list;			
	}
	catch(std::bad_alloc& error)
	{
		return NULL;
	}
}

size_t TMultiLang::GetLangListSize()
{
	return m_lang_db->size();
}