/*
 * Copyright (c) 1999, Jesper Hansen. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither name of the company nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//-----------------------------------------------------------------------------
#include <signal.h>
#include <stdio.h>
#include <string.h>
//-------------------------------------
#include <app/Roster.h>
#include <interface/Bitmap.h>
#include <interface/Menu.h>
#include <interface/MenuItem.h>
#include <interface/Screen.h>
#include <storage/Directory.h>
#include <storage/File.h>
#include <storage/Mime.h>
#include <storage/Path.h>
#include <support/Beep.h>
#include <support/Debug.h>
#include <translation/BitmapStream.h>
#include <translation/TranslatorRoster.h>
//-------------------------------------
#include "dle/BButton.h"
#include "dle/BStringView.h"
#include "dle/BTextControl.h"
#include "dle/ProgressBar.h"
#include "gfx/BitmapUtils.h"
#include "misc/Settings.h"
//-------------------------------------
#include "CAWin.h"
#include "CAApp.h"
#include "LoadView.h"
#include "ConfigWin.h"
#include "MiniSlider.h"
#include "HTTPFetch.h"
//-----------------------------------------------------------------------------

#define PALETTE_SIZE 22

static const uint8 gFirePalette[PALETTE_SIZE] =
{
	0, 51, 50, 49, 48, 47, 46, 45, 44, 43, 42,	// black->red
	157, 5*32-3, 4*32-3, 3*32-3, 256-7, 256-6,	// red->yellow
	256-5, 256-4, 256-3, 256-2, 31,				// yellow->white
};

//-----------------------------------------------------------------------------

CAWin::CAWin( const char *title ) :
	dle::Window( BRect(10,40,10+50-1,40+50-1), title, B_TITLED_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL, B_NOT_ZOOMABLE/*|B_NOT_RESIZABLE*/|B_ASYNCHRONOUS_CONTROLS )
{
	fBitmap = NULL;
	fOldBitmap = NULL;
//	fFilePanel = NULL;
	fHTTPFetcher = NULL;
	fConfigWindow = NULL;
	
	fUpdateTimeMap.AddEnum( MSG_UPDATE0S, 0 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE05S, 1000000 / 2 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE1S, 1 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE5S, 5 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE10S, 10 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE15S, 15 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE30S, 30 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE1M, 1*60 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE5M, 5*60 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE10M, 10*60 *1000000 );
	fUpdateTimeMap.AddEnum( MSG_UPDATE15M, 15*60 *1000000 );
	
	for( uint i=0; i<PALETTE_SIZE; i++ )
		fDeltaPaletteRev[gFirePalette[i]] = i;

	// Setup default config values:
	CAApp::Settings()->SetDefaultPoint( CFG_WINPOS, BPoint(10,40) );
	CAApp::Settings()->SetDefaultString( CFG_URL, "http://www.sakoman.com/codycam/codycam.jpg" );
	CAApp::Settings()->SetDefaultInt64( CFG_UPDATETIME, 5*60*1000000 );
	CAApp::Settings()->SetDefaultFloat( CFG_REFVALUE, 0.5f );
	CAApp::Settings()->SetDefaultFloat( CFG_SCALEVALUE, 1.0f );
	CAApp::Settings()->SetDefaultBool( CFG_PLAYALARM, false );
	CAApp::Settings()->SetDefaultBool( CFG_SAVEFRAME, false );
	CAApp::Settings()->SetDefaultString( CFG_SAVEPATH, "/boot/home" );
	CAApp::Settings()->SetDefaultRect( "selectrect", BRect(0,0,-1,-1) );
	CAApp::Settings()->SetDefaultMessage( CFG_URLHISTORY, BMessage('urls') );

	CAApp::Settings()->SetDefaultBool( CFG_RUNFILE, false );
	CAApp::Settings()->SetDefaultString( CFG_RUNFILENAME, "ShowImage {FILE}" );

	CAApp::Settings()->SetDefaultBool( CFG_PROXY, false );
	CAApp::Settings()->SetDefaultString( CFG_PROXY_HOST, "" );
	CAApp::Settings()->SetDefaultInt32( CFG_PROXY_PORT, 8080 );

	fBitmapWidth = 320;
	fBitmapHeight = 240;

	fDeltaBitmap = NULL;
	fNewURL = false;

	fViewMode = VIEW_NORMAL;
	fNormalRootView = NormalGui();
	fFullscreenRootView = FullScreenGui();
	AddObject( fNormalRootView );
//	AddObject( fFullscreenRootView );

	ReadHistroryFromConfig();

	fBitmapView->SetSelectionRect( CAApp::Settings()->GetRect("selectrect") );

	SetStatusMessage( "Init..." );

	MoveTo( CAApp::Settings()->GetPoint(CFG_WINPOS) );
	Show();

	fThreadQuitReq = false;
	fUpdateSem = create_sem( 0, "Fetcher timeout" );
	fThread = spawn_thread( _ImageFetcherThread, "Bitmap fetcher", B_NORMAL_PRIORITY, this );
	resume_thread( fThread );
	
//	new ConfigWin();
}

CAWin::~CAWin()
{
	status_t status;
	
	if( fConfigWindow != NULL )
	{
		fConfigWindow->Lock();
		fConfigWindow->Quit();
	}   

	WriteHistroryToConfig();
	if( fViewMode==VIEW_NORMAL )
		CAApp::Settings()->SetPoint( CFG_WINPOS, Frame().LeftTop() );
	CAApp::Settings()->Write(); // make sure that the config make it to the disk...

	if( fHTTPFetcher ) fHTTPFetcher->KillConnection();

	printf( "Requesting thread quit...\n" );
	fThreadQuitReq = true;
	delete_sem( fUpdateSem );
	kill( fThread, SIGUSR1 ); // get the thread out of Lock()
	wait_for_thread( fThread, &status );
	printf( "Requesting thread quit...ok\n" );
}

//--------------------------------------------------------------------------------

void CAWin::Show()
{
	// Make sure the windows does not open outside the screen:
	if( !BScreen().Frame().Contains(Frame()) )
		MoveTo( 10, 40 );

	dle::Window::Show();
}

//--------------------------------------------------------------------------------

void CAWin::SetStatusMessage( const char *status )
{
	if( LockWithTimeout(100000) == B_OK )
	{
		fStatusView->SetText( status );
		Flush();
		Unlock();
	}
}

void CAWin::AddHistoryItem( const char *url, bool first )
{
//	printf( "CAWin::AddHistoryItem(url:\"%s\" first:%d)\n", url, first );

	// Check if the url is in the history
	int size = fHistoryMenu->CountItems();
	for( int i=0; i<size; i++ )
	{
		BMenuItem *item = fHistoryMenu->ItemAt( i );
		if( strcmp(item->Label(),url) == 0 )
		{
			if( first && i!=0 )
			{
				fHistoryMenu->RemoveItem( i );
				fHistoryMenu->AddItem( item, 0 );
			}
			return;
		}
	}
	
	BMessage *msg = new BMessage( MSG_HISTORY_URL );
	msg->AddString( "url", url );

	fHistoryMenu->AddItem( new BMenuItem(url, msg), first?0:size );
}

void CAWin::ReadHistroryFromConfig()
{
	BMessage histmsg = CAApp::Settings()->GetMessage( CFG_URLHISTORY );

	const char *url;
	for( int i=0; histmsg.FindString("url_history",i,&url)==B_OK; i++ )
		AddHistoryItem( url );
}

void CAWin::WriteHistroryToConfig()
{
	BMessage histmsg( 'urls' );

	int size = fHistoryMenu->CountItems();
	for( int i=0; i<size; i++ )
	{
		BMenuItem *item = fHistoryMenu->ItemAt( i );
		histmsg.AddString( "url_history", item->Label() );
	}

	CAApp::Settings()->SetMessage( CFG_URLHISTORY, histmsg );
}

//--------------------------------------------------------------------------------

bool CAWin::QuitRequested()
{
	be_app->PostMessage( B_QUIT_REQUESTED );
	return true;
}

void CAWin::MessageReceived( BMessage *msg )
{
	switch( msg->what )
	{
		case MSG_NEW_URL:
			CAApp::Settings()->SetString( CFG_URL, fURLTextControl->Text() );
			if( fHTTPFetcher ) fHTTPFetcher->KillConnection();
			fNewURL = true;
			release_sem( fUpdateSem );
			break;
		case MSG_HISTORY_URL:
		{
			const char *url;
			if( msg->FindString("url",&url) == B_OK )
			{
				CAApp::Settings()->SetString( CFG_URL, url );
				if( fHTTPFetcher ) fHTTPFetcher->KillConnection();
				fURLTextControl->SetText( url );
				fNewURL = true;
				release_sem( fUpdateSem );
			}
			break;
		}
		
		case MSG_SETTINGS:
			if( fConfigWindow == NULL )
			{
				fConfigWindow = new ConfigWin( new BMessenger(NULL,this), new BMessage(MSG_CONFIG_CHANGED)  );
			}
			else
			{
				fConfigWindow->Lock();
				if( fConfigWindow->IsMinimized() || fConfigWindow->IsHidden() )
				{
					if( fConfigWindow->IsMinimized() ) fConfigWindow->Minimize( false );
					if( fConfigWindow->IsHidden() ) fConfigWindow->Show();
				}
				if( !fConfigWindow->IsActive() ) fConfigWindow->Activate( true );
				fConfigWindow->Unlock();
			}
			break;
		
		case MSG_ZOOM:
		{
			BRect src, dst;
			bool zoomin;
			SelectionBitmapView *bv = fViewMode==VIEW_NORMAL ? fBitmapView : fFullscreenBitmapView;

			if( bv->GetZoomRect().IsValid() )
			{
				zoomin = false;
				src = bv->GetSelectionRect();
				dst = bv->GetBitmap()->Bounds();
			}
			else
			{
				// Zoom in
				zoomin = true;
				dst = bv->GetSelectionRect();
				if( !dst.IsValid() )
					break;
				src = bv->GetBitmap()->Bounds();
			}

			BRect rect = src;
			for( int i=0; i<=10; i++ )
			{
				rect.left += (dst.left-rect.left)/2.5f;
				rect.right += (dst.right-rect.right)/2.5f;
				rect.top += (dst.top-rect.top)/2.5f;
				rect.bottom += (dst.bottom-rect.bottom)/2.5f;
					
				bv->SetZoomRect( rect );
				snooze( 1000000/50 );
			}
			bv->SetZoomRect( dst );

			if( !zoomin )
				bv->SetZoomRect( BRect(0,0,-1,-1) );

			break;
		}
		
		case MSG_FULLSCREEN:
		{
			DisableUpdates();
			if( fViewMode == VIEW_NORMAL )
			{
				CAApp::Settings()->SetPoint( CFG_WINPOS, Frame().LeftTop() );

				fFullscreenBitmapView->SetSelectionRect( fBitmapView->GetSelectionRect() );
				fFullscreenBitmapView->SetZoomRect( fBitmapView->GetZoomRect() );

				RemoveObject( fNormalRootView );
				AddObject( fFullscreenRootView );
				MoveTo( BPoint(0,0) );
				ResizeTo( BScreen().Frame().Width(), BScreen().Frame().Height() );
				fViewMode = VIEW_FULLSCREEN;
				Sync();
			}
			else
			{
				MoveTo( CAApp::Settings()->GetPoint(CFG_WINPOS) );

				fBitmapView->SetSelectionRect( fFullscreenBitmapView->GetSelectionRect() );
				fBitmapView->SetZoomRect( fFullscreenBitmapView->GetZoomRect() );
				RemoveObject( fFullscreenRootView );
				AddObject( fNormalRootView );
				fViewMode = VIEW_NORMAL;
				Sync();
			}
			EnableUpdates();
			break;
		}

		case MSG_UPDATE0S:
		case MSG_UPDATE05S:
		case MSG_UPDATE1S:
		case MSG_UPDATE5S:
		case MSG_UPDATE15S:
		case MSG_UPDATE10S:
		case MSG_UPDATE30S:
		case MSG_UPDATE1M:
		case MSG_UPDATE5M:
		case MSG_UPDATE10M:
		case MSG_UPDATE15M:
		{
			bigtime_t updatetime;
			status_t status = fUpdateTimeMap.GetVal( msg->what, &updatetime );
			if( status == B_OK )
			{
				CAApp::Settings()->SetInt64( CFG_UPDATETIME, updatetime );
				if( fHTTPFetcher ) fHTTPFetcher->KillConnection();
				release_sem( fUpdateSem );
			}
			break;
		}
	
		case MSG_DOWNLOAD_PROGRESS:
		{
			int32 length=-1;
			int32 position=-1;
			char string[32];

			msg->FindInt32( "length", &length );
			msg->FindInt32( "position", &position );

			if( length>=0 && position>=0 )
			{
				sprintf( string, "%ld/%ld", position, length );
				fProgressBar->SetProgress( float(position)/float(length) );
			}
			else if( position>=0 )
				sprintf( string, "%ld", position );
			else
				sprintf( string, "got data" );

			SetStatusMessage( string );
			
			break;
		}

		case MSG_NEWBITMAP:
			if( fBitmap )
			{
				if( fBitmap->Bounds().IntegerWidth()+1 != fBitmapWidth ||
					fBitmap->Bounds().IntegerHeight()+1 != fBitmapHeight )
				{
					fBitmapWidth = fBitmap->Bounds().IntegerWidth()+1;
					fBitmapHeight = fBitmap->Bounds().IntegerHeight()+1;

					fBitmapView->ForceWidth( fBitmapWidth );
					fBitmapView->ForceHeight( fBitmapHeight );
					fBitmapView->SetZoomRect( BRect(0,0,-1,-1) );
					
					fFullscreenBitmapView->SetZoomRect( BRect(0,0,-1,-1) );

					fDeltaBitmapView->ForceWidth( fBitmapWidth/2 );
					fDeltaBitmapView->ForceHeight( fBitmapHeight/2 );
					
					fBitmapWidth = fBitmap->Bounds().IntegerWidth()+1;
					fBitmapHeight = fBitmap->Bounds().IntegerHeight()+1;
					
					ReLayout();
				}
			}
			fBitmapView->SetBitmap( fBitmap );
			fFullscreenBitmapView->SetBitmap( fBitmap );
			fDeltaBitmapView->SetBitmap( fDeltaBitmap );
			Flush();
			break;
			
		case MSG_DELTASLIDE:
		{
			int32 value;
			if( msg->FindInt32("be:value", &value) >= B_OK )
			{
				float refvalue = float(value) / 256.0f;
				float scalevalue = CAApp::Settings()->GetFloat( CFG_SCALEVALUE );

				CAApp::Settings()->SetFloat( CFG_REFVALUE, refvalue*scalevalue );
				fDeltaLoadView->SetRefValue( refvalue*scalevalue );
			}
			break;
		}
		
		case MSG_SCALESLIDE:
		{
			int32 value;
			if( msg->FindInt32("be:value", &value) >= B_OK )
			{
				float scalevalue = float(value) / 256.0f;
				float refvalue = CAApp::Settings()->GetFloat( CFG_REFVALUE );

				CAApp::Settings()->SetFloat( CFG_SCALEVALUE, scalevalue );
				fDeltaLoadView->SetScale( 1.0f/scalevalue );
				
				fDeltaSlider->SetValue( (refvalue/scalevalue)*256 );
			}
			break;
		}
		
		// Possibly a drag'n'drop message
		case B_SIMPLE_DATA:
		{
			// Net+ is kind enough to include the URL in the first message, so that we don't have to go through the type negoatiotion...
			const char *url;
			if( msg->FindString("be:url",&url) == B_OK )
			{
				CAApp::Settings()->SetString( CFG_URL, url );
				if( fHTTPFetcher ) fHTTPFetcher->KillConnection();
				fURLTextControl->SetText( url );
				fNewURL = true;
				release_sem( fUpdateSem );
			}
			else
			{
				// Maybe a view can make sense of this...
				BWindow::MessageReceived( msg );
			}
			break;
		}

		default:
//			msg->PrintToStream();
			BWindow::MessageReceived( msg );
	}
}

//--------------------------------------------------------------------------------

int32 CAWin::_ImageFetcherThread( void *data )
{
	CAWin *win = (CAWin*)data;
	win->ImageFetcherThread();
	return 0;
}

// TODO: split this function...
void CAWin::ImageFetcherThread()
{
	printf( "Worker started\n" );
	
	while( !fThreadQuitReq )
	{
		bigtime_t nextframetime = system_time() + CAApp::Settings()->GetInt64( CFG_UPDATETIME ); // not totaly accurate, but close enough.
	
		BString url = CAApp::Settings()->GetString( CFG_URL );

		if( url.Length() == 0 )
		{
			SetStatusMessage( "Ok" );
		}
		else
		{
			SetStatusMessage( "Fetching bitmap..." );
	
			char strtime[64];
			time_t timet = time(NULL);
			strftime( strtime, sizeof(strtime), "%y%m%d-%H%M%S", localtime(&timet) );

			URL spliturl( url );
			if( spliturl.Host()=="" || spliturl.Request()=="" )
			{
				SetStatusMessage( "Invalid url..." );
			}
			else
			{
				if( fHTTPFetcher==NULL || fNewURL )
				{
					delete fHTTPFetcher;

					if( CAApp::Settings()->GetBool(CFG_PROXY) )
					{
						fHTTPFetcher = new HTTPFetch(
							spliturl.Host().String(), spliturl.Port()>0?spliturl.Port():80,
							CAApp::Settings()->GetString(CFG_PROXY_HOST), CAApp::Settings()->GetInt32(CFG_PROXY_PORT) );
					}
					else
					{
						fHTTPFetcher = new HTTPFetch( spliturl.Host().String(), spliturl.Port()>0?spliturl.Port():80 );
					}
#if 1
					if( !Lock() ) return;
					fProgressBar->SetProgress( 0 );
					fProgressBar->SetProgressColor( 152, 152, 255 );
					Flush();
					Unlock();
#endif
					fHTTPFetcher->ProcessMessage( new BMessage(MSG_DOWNLOAD_PROGRESS), new BMessenger(this) );
				}

	
				BMallocIO data;
				BString mimetype;
				status_t status = fHTTPFetcher->Get( spliturl.Request().String(), &data, &mimetype );
				if( status < 0 )
				{
					SetStatusMessage( "Could not download file..." );
				}
				else if( status == 304 )
				{
					SetStatusMessage( "Not changed" );
				}
				else
				{
					BBitmapStream stream;
					BTranslatorRoster *roster = BTranslatorRoster::Default();
//					if( roster->Translate(&data, NULL, NULL, &stream, B_TRANSLATOR_BITMAP,0,mimetype.String()) != B_OK )
					if( roster->Translate(&data, NULL, NULL, &stream, B_TRANSLATOR_BITMAP,0,NULL) != B_OK )
					{
						SetStatusMessage( "Count not decode image..." );
					}
					else
					{
						// Whey! the url worked! if if was a new url, add it to the history:
						if( fNewURL )
						{
							AddHistoryItem( url.String(), true );
							fNewURL = false;
						}
	
						BBitmap *bitmap = NULL;
						stream.DetachBitmap( &bitmap );
					
						SetStatusMessage( "Calculating delta..." );

						bool alarm=false;
						if( !Lock() ) return;
//						if( LockWithTimeout(1000000) >= B_OK )
						{
#ifdef OLD_DIFF
							fOldBitmap = NULL;
							fOldBitmap = fBitmap;
#else
							if( fOldBitmap == NULL )
								fOldBitmap = fBitmap;
#endif
							fBitmap = bitmap;
							alarm = CalcDeltaBitmap();
							PostMessage( MSG_NEWBITMAP );
							Unlock();
						}
	
						if( alarm )
						{
							SetStatusMessage( "ALARM!" );
#ifndef OLD_DIFF
							fOldBitmap = NULL;
							fOldBitmap = fBitmap;
#endif
	
							if( CAApp::Settings()->GetBool(CFG_PLAYALARM) )
								system_beep( "Coffee Alarm" );
	
							BPath savefilepath;
							if( CAApp::Settings()->GetBool(CFG_SAVEFRAME) )
							{
								const char *savepath = CAApp::Settings()->GetString( CFG_SAVEPATH );

								int subframe = 0;
								while( 1 )
								{
									BString filename;
									filename << GetBaseName(url.String()) << "_" << strtime;
									if( subframe ) filename << "[" << subframe << "]";
									filename << "." << GetExt(url.String());
								
									subframe++;
								
									savefilepath.SetTo( savepath, filename.String() );
									BFile file( savefilepath.Path(), B_WRITE_ONLY|B_CREATE_FILE|B_FAIL_IF_EXISTS );
									if( file.InitCheck()==B_FILE_EXISTS && subframe<1000 )
										continue;

									file.Write( data.Buffer(), data.BufferLength() );
									update_mime_info( savefilepath.Path(), false, false, false );
									break;
								}
							}

							if( CAApp::Settings()->GetBool(CFG_RUNFILE) )
							{
								BString command = CAApp::Settings()->GetString( CFG_RUNFILENAME );
								BString file = " ";
								if( CAApp::Settings()->GetBool(CFG_SAVEFRAME) )
									file << "\"" << savefilepath.Path() << "\"";
								command.ReplaceAll( "{FILE}", file.String() );
								
								BEntry sh_entry( "/bin/sh" );
								entry_ref sh_eref;
								if( sh_entry.GetRef(&sh_eref) < 0 )
								{
									SetStatusMessage( "Could not find /bin/sh" );
								}
								else
								{
									const char *argv[2] = { "-c", command.String() };
									be_roster->Launch( &sh_eref, 2, argv );
								}
							}
						}
						SetStatusMessage( "Ok" );
					}
				}
			}
		}

#if 1
		if( !Lock() ) return;
		fProgressBar->SetProgressColor( 255, 152, 152 );
		fProgressBar->SetProgress( 1.0f );
		Flush();
		Unlock();

		bigtime_t starttime = system_time();

		bigtime_t steptime = (nextframetime-starttime)/PROGRESS_BAR_WIDTH;
		if( steptime < 25000 ) steptime = 25000;
		
		while( true )
		{
			bigtime_t nowtime = system_time();
			if( nowtime > nextframetime ) break;
			if( !Lock() ) return;
			fProgressBar->SetProgress( 1.0f - float(nowtime-starttime)/float(nextframetime-starttime) );
			Flush();
//			printf( "%f\n", float(nowtime-starttime)/float(nextframetime-starttime) );
			Unlock();
			status_t status = acquire_sem_etc( fUpdateSem, 1, B_ABSOLUTE_TIMEOUT, min_c(nextframetime,nowtime+steptime) );
			if( status >= 0 ) break;
//			printf( "ACQUIRE: %d\n", status );
		}
		if( !Lock() ) return;
		fProgressBar->SetProgress( 0.0f );
		Flush();
		Unlock();
#else
		acquire_sem_etc( fUpdateSem, 1, B_ABSOLUTE_TIMEOUT, nextframetime );
#endif
	}
}

BString CAWin::GetBaseName( const char *path )
{
	const char *slash = strrchr( path, '/' );
	slash = slash?slash+1:path;

	const char *dot = strrchr( path, '.' );
	if( !dot ) dot = path+strlen(path);

	return BString( slash, dot-slash );
}

BString CAWin::GetExt( const char *path )
{
	const char *slash = strrchr( path, '/' );
	const char *dot = strrchr( path, '.' );
	if( dot && dot>slash )
		return BString( dot+1 );
	return BString( "" );
}


//-----------------------------------------------------------------------------

bool CAWin::CalcDeltaBitmap()
{
	if( fBitmap==NULL )
		return false;
	if( fBitmap->ColorSpace()!=B_RGB32 && fBitmap->ColorSpace()!=B_RGBA32 )
		return false;

	int width = fBitmap->Bounds().IntegerWidth()+1;
	int height= fBitmap->Bounds().IntegerHeight()+1;

	if( fOldBitmap==NULL ||
		fOldBitmap->Bounds().IntegerWidth()+1 != width ||
		fOldBitmap->Bounds().IntegerHeight()+1 != height )
	{
		fOldBitmap = NULL;
		fOldBitmap = new BBitmap( BRect(0,0,width-1,height-1), B_RGB32 );
		ASSERT( fOldBitmap != NULL );
		ASSERT( fOldBitmap->IsValid() );
		memset( fOldBitmap->Bits(), 0, fOldBitmap->BitsLength() );
	}

	if( fDeltaBitmap==NULL ||
		fDeltaBitmap->Bounds().IntegerWidth()+1 != width/2 ||
		fDeltaBitmap->Bounds().IntegerHeight()+1 != height/2 )
	{
		fDeltaBitmap = NULL;
		fDeltaBitmap = new BBitmap( BRect(0,0,width/2-1,height/2-1), B_CMAP8 );
		ASSERT( fDeltaBitmap != NULL );
		ASSERT( fDeltaBitmap->IsValid() );
	}
	memset( fDeltaBitmap->Bits(), 0, fDeltaBitmap->BitsLength() );

	BRect roi = fBitmapView->GetSelectionRect();
	int left = 0;
	int right = width-1;
	int top = 0;
	int bottom = height-1;
	if( roi.IsValid() )
	{
		left = int( (roi.left*width)/(fBitmapView->Bounds().IntegerWidth()+1) );
		right = int( (roi.right*width)/(fBitmapView->Bounds().IntegerWidth()+1) );
		top = int( (roi.top*height)/(fBitmapView->Bounds().IntegerHeight()+1) );
		bottom = int( (roi.bottom*height)/(fBitmapView->Bounds().IntegerHeight()+1) );
//		printf( "%d %d %d %d\n", left,right,top,bottom );
	}

	int globaldiff = 0;
	
	int deltawidth = fDeltaBitmap->Bounds().IntegerWidth()+1;
	int deltaheight = fDeltaBitmap->Bounds().IntegerHeight()+1;

	float scalevalue = CAApp::Settings()->GetFloat( CFG_SCALEVALUE );
	int mul = int( ((PALETTE_SIZE-1)*(1.0f/scalevalue))/3.0f );

	for( int iy=0; iy<height; iy++ )
	{
		bitmap_rgb *bits = (bitmap_rgb*)(((uint8*)fBitmap->Bits())+iy*fBitmap->BytesPerRow());
		bitmap_rgb *oldbits = (bitmap_rgb*)(((uint8*)fOldBitmap->Bits())+iy*fOldBitmap->BytesPerRow());
		for( int ix=0; ix<width; ix++ )
		{
			bitmap_rgb col = bits[ix];
			bitmap_rgb oldcol = oldbits[ix];
#if 0
			int diff = (col.red-oldcol.red)*(col.red-oldcol.red)*299 +
				(col.green-oldcol.green)*(col.green-oldcol.green)*587 +
				(col.blue-oldcol.blue)*(col.blue-oldcol.blue)*114;
//			diff = int(pow(sqrt(float(diff)/1000.0f)/256.0f,0.4f)*31.0f);
			diff = int(float(diff)/(65536.0f*1000.0f)*31.0f);
#else
			int rdiff = int(col.red) - int(oldcol.red);
			int gdiff = int(col.green) - int(oldcol.green);
			int bdiff = int(col.blue) - int(oldcol.blue);

			int diff = abs(rdiff)+abs(gdiff)+abs(bdiff);
			int renderdiff = (diff*mul) >> 8; if(renderdiff>(PALETTE_SIZE-1)) renderdiff=PALETTE_SIZE-1;
//			diff = diff*31 / (256*3);
#endif
			
			int dstx = (ix*deltawidth)/width;
			int dsty = (iy*deltaheight)/height;
			uint8 *dstbits = ((uint8*)fDeltaBitmap->Bits())+dsty*fDeltaBitmap->BytesPerRow()+dstx;
			if( renderdiff > fDeltaPaletteRev[*dstbits] )
				*dstbits = gFirePalette[renderdiff];
				
			if( ix>=left && ix<=right && iy>=top && iy<=bottom )
				globaldiff += diff;
		}
	}

//	float load = (float(globaldiff)/31.0f)/(((right-left)+1)*((bottom-top)+1));
	float load = (float(globaldiff)/(256.0f*3.0f))/(((right-left)+1)*((bottom-top)+1));
	load = pow( load, 0.65f );
	fDeltaLoadView->AddValue( load );
	
	return load > CAApp::Settings()->GetFloat( CFG_REFVALUE );
}



