/*
 * Copyright (c) 1999, Jesper Hansen. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither name of the company nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//-----------------------------------------------------------------------------
//-------------------------------------
#include <interface/Window.h>
#include <storage/Entry.h>
#include <support/Locker.h>
#include <support/String.h>
class BSlider;
//-------------------------------------
#include "dle/Window.h"
namespace dle { class BSlider; class BTextControl; class BStringView; class BCheckBox; class BButton; class ProgressBar; }
#include "misc/List.h"
//-------------------------------------
#include "BitmapView.h"

class MiniSlider;
class LoadView;
class MiniSlider;
class HTTPFetch;
class ConfigWin;
//-----------------------------------------------------------------------------

#define CFG_WINPOS		"winpos"
#define CFG_URL			"url"
#define CFG_URLHISTORY	"urlhistory"
#define CFG_PLAYALARM	"playalarm:enable"
#define CFG_UPDATETIME	"updatetime"
#define CFG_REFVALUE	"refvalue"
#define CFG_SCALEVALUE	"scalevalue"
#define CFG_SAVEFRAME	"save:enable"
#define CFG_SAVEPATH	"save:savepath"
#define CFG_RUNFILE		"runfile:enable"
#define CFG_RUNFILENAME	"runfile:filename"
#define CFG_PROXY		"proxy:enable"
#define CFG_PROXY_HOST	"proxy:host"
#define CFG_PROXY_PORT	"proxy:port"

//-----------------------------------------------------------------------------

enum
{
	MSG_NEWBITMAP		= 'newb',
	MSG_DELTASLIDE		= 'dlts',
	MSG_SCALESLIDE		= 'scsl',
	MSG_FULLSCREEN		= 'fuls',
	MSG_NEW_URL			= 'nurl',
	MSG_HISTORY_URL 	= 'hurl',
	MSG_ZOOM			= 'zoom',
	MSG_SETTINGS		= 'conf',
	MSG_CONFIG_CHANGED	= 'cnfc',

	MSG_DOWNLOAD_PROGRESS = 'dlpr',

	// FIXME: use one message for these, and then have a message member that specifies the timeout...
	MSG_UPDATE0S		= 'u0s ',
	MSG_UPDATE05S		= 'u05s',
	MSG_UPDATE1S		= 'u1s ',
	MSG_UPDATE5S		= 'u5s ',
	MSG_UPDATE15S		= 'u15s',
	MSG_UPDATE10S		= 'u10s',
	MSG_UPDATE30S		= 'u30s',
	MSG_UPDATE1M		= 'u1m ',
	MSG_UPDATE5M		= 'u5m ',
	MSG_UPDATE10M		= 'u10m',
	MSG_UPDATE15M		= 'u15m',
	
	MSG_PROGRESS_CHANGE	= 'prch',
};

#define PROGRESS_BAR_WIDTH 64

//-----------------------------------------------------------------------------

template<class T>
class EnumMapping
{
public:
	EnumMapping() {};
	~EnumMapping()
	{
		for( int i=0; i<fKeyList.CountItems(); i++ )
			delete fKeyList.ItemAtFast( i );
	}
	
	void AddEnum( int e_num, T val )
	{
		keyval *k = new keyval( e_num, val );
		fKeyList.AddItem( k );
	}

	status_t GetVal( int e_num, T *val )
	{
		assert( val != NULL );

		for( int i=0; i<fKeyList.CountItems(); i++ )
		{
			keyval *k = fKeyList.ItemAtFast( i );
			if( k->key == e_num )
			{
				*val = k->val;
				return B_OK;
			}
		}
		return B_ERROR;
	}

	status_t GetEnum( T val, int *e_num );

	status_t GetNearestEnum( T val, int *e_num )
	{
		assert( e_num != NULL );

		if( fKeyList.CountItems() == 0 )
			return B_ERROR;

		keyval *best = fKeyList.ItemAtFast( 0 );
		T best_dist = best->val - val;
		if( best_dist < 0 ) best_dist = -best_dist;
		
		for( int i=1; i<fKeyList.CountItems(); i++ )
		{
			keyval *k = fKeyList.ItemAtFast( i );
			T dist = k->val - val;
			if( dist < 0 ) dist = -dist;
			if( dist < best_dist )
			{
				best = k;
				best_dist = dist;
			}
		}

		*e_num = best->key;
		return B_OK;
	}

private:
	struct keyval { keyval(int k,T v) : key(k), val(v) {}; int key; T val;  };
	damn::List<keyval>	fKeyList;
};

//-----------------------------------------------------------------------------

class CAWin : public dle::Window
{
public:
	CAWin( const char *title );
	~CAWin();

	void Show();

protected:
	bool QuitRequested();
	void MessageReceived( BMessage *msg );

private:
	static int32 _ImageFetcherThread( void *data );
	void ImageFetcherThread();

	dle::Object *NormalGui();
	dle::Object *GuiMenu();
	dle::Object *GuiAddress();
	dle::Object *GuiMain();
	dle::Object *GuiConfig();
	dle::Object *GuiStatusBar();
	
	dle::Object *FullScreenGui();

	bool CalcDeltaBitmap();
	
	void AddHistoryItem( const char *url, bool first=false );
	void ReadHistroryFromConfig();
	void WriteHistroryToConfig();
	
	void SetStatusMessage( const char *status );
	
	BString GetBaseName( const char *path );
	BString GetExt( const char *path );

	volatile bool			fThreadQuitReq;
	thread_id				fThread;
	sem_id					fUpdateSem;
	HTTPFetch				*fHTTPFetcher;
	
	EnumMapping<bigtime_t> fUpdateTimeMap;
	
	BView					*fRootView;
	
	bool					fNewURL;

	int						fBitmapWidth;
	int						fBitmapHeight;

	damn::AutoPtr<BBitmap>	fOldBitmap;
	damn::AutoPtr<BBitmap>	fBitmap;
	damn::AutoPtr<BBitmap>	fDeltaBitmap;
	
	uint8					fDeltaPaletteRev[256];
	
	ConfigWin				*fConfigWindow;
	
	typedef enum viewmode { VIEW_NORMAL, VIEW_FULLSCREEN };
	viewmode				fViewMode;

	// Views - Normal
	dle::Object				*fNormalRootView;
	BMenu					*fHistoryMenu;
	SelectionBitmapView		*fBitmapView;
	SelectionBitmapView		*fDeltaBitmapView;
	LoadView				*fDeltaLoadView;
	MiniSlider				*fDeltaSlider;
	MiniSlider				*fScaleSlider;
	dle::BTextControl		*fURLTextControl;
	dle::BCheckBox			*fAlarmCheckbox;
	dle::BCheckBox			*fSaveCheckbox;
	dle::BTextControl		*fSavePathTextControl;
	dle::BButton			*fSavePathBrowse;
	dle::BStringView		*fStatusView;
	dle::ProgressBar		*fProgressBar;
	// Views - FullScreen
	dle::Object				*fFullscreenRootView;
	SelectionBitmapView		*fFullscreenBitmapView;
};

//-----------------------------------------------------------------------------
