/*
 * Copyright (c) 1999, Jesper Hansen. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither name of the company nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//-----------------------------------------------------------------------------
#include <assert.h>
#include <errno.h>
#include <netdb.h>
#include <sys/socket.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
//-------------------------------------
#include <app/Message.h>
#include <app/Messenger.h>
#include <support/Autolock.h>
#include <support/String.h>
//-------------------------------------
#include "HTTPFetch.h"
//-----------------------------------------------------------------------------

//#define PRINT_TCP(arglist) printf arglist
#define PRINT_TCP(arglist) 

//#define PRINT_HTTP(arglist) printf arglist
#define PRINT_HTTP(arglist) 

//#define PRINT_CTRL(arglist) printf arglist
#define PRINT_CTRL(arglist) 

//-----------------------------------------------------------------------------

URL::URL( const BString &url )
{
	fProtocol = url;
	int protocolindex = fProtocol.FindFirst( "://" );
	if( protocolindex >= 0 )
	{
		fHost.SetTo( fProtocol.String()+protocolindex+3 );
		fProtocol.Truncate( protocolindex );
	}
	else
	{
		fHost = fProtocol;
		fProtocol = "";
	}

//	fHost = url;
	int requestindex = fHost.FindFirst( '/' );
	if( requestindex >= 0 )
	{
		fRequest.SetTo( fHost.String()+requestindex );
		fHost.Truncate( requestindex );

		int portindex = fHost.FindFirst( ':' );
		if( portindex >= 0 )
		{
			fPort = atoi( fHost.String()+portindex+1 );
			fHost.Truncate( portindex );
		}
		else
		{
			fPort = 0;
		}
	}
	
#if 0
	printf( "URL:Protocol... \"%s\"\n", fProtocol.String() );
	printf( "URL:Host....... \"%s\"\n", fHost.String() );
	printf( "URL:Port....... %d\n", fPort );
	printf( "URL:Req........ \"%s\"\n", fRequest.String() );
#endif
}

//-----------------------------------------------------------------------------

class SocketHelper
{
public:
	SocketHelper( int socket );
	~SocketHelper();
	
	bool Send( const BString &string ) { return Send(string.String(),string.Length()); }
	bool Send( const char *string ) { return Send(string,strlen(string)); }
	bool Send( const void *buffer, size_t length );
	
	bool ReadString( char *string, size_t maxlength );
	bool ReadString( BString *string );

	size_t Read( void *buffer, size_t length );

private:
	size_t ReadWithTimeout( void *buffer, size_t length );

	int		fSocket;

    enum { fDataSize=4096 };
    uint8	fData[fDataSize];
    int		fDataPosL, fDataPosH;
};

SocketHelper::SocketHelper( int socket )
{
	fSocket = socket;
    fDataPosL = 0;
    fDataPosH = 0;

	int nonblock = 1;
	status_t status = setsockopt( fSocket, SOL_SOCKET, SO_NONBLOCK, &nonblock, sizeof(nonblock) );
	if( status < B_NO_ERROR )
	{
		fprintf( stderr, "SocketHelper::SocketHelper(): could not set socket to non-blocking\n" );
	}
}

SocketHelper::~SocketHelper()
{
}

bool SocketHelper::Send( const void *buffer, size_t length )
{
    assert( buffer != NULL );
    assert( length > 0 );

	while( length > 0 )
	{
		int sendlen = send( fSocket, (const char*)buffer, length, 0 );

		if( sendlen < 0 )
		{
			fprintf( stderr, "SocketHelper::Send(): send failed\n" );
			return false;
		}

		buffer = ((const char*)buffer) + sendlen;
		length -= sendlen;
    }

	return true;
}

bool SocketHelper::ReadString( char *string, size_t maxlength )
{
    assert( string != NULL );
    assert( maxlength > 1 );

    int readlen = 0;
    while( 1 )
    {
        while( fDataPosL!=fDataPosH )
        {
            char data = fData[fDataPosL];
            if( data == '\r' )
                ;
            else if( data == '\n' )
            {
                fDataPosL++;
                string[readlen++] = 0;
                return true;
            }
            else
            {
                if( (size_t)readlen == maxlength-1 )
                {
                    string[readlen++] = 0;
					return true;
                }
                string[readlen++] = data;
            }
            fDataPosL++;
        }
        
        // Fill the buffer with data:
        status_t recvstatus = 0;

		recvstatus = ReadWithTimeout( fData, fDataSize-1 );
		if( recvstatus < B_OK  )
		{
			if( errno == B_WOULD_BLOCK )
			{
				recvstatus = 0;
			}
			else
			{
				fprintf( stderr, "SocketHelper::ReadString(): could not get data in non-blocked mode\n" );
				return false;
			}
		}
		else if( recvstatus == 0 )
		{
			fprintf( stderr, "SocketHelper::ReadString(): got 0 bytes in non-blocked mode\n" );
			return false;
		}

        fDataPosL = 0;
        fDataPosH = recvstatus;
    }
}

bool SocketHelper::ReadString( BString *string )
{
	bool ret = ReadString( string->LockBuffer(4096), 4096 );
	PRINT_TCP(( "RECV: [%s]\n", string->String() ));
	string->UnlockBuffer( ret?-1:0 );
	return ret;
}

#if 0
size_t SocketHelper::Read( void *buffer, size_t length )
{
	size_t recvlength = 0;

	// First, get the data from the queue:
//	printf( "READ: %d, buffer:%d\n", length, fDataPosH-fDataPosL );
	if( fDataPosL!=fDataPosH )
	{
		size_t datasize = fDataPosH-fDataPosL;
		if( datasize > length )
			datasize = length;
		memcpy( buffer, fData+fDataPosL, datasize );
		buffer = ((char*)buffer) + datasize;
		recvlength += datasize;
		fDataPosL += datasize;
	}
	
	while( length != recvlength )
	{
//		printf( "READ: need:%d, got:%d\n", length, recvlength );
		// recevive network data
		status_t recvstatus = ReadWithTimeout( buffer, length-recvlength );
//		printf( "READ: ret:%d\n", recvstatus );
		if( recvstatus < B_OK )
		{
			fprintf( stderr, "SocketHelper::Read(): recv failed\n" );
			return recvlength;
		}
		if( recvstatus == 0 )
		{
			fprintf( stderr, "SocketHelper::Read(): got 0 bytes\n" );
			return recvlength;
		}

		buffer = ((char*)buffer) + recvstatus;
		recvlength += recvstatus;
	}
	return recvlength;
}
#else
size_t SocketHelper::Read( void *buffer, size_t length )
{
	// First, get the data from the queue:
	if( fDataPosL!=fDataPosH )
	{
		size_t datasize = fDataPosH-fDataPosL;
		if( datasize > length )
			datasize = length;
		memcpy( buffer, fData+fDataPosL, datasize );
		buffer = ((char*)buffer) + datasize;
		fDataPosL += datasize;

		return datasize;
	}
	
	status_t recvstatus = ReadWithTimeout( buffer, length );
	if( recvstatus < B_OK )
	{
		fprintf( stderr, "SocketHelper::Read(): recv failed\n" );
		return 0;
	}
	if( recvstatus == 0 )
	{
		fprintf( stderr, "SocketHelper::Read(): got 0 bytes\n" );
		return 0;
	}

	return recvstatus;
}
#endif

size_t SocketHelper::ReadWithTimeout( void *buffer, size_t length )
{
	fd_set read_bits;
	status_t result;
	
	FD_ZERO( &read_bits );
	FD_SET( fSocket, &read_bits );
	
	struct timeval timeout;
	timeout.tv_sec = 30;
	timeout.tv_usec = 0;
	result = select( fSocket+1, &read_bits, NULL, NULL, &timeout );
	PRINT_TCP(( "SocketHelper::ReadWithTimeout(): select:%d FD_ISSET:%d\n", result, FD_ISSET(fSocket,&read_bits) ));

	if( result <= 0 )
	{
		PRINT_TCP(( "SocketHelper::ReadWithTimeout(): select timed out...\n" ));
		return 0;
	}

	result = recv( fSocket, buffer, length, 0 );
	PRINT_TCP(( "SocketHelper::ReadWithTimeout(): read:%d errno:\"%s\"\n", result, strerror(errno) ));
	
	return result;

//	return read( fSocket, buffer, length );
}


//-----------------------------------------------------------------------------

HTTPFetch::HTTPFetch( const char *host, int port, const char *proxy_host, int proxy_port ) :
	fLock( "HTTPFetch lock", true )
{
	assert( host != NULL );

	fHost = host;
	fPort = port;

	fProxyUse = proxy_host!=NULL;
	fProxyHost = proxy_host ? proxy_host : "";
	fProxyPort = proxy_port;

	fSocket = -1;
	fKillConnection = false;

	fMessage = NULL;
	fMessenger = NULL;
}

HTTPFetch::~HTTPFetch()
{
	KillConnection();

	PRINT_CTRL(( "HTTPFetch::~HTTPFetch(): waiting for lock...\n" ));
	fLock.Lock();
	PRINT_CTRL(( "HTTPFetch::~HTTPFetch(): got lock\n" ));
	
	delete fMessage;
	delete fMessenger;

}

void HTTPFetch::ProcessMessage( BMessage *message, BMessenger *messenger )
{
	assert( message != NULL );
	assert( messenger != NULL );
	
	fMessage = message;
	fMessenger = messenger;
}

void HTTPFetch::KillConnection()
{
	if( fSocket >= 0 )
	{
		close( fSocket );
		fSocket = -1;
	}
	fKillConnection = true;
}


status_t HTTPFetch::Get( const char *file, BDataIO *io, BString *mimetype )
{
	bool keepalive = false;
	size_t content_length = 0;
	size_t received = 0;
	BString str;

	fKillConnection = false;
	BAutolock lock( fLock );
	
	BString last_lastmodified = fLastModified;

retry:
	bool mustretry = fSocket>=0;

	// NOTE: do not use "this" before the next lock, out class might be deleted!!!
	if( fSocket < 0 )
	{
		PRINT_CTRL(( "Looking up host\n" ));
	    struct hostent *hostent = gethostbyname( fProxyUse ? fProxyHost.String() : fHost.String() );
	    if( hostent == NULL )
		{
			fprintf( stderr, "HTTPFetch::Get(): could not look up host\n" );
			return ERR_UNKNOWN_HOST;
		}

		if( fKillConnection )
			return B_INTERRUPTED;

		struct sockaddr_in sockaddr;
		memcpy( &sockaddr.sin_addr, hostent->h_addr, hostent->h_length );
		sockaddr.sin_port = htons( fProxyUse ? fProxyPort : fPort );
		sockaddr.sin_family = AF_INET;
	
		PRINT_CTRL(( "Creating socket\n" ));
		fSocket = socket( AF_INET, SOCK_STREAM, 0 );
		if( fSocket < 0 )
		{
			fprintf( stderr, "HTTPFetch::Get(): get stuff: could not create socket\n" );
			return B_ERROR;
		}

		if( fKillConnection )
			return B_INTERRUPTED;

		PRINT_CTRL(( "Connecting\n" ));
		int status = connect( fSocket, (struct sockaddr*)&sockaddr, sizeof(sockaddr) );
		if( status < 0 )
		{
			close( fSocket );
			fSocket = -1;
			fprintf( stderr, "HTTPFetch::Get(): get stuff: could not connect to server\n" );
			return B_ERROR;
		}

		PRINT_CTRL(( "Connected\n" ));
	}
	
	SocketHelper sockhelp( fSocket );

	BString req;
	if( fProxyUse )
		req << "GET " << fHost << ":" << fPort << "/" << file << " HTTP/1.1\n";
	else
		req << "GET " << file << " HTTP/1.1\n";
	req << "Accept: */*\n";
	req << "User-Agent: CoffeeAlarm/0.1\n";
	if( fLastModified != "" )
		req << "If-Modified-Since: " << fLastModified << "\n";
	req << "Host: " << fHost << "\n";
	req << "Connection: Keep-Alive\n";
	req << "\n";

	PRINT_CTRL(( "Sending request\n" ));
	if( !sockhelp.Send(req) )
	{
		fprintf( stderr, "HTTPFetch::Get(): sendreq: could not send request\n" );
		goto error;
	}

	// get reply
	PRINT_CTRL(( "Waiting for reply\n" ));
	if( !sockhelp.ReadString(&str) )
	{
		fprintf( stderr, "HTTPFetch::Get(): statusreply: could not read string\n" );
		goto error;
	}
	int version1,version2;
	int errorcode;
	if( sscanf(str.String(),"HTTP/%d.%d %d", &version1, &version2, &errorcode)!=3 )
	{
		fprintf( stderr, "HTTPFetch::Get(): statusreply: could not parse reply from server: [%s]\n", str.String() );
		goto error;
	}
	if( errorcode == 100 )
	{
		if( !sockhelp.ReadString(&str) )
		{
			fprintf( stderr, "HTTPFetch::Get(): statusreply: could not read HTTP/1.1 string1\n" );
			goto error;
		}
		if( str.Length() != 0 )
		{
			fprintf( stderr, "HTTPFetch::Get(): statusreply: the HTTP/1.1 string1 should be empty (%s)\n", str.String() );
			goto error;
		}

		if( !sockhelp.ReadString(&str) )
		{
			fprintf( stderr, "HTTPFetch::Get(): statusreply: could not read HTTP/1.1 string2\n" );
			goto error;
		}
		if( sscanf(str.String(),"HTTP/%d.%d %d", &version1, &version2, &errorcode)!=3 )
		{
			fprintf( stderr, "HTTPFetch::Get(): statusreply: could not parse HTTP/1.1 string2 reply from server: [%s]\n", str.String() );
			goto error;
		}
	}
	if( errorcode == 304 )
	{
		// 304 Not Modified
		return 304;
	}
	
	if( errorcode != 200 )
	{
		fprintf( stderr, "HTTPFetch::Get(): statusreply: error replyed with err %d: [%s]\n", errorcode, str.String() );
		goto error;
	}

	if( version1 >= 1 )
	{
		while( 1 )
		{
			BString str;
			if( !sockhelp.ReadString(&str) )
			{
				fprintf( stderr, "HTTPFetch::Get(): statusreply: could not read string\n" );
				goto error;
			}
			if( str == "" )
				break;
//			printf( "[%s]\n", str.String() );
			BString parm;
			BString value;
			int colon = str.FindFirst( ':' );
			if( colon < 0 )
			{
				fprintf( stderr, "HTTPFetch::Get(): statusreply: server sent junk reply: [%s]\n", str.String() );
			}
			else
			{
				parm.SetTo( str, colon );
				parm.RemoveLast( " " );
				value.SetTo( str.String()+colon+1 );
				value.RemoveFirst( " " );
//				value.RemoveLast( " " );
				
				PRINT_HTTP(( "HTTPFetch::Get(): Read http arg: \"%s\":\"%s\"\n", parm.String(), value.String() ));
				
				if( mimetype && parm.ICompare("Content-Type")==0 )
					mimetype->SetTo( value );
				if( parm.ICompare("Content-Length")==0 )
					content_length = atoi( value.String() );
				else if( parm.ICompare("Connection")==0 && value.ICompare("Keep-Alive")==0 )
					keepalive = true;
				else if( parm.ICompare("Last-Modified")==0 )
					fLastModified = value;
			}
		}
	}
	
	if( content_length == 0 )
		content_length = INT_MAX;

	if( fMessenger )
	{
		fMessage->RemoveName( "length" );
		if( content_length != INT_MAX )
			fMessage->AddInt32( "length", content_length );
		fMessage->RemoveName( "position" );
		fMessage->AddInt32( "position", 0 );
				
		fMessenger->SendMessage( fMessage );
	}

//	printf( "READ_CONTENT_LENGTH:%d keepalive:%d\n", content_length, keepalive );
	while( received < content_length )
	{
		char buffer[4096];
		PRINT_CTRL(( "READING:%d\n", min_c(content_length-received,sizeof(buffer)) ));
		size_t readlen = sockhelp.Read( buffer, min_c(content_length-received,sizeof(buffer)) );
		PRINT_CTRL(( "READ:%d\n", readlen ));
		if( readlen<=0 )
		{
			if( !keepalive )
				break;
			else
			{
				fprintf( stderr, "HTTPFetch::Get(): getdata: Read returned %ld\n", readlen );
				goto error;
			}
		}

		if( fMessenger )
		{
			fMessage->RemoveName( "length" );
			if( content_length != INT_MAX )
				fMessage->AddInt32( "length", content_length );
			fMessage->RemoveName( "position" );
			fMessage->AddInt32( "position", received+readlen );
				
			fMessenger->SendMessage( fMessage );
		}
		
		io->Write( buffer, readlen );
		received += readlen;
	}

//	if( !keepalive )
//	{
//		closesocket( sock );
//		fSocket = -1;
//	}

//	printf( "READ OK\n" );
	return B_OK;

error:
	fLastModified = last_lastmodified;

	// Some error occured, so we can't reuse the socket, and have to close it...
	close( fSocket );
	fSocket = -1;
	if( mustretry && !fKillConnection )
	{
		fprintf( stderr, "HTTPFetch::Get(): retrying...\n" );
		goto retry;
	}
	return B_ERROR;
}
	
//-----------------------------------------------------------------------------
