/*
 * Copyright (c) 1999, Jesper Hansen. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither name of the company nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//-----------------------------------------------------------------------------
#include <assert.h>
#include <stdlib.h>
#include <string.h>
//-------------------------------------
#include <interface/Bitmap.h>
#include <support/Debug.h>
//-------------------------------------
#include "MiniSlider.h"
//-----------------------------------------------------------------------------

#define TRIANGLE_WIDTH 6
#define TRIANGLE_HEIGHT 7

#define C0 0
#define C1 8
#define C2 16
#define C3 24
#define C4 31
#define CT 0 // 26 //B_TRANSPARENT_MAGIC_CMAP8

static uint8 gRightArrowGfx[TRIANGLE_HEIGHT][TRIANGLE_WIDTH] =
{
	{C4,C4,C4,C0,CT,CT},
	{C4,C3,C3,C4,C0,CT},
	{C4,C3,C3,C3,C4,C0},
	{C4,C3,C3,C3,C3,C2},
	{C4,C3,C3,C3,C2,C0},
	{C4,C3,C3,C2,C0,CT},
	{C4,C2,C2,C0,CT,CT},
};

static uint8 gLeftArrowGfx[TRIANGLE_HEIGHT][TRIANGLE_WIDTH] =
{
	{CT,CT,C0,C4,C4,C4},
	{CT,C0,C4,C3,C3,C2},
	{C0,C4,C3,C3,C3,C2},
	{C4,C3,C3,C3,C3,C2},
	{C0,C2,C3,C3,C3,C2},
	{CT,C0,C2,C3,C3,C2},
	{CT,CT,C0,C2,C2,C2},
};

//-----------------------------------------------------------------------------

MiniSlider::MiniSlider( BMessage *message, look_t look, int32 minvalue, int32 maxvalue ) :
	BControl( BRect(0,0,0,0), "MiniSlider", NULL, message, B_FOLLOW_NONE, B_WILL_DRAW|B_FULL_UPDATE_ON_RESIZE ),
	dle::Object( this )
{
	ASSERT( minvalue<maxvalue );

	fLook = look;
	
	fSliding = false;
	
	fMinValue = minvalue;
	fMaxValue = maxvalue;

	fArrowBitmap = new BBitmap( BRect(0,0,TRIANGLE_WIDTH-1,TRIANGLE_HEIGHT-1), B_CMAP8 );
	if( fLook == TRIANGLE_RIGHT )
	{
		for( int y=0; y<TRIANGLE_HEIGHT; y++ )
			memcpy( ((uint8*)fArrowBitmap->Bits())+y*fArrowBitmap->BytesPerRow(), gRightArrowGfx[y], TRIANGLE_WIDTH );
	}
	else
	{
		for( int y=0; y<TRIANGLE_HEIGHT; y++ )
			memcpy( ((uint8*)fArrowBitmap->Bits())+y*fArrowBitmap->BytesPerRow(), gLeftArrowGfx[y], TRIANGLE_WIDTH );
	}

	SetValue( (fMinValue+fMaxValue) / 2 );
}

MiniSlider::~MiniSlider()
{
	delete fArrowBitmap;
}

//-----------------------------------------------------------------------------

dle::MinMax2 MiniSlider::GetMinMaxSize()
{
	if( fLook==TRIANGLE_LEFT || fLook==TRIANGLE_RIGHT )
		return dle::MinMax2( TRIANGLE_WIDTH,TRIANGLE_WIDTH, 0,kMaxSize );

	TRESPASS();
	return dle::MinMax2( 0,kMaxSize, 0,kMaxSize );
}

// FIXME: remove flicker!
void MiniSlider::Draw( BRect updaterect )
{
	int32 value = Value();
	float fvalue = float(value-fMinValue)/(float(fMaxValue-fMinValue));
	
	BRect bounds = Bounds();
	int top = (int)bounds.top;
	int bottom = (int)bounds.bottom;
	int iref = (int)floor(((bottom+1)-top)*(1.0f-fvalue)+0.5f);
	
	DrawBitmap( fArrowBitmap, BPoint(bounds.left,iref-(TRIANGLE_HEIGHT/2)) );

	SetHighColor( 0,0,0 );
	
	int tritop = iref-(TRIANGLE_HEIGHT/2+1);
	int tribot = iref+(TRIANGLE_HEIGHT/2+1);
	
	if( top <= tritop )
		FillRect( BRect(bounds.left,bounds.top,bounds.right,tritop) );
	if( bottom >= tribot )
		FillRect( BRect(bounds.left,tribot,bounds.right,bounds.bottom) );
}

void MiniSlider::MouseDown( BPoint position )
{
	SetValue( PosToVal(position) );
	Invoke();

	fSliding = true;
	SetMouseEventMask( B_POINTER_EVENTS, B_NO_POINTER_HISTORY|B_SUSPEND_VIEW_FOCUS|B_LOCK_WINDOW_FOCUS );
}

void MiniSlider::MouseUp( BPoint position )
{
	if( fSliding )
	{
		SetValue( PosToVal(position) );
		Invoke();
		fSliding = false;
	}
}

void MiniSlider::MouseMoved( BPoint position, uint32 code, const BMessage *message )
{
	if( fSliding )
	{
		SetValue( PosToVal(position) );
		Invoke();
	}
}

int32 MiniSlider::PosToVal( BPoint position ) const
{
	BRect bounds = Bounds();
	int top = (int)bounds.top;
	int bottom = (int)bounds.bottom;

	float fvalue = 1.0f - ((position.y-0.5f)/((bottom+1)-top));

	int value = int((fvalue)*float(fMaxValue-fMinValue)) + fMinValue;
	
//	printf( "%f->%f->%d\n", position.y, fvalue, value );
	return value;
}

//-----------------------------------------------------------------------------

void MiniSlider::SetValue( int32 value )
{
	if( value<fMinValue ) value = fMinValue;
	if( value>fMaxValue ) value = fMaxValue;

	BControl::SetValue( value );
}

//-----------------------------------------------------------------------------
