/*
 * Copyright (c) 1999, Jesper Hansen. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither name of the company nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//-----------------------------------------------------------------------------
#include <stdio.h>
//-------------------------------------
#include <interface/Bitmap.h>
#include <interface/Window.h>
//-------------------------------------
#include "gfx/BitmapUtils.h"

#include "BitmapView.h"
//-----------------------------------------------------------------------------

#define SELECTION_BITMAP_SIZE 4

class SelectionBitmapView::SelectView : public BView
{
public:
	SelectView();
	~SelectView();
	
	void AttachedToWindow();
	
	void Pulse();

	void MouseDown( BPoint where );
	void MouseUp( BPoint where );
	void MouseMoved( BPoint where, uint32 code, const BMessage *a_message );

//	void Draw();

private:
	int		fFrame;
	BBitmap	*fBackBitmap; // could be shared between all views...
};

SelectionBitmapView::SelectView::SelectView() :
	BView( BRect(0,0,0,0), "SelectView", B_FOLLOW_NONE, B_WILL_DRAW|B_PULSE_NEEDED )
{
	fBackBitmap = new BBitmap( BRect(0,0,SELECTION_BITMAP_SIZE-1,SELECTION_BITMAP_SIZE-1), B_CMAP8 );
	fFrame = 0;
}

SelectionBitmapView::SelectView::~SelectView()
{
	delete fBackBitmap;
}

void SelectionBitmapView::SelectView::AttachedToWindow()
{
	SetViewBitmap( fBackBitmap );
}

// TODO: let all the SelectionBitmapView's share one bitmap...
void SelectionBitmapView::SelectView::Pulse()
{
	if( Window() )
	{
		fFrame++;
		for( int iy=0; iy<SELECTION_BITMAP_SIZE; iy++ )
		{
			uint8 *bits = BBITMAP8_BITS( fBackBitmap, 0, iy );
			for( int ix=0; ix<SELECTION_BITMAP_SIZE; ix++ )
			{
				bits[ix] = (((ix^iy)-fFrame)&2) ? 0 : 31;
			}
		}
		Invalidate();
	}
}

void SelectionBitmapView::SelectView::MouseDown( BPoint where )
{
	if( Parent() )
		Parent()->MouseDown( ConvertToParent(where) );
}

void SelectionBitmapView::SelectView::MouseUp( BPoint where )
{
	if( Parent() )
		Parent()->MouseUp( ConvertToParent(where) );
}

void SelectionBitmapView::SelectView::MouseMoved( BPoint where, uint32 code, const BMessage *a_message )
{
	if( Parent() )
		Parent()->MouseMoved( ConvertToParent(where), code, a_message ); // note: the code will be wrong
}



//-----------------------------------------------------------------------------

SelectionBitmapView::SelectionBitmapView( bool selectable=false ) :
	BView( BRect(0,0,0,0), "BitmapView", B_FOLLOW_NONE, B_WILL_DRAW|B_FULL_UPDATE_ON_RESIZE|B_FRAME_EVENTS ),
	dle::Object( this )
{
	SetViewColor( B_TRANSPARENT_COLOR );

	fBitmap = NULL;
	fZoomRect.Set( 0,0, -1,-1 );

//	fZoomedBitmap = NULL;
//	fZoomFilter = damn::filter_point;
//	fZoomFilter = damn::filter_sinc;

	fSelectable = selectable;
	
	if( fSelectable )
	{
		fSelectViewsLeft = new SelectView;
		AddChild( fSelectViewsLeft );
		fSelectViewsRight = new SelectView;
		AddChild( fSelectViewsRight );
		fSelectViewsTop = new SelectView;
		AddChild( fSelectViewsTop );
		fSelectViewsBottom = new SelectView;
		AddChild( fSelectViewsBottom );
	}
	else
	{
		fSelectViewsLeft = NULL;
		fSelectViewsRight = NULL;
		fSelectViewsTop = NULL;
		fSelectViewsBottom = NULL;
	}
	
	ClearSelectionRect();
	DrawSelection();

	fDoSelect = false;
}

SelectionBitmapView::~SelectionBitmapView()
{
}

dle::MinMax2 SelectionBitmapView::GetMinMaxSize()
{
	return dle::MinMax2( 0,kMaxSize, 0,kMaxSize );
}

//-----------------------------------------------------------------------------

void SelectionBitmapView::FrameResized( float new_width, float new_height )
{
//	BView::FrameResized( new_width, new_height );

	printf( "FRAME RESIZE: %fx%f\n", new_width, new_height );

//	CalcZoomBitmap();
}

//-----------------------------------------------------------------------------

#define ROUNDNEAREST(v) floor((v)+0.5f)

void SelectionBitmapView::MouseDown( BPoint where )
{
	if( fSelectable && fBitmap && !fZoomRect.IsValid() )
	{
		fDoSelect = true;
		fSelectCorner[0].x = ROUNDNEAREST( where.x / (Bounds().Width()+1) * (fBitmap->Bounds().Width()+1) );
		fSelectCorner[0].y = ROUNDNEAREST( where.y / (Bounds().Height()+1) * (fBitmap->Bounds().Height()+1) );
		fSelectCorner[1] = fSelectCorner[0];
		DrawSelection();

		SetMouseEventMask( B_POINTER_EVENTS, B_NO_POINTER_HISTORY|B_SUSPEND_VIEW_FOCUS|B_LOCK_WINDOW_FOCUS );
	}
}

void SelectionBitmapView::MouseUp( BPoint where )
{
	fDoSelect = false;
}

void SelectionBitmapView::MouseMoved( BPoint where, uint32 code, const BMessage *a_message )
{
	if( fSelectable && fDoSelect )
	{
		fSelectCorner[1].x = ROUNDNEAREST( where.x / (Bounds().Width()+1) * (fBitmap->Bounds().Width()+1) );
		fSelectCorner[1].y = ROUNDNEAREST( where.y / (Bounds().Height()+1) * (fBitmap->Bounds().Height()+1) );
		ClampSelectionRect();
		DrawSelection();
	}
}

//-----------------------------------------------------------------------------

void SelectionBitmapView::SetBitmap( damn::AutoPtr<BBitmap> bitmap )
{
	fBitmap = bitmap;
//	CalcZoomBitmap();

	ClampSelectionRect();

	if( fBitmap )
		Draw( Bounds() );
}

damn::AutoPtr<BBitmap> SelectionBitmapView::GetBitmap() const
{
	return fBitmap;
}

void SelectionBitmapView::SetZoomRect( const BRect &rect )
{
	fZoomRect=rect;
//	CalcZoomBitmap();

	Draw( Bounds() );
}

#if 0
void SelectionBitmapView::CalcZoomBitmap()
{
	if( fZoomedBitmap != NULL )
	{
		delete fZoomedBitmap;
		fZoomedBitmap = NULL;
	}

	if( fBitmap!=NULL && fZoomFilter!=damn::filter_point )
	{
		if( fZoomRect.IsValid() )
		{
			fZoomedBitmap = new BBitmap( Bounds(), B_RGB32 );
			printf( "SCALE1\n" );
			damn::Scale( fBitmap, fZoomRect, fZoomedBitmap, fZoomFilter );
		}
		else if( fBitmap->Bounds()!=Bounds() && Bounds().Width()>=8 && Bounds().Height()>=8 )
		{
			fBitmap->Bounds().PrintToStream();
			Bounds().PrintToStream();
			fZoomedBitmap = new BBitmap( Bounds(), B_RGB32 );
			printf( "SCALE2\n" );
			damn::Scale( fBitmap, fZoomedBitmap, fZoomFilter );
		}
	}
}
#endif

//-----------------------------------------------------------------------------

BRect SelectionBitmapView::MakeSelectionRect() const
{
	if( fSelectCorner[0].x==fSelectCorner[1].x && fSelectCorner[0].y==fSelectCorner[1].y )
		return BRect(0,0,-1,-1);
	else
		return BRect(
			min_c(fSelectCorner[0].x, fSelectCorner[1].x), min_c(fSelectCorner[0].y, fSelectCorner[1].y),
			max_c(fSelectCorner[0].x, fSelectCorner[1].x)-1, max_c(fSelectCorner[0].y, fSelectCorner[1].y)-1 );
}

#define CLAMP(v,l,h) max_c(min_c((v),(h)),(l))

void SelectionBitmapView::ClampSelectionRect()
{
	if( fBitmap == NULL )
	{
		ClearSelectionRect();
	}
	else
	{
		float bitmapwidth = fBitmap->Bounds().Width()+1;
		float bitmapheight = fBitmap->Bounds().Height()+1;

		fSelectCorner[0].x = CLAMP( fSelectCorner[0].x, 0, bitmapwidth-1 );
		fSelectCorner[0].y = CLAMP( fSelectCorner[0].y, 0, bitmapheight-1 );
		fSelectCorner[1].x = CLAMP( fSelectCorner[1].x, 0, bitmapwidth-1 );
		fSelectCorner[1].y = CLAMP( fSelectCorner[1].y, 0, bitmapheight-1 );
	}
}

void SelectionBitmapView::ClearSelectionRect()
{
	fSelectCorner[0].Set( 0, 0 );
	fSelectCorner[1].Set( 0, 0 );
}

//--------------------------------------

void SelectionBitmapView::SetSelectionRect( const BRect &rect )
{
	fSelectCorner[0] = rect.LeftTop();
	fSelectCorner[1] = rect.RightBottom() + BPoint(1,1);
	DrawSelection();
}

BRect SelectionBitmapView::GetSelectionRect() const
{
	if( fDoSelect )
		return BRect(0,0,-1,-1);
	else
		return MakeSelectionRect();
}

//-----------------------------------------------------------------------------

void SelectionBitmapView::DrawSelection()
{
	BRect selrect = MakeSelectionRect();

	if( !selrect.IsValid() || fZoomRect.IsValid() )
	{
		if( fSelectable )
		{
			fSelectViewsLeft->MoveTo( 100000, 10000 );
			fSelectViewsRight->MoveTo( 100000, 10000 );
			fSelectViewsTop->MoveTo( 100000, 10000 );
			fSelectViewsBottom->MoveTo( 100000, 10000 );
		}
		return;
	}
	
	if( fBitmap )
	{
		float left = ROUNDNEAREST( selrect.left / (fBitmap->Bounds().Width()+1) * (Bounds().Width()+1) );
		float right = ROUNDNEAREST( (selrect.right+1) / (fBitmap->Bounds().Width()+1) * (Bounds().Width()+1) ) -1;
		float top = ROUNDNEAREST( selrect.top / (fBitmap->Bounds().Height()+1) * (Bounds().Height()+1) );
		float bottom = ROUNDNEAREST( (selrect.bottom+1) / (fBitmap->Bounds().Height()+1) * (Bounds().Height()+1) ) -1;

		fSelectViewsLeft->ResizeTo( 0, (bottom-top)-1 );
		fSelectViewsLeft->MoveTo( left, top+1 );

		fSelectViewsTop->ResizeTo( (right-left)-1, 0 );
		fSelectViewsTop->MoveTo( left, top );

		fSelectViewsRight->ResizeTo( 0, (bottom-top)-1 );
		fSelectViewsRight->MoveTo( right, top );

		fSelectViewsBottom->ResizeTo( (right-left)-1, 0 );
		fSelectViewsBottom->MoveTo( left+1, bottom );
	}
}

//-----------------------------------------------------------------------------

void SelectionBitmapView::Draw( BRect area )
{
	if( fBitmap )
	{
		if( fZoomRect.IsValid() )
			DrawBitmap( fBitmap, fZoomRect, Bounds() );
		else
			DrawBitmap( fBitmap, fBitmap->Bounds(), Bounds() );
		DrawSelection();
	}
	else
	{
		SetHighColor( 0, 0, 0 );
		FillRect( Bounds() );
	}
}

//-----------------------------------------------------------------------------
