/*
 * Copyright (c) 1999, Jesper Hansen. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither name of the company nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//-----------------------------------------------------------------------------
//#include <stdio.h>
#include <stdlib.h>
//-------------------------------------
#include <interface/CheckBox.h>
#include <interface/Screen.h>
#include <storage/FilePanel.h>
#include <storage/Path.h>
//-------------------------------------
#include "dle/BBox.h"
#include "dle/BButton.h"
#include "dle/BCheckBox.h"
#include "dle/BStringView.h"
#include "dle/BTextControl.h"
#include "dle/Frame.h"
#include "dle/Line.h"
#include "dle/Space.h"
#include "dle/Split.h"
#include "misc/Settings.h"
//-------------------------------------
#include "ConfigWin.h"
#include "CAWin.h"
#include "CAApp.h"
//-----------------------------------------------------------------------------

enum
{
	MSG_ENABLE_DISABLE	= 'endi',
	
	MSG_SAVEPATH_BROWSE	= 'spbr',
	MSG_RUNPATH_BROWSE	= 'sprn',

	MSG_FILEPANEL_SAVE	= 'fpsa',
	MSG_FILEPANEL_RUN	= 'fprn',

	MSG_CANCEL			= 'cncl',
	MSG_USE				= 'use ',
	MSG_SAVE			= 'save'
};

//-----------------------------------------------------------------------------

ConfigWin::ConfigWin( BMessenger *target, BMessage *message ) :
	dle::Window( BRect(0,0,0,0), "Config", B_TITLED_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL, B_NOT_ZOOMABLE/*|B_NOT_RESIZABLE*/|B_ASYNCHRONOUS_CONTROLS )
{
	fMessenger = target;
	fMessage = message;
	
	fSaveFilePanel = NULL;
	fRunFilePanel = NULL;

	// Setup default config values:
	CAApp::Settings()->SetDefaultRect( "config:winpos", BRect(10,40,10+400-1,40+50-1) );

	BRect winrect = CAApp::Settings()->GetRect( "config:winpos" );
	MoveTo( winrect.LeftTop() );
	ResizeTo( winrect.Width(), winrect.Height() );

	// ========== DLE GUI ==========
	dle::VSplit *vs = new dle::VSplit();
	{
		dle::Frame *frame = new dle::Frame( dle::Frame::RAISED );
		frame->InnerSpacing( false );
		{
			dle::VSplit *vs = new dle::VSplit();
			{
				dle::BBox *box = new dle::BBox( NULL );
				fProxyCheckBox = new BCheckBox( BRect(0,0,100-1,40-1), NULL, "Proxy", new BMessage(MSG_ENABLE_DISABLE) );
				fProxyCheckBox->SetFont( be_bold_font );
				box->SetLabel( fProxyCheckBox );
				{
					dle::HSplit *hs = new dle::HSplit();
					{
						{
						dle::BStringView *sv = new dle::BStringView( "Proxy" );
						sv->SetInnerRight( 0 );
						hs->AddObject( sv, 1.0f );
						fProxyHost = new dle::BTextControl( "", NULL );
						fProxyHost->SetInnerLeft( 0 );
						hs->AddObject( fProxyHost, 1.0f );
						}

						{
						dle::BStringView *sv = new dle::BStringView( "Port" );
						sv->SetInnerRight( 0 );
						hs->AddObject( sv, 1.0f );
						fProxyPort = new dle::BTextControl( "", NULL );
						fProxyPort->SetInnerLeft( 0 );
						static_cast<BTextView*>(fProxyPort->ChildAt(0))->SetMaxBytes( 5 );
						for( int i=0; i<=255; i++ ) static_cast<BTextView*>(fProxyPort->ChildAt(0))->DisallowChar( i );
						for( int i='0'; i<='9'; i++ ) static_cast<BTextView*>(fProxyPort->ChildAt(0))->AllowChar( i );
						hs->AddObject( fProxyPort, 0.2f );
						}
					}
					box->AddObject( hs );
				}
				vs->AddObject( box );

				box = new dle::BBox( "Frame-change action" );
				{
					dle::GridSplit *grid = new dle::GridSplit( 2, 3 );
					{
						{
							fPlaySoundCheckBox = new dle::BCheckBox( "Play sound", new BMessage(MSG_ENABLE_DISABLE) );
							grid->AddObject( fPlaySoundCheckBox, dle::LEFT );

							grid->AddObject( new dle::Space() );
						}

						{
							fSaveFrameCheckBox = new dle::BCheckBox( "Save frame", new BMessage(MSG_ENABLE_DISABLE) );
							grid->AddObject( fSaveFrameCheckBox, dle::LEFT );

							dle::HSplit *hs = new dle::HSplit();
							{
								fSaveFrameTextControl = new dle::BTextControl( "/boot/home/", NULL );
								hs->AddObject( fSaveFrameTextControl, 1.0f );

								fSaveFrameBrowseButton = new dle::BButton( "Browse", new BMessage(MSG_SAVEPATH_BROWSE) );
								hs->AddObject( fSaveFrameBrowseButton, 0.0f );
							}
							grid->AddObject( hs );
						}

						{
							fRunFileCheckBox = new dle::BCheckBox( "Run file", new BMessage(MSG_ENABLE_DISABLE) );
							grid->AddObject( fRunFileCheckBox, dle::LEFT );

							dle::HSplit *hs = new dle::HSplit();
							{
								fRunFileTextControl = new dle::BTextControl( NULL, NULL );
								hs->AddObject( fRunFileTextControl, 1.0f );

								fRunFileBrowseButton = new dle::BButton( "Browse", new BMessage(MSG_RUNPATH_BROWSE) );
								hs->AddObject( fRunFileBrowseButton, 0.0f );
							}
							grid->AddObject( hs );
						}
					}
					box->AddObject( grid );
				}
				vs->AddObject( box );
#if 0
				box = new dle::BBox( "Net/HTTP" );
				{
					dle::GridSplit *grid = new dle::GridSplit( 2, 1 );
					{
						dle::BCheckBox *cb = new dle::BCheckBox( "Use keepalive", NULL );
						grid->AddObject( cb, dle::LEFT );

						grid->AddObject( new dle::Space() );
					}
					box->AddObject( grid );
				}
				vs->AddObject( box );
#endif
			}
			frame->AddObject( vs );
		}
		vs->AddObject( frame, 1.0f );

		dle::HSplit *hs = new dle::HSplit();
		hs->SetInner( 2 );
		{
			hs->AddObject( new dle::Space() );
			
			dle::BButton *button = new dle::BButton( "Cancel", new BMessage(MSG_CANCEL) );
			hs->AddObject( button, 0.0f );

			button = new dle::BButton( "Use", new BMessage(MSG_USE) );
			hs->AddObject( button, 0.0f );

			button = new dle::BButton( "Save", new BMessage(MSG_SAVE) );
			button->MakeDefault( true );
			hs->AddObject( button, 0.0f );
		}
		vs->AddObject( hs );
	}
	AddObject( vs );

	Show();
}

ConfigWin::~ConfigWin()
{
	CAApp::Settings()->SetRect( "config:winpos", Frame() );

	delete fSaveFilePanel;
	delete fRunFilePanel;

	delete fMessenger;
	delete fMessage;
}

//--------------------------------------------------------------------------------

void ConfigWin::Show()
{
	ReadConfig();

	// Make sure the windows does not open outside the screen:
	if( !BScreen().Frame().Contains(Frame()) )
		MoveTo( 10, 40 );

	dle::Window::Show();
}

//--------------------------------------------------------------------------------

static BString itobs( int value )
{
	BString str;
	sprintf( str.LockBuffer(16), "%d", value );
	str.UnlockBuffer();
	return str;
}

void ConfigWin::ReadConfig()
{
	fProxyCheckBox->SetValue( CAApp::Settings()->GetBool(CFG_PROXY) );
	fProxyHost->SetText( CAApp::Settings()->GetString(CFG_PROXY_HOST) );
	fProxyPort->SetText( itobs(CAApp::Settings()->GetInt32(CFG_PROXY_PORT)).String() );

	fPlaySoundCheckBox->SetValue( CAApp::Settings()->GetBool(CFG_PLAYALARM) );

	fSaveFrameCheckBox->SetValue( CAApp::Settings()->GetBool(CFG_SAVEFRAME) );
	fSaveFrameTextControl->SetText( CAApp::Settings()->GetString(CFG_SAVEPATH) );

	fRunFileCheckBox->SetValue( CAApp::Settings()->GetBool(CFG_RUNFILE) );
	fRunFileTextControl->SetText( CAApp::Settings()->GetString(CFG_RUNFILENAME) );
	
	EnableDisableControls();
}

void ConfigWin::SetConfig( bool save )
{
	int32 changed = 0;
	damn::Settings *s = CAApp::Settings();

//--

	bool proxy_enabled = fProxyCheckBox->Value();
	BString proxy_host = fProxyHost->Text();
	int proxy_port = atoi(fProxyPort->Text());
	if( proxy_enabled!=s->GetBool(CFG_PROXY) || proxy_host!=s->GetString(CFG_PROXY_HOST) || proxy_port!=s->GetInt32(CFG_PROXY_PORT) )
		changed|=CFG_CHANGE_PROXY;
	CAApp::Settings()->SetBool( CFG_PROXY, proxy_enabled, save );
	CAApp::Settings()->SetString( CFG_PROXY_HOST, proxy_host.String(), save );
	CAApp::Settings()->SetInt32( CFG_PROXY_PORT, proxy_port, save );

//--

	CAApp::Settings()->SetBool( CFG_PLAYALARM, fPlaySoundCheckBox->Value(), save );

	CAApp::Settings()->SetBool( CFG_SAVEFRAME, fSaveFrameCheckBox->Value(), save );
	CAApp::Settings()->SetString( CFG_SAVEPATH, fSaveFrameTextControl->Text(), save );

	CAApp::Settings()->SetBool( CFG_RUNFILE, fRunFileCheckBox->Value(), save );
	CAApp::Settings()->SetString( CFG_RUNFILENAME, fRunFileTextControl->Text(), save );

	if( changed )
	{
		BMessage msg( *fMessage );
		msg.AddInt32( "changed", changed );
		fMessenger->SendMessage( &msg );
	}
}

void ConfigWin::EnableDisableControls()
{
	bool proxy_enabled = fProxyCheckBox->Value();
	fProxyHost->SetEnabled( proxy_enabled );
	fProxyPort->SetEnabled( proxy_enabled );

	bool save_enabled = fSaveFrameCheckBox->Value();
	fSaveFrameTextControl->SetEnabled( save_enabled );
	fSaveFrameBrowseButton->SetEnabled( save_enabled );

	bool runfile_enabled = fRunFileCheckBox->Value();
	fRunFileTextControl->SetEnabled( runfile_enabled );
	fRunFileBrowseButton->SetEnabled( runfile_enabled );
}


//--------------------------------------------------------------------------------

bool ConfigWin::QuitRequested()
{
	Hide();
	return false;
}

void ConfigWin::MessageReceived( BMessage *msg )
{
	switch( msg->what )
	{
		case MSG_ENABLE_DISABLE:
			EnableDisableControls();
			break;
			

		case MSG_SAVEPATH_BROWSE:
		{
			BEntry entry( fSaveFrameTextControl->Text() );
			entry_ref ref;
			bool refvalid = entry.GetRef(&ref) == B_OK;
			
			if( fSaveFilePanel == NULL )
			{
				fSaveFilePanel = new BFilePanel( B_OPEN_PANEL, NULL, refvalid?&ref:NULL, B_DIRECTORY_NODE, false, new BMessage(MSG_FILEPANEL_SAVE) );
				fSaveFilePanel->SetTarget( BMessenger(NULL,this) );
			}
			fSaveFilePanel->Show();
			break;
		}
		case MSG_FILEPANEL_SAVE:
		{
			entry_ref ref;
			if( msg->FindRef("refs",&ref) == B_OK )
			{
				BEntry entry( &ref, true );
				BPath path;
				if( entry.GetPath(&path) == B_OK )
				{
					fSaveFrameTextControl->SetText( path.Path() );
				}
			}
			break;
		}


		case MSG_RUNPATH_BROWSE:
		{
			BEntry entry( fRunFileTextControl->Text() );
			entry_ref ref;
			bool refvalid = entry.GetRef(&ref) == B_OK;
			
			if( fRunFilePanel == NULL )
			{
				fRunFilePanel = new BFilePanel( B_OPEN_PANEL, NULL, refvalid?&ref:NULL, B_DIRECTORY_NODE, false, new BMessage(MSG_FILEPANEL_RUN) );
				fRunFilePanel->SetTarget( BMessenger(NULL,this) );
			}
			fRunFilePanel->Show();
			break;
		}
		case MSG_FILEPANEL_RUN:
		{
			entry_ref ref;
			if( msg->FindRef("refs",&ref) == B_OK )
			{
				BEntry entry( &ref, true );
				BPath path;
				if( entry.GetPath(&path) == B_OK )
				{
					fRunFileTextControl->SetText( path.Path() );
				}
			}
			break;
		}
		
		


		case MSG_CANCEL:
			Hide();
			break;

		case MSG_USE:
			SetConfig( false );
			Hide();
			break;

		case MSG_SAVE:
			SetConfig( true );
			Hide();
			break;
	
		case B_KEY_DOWN:
		{
			int32 raw_key;
			if( msg->FindInt32("raw_char",&raw_key)>=0 && raw_key==B_ESCAPE )
				Hide();
			else
				BWindow::MessageReceived( msg );
			break;
		}
	
		default:
//			msg->PrintToStream();
			BWindow::MessageReceived( msg );
	}
}

//--------------------------------------------------------------------------------

