/*
 * Copyright (c) 1999, Jesper Hansen. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither name of the company nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//-----------------------------------------------------------------------------
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
//-------------------------------------
#include <storage/File.h>
#include <storage/FindDirectory.h>
#include <storage/Path.h>
#include <support/Debug.h>
//-------------------------------------
#include "Settings.h"
//-----------------------------------------------------------------------------

damn::Settings::Settings( const char *filename, bool writeondestruct ) :
	fSaveMessage( 'damn' ), fUseMessage( 'damn' )
{
	status_t status;

	BPath pathname;
	status = find_directory( B_USER_SETTINGS_DIRECTORY, &pathname );
	if( status != B_OK )
	{
		fprintf( stderr, "Settings: could not find user config path: %s: %s\n", filename, strerror(errno) );
	}
	else
	{
		pathname.Append( filename );
	
		BFile file( pathname.Path(), B_READ_ONLY );
		if( !file.IsReadable() )
		{
			fprintf( stderr, "Settings: could not open file: %s: %s\n", pathname.Path(), strerror(errno) );
		}
		else
		{
			status = fSaveMessage.Unflatten( &file );
			if( status != B_OK )
			{
				fprintf( stderr,  "Settings: could not unflatten file: %s: %s\n", pathname.Path(), strerror(status) );
			}
		}
	}
	
	fWriteOnDestruct = writeondestruct;
	if( writeondestruct )
		fFilename = filename;
}

damn::Settings::~Settings()
{
	if( fWriteOnDestruct )
		Write( fFilename.String() );
}

//-----------------------------------------------------------------------------

status_t damn::Settings::Write( const char *filename )
{
	assert( filename!=NULL || fWriteOnDestruct );
	
	if( filename == NULL )
		filename = fFilename.String();

	status_t status;
	
	BPath pathname;

	status = find_directory( B_USER_SETTINGS_DIRECTORY, &pathname );
	if( status != B_OK )
	{
		fprintf( stderr, "Settings: could not find user config path: %s: %s\n", filename, strerror(errno) );
		return status;
	}
	pathname.Append( filename );
	
	BFile file( pathname.Path(), B_READ_WRITE|B_CREATE_FILE|B_ERASE_FILE );
	if( !file.IsWritable() )
	{
		fprintf( stderr, "Settings: could not open file: %s: %s\n", pathname.Path(), strerror(errno) );
		return B_ERROR;
	}
	
	status = fSaveMessage.Flatten( &file );
	if( status != B_OK )
	{
		fprintf( stderr,  "Settings: could not flatten to file: %s: %s\n", pathname.Path(), strerror(status) );
		return status;
	}
	
	return B_OK;
}

//-----------------------------------------------------------------------------

#define MAKEFUNCS( setdeff, setf, getf, hasm, addm, repm, findm, type, def ) \
void damn::Settings::setdeff( const char *name, type value ) \
{ \
	if( !fSaveMessage.hasm(name) ) \
		fSaveMessage.addm( name, value ); \
} \
void damn::Settings::setf( const char *name, type value, bool save=true ) \
{ \
	if( !save ) \
	{ \
		if( fUseMessage.repm(name, value) != B_OK ) \
			fUseMessage.addm( name, value ); \
	} \
	else \
	{ \
		fSaveMessage.repm( name, value ); \
	} \
} \
type damn::Settings::getf( const char *name, bool saved=false ) const \
{ \
	type value; \
	if( !saved && fUseMessage.findm(name,&value) == B_OK ) \
		return value; \
	if( fSaveMessage.findm(name,&value) == B_OK ) \
		return value; \
	TRESPASS(); \
	return def; \
}

//-----------------------------------------------------------------------------

MAKEFUNCS( SetDefaultString, SetString, GetString, \
	HasString, AddString, ReplaceString, FindString, const char *, "" );

MAKEFUNCS( SetDefaultInt32, SetInt32, GetInt32, \
	HasInt32, AddInt32, ReplaceInt32, FindInt32, int32, 0 );

MAKEFUNCS( SetDefaultInt64, SetInt64, GetInt64, \
	HasInt64, AddInt64, ReplaceInt64, FindInt64, int64, 0 );

MAKEFUNCS( SetDefaultFloat, SetFloat, GetFloat, \
	HasFloat, AddFloat, ReplaceFloat, FindFloat, float, 0.0f );

MAKEFUNCS( SetDefaultBool, SetBool, GetBool, \
	HasBool, AddBool, ReplaceBool, FindBool, bool, false );

MAKEFUNCS( SetDefaultPoint, SetPoint, GetPoint, \
	HasPoint, AddPoint, ReplacePoint, FindPoint, BPoint, BPoint(0,0) );

MAKEFUNCS( SetDefaultRect, SetRect, GetRect, \
	HasRect, AddRect, ReplaceRect, FindRect, BRect, BRect(0,0,-1,-1) );

//-----------------------------------------------------------------------------

void damn::Settings::SetDefaultMessage( const char *name, const BMessage &value )
{
	if( !fSaveMessage.HasMessage(name) )
		fSaveMessage.AddMessage( name, &value );
}

void damn::Settings::SetMessage( const char *name, const BMessage &value, bool save=true )
{
	if( !save )
	{
		if( fUseMessage.ReplaceMessage(name, &value) != B_OK )
			fUseMessage.AddMessage( name, &value );
	}
	else
	{
		fSaveMessage.ReplaceMessage( name, &value );
	}
}

static BMessage gEmptyMessage( 'BOOM' );
BMessage damn::Settings::GetMessage( const char *name, bool saved=false ) const
{
	BMessage value;

	if( !saved && fUseMessage.FindMessage(name,&value) == B_OK )
		return value;
	if( fSaveMessage.FindMessage(name,&value) == B_OK )
		return value;

	TRESPASS();
	return gEmptyMessage;
}

//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

damn::OldSettings::OldSettings( const char *filename, bool writeondestruct ) :
	fMessage( 'damn' )
{
	status_t status;


	BPath pathname;
	status = find_directory( B_USER_SETTINGS_DIRECTORY, &pathname );
	if( status != B_OK )
	{
		fprintf( stderr, "Settings: could not find user config path: %s: %s\n", filename, strerror(errno) );
	}
	else
	{
		pathname.Append( filename );
	
		BFile file( pathname.Path(), B_READ_ONLY );
		if( !file.IsReadable() )
		{
			fprintf( stderr, "Settings: could not open file: %s: %s\n", pathname.Path(), strerror(errno) );
		}
		else
		{
			status = fMessage.Unflatten( &file );
			if( status != B_OK )
			{
				fprintf( stderr,  "Settings: could not unflatten file: %s: %s\n", pathname.Path(), strerror(status) );
			}
		}
	}
	
	fWriteOnDestruct = writeondestruct;
	if( writeondestruct )
		fFilename = filename;
}

damn::OldSettings::OldSettings( const BMessage &message )
{
	fMessage = message;
}

damn::OldSettings::OldSettings( const OldSettings &settings )
{
	fMessage = settings.fMessage;
}

damn::OldSettings::~OldSettings()
{
	if( fWriteOnDestruct )
		Write( fFilename.String() );
}

//-----------------------------------------------------------------------------

status_t damn::OldSettings::Write( const char *filename )
{
	assert( filename!=NULL || fWriteOnDestruct );
	
	if( filename == NULL )
		filename = fFilename.String();

	status_t status;
	
	BPath pathname;

	status = find_directory( B_USER_SETTINGS_DIRECTORY, &pathname );
	if( status != B_OK )
	{
		fprintf( stderr, "Settings: could not find user config path: %s: %s\n", filename, strerror(errno) );
		return status;
	}
	pathname.Append( filename );
	
	BFile file( pathname.Path(), B_READ_WRITE|B_CREATE_FILE|B_ERASE_FILE );
	if( !file.IsWritable() )
	{
		fprintf( stderr, "Settings: could not open file: %s: %s\n", pathname.Path(), strerror(errno) );
		return B_ERROR;
	}
	
	status = fMessage.Flatten( &file );
	if( status != B_OK )
	{
		fprintf( stderr,  "Settings: could not flatten to file: %s: %s\n", pathname.Path(), strerror(status) );
		return status;
	}
	
	return B_OK;
}

//-----------------------------------------------------------------------------

const char *damn::OldSettings::GetString( const char *name, const char *defaultvalue ) const
{
	const char *value;
	if( fMessage.FindString(name,&value) == B_OK )
		return value;
	return defaultvalue;
}

void damn::OldSettings::SetString( const char *name, const char *value )
{
	if( fMessage.ReplaceString(name, value) != B_OK )
		fMessage.AddString( name, value );
}

//-----------------------------------------------------------------------------

bool damn::OldSettings::GetMessage( const char *name, BMessage *value ) const
{
	return fMessage.FindMessage(name,value) == B_OK;
}

void damn::OldSettings::SetMessage( const char *name, const BMessage *value )
{
	if( fMessage.ReplaceMessage(name, value) != B_OK )
		fMessage.AddMessage( name, value );
}

//-----------------------------------------------------------------------------

bool damn::OldSettings::GetBool( const char *name, bool defaultvalue ) const
{
	bool value;
	if( fMessage.FindBool(name,&value) == B_OK )
		return value;
	return defaultvalue;
}

void damn::OldSettings::SetBool( const char *name, bool value )
{
	if( fMessage.ReplaceBool(name, value) != B_OK )
		fMessage.AddBool( name, value );
}

//-----------------------------------------------------------------------------

float damn::OldSettings::GetFloat( const char *name, float defaultvalue ) const
{
	float value;
	if( fMessage.FindFloat(name,&value) == B_OK )
		return value;
	return defaultvalue;
}

void damn::OldSettings::SetFloat( const char *name, float value )
{
	if( fMessage.ReplaceFloat(name, value) != B_OK )
		fMessage.AddFloat( name, value );
}

//-----------------------------------------------------------------------------

int32 damn::OldSettings::GetInt32( const char *name, int32 defaultvalue ) const
{
	int32 value;
	if( fMessage.FindInt32(name,&value) == B_OK )
		return value;
	return defaultvalue;
}

void damn::OldSettings::SetInt32( const char *name, int32 value )
{
	if( fMessage.ReplaceInt32(name, value) != B_OK )
		fMessage.AddInt32( name, value );
}

//-----------------------------------------------------------------------------

int64 damn::OldSettings::GetInt64( const char *name, int64 defaultvalue ) const
{
	int64 value;
	if( fMessage.FindInt64(name,&value) == B_OK )
		return value;
	return defaultvalue;
}

void damn::OldSettings::SetInt64( const char *name, int64 value )
{
	if( fMessage.ReplaceInt64(name, value) != B_OK )
		fMessage.AddInt64( name, value );
}

//-----------------------------------------------------------------------------

BPoint damn::OldSettings::GetPoint( const char *name, BPoint defaultvalue ) const
{
	BPoint value;
	if( fMessage.FindPoint(name,&value) == B_OK )
		return value;
	return defaultvalue;
}

void damn::OldSettings::SetPoint( const char *name, BPoint value )
{
	if( fMessage.ReplacePoint(name, value) != B_OK )
		fMessage.AddPoint( name, value );
}

//-----------------------------------------------------------------------------

BRect damn::OldSettings::GetRect( const char *name, BRect defaultvalue ) const
{
	BRect value;
	if( fMessage.FindRect(name,&value) == B_OK )
		return value;
	return defaultvalue;
}

void damn::OldSettings::SetRect( const char *name, BRect value )
{
	if( fMessage.ReplaceRect(name, value) != B_OK )
		fMessage.AddRect( name, value );
}

//-----------------------------------------------------------------------------
