#include <Debug.h>
#include "PeopleData.h"

#include <Locker.h>
#include <Autolock.h>
#include <OutlineListView.h>
#include <Window.h>
#include <File.h>
#include <Path.h>

#include <ctype.h>
#include <stdlib.h>

#include "GManager.h"
#include "Gicq.h"
#include "classSoundMaster.h"
#include "WindowManager.h"
#include "MiscStuff.h"
#include "Locale.h"
#include "PList.h"
#include "Prefs.h"

//-------------------------------------------------------------------------

GroupItem::GroupItem(const char *aType )
{
	mType = aType;
}

//-------------------------------------------------------------------------

GroupItem::~GroupItem() {
}

//-------------------------------------------------------------------------

void GroupItem::DrawItem( BView *owner, BRect frame, bool complete ) {

	rgb_color color;
	
	// Make the selection color (shouldn't be hardcoded!)
	rgb_color kHighlight; 
	kHighlight.red = kHighlight.blue = 222;
	kHighlight.green = 219;

	// Grab the owner's font, put it into boldface if it's expanded
	BFont baseFont;
	owner->GetFont( &baseFont );
	BFont ownerFont( baseFont );
	ownerFont.SetFace( B_BOLD_FACE );

	// does the background need to be redrawn?
	if( IsSelected() || complete ) {
		
		// pick the appropriate background color
        if( IsSelected() ) {
        	color = kHighlight;
        	owner->SetLowColor( 222, 219, 222 );
        } else {
			color = owner->ViewColor();
			owner->SetLowColor( 255, 255, 255 );
		}

		// draw the background		
		owner->SetHighColor(color);
		owner->FillRect(frame);
	}

	// set the font and draw the string
	//owner->SetHighColor(0,0,0);
	owner->SetHighColor(0,77,77);
	owner->SetFont( &ownerFont );
	owner->MovePenTo(frame.left + 4, frame.bottom - 2);
	owner->DrawString( mType.String() );

	// reset the font and colors
	owner->SetFont( &baseFont );
	owner->SetLowColor( 255, 255, 255 );
}

//=========================================================================
//=========================================================================

PeopleItem::PeopleItem( uint32 aUin, const char *aNick, ICQStatus aStatus, uint32 aEncoding, bool aOnlineNotify, GManager *aManager )
	:mUin( aUin )
	,mNick("")
	,mStrUin("")
	,mEncoding( aEncoding )
	,mOnlineNotify( aOnlineNotify )
	,mUnreadMsg( false )
{

	mStrUin << mUin;
	mNick = aNick;
	mStatus = aStatus;

	mLock = new BLocker( "item_lock" );
	mManager = aManager;
}

//-------------------------------------------------------------------------

PeopleItem::~PeopleItem() {
	delete mLock;
}

//-------------------------------------------------------------------------

const char *PeopleItem::GetDisplayName() {
	const char *ptr;

	if( *(mNick.String()) )
		ptr = mNick.String();
	else
		ptr = mStrUin.String();
	return ptr;
}

//-------------------------------------------------------------------------

uint32 PeopleItem::GetUIN() {
	return mUin;
}

//-------------------------------------------------------------------------

uint32 PeopleItem::GetEncoding() {
#warning 42?!? Meaning of life!
	if( mEncoding == 42 )
		return mManager -> defEncoding;
	return mEncoding;
}

//-------------------------------------------------------------------------

bool PeopleItem::GetOnlineNotify() {
	return mOnlineNotify;
}

//-------------------------------------------------------------------------

ICQStatus PeopleItem::GetStatus() {
	return mStatus;
}

//-------------------------------------------------------------------------

void PeopleItem::SetOnlineNotify( bool aOnlineNotify ) {
	mOnlineNotify = aOnlineNotify;
}

//-------------------------------------------------------------------------

void PeopleItem::SetGotNewMessage(bool aUnreadMsg)
{
	mUnreadMsg = aUnreadMsg;
}


//-------------------------------------------------------------------------

void PeopleItem::SetEncoding( uint32 aEncoding ) {
	mEncoding = aEncoding;
}

//-------------------------------------------------------------------------

void PeopleItem::SetStatus( ICQStatus aStatus ) {
	mStatus = aStatus;
}

//-------------------------------------------------------------------------

void PeopleItem::SetNick( const char *aNick ) {
	mLock->Lock();
	mNick=aNick;
	mLock->Unlock();
}

//-------------------------------------------------------------------------

void PeopleItem::DrawItem( BView *owner, BRect frame, bool complete ) {

	rgb_color color;
	frame.left -= 10;

	// Make the selection color (shouldn't be hardcoded!)
	rgb_color kHighlight; 
	kHighlight.red = kHighlight.blue = 222;
	kHighlight.green = 219;
	
	// Grab the owner's font, to be fiddled with if needed
	BFont baseFont;
	owner->GetFont( &baseFont );
	BFont ownerFont( baseFont );
	
	// does the background need to be redrawn?
	if( IsSelected() || complete ) {
		
		// pick the appropriate background color
        if( IsSelected() ) {
        	color = kHighlight;
        	owner->SetLowColor( 222, 219, 222 );
        } else {
					color = owner->ViewColor();
					owner->SetLowColor( 255, 255, 255 );
			}
	
		// draw the background		
		owner->SetHighColor(color);
		owner->FillRect(frame);
	}
	
	// Set colors and font attributes based on buddy status
	//if( status & BS_ENTERING )
	//	ownerFont.SetFace( B_BOLD_FACE );
	//if( status & BS_LEAVING )
	//	ownerFont.SetFace( B_ITALIC_FACE );
	//if( status & BS_IDLE )
	//	owner->SetHighColor(105,106,105);
	//else
	//	owner->SetHighColor(0,0,0);
	
	switch( mStatus ) {
		case IS_OFFLINE:
			owner->SetHighColor(90,90,90);
			break;
		case IS_PENDING:
			owner->SetHighColor(255,0,0);
				break;
		default:
			//owner->SetHighColor(0,107,107);
			owner->SetHighColor(0,0,0);
			break;
	}
	
	BPoint bitPoint( frame.left+3, frame.top+4 );
	BBitmap* drawMap = mManager -> statBitmaps[StatusC(mStatus)];
	if( drawMap ) {
		owner->PushState();
		owner->SetDrawingMode(B_OP_ALPHA);
		owner->DrawBitmap( drawMap, bitPoint );
		owner->PopState();

		owner->MovePenTo(frame.left + 19, frame.bottom - 2);
	} else
		owner->MovePenTo(frame.left + 4, frame.bottom - 2);

	// set the font and draw the string
	if( mUnreadMsg )
		ownerFont.SetFace( B_BOLD_FACE );
	owner->SetFont( &ownerFont );
	mLock->Lock();														//Some thread might be setting the mNick.
	owner->DrawString( GetDisplayName() );
	mLock->Unlock();

	// reset the font and colors
	owner->SetFont( &baseFont );
	owner->SetLowColor( 255, 255, 255 );
	owner->SetHighColor( 0, 0, 0 );
}

//============================================================================
//============================================================================

bool global_delete( PersonInfo *item ) {
	delete item;
	return false;
}

//-------------------------------------------------------------------------

bool global_attach( PersonInfo *pi, void *arg2 ) {
	PeopleList *list = static_cast<PeopleList *>( arg2 );
	
	list->Attach( pi );
	return false;
}

//-------------------------------------------------------------------------

bool global_save( PersonInfo *pi, void *arg2 ) {
	FILE *ptr = static_cast<FILE *>( arg2 );
	char encWrite;

	PRINT( ("global_save. uin: %d, name: %s, ptr: 0x%u\n", pi->uin, pi->item->GetDisplayName(), (uint32)ptr) );
	encWrite = pi->item->GetEncoding() == 42 ? 'Z' : TranslateFromEncoding(pi->item->GetEncoding());
	if( pi->item->GetOnlineNotify() )
		encWrite = tolower(encWrite);
	fprintf( ptr, "%c%lu%c%s\n", encWrite, pi->uin, 9, pi->item->GetDisplayName() );
	return false;
}

bool global_find( PersonInfo *pi, void *arg2 ) {
	uint32 *uin = static_cast<uint32 *>( arg2 );
	
	return (pi->uin == *uin);
}

//-------------------------------------------------------------------------

PeopleList::PeopleList(GManager *aManager) {
	
	PRINT( ("PeopleList::PeopleList\n") );
	mManager = aManager;
			
	mDisplay = NULL;
	
	*mFilename = '\0';
	mPersonList = new PList<PersonInfo>( 30 );
	mLock = new BLocker( "people_lock" );
}

//-------------------------------------------------------------------------

PeopleList::~PeopleList() {

	mPersonList->DoForEach( global_delete );
	delete mPersonList;
	delete mLock;
}

//-------------------------------------------------------------------------

void PeopleList::Reset() {

	BAutolock a(mLock );
	mDisplay -> Window() -> Lock();
	mDisplay -> MakeEmpty();
	mDisplay->AddItem( mOnlineGroup = new GroupItem( mManager->mLocale->FindString("label170")) );
	mDisplay->AddItem( mOfflineGroup = new GroupItem( mManager->mLocale->FindString("label171")) );
//	mDisplay->AddItem( pendingGrup = new GroupItem( mManager->mLocale->FindString("label172")) );
	mDisplay -> Window() -> Unlock();

	*mFilename = '\0';
	mUin = 0;
	
	mPersonList->DoForEach( global_delete );
	mPersonList->MakeEmpty();
	
}

//-------------------------------------------------------------------------

void PeopleList::SetDisplayList( BOutlineListView* aDisplay ) {

	BAutolock a( mLock );

	mDisplay = aDisplay;

	// load in the mPersonList groups
	mDisplay->AddItem( mOnlineGroup = new GroupItem( mManager->mLocale->FindString("label170")) );
	mDisplay->AddItem( mOfflineGroup = new GroupItem( mManager->mLocale->FindString("label171")) );
//	mDisplay->AddItem( pendingGrup = new GroupItem( mManager->mLocale->FindString("label172")) );
	
	mPersonList->DoForEach( global_attach, static_cast<void *>(this) );
	
}

//-------------------------------------------------------------------------

PersonInfo* PeopleList::AddPerson( uint32 aUin, const char* aNick, ICQStatus aStatus, uint32 aEncoding, bool aOnlineNotify, bool commitNow, bool check )
{
	PersonInfo *pi;
	
	BAutolock a(mLock );
	
	// if we are supposed to check if the person is already there, do it
	if( check ) {
		pi = mPersonList->FindItem( global_find, &aUin );
		if( pi ) {
			return NULL;		
		}		
	}

	// Set some information
	pi = new PersonInfo;
	pi->item = new PeopleItem( aUin, aNick, aStatus, aEncoding, aOnlineNotify, mManager );
	pi->uin = aUin;

	// Attach it to the list
	mPersonList->AddItem( pi );
	Attach( pi );

	// add it to the ICQ list
	BMessage* addMsg = new BMessage(ICQ_ADD_CONTACT);
	addMsg->AddInt32( "UIN", (uint32)aUin );
	mManager->icq->PostMessage( addMsg );
	
	if( commitNow )
		Save();

	return pi;
}

//-------------------------------------------------------------------------

void PeopleList::RemovePerson( uint32 aUin, bool commitNow ) {

	BAutolock a(mLock );
	PersonInfo* pi;
	
	// Find the person in the list, and if successful, remove it
	pi = mPersonList->FindItem( global_find, &aUin );
	if( pi )
		RemovePerson( pi, commitNow );

}

//-------------------------------------------------------------------------

bool 
PeopleList::HasPerson(uint32 aUin)
{
	return( mPersonList->FindItem( global_find, &aUin ) );
}

//-------------------------------------------------------------------------

void PeopleList::RemovePerson( PersonInfo* person, bool commitNow ) {
	
	BAutolock a(mLock );
	
	// detach the person from the list
	//Detach( person );
	if( mDisplay ) {
		mDisplay->Window()->Lock();
		mDisplay->RemoveItem( person->item );
		mDisplay->Window()->Unlock();
	}
	
	mPersonList->RemoveItem( person );
	// delete the sucker
	delete person;

	if( commitNow )
		Save();
}

//-------------------------------------------------------------------------

const char *PeopleList::FindNameForUIN( uint32 aUin ) {

	PersonInfo* pi = mPersonList->FindItem( global_find, &aUin );
	if( pi )
		return pi->item->GetDisplayName();
	return NULL;
}

//-------------------------------------------------------------------------

bool PeopleList::FindEncodingForUIN( uint32 aUin, uint32& aEncoding ) {

	PersonInfo* pi = mPersonList->FindItem( global_find, &aUin );
	if( pi ) {
		aEncoding = pi->item->GetEncoding();
		return true;
	}
	return false;
}

//-------------------------------------------------------------------------

void PeopleList::SetPersonStatus( uint32 aUin, ICQStatus aStatus )
{
	PersonInfo* pi;

	BAutolock a(mLock );
	
	PRINT( ( "list::user( %d ) is logged in. status: %d\n", aUin, aStatus ) );
	// Try and find that person and leave if it fails
	pi = mPersonList->FindItem( global_find, &aUin );
	if( !pi )
		return;

	// call the function that does the real work
	SetPersonStatus( pi, aStatus );
}

//-------------------------------------------------------------------------

void PeopleList::SetPersonStatus( PersonInfo* person, ICQStatus aStatus )
{
	if( !person || !mDisplay )
		return;

	BAutolock a(mLock );

	//mManager -> sounds->PlaySound( WS_ENTER );

	mDisplay->Window()->Lock();
	if( aStatus == IS_OFFLINE ) {
		mDisplay->RemoveItem( person->item );
		mDisplay->AddUnder( person->item, mOfflineGroup );
		mManager -> sounds->PlaySound( WS_EXIT );
	} else if( person->item->GetStatus() == IS_OFFLINE ) {
		mDisplay->RemoveItem( person->item );
		mDisplay->AddUnder( person->item, mOnlineGroup );
	}
	
	// do the notify is needed...
	if( (aStatus == IS_AVAILABLE) || (aStatus == IS_FREEFORCHAT) ) {
		if( person->item->GetOnlineNotify() ) {
			BMessage* msg = new BMessage(GIMICQ_ONLINE_NOTIFY);
			msg->AddInt32( "UIN", (int32)person->uin );
			msg->AddString("NICK", person->item->GetDisplayName() );
			mManager -> windows -> SendBuddyListMessage(msg);
		} else
			mManager -> sounds->PlaySound( WS_ENTER );
	}
	

	// now set it for real
	person->item->SetStatus(aStatus);
	mDisplay->InvalidateItem( mDisplay->IndexOf(person->item) );
	mDisplay->Window()->Unlock();

}

//-------------------------------------------------------------------------

void PeopleList::SetPersonEncoding( uint32 aUin, uint32 aEncoding )
{
	PersonInfo* pi;
	
	BAutolock a(mLock );

	// Try and find that person and leave if it fails
	pi = mPersonList->FindItem( global_find, &aUin );
	if( !pi )
		return;

	// call the function that does the real work
	SetPersonEncoding( pi, aEncoding );
}

//-------------------------------------------------------------------------

void PeopleList::SetPersonEncoding( PersonInfo* person, uint32 aEncoding )
{
	if( !person || !mDisplay )
		return;

	BAutolock a(mLock );

	// set the encoding
	person->item->SetEncoding( aEncoding );

}

//-------------------------------------------------------------------------

void PeopleList::SetPersonNotify( uint32 aUin, bool aOnlineNotify )
{
	BAutolock a(mLock );
	PersonInfo* pi;

	// Try and find that person and leave if it fails
	pi = mPersonList->FindItem( global_find, &aUin );
	if( !pi )
		return;

	// call the function that does the real work
	SetPersonNotify( pi, aOnlineNotify );
}

//-------------------------------------------------------------------------

void PeopleList::SetPersonNotify( PersonInfo* person, bool aOnlineNotify )
{
	if( !person || !mDisplay )
		return;

	BAutolock a(mLock );

	// set the onlineNotify
	person->item->SetOnlineNotify( aOnlineNotify );

}

//-------------------------------------------------------------------------

void PeopleList::SetPersonMessage(uint32 aUin, bool aUnreadMsg )
{
	PersonInfo* pi;

	BAutolock a(mLock );

	// Try and find that person and leave if it fails
	pi = mPersonList->FindItem( global_find, &aUin );
	if( !pi )
		return;

	// call the function that does the real work
	SetPersonMessage( pi, aUnreadMsg );

}

//-------------------------------------------------------------------------

void PeopleList::SetPersonMessage(PersonInfo *person, bool aUnreadMsg )
{
	if( !person || !mDisplay )
		return;

	BAutolock a(mLock );
	
	person->item->SetGotNewMessage( aUnreadMsg );

	if (mDisplay->Window()->LockWithTimeout(100000) == B_OK)	// wait for the 10th of a second
	{
		mDisplay->InvalidateItem( mDisplay->IndexOf(person->item) );
		mDisplay->Window()->Unlock();
	}

}

//-------------------------------------------------------------------------

void PeopleList::SetPersonNick( uint32 aUin, const char* aNick )
{
	PersonInfo* pi;
	
	BAutolock a(mLock );
	// Try and find that person and leave if it fails
	pi = mPersonList->FindItem( global_find, &aUin );
	if( !pi )
		return;

	// call the function that does the real work
	SetPersonNick( pi, aNick );

}

//-------------------------------------------------------------------------

void PeopleList::SetPersonNick( PersonInfo* person, const char* aNick )
{
	if( !person || !mDisplay )
		return;

	BAutolock a(mLock );

	// set the nickname
	mDisplay->Window()->Lock();
	person->item->SetNick( aNick );
	mDisplay->InvalidateItem( mDisplay->IndexOf(person->item) );
	mDisplay->Window()->Unlock();

	Save();
}


//-------------------------------------------------------------------------

void PeopleList::Load() {
	
	FILE *ptr;
	char tmpchar = 0;
	uint32 i = 0, j = 0, k = 0;
	uint32 newUID;
	char data[512];
	char person[128];
	char group[128];
	bool onlineNotify = false;
	uint32 encoding;
	
	BAutolock a(mLock );
	
	BFile file;
	BPath path;
	mManager->prefs->FindConfigFile( NULL, mFilename, &file, B_READ_ONLY, &path );
	file.Unset();
	
	// open the people file
	ptr = fopen( path.Path(), "r" );
	if( ptr == 0 )
		return;

	// read through the lines in the file
	tmpchar = getc( ptr );
	while( tmpchar != -1 ) {
	
		i = 0;
		while( tmpchar != -1 ) {
			if( tmpchar == '\n' ) {
				data[i] = '\0';
				break;
			}
			data[i++] =	tmpchar;
			tmpchar = getc( ptr );
		}
		
		tmpchar = getc( ptr );
		k = 0;
		j = 0;
		
		// check: was this a group and not a person?
//		isgroup = false;
//		if( data[j++] == 'g' )
//			isgroup = true;	

		// get the UIN
		while( data[j] && data[j] != char(9) )
			person[k++] = data[j++];
		person[k] = '\0';
		newUID = atoi( person+1 );

		// get the nick
		k = 0; 	++j;
		while( data[j] && data[j] != char(10) )
			group[k++] = data[j++];
		group[k] = '\0';

		//stat = (person[0] == 'p' ? IS_PENDING : IS_OFFLINE);
		onlineNotify = false;
		if( islower(person[0]) ) {
			onlineNotify = true;
			person[0] = toupper( person[0] );
		}
		if( person[0] == 'o' )
			person[0] = 'Z';
		encoding = TranslateToEncoding( person[0], mManager );

		AddPerson( newUID, k != 0 ? group : NULL, IS_OFFLINE, encoding, onlineNotify, false );

	}
	
	// close the file output
	fclose( ptr );

	Debug();
}

//-------------------------------------------------------------------------

void PeopleList::Save() {

	FILE *ptr;

	BAutolock a(mLock );
	
	BFile file;
	BPath path;
	mManager->prefs->FindConfigFile( NULL, mFilename, &file, B_READ_ONLY, &path );
	file.Unset();

	// open the people file
	ptr = fopen( path.Path(), "w" );
	if( ptr == 0 )
		return;	

	// write out the entries
	PRINT( ("Save. ptr: 0x%u\n", (uint32)ptr) );
	mPersonList->DoForEach( global_save, static_cast<void *>(ptr) );
	
	fclose( ptr );

	Debug();
}


//-------------------------------------------------------------------------

void PeopleList::SetProgramUser( uint32 aUin ) {

	BAutolock a(mLock );
	
	Reset();
	// Change the name
	mUin = aUin;

	// make the filename based on the current username
	MakeFilenameFromUIN( mFilename, mUin );
		
	// read the contents from a config file
	Load();

}

//-------------------------------------------------------------------------

void PeopleList::Attach( PersonInfo* newPerson ) {

	PRINT( ("PeopleList::Attach\n") );
	if( !mDisplay )
		return;

	BListItem* addUnder = NULL;
	switch( newPerson->item->GetStatus() ) {
	case IS_OFFLINE:
		addUnder = mOfflineGroup;
		break;
	default:
		addUnder = mOnlineGroup;
		break;
	}
	
	if( mDisplay && addUnder ) {
		mDisplay->Window()->Lock();
		mDisplay->AddUnder( newPerson->item, addUnder );
		mDisplay->Window()->Unlock();
	}

}
	
//-------------------------------------------------------------------------

#if DEBUG
bool global_debug( PersonInfo *pi ) {
	PRINT( ("\tuser uin: %u, name: %s\n", pi->uin, pi->item->GetDisplayName()) );
	return false;
}

void PeopleList::Debug() {

	PRINT(("\nPeople List:\n"));
	mPersonList->DoForEach( global_debug );
	PRINT(("End of List.\n\n"));

}

#else
void PeopleList::Debug() {}
#endif	
//-------------------------------------------------------------------------

