#include <Debug.h>
#include "TrayIcon.h"

#include <Bitmap.h>
#include <MessageRunner.h>
#include <Window.h>
#include <PopUpMenu.h>
#include <MenuItem.h>

#include "GConstants.h"
#include "Gicq.h"
//---------------------------------------------------------

const uint32 TRAYICON_TICK = 'tiCK';
const bigtime_t mTimeout = 1000000;

//---------------------------------------------------------

TrayIcon::TrayIcon(BRect frame, BBitmap* active, BBitmap* inactive)
	: BView(frame, kViewName, B_FOLLOW_NONE, B_WILL_DRAW)
{
	ActiveIcon = active;
	InactiveIcon = inactive;
	runner = NULL;
	iconOn = true;
	flashing = false;
	doFlash = true;
	flashCount = 0;
	status = IS_AVAILABLE;
}


//---------------------------------------------------------

TrayIcon::TrayIcon( BMessage *archive )
	: BView( archive )
{
	BMessage bits;
	archive->FindMessage("ActiveIcon", &bits);
	ActiveIcon = new BBitmap(&bits);
	
	bits.MakeEmpty();
	archive->FindMessage("InactiveIcon", &bits);
	InactiveIcon = new BBitmap(&bits);

	runner = NULL;
	iconOn = false;
	flashing = false;
	flashCount = 0;
	status = IS_AVAILABLE;
}

//---------------------------------------------------------

TrayIcon::~TrayIcon()
{
	if(ActiveIcon) delete ActiveIcon;
	if(InactiveIcon) delete InactiveIcon;
}

//---------------------------------------------------------

void TrayIcon::FlashOn() {
	++flashCount;
	flashing = true;
}

//---------------------------------------------------------

void TrayIcon::FlashOff() {
	if( flashing && flashCount )
		--flashCount;
	if( !flashCount ) {
		flashing = false;
		Invalidate();
	}
}

//---------------------------------------------------------

void TrayIcon::AttachedToWindow() 
{ 
	// set view color based on parent
	if (Parent()) 
		SetViewColor(Parent()->ViewColor()); 

	// set up the ticker
	BMessage* tickMsg = new BMessage(TRAYICON_TICK);
	runner = new BMessageRunner( this, tickMsg, 500000 );

	// call inherited class
	BView::AttachedToWindow();
}

//---------------------------------------------------------

void TrayIcon::Draw(BRect updateRect)
{
	BView::Draw(updateRect);
	SetDrawingMode(B_OP_ALPHA);

	// just draw the active icon for now until we can implement status
	if( flashing && doFlash ) {
		if( iconOn ) {
			if( ActiveIcon )
				DrawBitmap(InactiveIcon);
		} else {
			BRect rect = Bounds();
			SetHighColor( 189, 186, 189 );
			FillRect( rect );
		}
	} else 
		DrawBitmap( ActiveIcon );
}

//---------------------------------------------------------

void TrayIcon::MouseDown(BPoint where)
{
	BMessage* statMsg;
	uint32 mouseButtons;
	Window()->CurrentMessage()->FindInt32("buttons", (long*)&mouseButtons);
	switch (mouseButtons)
	{
		// left: send a message to GIMICQ telling it to toggle minimization
		case B_PRIMARY_MOUSE_BUTTON: {
			
			// just toggle the contact list
			if( !flashing ) {
				BMessage msg(GIMICQ_TOGGLE_HIDDEN);
				BMessenger(APP_SIGNATURE).SendMessage(&msg);
				break;
			} 
			
			// put in the popup menu
			else {
				bool kg;
				flashItem temp;
				char menuString[100];
				BPopUpMenu* popup = new BPopUpMenu("Events",false,false);
				BMessage* sndMessage;
				kg = flashes.First(temp);
				while( kg ) {
					if( temp.count == 1 )
						sprintf( menuString, "Message from %s", temp.display.String() );
					else
						sprintf( menuString, "%d messages from %s", temp.count, temp.display.String() );
					sndMessage = new BMessage(GIMICQ_OPEN_USER_WINDOW);
					sndMessage->AddInt32( "wtype", (int32)USER_MESSAGE_TYPE );
					sndMessage->AddInt32( "UIN", temp.uin );
					popup->AddItem( new BMenuItem( menuString, sndMessage ) );
					kg = flashes.Next(temp);
				}
				ConvertToScreen(&where); 
				popup->SetTargetForItems(BMessenger(APP_SIGNATURE, -1, NULL));
				popup->Go( where,true,false,true );
				break;
			}
		}

		// right: popup a menu (which does very little at the moment...)
		case B_SECONDARY_MOUSE_BUTTON: {
			BPopUpMenu* popup = new BPopUpMenu("Status",false,false);

			//popup->AddItem(new BMenuItem("Quit Gim-ICQ", new BMessage(B_QUIT_REQUESTED)));
			popup->AddSeparatorItem();
			popup->AddItem(new BMenuItem("Search for Contacts" B_UTF8_ELLIPSIS, new BMessage(GIMICQ_SEARCH_BY_EMAIL) ));
			popup->AddItem(new BMenuItem("Update Info" B_UTF8_ELLIPSIS, new BMessage(GIMICQ_UPDATE_MY_INFO) ));
			popup->AddSeparatorItem();
			popup->AddItem(new BMenuItem("Preferences" B_UTF8_ELLIPSIS, new BMessage(GIMICQ_OPEN_PREFS) ));
			//popup->AddItem(new BMenuItem("About Gim-ICQ" B_UTF8_ELLIPSIS, new BMessage(B_ABOUT_REQUESTED) ));
			popup->SetTargetForItems(BMessenger(APP_SIGNATURE, -1, NULL));
			
			popup->AddSeparatorItem();
			statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			statMsg->AddInt32( "status", (int32)IS_AVAILABLE );
			popup->AddItem( stat1 = new BMenuItem("Available" B_UTF8_ELLIPSIS, statMsg ));
			stat1->SetMarked( bool(status == IS_AVAILABLE) );
			statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			statMsg->AddInt32( "status", (int32)IS_FREEFORCHAT );
			popup->AddItem( stat2 = new BMenuItem("Free For Chat" B_UTF8_ELLIPSIS, statMsg ));
			stat2->SetMarked( bool(status == IS_FREEFORCHAT) );
			statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			statMsg->AddInt32( "status", (int32)IS_AWAY );			
			popup->AddItem( stat3 = new BMenuItem("Away" B_UTF8_ELLIPSIS, statMsg ));
			stat3->SetMarked( bool(status == IS_AWAY) );
			statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			statMsg->AddInt32( "status", (int32)IS_EXTENDEDAWAY );
			popup->AddItem( stat4 = new BMenuItem("Not Available" B_UTF8_ELLIPSIS, statMsg ));
			stat4->SetMarked( bool(status == IS_EXTENDEDAWAY) );
			statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			statMsg->AddInt32( "status", (int32)IS_OCCUPIED );
			popup->AddItem( stat5 = new BMenuItem("Occupied" B_UTF8_ELLIPSIS, statMsg ));
			stat5->SetMarked( bool(status == IS_OCCUPIED) );
			statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			statMsg->AddInt32( "status", (int32)IS_DND );
			popup->AddItem( stat6 = new BMenuItem("Do Not Disturb" B_UTF8_ELLIPSIS, statMsg ));
			stat6->SetMarked( bool(status == IS_DND) );
			statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			statMsg->AddInt32( "status", (int32)IS_INVISIBLE );
			popup->AddItem( stat7 = new BMenuItem("Invisible" B_UTF8_ELLIPSIS, statMsg ));
			stat7->SetMarked( bool(status == IS_INVISIBLE) );
			
			ConvertToScreen(&where); 
			popup->SetTargetForItems(BMessenger(APP_SIGNATURE, -1, NULL));
			popup->Go( where,true,false,true );
		}
	}
}

//---------------------------------------------------------

void TrayIcon::MessageReceived(BMessage* msg)
{
	switch(msg->what) {
	
		case B_SET_PROPERTY: {

			switch( msg->FindInt32("what2") ) {
				case ICQ_URL_MESSAGE:
				case GIMICQ_FLASH_ON:
				case ICQ_TEXT_MESSAGE:
				case GIMICQ_FLASH_OFF:
					SetFlash(msg);
					break;
				case GIMICQ_ENABLE_FLASHY_STUFF:
					doFlash = msg->FindBool("enabled");
					break;
				case GIMICQ_MY_STATUS_CHANGE:
					status = (ICQStatus)msg->FindInt32("status");
					break;
			}
			break;
		}

		case TRAYICON_TICK:
			if( flashing && doFlash )
				Flash();
			break;

		default:
			BView::MessageReceived(msg);
			break;
	}
}	

//---------------------------------------------------------

void TrayIcon::SetFlash( BMessage* msg ) {

	uint32 msg2;
	bool kg, found = false, isOn;
	unsigned uin;
	int count = 0;
	flashItem temp;

	if( !msg ) 
		return;
		
	msg2 = (uint32)msg->FindInt32("what2");
	isOn = (msg2 == GIMICQ_FLASH_ON);
	uin = msg->FindInt32("UIN");

	// try and find this uin in the flash list
	kg = flashes.First(temp);
	while( kg ) {
		if( temp.uin == uin ) {
			found = true;
			break;
		}
		++count;
		kg = flashes.Next(temp);
	}
	
	// if we found it and it's a NEW flash, just increment it
	if( found && isOn ) {
		flashes[count].count++;
	}
	
	// if we found it and we're supposed to delete it, do that
	else if( found && !isOn ) {
		flashes.Delete(count);
		FlashOff();
	}
	
	// if we're adding a new flash and it ain't there, add it	
	else if( !found && isOn ) {
		flashItem newFlash;
		newFlash.uin = uin;
		newFlash.count = 1;
#warning Changed the display.
		msg->FindString("nick", &(newFlash.display) );
		flashes.Add( newFlash );
		FlashOn();
	}
}

//---------------------------------------------------------

void TrayIcon::Flash() {
	iconOn = !iconOn;
	Invalidate();
}

//---------------------------------------------------------

TrayIcon* TrayIcon::Instantiate(BMessage* archive)
{
	if(!validate_instantiation(archive, "TrayIcon"))
		return NULL;
		
	return new TrayIcon(archive);
}

//---------------------------------------------------------

status_t TrayIcon::Archive( BMessage *archive, bool deep ) const
{
	// archive this bad boy
	BView::Archive(archive, deep);
	
	// save the icons
	BMessage	bits(B_ARCHIVED_OBJECT);
	if(ActiveIcon) {
		ActiveIcon->Archive(&bits);
		archive->AddMessage("ActiveIcon", &bits);
		bits.MakeEmpty();
	}
	
	if(InactiveIcon) {
		InactiveIcon->Archive(&bits);
		archive->AddMessage("InactiveIcon", &bits);
	}
	
	archive->AddString("add_on", APP_SIGNATURE);
	return B_OK;
}

//---------------------------------------------------------
