#include <Debug.h>
#include "History.h"

#include <Message.h>
#include <Window.h>
#include <Locker.h>
#include <Autolock.h>

#include "GConstants.h"
#include "GManager.h"
#include "Gicq.h"
#include "HistoryIO.h"

//--------------- global --------------------

bool global_hist_read( struct chatEntry *aItem, void *aArg )
{
	BMessage *data = static_cast<BMessage *>(aArg);
	
	uint32 uin = data->FindInt32( "uin" );
	uint32 own_uin = data->FindInt32( "own_uin" );

	if( (aItem->uin != uin) || (aItem->own_uin!=own_uin) )
		return false;

	
	BMessage *msg = new BMessage;
	if( aItem->type == CT_OFFLINE_DONE ) {
	} else if( aItem->type == CT_INCOMING_MESSAGE ) {
		msg->what = ICQ_TEXT_MESSAGE;
		msg->AddString( "TEXT", aItem->message.String() );
	} else if( aItem->type == CT_INCOMING_URL ) {
		msg->what = ICQ_URL_MESSAGE;
		msg->AddString( "TEXT", aItem->message.String() );
		msg->AddString( "URL", aItem->url.String() );
	} else if( aItem->type == CT_OUTGOING_URL ) {
		msg->what = GIMICQ_OUTGOING_URL;
		msg->AddString( "TEXT", aItem->message.String() );
		msg->AddString( "URL", aItem->url.String() );
	} else if( aItem->type == CT_OUTGOING_MESSAGE ) {
		msg->what = GIMICQ_OUTGOING_IM;
		msg->AddString( "TEXT", aItem->message.String() );
	}
	
	BWindow *w=NULL;
	data->FindPointer( "window", reinterpret_cast<void **>(&w));

	w->LockLooper();
	w->PostMessage( msg );
	w->UnlockLooper();
	delete msg;
	return false;
}

bool global_hist_delete( struct chatEntry *aItem)
{
	delete aItem;
	return false;
}

//--------------- class --------------------
History::History(GManager *aManager)
{
	mIO = new HistoryIO(aManager);
	mLocker = new BLocker();
}

History::~History()
{

	mIO -> Lock();
	mIO -> Quit();
	delete mLocker;
	
	mStorage.DoForEach( global_hist_delete );
	mStorage.MakeEmpty();

}

void History::Add(BMessage *aMsg)
{
	BAutolock a( mLocker );
	struct chatEntry *entry = new chatEntry;
	
	mIO -> Lock();
	mIO -> PostMessage(aMsg);	
	mIO -> Unlock();
	
	switch(aMsg->what) {
	
	case GIMICQ_OUTGOING_URL:
		entry->type = CT_OUTGOING_URL;
		entry->message = aMsg->FindString("TEXT");
		entry->url = aMsg->FindString("URL");
		break;

	case GIMICQ_ATTEMPT_SEND_IM:
		entry->type = CT_OUTGOING_MESSAGE;
		entry->message = aMsg->FindString("TEXT");
		break;

	case ICQ_TEXT_MESSAGE:
		entry->type = CT_INCOMING_MESSAGE;
		entry->message = aMsg->FindString("TEXT");
		break;

	case ICQ_URL_MESSAGE:
		entry->type = CT_INCOMING_URL;
		entry->message = aMsg->FindString("TEXT");
		entry->url = aMsg->FindString("URL");
		break;
	}
	entry->uin = (uint32) aMsg->FindInt32("uin");
	entry->own_uin = (uint32) aMsg->FindInt32("own_uin");
	entry->time = (time_t) aMsg->FindInt64("time");

	mStorage.AddItem(entry);

}

void History::Clear(uint32 aUin, uint32 aOwnUin)
{
	BAutolock a( mLocker );
	int i;
	chatEntry *entry;
	for( i=0; i<mStorage.CountItems();) {
		entry = mStorage.ItemAt(i);
		if( (entry->uin != aUin) || (entry->own_uin!=aOwnUin) ) {
			i++;		//Only increase i if an element is not deleted.
			continue;
		}

		mStorage.RemoveItem( i );
		delete entry;
	}
}

void 
History::ReadAll(const uint32 aUin, const uint32 aOwnUin, BWindow *aWindow)
{
	BAutolock a( mLocker );
	BMessage data;
	data.AddInt32( "uin", aUin );
	data.AddInt32( "own_uin", aOwnUin );
	data.AddPointer( "window", aWindow );
	mStorage.DoForEach( global_hist_read, &data );

	aWindow -> LockLooper();
	aWindow -> PostMessage( GIMICQ_LAST_MESSAGE );
	aWindow -> UnlockLooper();
}

