/*	BitmapView.cpp
 */

#include <InterfaceKit.h>
#include <string.h>
#include "BitmapView.h"

BitmapView::BitmapView(
	BBitmap *		map,
	const char *	comment) :
	BView(map->Bounds(), "bitmap", B_FOLLOW_ALL, B_WILL_DRAW | B_FRAME_EVENTS),
	fMargins(-20,-20,20,20)
{
	fBitmap = map;
	fOwnership = true;
	fAdjustBars = true;
	rgb_color gray = { 220, 220, 220, 0 };
	SetViewColor(gray);
	if (comment != NULL)
		strncpy(fComment, comment, 255);
	else
		fComment[0] = 0;
	fDrawComment = false;
	fComment[255] = 0;

	//	reveal comment under picture
	if (fComment[0]) {
		BFont font;
		GetFont(&font);
		font_height fh;
		font.GetHeight(&fh);
		fCommentPos.y = map->Bounds().bottom+fh.ascent+fh.leading+4;
		fCommentPos.x = (map->Bounds().right-font.StringWidth(fComment))/2;
		fDrawComment = true;
		if (fMargins.bottom < fh.ascent+fh.descent+fh.leading+8)
			fMargins.bottom = fh.ascent+fh.descent+fh.leading+8;
	}
}


BitmapView::~BitmapView()
{
	if (fOwnership && fBitmap)
		delete fBitmap;
}


void
BitmapView::SetOwnership(
	bool			ownership)
{
	fOwnership = ownership;
}


bool
BitmapView::GetOwnership()
{
	return fOwnership;
}


void
BitmapView::SetBitmap(
	BBitmap *		map)
{
	if (fOwnership && fBitmap)
		delete fBitmap;
	fBitmap = map;
	Invalidate();
	if (fAdjustBars)
		AdjustBars();
}


BBitmap *
BitmapView::GetBitmap()
{
	return fBitmap;
}


void
BitmapView::SetMargins(
	BRect			margins)
{
	fMargins = margins;
	if (fAdjustBars)
		AdjustBars();
}


BRect
BitmapView::GetMargins()
{
	return fMargins;
}


void
BitmapView::SetAdjustBars(
	bool			adjust)
{
	fAdjustBars = adjust;
	if (fAdjustBars)
		AdjustBars();
}


bool
BitmapView::GetAdjustBars()
{
	return fAdjustBars;
}


void
BitmapView::Draw(
	BRect			area)
{
	if (!fBitmap)
		return;
	BRect frame = fBitmap->Bounds();
	area = area & frame;
	if ((area.right >= area.left) && (area.bottom >= area.top))
		DrawBitmap(fBitmap, area, area);
	frame.InsetBy(-1, -1);
	StrokeRect(frame);
	if (fDrawComment) {
		BFont font;
		GetFont(&font);
		float width = font.StringWidth(fComment);
		font_height fh;
		font.GetHeight(&fh);
		float baseline = fh.ascent+fh.leading;
		BRect frame(fCommentPos.x-2, fCommentPos.y-baseline-1, 
			fCommentPos.x+width+1, fCommentPos.y+fh.descent+1);
		rgb_color low_color = { 255, 255, 222, 0 };
		rgb_color high_color = { 0, 0, 0, 0 };
		SetHighColor(high_color);
		SetLowColor(low_color);
		FillRect(frame, B_SOLID_LOW);
		StrokeRect(frame);
		SetDrawingMode(B_OP_COPY);
		DrawString(fComment, fCommentPos);
	}
}


void
BitmapView::MouseDown(
	BPoint		where)
{
	//	this may be changed sometime later
	if (!fComment[0]) return;
	BFont font;
	GetFont(&font);
	float width = font.StringWidth(fComment);
	font_height fh;
	font.GetHeight(&fh);
	float baseline = fh.ascent+fh.leading;
	if (fDrawComment && 
			(where.x >= fCommentPos.x-1) && (where.x <= fCommentPos.x+width+1) &&
			(where.y >= fCommentPos.y-baseline) && (where.y <= fCommentPos.y+fh.descent)) {
		fDrawComment = false;
	} else {
		if (fDrawComment)
			Invalidate(BRect(fCommentPos.x-2, fCommentPos.y-baseline-1, 
				fCommentPos.x+width+1, fCommentPos.y+fh.descent+1));
		fDrawComment = true;
		fCommentPos = where;
		fCommentPos.x -= width/2;
	}
	Invalidate(BRect(fCommentPos.x-2, fCommentPos.y-baseline-1, 
		fCommentPos.x+width+1, fCommentPos.y+fh.descent+1));
	return;
}


void
BitmapView::FrameResized(
	float			newWidth,
	float			newHeight)
{
	BView::FrameResized(newWidth, newHeight);
	if (fAdjustBars)
		AdjustBars();
}


	static void
	AdjustBar(
		BScrollBar *		bar,
		float				page,
		float				total,
		float				start)
	{
		if (total <= page) {
			bar->SetRange(start, start);
		} else {
			bar->SetRange(start, start+total-page);
		}
		float pgStep = page-4.0;
		if (pgStep<16.0)
			pgStep = 16.0;
		bar->SetSteps(4.0, pgStep);
	}

void
BitmapView::AdjustBars()
{
	BRect area(0,0,0,0);
	BRect bounds(Bounds());
	if (fBitmap)
		area = fBitmap->Bounds();

	BScrollBar *bar = ScrollBar(B_HORIZONTAL);
	if (bar)
		AdjustBar(bar, bounds.Width(), area.Width()+fMargins.Width()+1, fMargins.left);
	bar = ScrollBar(B_VERTICAL);
	if (bar)
		AdjustBar(bar, bounds.Height(), area.Height()+fMargins.Height()+1, fMargins.top);
}


void
BitmapView::WindowActivated(
	bool			state)
{
	BView::WindowActivated(state);
	if (fAdjustBars && state) {
		AdjustBars();
	}
}






