// MediaConverter.cpp
//
//   Copyright 1999, Be Incorporated.   All Rights Reserved.
//   This file may be used under the terms of the Be Sample Code License.

#include "MediaConverter.h"
#include "EncoderView.h"

BDirectory dir;
bool openflag = false;
bool saveflag = false;
bool durationchanged = false;
int32 priority = B_NORMAL_PRIORITY;
int32 videoquality = 75;
int32 audioquality = 75;

// Save File Panel Filter;

class DirectoryFilter : public BRefFilter 
{ 
public: 
        DirectoryFilter(){}; 
        virtual bool Filter(const entry_ref *ref, 
                BNode *node, struct stat *st, const char *filetype)
		{ 
			return node->IsDirectory(); 
		}
private: 
}; 

// ------------------- MediaFileListItem -------------------

class MediaFileListItem : public BStringItem
{
public:
				MediaFileListItem(BMediaFile *f, entry_ref *ref);
	virtual		~MediaFileListItem();

	entry_ref	fRef;
	BMediaFile	*fMediaFile;
};

MediaFileListItem::MediaFileListItem(BMediaFile *f, entry_ref *ref)
	: BStringItem(ref->name),
	  fRef(*ref),
	  fMediaFile(f)
{
}

MediaFileListItem::~MediaFileListItem()
{
	delete fMediaFile;
}

// ------------------- FileFormatMenuItem -------------------

class FileFormatMenuItem : public BMenuItem
{
public:
				FileFormatMenuItem(media_file_format *format);
	virtual		~FileFormatMenuItem();
	
	media_file_format fFileFormat;
};

FileFormatMenuItem::FileFormatMenuItem(media_file_format *format)
	: BMenuItem(format->pretty_name, new BMessage(FORMAT_SELECT_MESSAGE))
{
	memcpy(&fFileFormat, format, sizeof(fFileFormat));
}


FileFormatMenuItem::~FileFormatMenuItem()
{
}

// ------------------- CodecMenuItem -------------------

class CodecMenuItem : public BMenuItem
{
public:
				CodecMenuItem(media_codec_info *ci, uint32 msg_type);
	virtual		~CodecMenuItem();
	
	media_codec_info fCodecInfo;
};

CodecMenuItem::CodecMenuItem(media_codec_info *ci, uint32 msg_type)
	: BMenuItem(ci->pretty_name, new BMessage(msg_type))
{
	memcpy(&fCodecInfo, ci, sizeof(fCodecInfo));
}


CodecMenuItem::~CodecMenuItem()
{
}

// ------------------- MediaFileInfoView implementation -------------------

MediaFileInfoView::MediaFileInfoView(BRect frame, uint32 resizingMode)
	: BView(frame, "MediaFileInfoView", resizingMode,
			B_FULL_UPDATE_ON_RESIZE | B_WILL_DRAW)
{
	fMediaFile = NULL;
	memset(&fRef, 0, sizeof(fRef));
	BFont font;
	GetFont(&font);
	font.SetSize(12);
	SetFont(&font);
}


MediaFileInfoView::~MediaFileInfoView()
{
}

void 
MediaFileInfoView::Update(BMediaFile *f, entry_ref *ref)
{
	fMediaFile = f;
	if (f != NULL) {
		fRef = *ref;
	}
	Invalidate();
}

void 
MediaFileInfoView::GetFileInfo(BString *audioFormat, BString *videoFormat,
							   BString *audioDetails, BString *videoDetails,
							   BString *duration)
{
	if (fMediaFile == NULL) {
		return;
	}
	
	BMediaTrack *track;
	media_format format;
	memset(&format, 0, sizeof(format));
	media_codec_info codecInfo;
	bool audioDone(false), videoDone(false);
	bigtime_t audioDuration(0), videoDuration(0);
	int32 tracks = fMediaFile->CountTracks();
	int64 videoFrames = 0;
	int64 audioFrames = 0;
	for (int32 i = 0; i < tracks && (!audioDone || !videoDone); i++) {
		track = fMediaFile->TrackAt(i);
		if (track != NULL) {
			track->EncodedFormat(&format);
			if (format.IsVideo()) {
				memset(&format, 0, sizeof(format));
				format.type = B_MEDIA_RAW_VIDEO;
				track->DecodedFormat(&format);
				media_raw_video_format *rvf = &(format.u.raw_video);

				track->GetCodecInfo(&codecInfo);
				*videoFormat << codecInfo.pretty_name;
				videoDuration = track->Duration();
				videoFrames = track->CountFrames();

				*videoDetails << (int32)format.Width() << "x" << (int32)format.Height()
							 << " " << (int32)(rvf->field_rate / rvf->interlace)
							 << " fps / "  << videoFrames << " frames";

				videoDone = true;

			} else if (format.IsAudio()) {
				memset(&format, 0, sizeof(format));
				format.type = B_MEDIA_RAW_AUDIO;
				track->DecodedFormat(&format);
				media_raw_audio_format *raf = &(format.u.raw_audio);
				char bytesPerSample = (char)(raf->format & 0xf);
				if (bytesPerSample == 1) {
					*audioDetails << "8 bit ";
				} else if (bytesPerSample == 2) {
					*audioDetails << "16 bit ";
				} else {
					*audioDetails << bytesPerSample << "byte ";
				}

				track->GetCodecInfo(&codecInfo);
				*audioFormat << codecInfo.pretty_name;
				audioDuration = track->Duration();
				audioFrames = track->CountFrames();

				*audioDetails << (float)(raf->frame_rate / 1000.0f) << " kHz";
				if (raf->channel_count == 1) {
					*audioDetails << " mono / ";
				} else if (raf->channel_count == 2) {
					*audioDetails << " stereo / ";
				} else {
					*audioDetails << (int32)raf->channel_count << " channel / " ;
				}
				*audioDetails << audioFrames << " frames";
				audioDone = true;
			}
			fMediaFile->ReleaseTrack(track);
		}	
	}
//	
	*duration << (int32)(MAX(audioDuration, videoDuration) / 1000000)
			  << " seconds";
	end_time = "";
	end_time << (int64)(MAX(audioDuration, videoDuration) / 1000);
	if (durationchanged)
	{
		textcontrol1->SetText("0");
		textcontrol2->SetText(end_time.String());
		durationchanged = false;
	}
//
}


void 
MediaFileInfoView::Draw(BRect /*update*/)
{
	font_height fh;
	GetFontHeight(&fh);
	BPoint p(2, fh.ascent + fh.leading), p2;
	BFont font;
	GetFont(&font);
	font.SetFace(B_BOLD_FACE);
	font.SetSize(12);
	SetFont(&font);
	
	if (fMediaFile != NULL) {
		BString aFmt, vFmt, aDetails, vDetails, duration;
		GetFileInfo(&aFmt, &vFmt, &aDetails, &vDetails, &duration);
		
		// draw filename
		DrawString(fRef.name, p);
		float lineHeight = fh.ascent + fh.descent + fh.leading;
		p.y += (float)ceil(lineHeight * 1.5);
		
		float durLen = StringWidth(DURATION_LABEL) + 5;
		float audLen = StringWidth(AUDIO_INFO_LABEL) + 5;
		float vidLen = StringWidth(VIDEO_INFO_LABEL) + 5;
		float maxLen = MAX(durLen, audLen);
		maxLen = MAX(maxLen, vidLen);
				
		// draw labels
		DrawString(AUDIO_INFO_LABEL, p + BPoint(maxLen - audLen, 0));
		p2 = p;
		p2.x += maxLen + 4;
		p.y += lineHeight * 2;
		DrawString(VIDEO_INFO_LABEL, p + BPoint(maxLen - vidLen, 0));
		p.y += lineHeight * 2;
		DrawString(DURATION_LABEL, p + BPoint(maxLen - durLen, 0));

		// draw audio/video/duration info
		font.SetFace(B_REGULAR_FACE);
		font.SetSize(10);
		SetFont(&font);
		
		DrawString(aFmt.String(), p2);
		p2.y += lineHeight;
		DrawString(aDetails.String(), p2);
		p2.y += lineHeight;
		DrawString(vFmt.String(), p2);
		p2.y += lineHeight;
		DrawString(vDetails.String(), p2);
		p2.y += lineHeight;
		DrawString(duration.String(), p2);
	} else {
		DrawString(NO_FILE_LABEL, p);
	}
}

void 
MediaFileInfoView::AttachedToWindow()
{
	rgb_color c = Parent()->LowColor();
	SetViewColor(c);
	SetLowColor(c);
}


// ------------------- MediaFileListView implementation -------------------

MediaFileListView::MediaFileListView(BRect frame, uint32 resizingMode)
	: BListView(frame, "MediaFileListView", B_SINGLE_SELECTION_LIST, resizingMode,
				B_WILL_DRAW | B_NAVIGABLE | B_FRAME_EVENTS)
{
	fEnabled = true;
}


MediaFileListView::~MediaFileListView()
{
	BListItem *item;
	while ((item = RemoveItem((int32)0)) != NULL) {
		delete item;
	}
}

void 
MediaFileListView::SetEnabled(bool enabled)
{
	if (enabled != fEnabled) {
		fEnabled = enabled;
	}
}

bool 
MediaFileListView::IsEnabled()
{
	return fEnabled;
}

void
MediaFileListView::AddItem(BMediaFile *f, entry_ref *ref)
{
	BListView::AddItem(new MediaFileListItem(f, ref));
	BMessenger(be_app).SendMessage(FILE_LIST_CHANGE_MESSAGE);
}

void 
MediaFileListView::KeyDown(const char *bytes, int32 numBytes)
{
	switch (bytes[0]) {
	case B_DELETE:
		if (IsEnabled()) {
			int32 selection = CurrentSelection();
			if (selection >= 0) {
				BListItem *item = RemoveItem(selection);
				if (item != NULL) {
					delete item;
				}
				int32 count = CountItems();
				if (selection >= count) {
					selection = count - 1;
				}
				Select(selection);
				BMessenger(be_app).SendMessage(FILE_LIST_CHANGE_MESSAGE);
			}		
		}
		break;
	default:
		BListView::KeyDown(bytes, numBytes);
	}
}

void 
MediaFileListView::SelectionChanged()
{
	MediaConverterWindow *win = dynamic_cast<MediaConverterWindow *>(Window());
	if (win != NULL) {
		win->SourceFileSelectionChanged();
	}
}

// ------------------- StatusView implementation -------------------


StatusView::StatusView(BRect frame, uint32 resizingMode)
	: BView(frame, "StatusView", resizingMode, B_WILL_DRAW | B_FULL_UPDATE_ON_RESIZE),
		fStatusString(B_EMPTY_STRING)
{
	frame.OffsetTo(0, 0);
	fStatusRect = frame;
	BFont font;
	GetFont(&font);
	font.SetSize(12);
	SetFont(&font);
}


StatusView::~StatusView()
{
}

void 
StatusView::AttachedToWindow()
{
	rgb_color c = Parent()->LowColor();
	SetViewColor(c);
	SetLowColor(c);
}


void
StatusView::Draw(BRect /*update*/)
{
	// Placement information should be cached here for better performance
	font_height fh;
	GetFontHeight(&fh);
	DrawString(fStatusString.String(),
			   fStatusRect.LeftBottom() + BPoint(1, -fh.descent));
}

void
StatusView::MessageReceived(BMessage *msg)
{
	BView::MessageReceived(msg);
}

void 
StatusView::SetStatus(const char *text)
{
	fStatusString.SetTo(text);
	Invalidate();
}

const char *
StatusView::Status()
{
	return fStatusString.String();
}

// ------------------- MediaConverterWindow implementation -------------------

void MediaConverterWindow::LanguageChanged()
{
	DestroyMenu();
	CreateMenu();
	SetLabel();
	BuildAudioVideoMenus();
	Lock();
	fInfoView->Invalidate();
	Unlock();
}

void MediaConverterWindow::SetLabel()
{
	char buffer[255];

	if (box1 != NULL)
		box1->SetLabel(SOURCE_BOX_LABEL);
	if (box2 != NULL)
		box2->SetLabel(INFO_BOX_LABEL);
	if (box3 != NULL)
		box3->SetLabel(OUTPUT_BOX_LABEL);

	BRect r(Bounds());
	float w = 0;

	if (fConvertButton != NULL)
	{
		fConvertButton->SetLabel(CONVERT_LABEL);
		fConvertButton->ResizeToPreferred();
		BRect buttonFrame(fConvertButton->Frame());
		w = buttonFrame.Width();
		buttonFrame.OffsetTo(r.right - w - 10, r.bottom - buttonFrame.Height() - 25);
		fConvertButton->MoveTo(buttonFrame.LeftTop());
	}

	if (fPreviewButton != NULL)
	{
		fPreviewButton->SetLabel(PREVIEW_BUTTON_LABEL);
		fPreviewButton->ResizeToPreferred();
		BRect buttonFrame(fPreviewButton->Frame());
		w = buttonFrame.Width();
		buttonFrame.OffsetTo(r.right - w - buttonFrame.Width() - 40,
						 r.bottom - buttonFrame.Height() - 25);
		fPreviewButton->MoveTo(buttonFrame.LeftTop());
	}
		
	if (fDestButton != NULL)
	{
		fDestButton->SetLabel(OUTPUT_FOLDER_LABEL);
		fDestButton->ResizeToPreferred();
	}

	if (fOutputFolder != NULL)
	{
		BRect destrect;
		destrect = fDestButton->Frame();
		fOutputFolder->MoveTo(destrect.right + 10, destrect.top + 5);
		fOutputFolder->ResizeToPreferred();
	}

	sprintf(buffer, VIDEO_QUALITY_LABEL, (int8)videoquality);
	if (VideoQuality_Slider != NULL)
	{
		VideoQuality_Slider->SetLabel(buffer);
	    VideoQuality_Slider->SetLimitLabels(SLIDER_LOW_LABEL,SLIDER_HIGH_LABEL);
	}

	sprintf(buffer, AUDIO_QUALITY_LABEL, (int8)audioquality);
	
	if (AudioQuality_Slider != NULL)
	{
		AudioQuality_Slider->SetLabel(buffer);
	    AudioQuality_Slider->SetLimitLabels(SLIDER_LOW_LABEL,SLIDER_HIGH_LABEL);
	}
	float maxLabelLen = 0;

	if (textcontrol1 != NULL)
	{
		textcontrol1->SetLabel(START_LABEL);
		maxLabelLen = textcontrol1->StringWidth(START_LABEL);
	}
	if (textcontrol2 != NULL)
	{
		textcontrol2->SetLabel(END_LABEL);
		maxLabelLen = MAX(maxLabelLen, textcontrol2->StringWidth(END_LABEL));
	}
	if (textcontrol1 != NULL)
		textcontrol1->SetDivider(maxLabelLen + 5);
	if (textcontrol2 != NULL)
		textcontrol2->SetDivider(maxLabelLen + 5);
	
	if (fFormatMenu != NULL)
	{
		fFormatMenu->SetLabel(FORMAT_LABEL);
		maxLabelLen = fFormatMenu->StringWidth(fFormatMenu->Label());
	}
	if (fAudioMenu != NULL)
	{
		fAudioMenu->SetLabel(AUDIO_LABEL);
		maxLabelLen = MAX(maxLabelLen, fAudioMenu->StringWidth(fAudioMenu->Label()));
	}
	if (fVideoMenu != NULL)
	{
		fVideoMenu->SetLabel(VIDEO_LABEL);
		maxLabelLen = MAX(maxLabelLen, fVideoMenu->StringWidth(fVideoMenu->Label()));
	}
	maxLabelLen += 10;

	if (fFormatMenu != NULL)
		fFormatMenu->SetDivider(maxLabelLen);
	if (fAudioMenu != NULL)
		fAudioMenu->SetDivider(maxLabelLen);
	if (fVideoMenu != NULL)
		fVideoMenu->SetDivider(maxLabelLen);

	SetFileMessage(DROP_MEDIA_FILE_LABEL);
}

void MediaConverterWindow::DestroyMenu()
{
	BMenu*	Menu;

	while  ( (Menu = MenuBar->SubmenuAt(0)) != NULL ) {
		MenuBar->RemoveItem(Menu);
		delete Menu;
	}
}

void MediaConverterWindow::CreateMenu()
{
    BMenuItem	*MenuItem;


    aMenu = new BMenu( FILE_MENU_LABEL );
	MenuItem = new BMenuItem( OPEN_MENU_LABEL,new BMessage(OPEN_FILE_MESSAGE));
	aMenu->AddItem( MenuItem );
	aMenu->AddSeparatorItem();
	MenuItem = new BMenuItem( ABOUT_MENU_LABEL,new BMessage(DISP_ABOUT_MESSAGE));
	aMenu->AddItem( MenuItem );
	aMenu->AddSeparatorItem();
	MenuItem = new BMenuItem( QUIT_MENU_LABEL,new BMessage(QUIT_MESSAGE));
	aMenu->AddItem( MenuItem );

    bMenu = new BMenu( PRIORITY_MENU_LABEL );
	MenuItem = new BMenuItem( PRIORITY_LOW, new BMessage(PRIORITY_CHANGED_MESSAGE));
	bMenu->AddItem( MenuItem );
	MenuItem = new BMenuItem( PRIORITY_NORMAL, new BMessage(PRIORITY_CHANGED_MESSAGE));
	bMenu->AddItem( MenuItem );
	MenuItem->SetMarked(true);
	MenuItem = new BMenuItem( PRIORITY_HIGH, new BMessage(PRIORITY_CHANGED_MESSAGE));
	bMenu->AddItem( MenuItem );
	MenuItem = new BMenuItem( PRIORITY_TOP, new BMessage(PRIORITY_CHANGED_MESSAGE));
	bMenu->AddItem( MenuItem );

	bMenu->SetRadioMode(true);

    MenuBar->AddItem( aMenu );
    MenuBar->AddItem( bMenu );
}

MediaConverterWindow::MediaConverterWindow(BRect frame)
	: BWindow(frame, "MediaConverter+", B_DOCUMENT_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL,
			  B_NOT_RESIZABLE | B_NOT_ZOOMABLE | B_ASYNCHRONOUS_CONTROLS)
{
	char defaultdirectory[] = "/boot/home";

	fEnabled = true;
	fConverting = false;
	fCancelling = false;

#if B_BEOS_VERSION >= 0x530 // 0x520 RC2 0x530 RC3

// Loading Locale files for Zeta RC3

	entry_ref mypath;
	app_info info;

	be_app->GetAppInfo(&info);
	mypath = info.ref;

	BPath path(&mypath);
	path.GetParent(&path);
	path.Append("Language/Dictionaries"); 
    path.Append("MediaConverter");
	be_locale.LoadLanguageFile(path.Path()); 

#endif
//
	dir = defaultdirectory;

    BRect dummyRect( 0, 0, 10, 10 );
    MenuBar = new BMenuBar( dummyRect, "menubar" );
	CreateMenu();
    AddChild( MenuBar );
	
	BRect r(frame);
	r.OffsetTo(0, 0);
	BView *background = new BView(r, NULL, B_FOLLOW_ALL_SIDES, 0);
	rgb_color c = ui_color(B_PANEL_BACKGROUND_COLOR);
	background->SetViewColor(c);
	background->SetLowColor(c);
	r.InsetBy(5, 25);

	BRect r2(r);
	r2.bottom = r2.top + 420;
	r2.right = r2.left + 150;
	box1 = new BBox(r2, NULL, B_FOLLOW_LEFT | B_FOLLOW_TOP_BOTTOM);

	BRect r3(r2);
	r3.OffsetTo(0, 0);
	r3.InsetBy(5, 5);
	r3.top += 12;
	r3.bottom -= B_H_SCROLL_BAR_HEIGHT;
	r3.right -= B_V_SCROLL_BAR_WIDTH;
	fListView = new MediaFileListView(r3, B_FOLLOW_LEFT | B_FOLLOW_TOP_BOTTOM);
	BScrollView *scroller = new BScrollView(NULL, fListView,
					B_FOLLOW_LEFT | B_FOLLOW_TOP_BOTTOM, 0, true, true);
	box1->AddChild(scroller);
	background->AddChild(box1);
	
	r2.left = r2.right + 5;
	r2.right = r.right - 5;
	r2.bottom = r2.top + 120;
	box2 = new BBox(r2, NULL, B_FOLLOW_LEFT | B_FOLLOW_TOP_BOTTOM);
	
	r3 = r2;
	r3.OffsetTo(0, 0);
	r3.InsetBy(5, 5);
	r3.top += 12;
	fInfoView = new MediaFileInfoView(r3, B_FOLLOW_LEFT | B_FOLLOW_TOP_BOTTOM);
	box2->AddChild(fInfoView);
	background->AddChild(box2);

	r2.top = r2.bottom + 5;
	r2.bottom = r2.top + 295;
	box3 = new BBox(r2, NULL, B_FOLLOW_LEFT | B_FOLLOW_TOP_BOTTOM);

	r3 = r2;
	r3.OffsetTo(0, 0);
	r3.InsetBy(5, 5);
	r3.top += 12;
	
	BRect r4(r3);
	r4.bottom = r4.top + 20;
	BPopUpMenu *popmenu = new BPopUpMenu("");
	fFormatMenu = new BMenuField(r4, NULL, FORMAT_LABEL, popmenu);
	box3->AddChild(fFormatMenu);
	
	r4.top = r4.bottom + 5;
	r4.bottom = r4.top + 20;
	popmenu = new BPopUpMenu("");
	fAudioMenu = new BMenuField(r4, NULL, AUDIO_LABEL, popmenu);
	box3->AddChild(fAudioMenu);

	r4.top = r4.bottom + 5;
	r4.bottom = r4.top + 20;
	popmenu = new BPopUpMenu("");
	fVideoMenu = new BMenuField(r4, NULL, VIDEO_LABEL, popmenu);
	box3->AddChild(fVideoMenu);
//
	r4.top = r4.bottom + 5;
	r4.bottom = r4.top + 20;
	r4.right = 80;
	fDestButton = new BButton(r4, NULL, OUTPUT_FOLDER_LABEL,
								 new BMessage(OUTPUT_FOLDER_MESSAGE));
	box3->AddChild(fDestButton);
	fDestButton->ResizeToPreferred();
	BRect buttonFrame2(fDestButton->Frame());
	buttonFrame2.OffsetTo(r.right - buttonFrame2.Width(),
						 r.bottom - buttonFrame2.Height());

	fOutputFolder = new BStringView(r4,NULL,defaultdirectory,B_FOLLOW_LEFT|B_FOLLOW_TOP,B_WILL_DRAW);
	box3->AddChild(fOutputFolder);
	fOutputFolder->MoveBy(buttonFrame2.Width() + 10, 5);
	fOutputFolder->ResizeToPreferred();

	r4.top = r4.bottom + 10;
	r4.bottom = r4.top + 20;
	r4.right = 250;
	textcontrol1 = new BTextControl(r4,NULL,"","0",NULL,B_FOLLOW_LEFT|B_FOLLOW_TOP,B_WILL_DRAW);
	box3->AddChild(textcontrol1);
	textcontrol1->SetText("0");
	
	r4.top = r4.bottom + 5;
	r4.bottom = r4.top + 20;
	textcontrol2 = new BTextControl(r4,NULL,"","0",NULL,B_FOLLOW_LEFT|B_FOLLOW_TOP,B_WILL_DRAW);
	box3->AddChild(textcontrol2);
	textcontrol2->SetText("0");
 	
	r4.top = r4.bottom + 5;
	r4.bottom = r4.top + 50;

//Video Quality

	VideoQuality_Slider = new BSlider(r4,"VSlider", "" ,new BMessage(VIDEO_QUALITY_CHANGED_MESSAGE), 
                     1,100,B_HORIZONTAL);
	VideoQuality_Slider->SetValue(75);
	VideoQuality_Slider->SetEnabled(false);
    box3->AddChild(VideoQuality_Slider);

	r4.top = r4.bottom + 5;
	r4.bottom = r4.top + 50;

//Audio Quality

	AudioQuality_Slider = new BSlider(r4,"ASlider", "" ,new BMessage(AUDIO_QUALITY_CHANGED_MESSAGE), 
                     1,100,B_HORIZONTAL);
	AudioQuality_Slider->SetValue(75);
	AudioQuality_Slider->SetEnabled(false);
    box3->AddChild(AudioQuality_Slider);
	background->AddChild(box3);

//

	r2.top = r2.bottom + 15;
	r2.bottom = r2.top + 20;
	r2.left = r.left - 120;
	r2.right = r.right;

	fPreviewButton = new BButton(r2, NULL, PREVIEW_BUTTON_LABEL,
								 new BMessage(PREVIEW_MESSAGE));
	background->AddChild(fPreviewButton);
	fPreviewButton->SetEnabled(false);

	fConvertButton = new BButton(r2, NULL, CONVERT_LABEL,
								 new BMessage(CONVERT_BUTTON_MESSAGE));
	background->AddChild(fConvertButton);

// Status view

	r2.bottom = r2.top + 20;
	r2.left = r.left;
	r2.right = r.right;

	fStatusView2 = new StatusView(r2, B_FOLLOW_LEFT | B_FOLLOW_TOP);
	background->AddChild(fStatusView2);

	r2.top += 15;
	r2.bottom += 20;

	fStatusView = new StatusView(r2, B_FOLLOW_LEFT | B_FOLLOW_TOP);
	background->AddChild(fStatusView);
	AddChild(background);

	SetStatusMessage("");
	SetLabel();
}


MediaConverterWindow::~MediaConverterWindow()
{
}

void
MediaConverterWindow::BuildAudioVideoMenus()
{
	BMenu *menu = fAudioMenu->Menu();
	BMenuItem *item;
	// clear out old audio codec menu items
	while ((item = menu->RemoveItem((int32)0)) != NULL) {
		delete item;
	}

	// get selected file format
	FileFormatMenuItem *ffmi = (FileFormatMenuItem*)fFormatMenu->Menu()->FindMarked();
	media_file_format *mf_format = &(ffmi->fFileFormat);

	media_format format, outfmt;
	memset(&format, 0, sizeof(format));
	media_codec_info codec_info;
	int32 cookie = 0;
	CodecMenuItem *cmi;

	// add available audio encoders to menu
	format.type = B_MEDIA_RAW_AUDIO;
	format.u.raw_audio = media_raw_audio_format::wildcard;	
	while (get_next_encoder(&cookie, mf_format, &format, &outfmt, &codec_info) == B_OK) {
		cmi = new CodecMenuItem(&codec_info, AUDIO_CODEC_SELECT_MESSAGE);
		menu->AddItem(cmi);
		// reset media format struct
/*
		format.type = B_MEDIA_RAW_AUDIO;
		format.u.raw_audio = media_raw_audio_format::wildcard;
*/
	}

	// mark first audio encoder
	item = menu->ItemAt(0);
	if (item != NULL) {
		fAudioMenu->SetEnabled(fEnabled);
		AudioQuality_Slider->SetEnabled(fEnabled);
		item->SetMarked(true);
		((BInvoker *)item)->Invoke();
	} else {
		item = new BMenuItem(NONE_LABEL, NULL);
		menu->AddItem(item);
		item->SetMarked(true);
		fAudioMenu->SetEnabled(false);
		AudioQuality_Slider->SetEnabled(false);
	}
	
	// clear out old video codec menu items
	menu = fVideoMenu->Menu();
	while ((item = menu->RemoveItem((int32)0)) != NULL) {
		delete item;
	}

	// construct a generic video format.  Some of these parameters
	// seem silly, but are needed for R4.5.x, which is more picky
	// than subsequent BeOS releases will be.
	memset(&format, 0, sizeof(format));
	format.type = B_MEDIA_RAW_VIDEO;
	format.u.raw_video.last_active = (uint32)(320 - 1);
	format.u.raw_video.orientation = B_VIDEO_TOP_LEFT_RIGHT;
	format.u.raw_video.display.format = B_RGB32;
	format.u.raw_video.display.line_width = (int32)320;
	format.u.raw_video.display.line_count = (int32)240;
	format.u.raw_video.display.bytes_per_row = 4 * 320;

	// add available video encoders to menu
	cookie = 0;
	while (get_next_encoder(&cookie, mf_format, &format, &outfmt, &codec_info) == B_OK) {
		cmi = new CodecMenuItem(&codec_info, VIDEO_CODEC_SELECT_MESSAGE);
		menu->AddItem(cmi);
	}

	// mark first video encoder
	item = menu->ItemAt(0);
	if (item != NULL) {
		fVideoMenu->SetEnabled(fEnabled);
		VideoQuality_Slider->SetEnabled(fEnabled);
		item->SetMarked(true);
		((BInvoker *)item)->Invoke();
	} else {
		item = new BMenuItem(NONE_LABEL, NULL);
		menu->AddItem(item);
		item->SetMarked(true);
		fVideoMenu->SetEnabled(false);
		VideoQuality_Slider->SetEnabled(false);
	}
}

void
MediaConverterWindow::GetSelectedFormatInfo(media_file_format **format,
											media_codec_info **audio,
											media_codec_info **video)
{
	*format = NULL;
	FileFormatMenuItem *formatItem =
		dynamic_cast<FileFormatMenuItem *>(fFormatMenu->Menu()->FindMarked());
	if (formatItem != NULL) {
		*format = &(formatItem->fFileFormat);
	}
	
	*audio = *video = NULL;
	CodecMenuItem *codecItem =
		dynamic_cast<CodecMenuItem *>(fAudioMenu->Menu()->FindMarked());
	if (codecItem != NULL) {
		*audio =  &(codecItem->fCodecInfo);
	}
	codecItem = dynamic_cast<CodecMenuItem *>(fVideoMenu->Menu()->FindMarked());
	if (codecItem != NULL) {
		*video =  &(codecItem->fCodecInfo);
	}
}


void 
MediaConverterWindow::BuildFormatMenu()
{
	BMenu *menu = fFormatMenu->Menu();
	BMenuItem *item;
	// clear out old format menu items
	while ((item = menu->RemoveItem((int32)0)) != NULL) {
		delete item;
	}

	// add menu items for each file format
	media_file_format mfi;
	int32 cookie = 0;
	FileFormatMenuItem *ff_item;
	while (get_next_file_format(&cookie, &mfi) == B_OK) {
		ff_item = new FileFormatMenuItem(&mfi);
		menu->AddItem(ff_item);
	}
	
	// mark first item
	item = menu->ItemAt(0);
	if (item != NULL) {
		item->SetMarked(true);
		((BInvoker *)item)->Invoke();
	}
}

void 
MediaConverterWindow::SetFileMessage(const char *message)
{
	fStatusView->SetStatus(message);
}

void 
MediaConverterWindow::SetStatusMessage(const char *message)
{
	fStatusView2->SetStatus(message);
}

void
MediaConverterWindow::AddSourceFile(BMediaFile *f, entry_ref *ref)
{
	fListView->AddItem(f, ref);
}

void
MediaConverterWindow::RemoveSourceFile(int32 index)
{
	BListItem *item = fListView->RemoveItem(index);
	if (item != NULL) {
		delete item;
	}
}

int32 
MediaConverterWindow::CountSourceFiles()
{
	return fListView->CountItems();
}

status_t
MediaConverterWindow::GetSourceFileAt(int32 index, BMediaFile **f, entry_ref *ref)
{
	MediaFileListItem *item = dynamic_cast<MediaFileListItem*>(fListView->ItemAt(index));
	if (item != NULL) {
		*f = item->fMediaFile;
		*ref = item->fRef;
		return B_OK;
	} else {
		return B_ERROR;
	}
}

void 
MediaConverterWindow::SourceFileSelectionChanged()
{
	int32 selected = fListView->CurrentSelection();
	BMediaFile *f = NULL;
	entry_ref *ref = NULL;
	if (selected >= 0) {
		MediaFileListItem* mfli = dynamic_cast<MediaFileListItem*>(fListView->ItemAt(selected));
		if (mfli != NULL) {
			f = mfli->fMediaFile;
			ref = &(mfli->fRef);
		}
		fPreviewButton->SetEnabled(true);
		VideoQuality_Slider->SetEnabled(true);
		AudioQuality_Slider->SetEnabled(true);
	} else {
		fPreviewButton->SetEnabled(false);
		VideoQuality_Slider->SetEnabled(false);
		AudioQuality_Slider->SetEnabled(false);
	}
	fInfoView->Update(f, ref);
	durationchanged = true;
}

void 
MediaConverterWindow::SetEnabled(bool enabled, bool buttonEnabled)
{
	fConvertButton->SetEnabled(buttonEnabled);
	if (enabled != fEnabled) {
		fFormatMenu->SetEnabled(enabled);
		fAudioMenu->SetEnabled(enabled);
		fVideoMenu->SetEnabled(enabled);
		fListView->SetEnabled(enabled);
		fDestButton->SetEnabled(enabled);
		textcontrol1->SetEnabled(enabled);
		textcontrol2->SetEnabled(enabled);
		fEnabled = enabled;
	}
}

bool 
MediaConverterWindow::IsEnabled()
{
	return fEnabled;
}

/*
void 
MediaConverterWindow::DispatchMessage(BMessage *msg, BHandler *handler)
{
	if (msg->WasDropped() && msg->what == B_SIMPLE_DATA) {
	
		printf("Dispatch 1\n");
		DetachCurrentMessage();
		msg->what = B_REFS_RECEIVED;
		BMessenger(be_app).SendMessage(msg);
		delete msg;
	} else {
		BWindow::DispatchMessage(msg, handler);
	}
}
*/

void 
MediaConverterWindow::MessageReceived(BMessage *msg)
{
	int32 prioritylevel[]= {
		B_LOW_PRIORITY,
		B_NORMAL_PRIORITY,
		B_REAL_TIME_DISPLAY_PRIORITY,
		B_REAL_TIME_PRIORITY,
	};

	status_t status;
	entry_ref ref;
	entry_ref inRef;

	BString string, string2;

	BEntry entry("/boot/beos/apps/MediaPlayer",true);
	char buffer[40];
	char buffer2[B_PATH_NAME_LENGTH];
	char *argv[3];
	argv[0] = "-pos";
	BMediaFile *inFile(NULL);
	int32 srcIndex = 0;
	BPath name;
	BEntry inentry;
	int32 value;
   	BRect ButtonRect;

	switch (msg->what) {
#if B_BEOS_VERSION <= B_BEOS_VERSION_6
	case B_LANGUAGE_CHANGED:
		LanguageChanged();
		break;
#endif
	// Dispatch Message
	case B_SIMPLE_DATA:
		if (msg->WasDropped())
		{
			DetachCurrentMessage();
			msg->what = B_REFS_RECEIVED;
			BMessenger(be_app).SendMessage(msg);
			delete msg;
		}
		break;
	case FORMAT_SELECT_MESSAGE:
		BuildAudioVideoMenus();
		break;
	case AUDIO_CODEC_SELECT_MESSAGE:
		break;
	case VIDEO_CODEC_SELECT_MESSAGE:
		break;
	case CONVERT_BUTTON_MESSAGE:
		if (!fConverting) {
			fConvertButton->SetLabel(CANCEL_LABEL);
			fConverting = true;
			SetStatusMessage(CONVERT_LABEL);
			SetEnabled(false, true);
			BMessenger(be_app).SendMessage(START_CONVERSION_MESSAGE);
		} else if (!fCancelling) {
			fCancelling = true;
			SetStatusMessage(CANCELING_LABEL);
			BMessenger(be_app).SendMessage(CANCEL_CONVERSION_MESSAGE);			
		}
		break;
	case CONVERSION_DONE_MESSAGE:
		SetStatusMessage(fCancelling ? CONV_CANCEL_LABEL : CONV_COMPLETE_LABEL);
		fConverting = false;
		fCancelling = false;
		{
			bool enable = CountSourceFiles() > 0;
			SetEnabled(enable, enable);
		}
		fConvertButton->SetLabel(CONVERT_LABEL);


		break;
//
	case OUTPUT_FOLDER_MESSAGE:
	//	 Execute Save Panel
		if (!openflag && !saveflag) {
		    BButton *SelectThisDir;

			fFilePanelS = new BFilePanel(B_OPEN_PANEL, NULL, NULL, B_DIRECTORY_NODE, true, new BMessage(FOLDER_SELECT_MESSAGE), NULL, false, true);
			fFilePanelS->SetButtonLabel(B_DEFAULT_BUTTON, SELECT_LABEL);
			fFilePanelS->Window()->SetTitle(SAVE_DIR_LABEL);
			fFilePanelS->SetTarget(this);

			fFilePanelS->Window()->Lock();
			ButtonRect = fFilePanelS->Window()->ChildAt(0)->FindView("cancel button")->Frame();
			ButtonRect.right  = ButtonRect.left - 20;
			ButtonRect.left = ButtonRect.right - 130;
			SelectThisDir = new BButton(ButtonRect, NULL, SELECT_DIR_LABEL,
								 new BMessage(SELECT_THIS_DIR_MESSAGE), B_FOLLOW_BOTTOM | B_FOLLOW_RIGHT);
			SelectThisDir->SetTarget(this);
			fFilePanelS->Window()->ChildAt(0)->AddChild(SelectThisDir);
			fFilePanelS->Window()->Unlock();

			BRefFilter *filter;
			filter = new DirectoryFilter;
			fFilePanelS->SetRefFilter(filter);
			saveflag = true;
			fFilePanelS->Show();
		}
		break;
	case FOLDER_SELECT_MESSAGE:
	//	 "SELECT" Button at Save Panel Pushed
		fFilePanelS->GetNextSelectedRef(&inRef);
		inentry.SetTo(&inRef, true);
		inentry.GetPath(&name);
		fOutputFolder->SetText(name.Path());
		fOutputFolder->ResizeToPreferred();
		dir.SetTo(name.Path());
		saveflag = false;
		break;
	case SELECT_THIS_DIR_MESSAGE:
	//	 "THIS DIR" Button at Save Panel Pushed
		fFilePanelS->GetPanelDirectory(&inRef);
		fFilePanelS->Hide();
		inentry.SetTo(&inRef, true);
		inentry.GetPath(&name);
		fOutputFolder->SetText(name.Path());
		fOutputFolder->ResizeToPreferred();
		dir.SetTo(name.Path());
		saveflag = false;
		break;
	case OPEN_FILE_MESSAGE:
	//	 Execute Open Panel
		if (!openflag && !saveflag) {
			fFilePanel = new BFilePanel(B_OPEN_PANEL, NULL, NULL, B_FILE_NODE, true, NULL, NULL, false, true);
			fFilePanel->SetTarget(this);
			openflag = true;
			fFilePanel->Show();
		}
		break;
	case B_REFS_RECEIVED:
	//	Media Files Seleced by Open Panel
		DetachCurrentMessage();
		msg->what = B_REFS_RECEIVED;
		BMessenger(be_app).SendMessage(msg);
	case B_CANCEL:
	//	Open & Save Panel Cancelling
		if (openflag)
			openflag = false;
		if (saveflag)
			saveflag = false;
		break;
	case DISP_ABOUT_MESSAGE:
		(new BAlert(ABOUT_TITLE_LABEL,
"MediaConverter+ for " OS"\n"
VERSION "\n" 
"(C)1999, Be Incorporated.\n"
"(C)2000-2004 Jun Suzuki",
			OK_LABEL))->Go();
		break;
	case QUIT_MESSAGE:
		MediaConverterWindow::QuitRequested();
		break;
	case PREVIEW_MESSAGE:
		entry.GetRef(&ref);
		string = "";
		string << textcontrol1->Text();
		string << "000";
		
		strcpy(buffer,string.String());
		argv[1] = buffer;
		srcIndex = fListView->CurrentSelection();
		status = GetSourceFileAt(srcIndex, &inFile, &inRef);
		if (status == B_OK) {
			inentry.SetTo(&inRef);
			inentry.GetPath(&name);

			strcpy(buffer,string.String());
			
			strcpy(buffer2,name.Path());
			argv[2]= buffer2;
		}

		status = be_roster->Launch(&ref,3,argv);

		if (status != B_NO_ERROR)
		{
			string2 << LAUNCH_ERROR << strerror(status);
			(new BAlert("", string2.String(), OK_LABEL))->Go();
		}
		break;
	case PRIORITY_CHANGED_MESSAGE:
		priority = prioritylevel[(int8)bMenu->IndexOf(bMenu->FindMarked())];
	break;
	case VIDEO_QUALITY_CHANGED_MESSAGE:
		msg->FindInt32("be:value",&value); 
		sprintf(buffer, VIDEO_QUALITY_LABEL, (int8)value);
		VideoQuality_Slider->SetLabel(buffer);
		videoquality = value;
	break;
	case AUDIO_QUALITY_CHANGED_MESSAGE:
		msg->FindInt32("be:value",&value); 
		sprintf(buffer, AUDIO_QUALITY_LABEL, (int8)value);
		AudioQuality_Slider->SetLabel(buffer);
		audioquality = value;
	break;
	default:
		BWindow::MessageReceived(msg);
	}
}

bool
MediaConverterWindow::QuitRequested()
{
	if (!fConverting) {
		BMessenger(be_app).SendMessage(B_QUIT_REQUESTED);
		return true;
	} else if (!fCancelling) {
		fCancelling = true;
		SetStatusMessage(CANCELLING_LABEL);
		BMessenger(be_app).SendMessage(CANCEL_CONVERSION_MESSAGE);
	}
	return false;
}

// ------------------- MediaConverterApp implementation -------------------

MediaConverterApp::MediaConverterApp()
	: BApplication(APP_SIGNATURE)
{
	fConverting = false;
	fCancel = false;
	fConvertThreadID = -1;
	fWin = new MediaConverterWindow(BRect(50, 50, 520, 555));
}


MediaConverterApp::~MediaConverterApp()
{
}

BEntry MediaConverterApp::CreateOutputFile(BMediaFile */*input*/,  entry_ref *ref,
									media_file_format *outputFormat)
{
	BString name(ref->name);
	// create output file name
	int32 extIndex = name.FindLast('.');
	if (extIndex != B_ERROR)
		name.Truncate(extIndex + 1);
	else
		name.Append(".");
	
	name.Append(outputFormat->file_extension);

	BEntry inEntry(ref);
	BEntry outEntry;

//	if (inEntry.InitCheck() == B_OK && inEntry.GetParent(&dir) == B_OK) {
	if (inEntry.InitCheck() == B_OK) {
		// ensure that output name is unique
		int32 len = name.Length();
		int32 i = 1;
		while (dir.Contains(name.String())) {
			name.Truncate(len);
			name << " " << i;
			i++;
		}
		outEntry.SetTo(&dir, name.String());
	}

	return outEntry;
}

bool 
MediaConverterApp::IsConverting()
{
	return fConverting;
}

void 
MediaConverterApp::StartConverting()
{
	bool locked = fWin->Lock();

	if (locked && (fWin->CountSourceFiles() > 0)) {
		fConvertThreadID = spawn_thread(MediaConverterApp::RunConvert, "converter thread",
										priority, (void *)this);
		if (fConvertThreadID > 0) {
			fConverting = true;
			fCancel = false;
			resume_thread(fConvertThreadID);			
		} else {
			fConvertThreadID = -1;
		}
	}

	if (locked) {
		fWin->Unlock();
	}
}

void 
MediaConverterApp::ConvertLoop()
{
//
	bigtime_t start;
	bigtime_t end;
	char *a;
	start = strtoimax(textcontrol1->Text(), &a,0) * 1000;
	end = strtoimax(textcontrol2->Text(), &a,0) * 1000;

	int32 srcIndex = 0;

	BMediaFile *inFile(NULL), *outFile(NULL);
	BEntry outEntry;
	entry_ref inRef;
	entry_ref outRef;
	BPath path;
	BString name;

	while (!fCancel) {
		if (fWin->Lock()) {
			status_t r = fWin->GetSourceFileAt(srcIndex, &inFile, &inRef);
			if (r == B_OK) {
				media_codec_info *audioCodec, *videoCodec;
				media_file_format *fileFormat;
				fWin->GetSelectedFormatInfo(&fileFormat, &audioCodec, &videoCodec);
				fWin->Unlock();
				outEntry = CreateOutputFile(inFile, &inRef, fileFormat);

				// display file name

				outEntry.GetPath(&path);
				name.SetTo(path.Leaf());

				if (outEntry.InitCheck() == B_OK) {
					entry_ref outRef;
					outEntry.GetRef(&outRef);
					outFile = new BMediaFile(&outRef, fileFormat);

					name.Prepend(" '");
					name.Prepend(OUTPUT_FILE_STRING1);
					name.Append("' ");
					name.Append(OUTPUT_FILE_STRING2);
				} else {
					name.Prepend(" '");
					name.Prepend(OUTPUT_FILE_STRING3);
					name.Append("'");
				}

				if (fWin->Lock()) {
					fWin->SetFileMessage(name.String());
					fWin->Unlock();
				}

				if (outFile != NULL) {
					r = ConvertFile(inFile, outFile, audioCodec, videoCodec, start, end);

					// set mime
					update_mime_info(path.Path(), false, false, false);

					fWin->Lock();
					if (r == B_OK) {
						fWin->RemoveSourceFile(srcIndex);
						textcontrol1->SetText("0");
						textcontrol2->SetText("0");
					} else {
						srcIndex++;
						BString error(CONVERT_ERROR_STRING);
  						error << " '" << inRef.name << "'";
						fWin->SetStatusMessage(error.String());
					}
					fWin->Unlock();
				}
				
				
			} else {
				fWin->Unlock();
				break;
			}
		} else {
			break;
		}
	}

	BMessenger(this).SendMessage(CONVERSION_DONE_MESSAGE);
}

status_t 
MediaConverterApp::ConvertFile(BMediaFile *inFile, BMediaFile *outFile,
							   media_codec_info *audioCodec,
							   media_codec_info *videoCodec, bigtime_t StartTime, bigtime_t EndTime)
{
	BMediaTrack *inVidTrack(NULL), *inAudTrack(NULL);
	BMediaTrack *outVidTrack(NULL), *outAudTrack(NULL);
	media_format inFormat, outAudFormat, outVidFormat;
	media_raw_audio_format *raf(NULL);
	media_raw_video_format *rvf(NULL);
	int32 width(-1), height(-1);
	short audioFrameSize(1);
	char *videoData(NULL);
	char *audioData(NULL);
	BView *encoderView = NULL;

	// gather the necessary format information and construct output tracks
	int32 tracks = inFile->CountTracks();
	int64 vsize;
	int64 asize;

	for (int32 i = 0; i < tracks && (!outAudTrack || !outVidTrack); i++) {
		BMediaTrack *inTrack = inFile->TrackAt(i);
		inTrack->EncodedFormat(&inFormat);
		if (inFormat.IsAudio() && (audioCodec != NULL)) {
			inAudTrack = inTrack;
			memcpy(&outAudFormat, &inFormat, sizeof(outAudFormat));
			outAudFormat.type = B_MEDIA_RAW_AUDIO;
			raf = &(outAudFormat.u.raw_audio);
			inTrack->DecodedFormat(&outAudFormat);
			
			audioData = (char*)malloc(raf->buffer_size);
//			audioFrameSize = (raf->format & media_raw_audio_format::B_AUDIO_SIZE_MASK)
			audioFrameSize = (raf->format & 0xf)
								* raf->channel_count;
			outAudTrack = outFile->CreateTrack(&outAudFormat, audioCodec);
//
			if (outAudTrack != NULL)
			{
				if (outAudTrack->SetQuality(audioquality / 100.0f) != B_OK)
				{
					fWin->Lock();
					AudioQuality_Slider->SetLabel(AUDIO_SUPPORT_STRING);
					fWin->Unlock();
				}
			}
//

		} else if (inFormat.IsVideo() && (videoCodec != NULL)) {
			inVidTrack = inTrack;
			width = (int32)inFormat.Width();
			height = (int32)inFormat.Height();
			
			// construct desired decoded video format
//			memset(&outVidFormat, 0, sizeof(outVidFormat));
			outVidFormat.type = B_MEDIA_RAW_VIDEO;
			rvf = &(outVidFormat.u.raw_video);
			rvf->last_active = (uint32)(height - 1);
			rvf->orientation = B_VIDEO_TOP_LEFT_RIGHT;
			rvf->pixel_width_aspect = 1;
			rvf->pixel_height_aspect = 3;
			rvf->display.format = B_RGB32;
			rvf->display.bytes_per_row = 4 * width;
			rvf->display.line_width = width;
			rvf->display.line_count = height;

			inVidTrack->DecodedFormat(&outVidFormat);
			videoData = (char *)malloc(width * height * 4);
			outVidTrack = outFile->CreateTrack(&outVidFormat, videoCodec);
//
			if (outVidTrack != NULL)
			{
				// DLM Added to use 3ivx Parameter View
				encoderView = outVidTrack->GetParameterView();
				
				if (encoderView) {
					MediaEncoderWindow *encoderWin = new MediaEncoderWindow(BRect(50, 50, 520, 555),encoderView);
					encoderWin->Go();
					
					// The quality setting is ignored by the 3ivx encoder if the View was displayed
					// But this method is the trigger to read all the parameter settings
					outVidTrack->SetQuality(videoquality / 100.0f);

					// We can now delete the encoderView created for us by the encoder
					delete encoderView;
					encoderView = NULL;
					
					fWin->Lock();
					VideoQuality_Slider->SetLabel(VIDEO_PARAMFORM_STRING);
					fWin->Unlock();
				} else {
			
					if (outVidTrack->SetQuality(videoquality / 100.0f) != B_OK)
					{
						fWin->Lock();
						VideoQuality_Slider->SetLabel(VIDEO_SUPPORT_STRING);
						fWin->Unlock();
					}
				}
//
			}
		} else {
			inFile->ReleaseTrack(inTrack);		
		}
	}
	
	if (fCancel || (!outVidTrack && !outAudTrack)) {
		// don't have any video or audio tracks here, or cancelled
		delete outFile;
		return B_ERROR;
	} else {
		outFile->CommitHeader();
		// this is where you would call outFile->AddCopyright(...)
	}

	int64 framesRead;
	status_t err;
	media_header mh;
	int32 lastPercent, currPercent;
	float completePercent;
	BString status;

	int64 start;
	int64 end;
	int32 stat = 0;

	// read video from source and write to destination, if necessary
	if (outVidTrack != NULL) {
		lastPercent = -1;
		vsize = inVidTrack->CountFrames();
		if ((StartTime == 0) && (EndTime == 0))
		{
			start = 0;
			end = vsize;
		} else {
			inVidTrack->SeekToTime(&EndTime, stat);
			end = inVidTrack->CurrentFrame();
			inVidTrack->SeekToTime(&StartTime, stat);
			start = inVidTrack->CurrentFrame();
			if (end > vsize)
				end =  vsize;
			if (start > end)
				start = 0;
		}


		framesRead = 0;
//		for (int64 i = start; (B_INFINITE_TIMEOUT) && !fCancel; i+=framesRead) {
		for (int64 i = start; (i <= end) && !fCancel; i+=framesRead) {
			if ((err = inVidTrack->ReadFrames(videoData, &framesRead, &mh)) != B_OK) {
				fprintf(stderr, "Error reading video frame %Ld: %s\n", i,
						strerror(err));
				status.SetTo(ERROR_READ_VIDEO_STRING);
				status << i;
				SetStatusMessage(status.String());

				break;
			}
			if ((err = outVidTrack->WriteFrames(videoData, framesRead,
					mh.u.encoded_video.field_flags)) != B_OK)
			{
				fprintf(stderr, "Error writing video frame %Ld: %s\n", i,
						strerror(err));
				status.SetTo(ERROR_WRITE_VIDEO_STRING);
				status << i;
				SetStatusMessage(status.String());
				break;
			}
			completePercent = (float)(i - start) / (float)(end - start) * 100;
			currPercent = (int16)floor(completePercent);
			if (currPercent > lastPercent) {
				lastPercent = currPercent;
				status.SetTo(WRITE_VIDEO_STRING);
				status.Append(" ");
				status << currPercent << "% " << COMPLETE_STRING;
				SetStatusMessage(status.String());

			}
		}
		outVidTrack->Flush();
		inFile->ReleaseTrack(inVidTrack);
	}
	
	// read audio from source and write to destination, if necessary
	if (outAudTrack != NULL) {
		lastPercent = -1;

		asize =  inAudTrack->CountFrames();

		if ((StartTime == 0) && (EndTime == 0))
		{
			start = 0;
			end = asize;
		} else {
			inAudTrack->SeekToTime(&EndTime, stat);
			end = inAudTrack->CurrentFrame();
			inAudTrack->SeekToTime(&StartTime, stat);
			start = inAudTrack->CurrentFrame();
			if (end > asize)
				end = asize;
			if (start > end)
				start = 0;
		}

		for (int64 i = start; (i <= end) && !fCancel; i+=framesRead) {
			if ((err = inAudTrack->ReadFrames(audioData, &framesRead, &mh)) != B_OK) {
				fprintf(stderr, "Error reading audio frames: %s\n", strerror(err));
				status.SetTo(ERROR_READ_AUDIO_STRING);
				status << i;
				SetStatusMessage(status.String());
				break;
			}
			if ((err = outAudTrack->WriteFrames(audioData, framesRead)) != B_OK)
			{
				fprintf(stderr, "Error writing audio frames: %s\n", strerror(err));
				status.SetTo(ERROR_WRITE_AUDIO_STRING);
				status << i;
				SetStatusMessage(status.String());
				break;
			}
			completePercent = (float)(i - start) / (float)(end - start) * 100;
			currPercent = (int16)floor(completePercent);
			if (currPercent > lastPercent) {
				lastPercent = currPercent;
				status.SetTo(WRITE_AUDIO_STRING);
				status.Append(" ");
				status << currPercent << "% " << COMPLETE_STRING;
				SetStatusMessage(status.String());
			}
		}
		outAudTrack->Flush();
		inFile->ReleaseTrack(inAudTrack);

	}
	outFile->CloseFile();
	delete outFile;
	free(videoData);
	free(audioData);

	return B_OK;
}

void 
MediaConverterApp::SetStatusMessage(const char *message)
{
	if (fWin != NULL && fWin->Lock()) {
		fWin->SetStatusMessage(message);
		fWin->Unlock();
	}
}

void 
MediaConverterApp::ReadyToRun()
{
	fWin->Show();
	if (fWin->Lock()) {
		fWin->BuildFormatMenu();
		if (fWin->CountSourceFiles() == 0) {
			fWin->SetEnabled(false, false);
		}
		fWin->Unlock();
	}
}

void 
MediaConverterApp::RefsReceived(BMessage *msg)
{
	entry_ref ref;
	int32 i = 0;
	BMediaFile *f;
	BString errorFiles;
	int32 errors = 0;
	
	//	From Open Dialog and D&D

	while (msg->FindRef("refs", i++, &ref) == B_OK) {
		f = new BMediaFile(&ref/*, B_MEDIA_FILE_NO_READ_AHEAD*/);
		if (f->InitCheck() != B_OK) {
			errorFiles << ref.name << "\n";
			errors++;
			delete f;
			continue;
		}
		if (fWin->Lock()) {
			fWin->AddSourceFile(f, &ref);
			fWin->Unlock();
		}
	}

	if (errors) {
		BString alertText;
		alertText << errors << ((errors > 1) ? FILES : FILE)
				  << NOTRECOGNIZE << "\n";
		alertText << errorFiles;
		BAlert *alert = new BAlert(ERROR_LOAD_STRING, alertText.String(),
						CONTINUE_STRING	, NULL, NULL, B_WIDTH_AS_USUAL, B_STOP_ALERT);
		alert->Go();
	}
	openflag = false;
}


void 
MediaConverterApp::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
	case FILE_LIST_CHANGE_MESSAGE:
		if (fWin->Lock()) {
			bool enable = fWin->CountSourceFiles() > 0;
			fWin->SetEnabled(enable, enable);
			fWin->Unlock();
		}
		break;
	case START_CONVERSION_MESSAGE:
		if (!fConverting) {
			StartConverting();			
		}
		break;
	case CANCEL_CONVERSION_MESSAGE:
		fCancel = true;
		break;
	case CONVERSION_DONE_MESSAGE:
		fCancel = false;
		fConverting = false;
		DetachCurrentMessage();
		BMessenger(fWin).SendMessage(msg);
		break;

	default:
		BApplication::MessageReceived(msg);
	}
}

int32 
MediaConverterApp::RunConvert(void *castToMediaConverterApp)
{
	MediaConverterApp *app = (MediaConverterApp *)castToMediaConverterApp;
	app->ConvertLoop();
	return 0;
}


int main(int, char **)
{
	MediaConverterApp app;
	app.Run();
	
	return 0;
}
