#
#  Demonstrates part of a plugin architecture for authentication
#  methods in IMAP4 and POP3 Python applications.
#
#  Copyright 1999 University of Washington
#  Written 1999 by Donn Cave, donn@u.washington.edu

#  Permission to use, copy, modify, and distribute this software
#  and its documentation for any purpose and without fee to the
#  University of Washington is hereby granted, provided that the
#  above copyright notice appears in all copies, and that the name
#  of the University of Washington not be used in advertising or
#  publicity pertaining to distribution of the software without specific,
#  written prior permission. This software is made available as is.

#  THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR
#  IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION
#  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
#  PURPOSE, AND IN NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE
#  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
#  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
#  AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY,
#  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
#  SOFTWARE.

import gssapi
import os
import socket
import string
import sys
import traceback

KRB5_FCC_NOFILE = -1765328189
# Possibly due to error in my gssapi module, this value doesn't
# sign extend to a negative 64-bit integer on Alpha.
# KRB5_FCC_NOFILE = 2529639107

#  Module exports "plugins", a list of authentication plugins.
#  The only one supported at this time is GSSAPI, it's a short list.

#  GSSAPI-Kerberos5 class for imaplib.IMAP4 and poplib.POP3 requires
#  gssapimodule.so.

#  An instance of this class is to be passed to IMAP4.authenticate(),
#  which requires only that the object it gets
#  1) be callable, with one string type argument for data from the server, and
#  2) return a data string from the server (or None on error.)
#
#  After the GSSAPI ritual, client also sends server flags and an intended
#  login name, and server sends flags.
#
#  Raises exceptions that the application is expected to account for.
#  The application can give up on this authentication option and go on
#  with whatever other options it may have, such as clear text password.
#  It may also initiate a Kerberos kinit login to remedy the problem, but
#  that's complicated and questionable and no example is shown here of
#  how that would be done.
#
#  Some errors relate to initial credentials state, missing or expired.
#  This example class will raise these errors right away on instantiation.
#  Other errors may crop up during the IMAP4 protocol and result in a
#  failure and server.error exception, but these are of a more truly
#  exceptional nature.
#
#  gssapi.error has the form ((major, (majorstr,...)), (minor, (minorstr,...)))
#
class GSSAPI_Plugin:
	class LocalInvalid(Exception):
		pass
	class UnknownError(Exception):
		pass
	name = 'gssapi'
	def __init__(self, service, host, account = None):
		#  Hack for PPC BeOS FILE: problem...
		# os.putenv('KRB5CCNAME', 'STDIO:/tmp/krb5cc_0')
		if account is None:
			#  Ideally, gets "user@realm" from the user's
			#  Kerberos credentials.
			#
			try:
				princ = gssapi.default_principal()
			except gssapi.error, val:
				raise self.gss_exception(val)

			princ = string.split(princ, '@')
			self.remotelogin = princ[0]
		else:
			#  Caller appears to know remote account.
			self.remotelogin = account

		#  Remote service principal
		target = '%s@%s' % (service, host)
		print 'Kerberos authentication', repr(self.remotelogin), 'to', repr(target)
		self.context = gssapi.Context(target)

		#  Possible exceptions here include missing credentials
		#  file or expired credentials.
		try:
			self.clidat = self.context.init()
		except gssapi.error, val:
			raise self.gss_exception(val)

	def __call__(self, data):
		#
		#  imaplib calls here with decoded data from the server.
		#  If all goes well, return data for the server, else
		#  return None to abort.

		#  Initial GSSAPI server data is null, and the initial
		#  client data was already calculated in __init__().
		#  In this special case, just return the initial data.

		if self.clidat:
			if data:
				sys.stderr.write('!!! negotiation collision: %s\n', repr(data))
			data = self.clidat
			self.clidat = None
		elif self.context.complete:
			#  Credentials appear to be in order, now imapd
			#  just wants to match the intended user name and
			#  sort of negotiate encryption options.  The
			#  option negotiation here may not be correct, I'm
			#  just relying on observation that MRC's imapd doesn't
			#  support any data privacy.
			#
			if data:
				data = self.context.unwrap(data)
				self.server_security_layers = ord(data[0])

			clisec = '\001\000\000\000'
			data = self.context.wrap(clisec + self.remotelogin)
		else:
			try:
				data = self.context.init(data)
			except gssapi.error:
				#  Application may actually be interested in
				#  the nature of the failure, but probably not.
				return None
			if self.context.complete:
				#  Just diagnostic.
				print 'GSSAPI name "%s"' % self.context.service
		return data

	def gss_exception(self, val):
		#
		#  Convert gssapi errors for application.
		#  Intended for use internal to the present class.
		#
		((maj, majstrs), (min, minstrs)) = val
		if maj == gssapi.GSS_S_CREDENTIALS_EXPIRED:
			print 'Credentials expired, run kinit!'
			return self.LocalInvalid(minstrs[0] + ', run kinit, OK?')
		elif min == KRB5_FCC_NOFILE:
			print 'No credentials, run kinit!'
			return self.LocalInvalid(minstrs[0] + ', run kinit, OK?')
		else:
			return self.UnknownError(val)

plugins = [('GSSAPI', GSSAPI_Plugin),]
