//  Copyright 1999 by Donn Cave, Seattle, Washington, USA.
//  All rights reserved.  Permission to copy, modify and distribute this
//  material is hereby granted, without fee, provided that the above
//  copyright notice appear in all copies.
/*
**  GSS API
**
*/

#include <stdio.h>
#include <Python.h>

#include <gssapi/gssapi_generic.h>

/* Exception exc_type. */	
static PyObject *GSSError = 0;

/*
**  Assemble the error strings for a maj/min pair,
**  give result in the form  ((maj, ('string', ...)), (min, ('string',...)))
*/
static PyObject *
gsserr(OM_uint32 err, OM_uint32 err2)
{
    PyObject *e, *er, *s;
    PyObject *elist[16];
    PyObject *ee;
    char ebuf[2048];
    struct {
	OM_uint32 stat;
	int type;
    } errs[2];
    int nerrs;
    int ie, eoff;

    errs[0].stat = err;
    errs[0].type = GSS_C_GSS_CODE;
    errs[1].stat = err2;
    errs[1].type = GSS_C_MECH_CODE;
    nerrs = 2;
    e = PyTuple_New(nerrs);
    if (e) {
	int ierr, i;
	for (ierr = 0; ierr < nerrs; ++ierr) {
	    OM_uint32 maj_stat, min_stat;
	    gss_buffer_desc msg;
	    OM_uint32 msg_ctx;

	    er = PyTuple_New(2);
	    PyTuple_SetItem(e, ierr, er);
	    PyTuple_SetItem(er, 0, PyInt_FromLong(errs[ierr].stat));
	    msg_ctx = 0;
	    for (ie = 0;; ie < 16) {
		OM_uint32 min_stat;
		maj_stat = gss_display_status(&min_stat, errs[ierr].stat,
					      errs[ierr].type, GSS_C_NULL_OID,
					      &msg_ctx, &msg);
		elist[ie++] = PyString_FromString(msg.value);
		gss_release_buffer(&min_stat, &msg);
		if (!msg_ctx)
		    break;
	    }
	    ee = PyTuple_New(ie);
	    for (i = 0; i < ie; ++i)
		PyTuple_SetItem(ee, i, elist[i]);
	    PyTuple_SetItem(er, 1, ee);
	}
    }
    PyErr_SetObject(GSSError, e);
    Py_DECREF(e);
    return 0;
}

typedef struct {
    PyObject_HEAD
    gss_ctx_id_t context_id;
    gss_name_t gss_name;
    OM_uint32 status;
} contextobject;

/*
**  Convert svc@host to service name.
**  Should take an optional socket, because this API is prone to error with
**  services on cluster domain names.
**  - Say you've connected to xsrv@x.org.com;
**  - now you want to authenticate, so you naturally use that name
**    to initialize the context.
**  - x.org.com is a cluster, whose various actual IP addresses
**    resolve back (cf. "reverse lookup") to x1.org.com, x2.org.com, etc.,
**    and gss_import_name() will pick one of those at random.  Not
**    necessarily the one you connected to, and if not the authentication
**    will fail.
**  Given the socket, it could fix the host.  However, that's actually
**  easier to handle higher up, and if you know enough to supply the socket,
**  you probably know what to do.
*/
static int
setcontextname(contextobject *obj, char *service_name)
{
    gss_buffer_desc send_tok;
    OM_uint32 maj_stat, min_stat;

    send_tok.value = service_name;
    /* fprintf(stderr, "name initialized to \"%s\".\n", service_name); */
    send_tok.length = strlen(service_name) + 1;
    maj_stat = gss_import_name(&min_stat, &send_tok,
			       (gss_OID) gss_nt_service_name,
			       &obj->gss_name);
    if (maj_stat == GSS_S_COMPLETE)
	return 1;
    else {
	gsserr(maj_stat, min_stat);
	return 0;
    }
}

/*
**  Process data from peer.
**  Usage:
**     g = Context('imap@mail.my.org')
**     svc.send(g.process())
**     svc.send(g.process(svc.read()))
**     ... repeat last as necessary while process() produces more data
**         for the service, and while the GSSAPI context is incomplete.
**  Data seems to typically include NUL terminators.
*/
static PyObject *
context_process(contextobject *self, PyObject *args)
{
    gss_buffer_desc clidat, srvdat, *clidatp;
    char *clistr;
    int clistrlen;
    OM_uint32 min, ret_flags;
    PyObject *ret;

    clistr = 0;
    clistrlen = 0;
    if (!PyArg_ParseTuple(args, "|s#:init", &clistr, &clistrlen))
	return 0;
    if (clistr && clistrlen > 0) {
	clidat.value = clistr;
	clidat.length = clistrlen;
	clidatp = &clidat;
	/* fprintf(stderr, "clistrlen %d\n", clistrlen); */
    } else
	clidatp = 0;

    self->status = gss_init_sec_context(
	&min, GSS_C_NO_CREDENTIAL,
	&self->context_id, self->gss_name, GSS_C_NULL_OID,
	GSS_C_MUTUAL_FLAG | GSS_C_REPLAY_FLAG, 0, 0, clidatp, 0,
	&srvdat, &ret_flags, 0
	);

    if (self->status != GSS_S_COMPLETE
	&& self->status != GSS_S_CONTINUE_NEEDED)
	    return gsserr(self->status, min);
    ret = PyString_FromStringAndSize(srvdat.value, srvdat.length);
    /* fprintf(stderr, "return string length %d\n", srvdat.length); */
    gss_release_buffer(&min, &srvdat);
    return ret;
}

static PyObject *
context_unwrap(contextobject *self, PyObject *args)
{
    gss_buffer_desc plain, wrapped;
    int state;
    OM_uint32 maj, min;
    int len;
    char *str;
    PyObject *ret;

    if (!PyArg_ParseTuple(args, "s#:unwrap", &str, &len))
	return 0;
    wrapped.value = str;
    wrapped.length = len;
    maj = gss_unwrap(&min, self->context_id, &wrapped, &plain, &state, 0);
    if (maj != GSS_S_COMPLETE)
	return gsserr(maj, min);

    ret = PyString_FromStringAndSize(plain.value, plain.length);
    gss_release_buffer(&min, &plain);
    return ret;
}

static PyObject *
context_wrap(contextobject *self, PyObject *args)
{
    gss_buffer_desc plain, wrapped;
    int state;
    OM_uint32 maj, min;
    int len;
    char *str;
    PyObject *ret;

    if (!PyArg_ParseTuple(args, "s#:wrap", &str, &len))
	return 0;
    plain.length = len;
    plain.value = str;
    maj = gss_wrap(&min, self->context_id, 1, GSS_C_QOP_DEFAULT,
		&plain, &state, &wrapped);
    if (maj != GSS_S_COMPLETE)
	return gsserr(maj, min);
 
    ret = PyString_FromStringAndSize(wrapped.value, wrapped.length);
    gss_release_buffer(&min, &wrapped);
    return ret;
}

static struct PyMethodDef context_methods[] = {
    {"init",		(PyCFunction) context_process, 1},
    {"unwrap",		(PyCFunction) context_unwrap, 1},
    {"wrap",		(PyCFunction) context_wrap, 1},
    {0, 0}
};

static void
context_dealloc(contextobject *self)
{
    OM_uint32 min_stat;
    /* fprintf(stderr, "context_dealloc(0x%08x)\n", self); */
    if (self->context_id)
	gss_delete_sec_context(&min_stat, &self->context_id, GSS_C_NO_BUFFER);
    if (self->gss_name)
	gss_release_name(&min_stat, &self->gss_name);
    PyMem_DEL(self);
}

static PyObject *
context_getattr(contextobject *self, char *name)
{
    /*
    **  There's a gss_inquire_context() that returns all kinds of
    **  information - client and service names, lifetime etc. -
    **  possibility to add here if anyone has a use for it.
    */
    if (!strcmp(name, "service") && self->gss_name) {
	gss_buffer_desc nmb;
	OM_uint32 st0, st1;

	st0 = gss_display_name(&st1, self->gss_name, &nmb, 0);
	if (st0 != GSS_S_COMPLETE) {
	    gsserr(st0, st1);
	    return 0;
	}
	return PyString_FromString(nmb.value);
    } else if (!strcmp(name, "complete"))
	return PyInt_FromLong(self->status == GSS_S_COMPLETE);
    else
	return Py_FindMethod(context_methods, (PyObject *) self, name);
}

static PyObject *
context_repr(contextobject *self)
{
    char buf[128];
    char str[256];
    sprintf(buf, "???");
    if (self->gss_name) {
	gss_buffer_desc nmb;
	OM_uint32 st0, st1;

	st0 = gss_display_name(&st1, self->gss_name, &nmb, 0);
	if (st0 == GSS_S_COMPLETE)
	     sprintf(buf, "service \"%.80s\", status 0x%08x",
		nmb.value, self->status);
    }
    sprintf(str, "<GSSAPI context: %.128s>", buf);
    return PyString_FromString(str);
}

static PyTypeObject ContextType = {
    PyObject_HEAD_INIT(&PyType_Type)
    0,			/*ob_size*/
    "context",		/*tp_name*/
    sizeof(contextobject),	/*tp_size*/
    0,			/*tp_itemsize*/
    /* methods */
    (destructor) context_dealloc, /*tp_dealloc*/
    0,			/*tp_print*/
    (getattrfunc) context_getattr, /*tp_getattr*/
    0,			/*tp_setattr*/
    0,			/*tp_compare*/
    (reprfunc) context_repr, /*tp_repr*/
};

#define Context_Check(v)  ((v)->ob_type == &ContextType)

/*
**  gssapi.Context(socket, service)
**
*/
static PyObject *
Context(PyObject *module, PyObject *args)
{
    gss_ctx_id_t c;
    contextobject *obj;
    int socket;
    char *srv;

    if (!PyArg_ParseTuple(args, "s:Context", &srv))
	return 0;
    obj = PyObject_NEW(contextobject, &ContextType);
    if (obj) {
	if (!setcontextname(obj, srv)) {
	    Py_DECREF(obj);
	    return 0;
	}
	obj->context_id = GSS_C_NO_CONTEXT;
	obj->status = GSS_S_CONTINUE_NEEDED;
    }
    return (PyObject *) obj;
}

static PyObject *
default_principal(PyObject *mod, PyObject *args)
{
    gss_buffer_desc nmb;
    gss_name_t name;
    OM_uint32 st0, st1;
    PyObject *ret;

    if (!PyArg_ParseTuple(args, ":default_principal"))
	return 0;
    st0 = gss_inquire_cred(&st1, GSS_C_NO_CREDENTIAL, &name, 0, 0, 0);
    if (st0 != GSS_S_COMPLETE) {
	gsserr(st0, st1);
	return 0;
    }
    st0 = gss_display_name(&st1, name, &nmb, 0);
    if (st0 != GSS_S_COMPLETE) {
	gsserr(st0, st1);
	return 0;
    }
    ret = PyString_FromStringAndSize(nmb.value, nmb.length - 1);
    gss_release_buffer(&st1, &nmb);
    gss_release_name(&st1, &name);
    return ret;
}


static struct PyMethodDef gss_global_methods[] = {
    {"Context", Context, 1},
    {"default_principal", default_principal, 1},
    {NULL, NULL}		/* sentinel */
};

#define set(v) PyDict_SetItemString(dict, #v, PyInt_FromLong(v))

initgssapi()
{
    PyObject *module;
    PyObject *dict;

    module = Py_InitModule("gssapi", gss_global_methods);
    dict = PyModule_GetDict(module);
    
    /* Initialize pwicon.error exception */
    GSSError = PyString_FromString("gssapi.error");
    if (GSSError == NULL
	|| PyDict_SetItemString(dict, "error", GSSError) != 0)
	Py_FatalError("can't define gssapi.error");

    set(GSS_S_COMPLETE);
    set(GSS_S_CREDENTIALS_EXPIRED);
    set(GSS_S_CONTINUE_NEEDED);
    set(GSS_S_BAD_MECH);
    set(GSS_S_BAD_NAME);
    set(GSS_S_NO_CRED);
    set(GSS_S_DEFECTIVE_TOKEN);
    set(GSS_S_DEFECTIVE_CREDENTIAL);
    set(GSS_S_FAILURE);
    set(GSS_S_UNAUTHORIZED);
    set(GSS_S_UNAVAILABLE);

    /* set(GSS_C_INDEFINITE);
    set(GSS_S_COMPLETE);
    set(GSS_S_BAD_MECH);
    set(GSS_S_BAD_NAME);
    set(GSS_S_BAD_NAMETYPE);
    set(GSS_S_BAD_BINDINGS);
    set(GSS_S_BAD_STATUS);
    set(GSS_S_BAD_SIG);
    set(GSS_S_NO_CRED);
    set(GSS_S_NO_CONTEXT);
    set(GSS_S_DEFECTIVE_TOKEN);
    set(GSS_S_DEFECTIVE_CREDENTIAL);
    set(GSS_S_CREDENTIALS_EXPIRED);
    set(GSS_S_CONTEXT_EXPIRED);
    set(GSS_S_FAILURE);
    set(GSS_S_BAD_QOP);
    set(GSS_S_UNAUTHORIZED);
    set(GSS_S_UNAVAILABLE);
    set(GSS_S_DUPLICATE_ELEMENT);
    set(GSS_S_CONTINUE_NEEDED);
    set(GSS_S_DUPLICATE_TOKEN);
    set(GSS_S_OLD_TOKEN);
    set(GSS_S_UNSEQ_TOKEN);
    set(GSS_S_GAP_TOKEN);
    */
    return 1;
}
