//  Copyright 1999 by Donn Cave, Seattle, Washington, USA.
//  All rights reserved.  Permission to copy, modify and distribute this
//  material is hereby granted, without fee, provided that the above
//  copyright notice appear in all copies.

#include <stdio.h>
#include <string.h>

#include <kernel/image.h>
#include <kernel/OS.h>

#include <Python.h>

static PyObject *PykernelError;
static PyObject *module_dict;

static PyObject *
Pykernel_load_image(PyObject *module, PyObject *args)
{
	PyObject *argv, *env;
	char **argvlist;
	char **envlist;
	PyObject *key, *val, *keys=NULL, *vals=NULL;
	int i, pos, argc, envc;
	thread_id thread;
	PyObject *(*getitem)(PyObject *, int);

	/* load_image has two arguments: (argv, env),
	   where argv is a list or tuple of strings and env is a dictionary
	   like posix.environ. argc is determined from argv. */

	if (!PyArg_Parse(args, "(OO)", &argv, &env))
		return NULL;
	if (PyList_Check(argv)) {
		argc = PyList_Size(argv);
		getitem = PyList_GetItem;
	}
	else if (PyTuple_Check(argv)) {
		argc = PyTuple_Size(argv);
		getitem = PyTuple_GetItem;
	}
	else {
		PyErr_SetString(PyExc_TypeError, "argv must be tuple or list");
		return NULL;
	}
	if (!PyMapping_Check(env)) {
		PyErr_SetString(PyExc_TypeError, "env must be mapping object");
		return NULL;
	}

	argvlist = PyMem_NEW(char *, argc+1);
	if (argvlist == NULL) {
		PyErr_NoMemory();
		return NULL;
	}
	for (i = 0; i < argc; i++) {
		if (!PyArg_Parse((*getitem)(argv, i),
				 "s;argv must be list of strings",
				 &argvlist[i]))
		{
			goto fail_1;
		}
	}
	argvlist[argc] = NULL;

	i = PyMapping_Length(env);
	envlist = PyMem_NEW(char *, i + 1);
	if (envlist == NULL) {
		PyErr_NoMemory();
		goto fail_1;
	}
	envc = 0;
	keys = PyMapping_Keys(env);
	vals = PyMapping_Values(env);
	if (!keys || !vals)
		goto fail_2;
	
	for (pos = 0; pos < i; pos++) {
		char *p, *k, *v;

		key = PyList_GetItem(keys, pos);
		val = PyList_GetItem(vals, pos);
		if (!key || !val)
			goto fail_2;
		
		if (!PyArg_Parse(key, "s;non-string key in env", &k) ||
		    !PyArg_Parse(val, "s;non-string value in env", &v))
		{
			goto fail_2;
		}

		p = PyMem_NEW(char, PyString_Size(key)+PyString_Size(val) + 2);
		if (p == NULL) {
			PyErr_NoMemory();
			goto fail_2;
		}
		sprintf(p, "%s=%s", k, v);
		envlist[envc++] = p;
	}
	envlist[envc] = 0;

	thread = load_image(argc, (const char **) argvlist, (const char **) envlist);

	if (thread <= 0) {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", thread, "load_image() failed"));
	}

	/*
	**  Unlike execve(), load_image() does return.
	**  It's OK to free argv, envlist.
	*/
 fail_2:
	/* while (--envc >= 0)
		PyMem_DEL(envlist[envc]);
	PyMem_DEL(envlist); */
 fail_1:
	/* PyMem_DEL(argvlist);
	Py_XDECREF(vals);
	Py_XDECREF(keys); */

	if (thread > 0)
		return PyInt_FromLong(thread);
	else
		return 0;
}

static PyObject *
Pykernel_find_thread(PyObject *module, PyObject *args)
{
	thread_id thread;
	char *name;
	status_t status, exit_value;

	if (args) {
		if (PyString_Check(args))
			name = PyString_AsString(args);
		else {
			PyErr_SetString(PyExc_TypeError, "find_thread(int)");
			return 0;
		}
	} else
		name = 0;
	thread = find_thread(name);
	if (thread > 0)
		return PyInt_FromLong(thread);
	else {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", thread, strerror(thread)));
		return 0;
	}
}

static PyObject *
Pykernel_set_thread_priority(PyObject *module, PyObject *args)
{
	thread_id thread;
	int32 priority;
	status_t status;
	if (!PyArg_ParseTuple(args, "ii", &thread, &priority)) {
		PyErr_SetString(PyExc_TypeError, "set_thread_priority(thread_id, int32)");
		return 0;
	}
	status = set_thread_priority(thread, priority);
	if (status > 0) {
		return PyInt_FromLong(status);
	} else {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", status, strerror(status)));
		return 0;
	}
}

static PyObject *
Pykernel_resume_thread(PyObject *module, PyObject *args)
{
	int thread, options;
	status_t status;

	if (!PyInt_Check(args)) {
		PyErr_SetString(PyExc_TypeError, "resume_thread(int)");
		return 0;
	}
	thread = PyInt_AsLong(args);
	Py_BEGIN_ALLOW_THREADS
	status = resume_thread(thread);
	Py_END_ALLOW_THREADS
	if (status == B_OK) {
		Py_INCREF(Py_None);
		return Py_None;
	} else {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", status, strerror(status)));
		return 0;
	}
}

static PyObject *
Pykernel_wait_for_thread(PyObject *module, PyObject *args)
{
	int thread, options;
	status_t status, exit_value;

	if (!PyInt_Check(args)) {
		PyErr_SetString(PyExc_TypeError, "wait_for_thread(int)");
		return 0;
	}
	thread = PyInt_AsLong(args);
	Py_BEGIN_ALLOW_THREADS
	status = wait_for_thread(thread, &exit_value);
	Py_END_ALLOW_THREADS
	if (status == B_OK)
		return PyInt_FromLong(exit_value);
	else {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", status, strerror(status)));
		return 0;
	}
}

static PyObject *
Pykernel_acquire_sem(PyObject *module, PyObject *args)
{
	status_t status;
	sem_id sem;

	if (!PyInt_Check(args)) {
		PyErr_SetString(PyExc_TypeError, "acquire_sem(int)");
		return 0;
	}
	sem = PyInt_AsLong(args);
	Py_BEGIN_ALLOW_THREADS
	status = acquire_sem(sem);
	Py_END_ALLOW_THREADS
	if (status == B_NO_ERROR) {
		Py_INCREF(Py_None);
		return Py_None;
	} else {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", status, strerror(status)));
		return 0;
	}
}

static PyObject *
Pykernel_create_sem(PyObject *module, PyObject *args)
{
	uint32 thread_count;
	sem_id sem;
	const char *name = 0;

	if (!PyArg_ParseTuple(args, "i|s", &thread_count, &name)) {
		PyErr_SetString(PyExc_TypeError, "create_sem(int, string)");
		return 0;
	}
	sem = create_sem(thread_count, name);
	if (sem > 0)
		return PyInt_FromLong(sem);
	else {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", sem, strerror(sem)));
		return 0;
	}
}

static PyObject *
Pykernel_delete_sem(PyObject *module, PyObject *args)
{
	status_t status;
	sem_id sem;

	if (!PyInt_Check(args)) {
		PyErr_SetString(PyExc_TypeError, "delete_sem(int)");
		return 0;
	}
	sem = PyInt_AsLong(args);
	status = delete_sem(sem);
	if (status == B_NO_ERROR) {
		Py_INCREF(Py_None);
		return Py_None;
	} else {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", status, strerror(status)));
		return 0;
	}
}

static PyObject *
Pykernel_release_sem(PyObject *module, PyObject *args)
{
	status_t status;
	sem_id sem;

	if (!PyInt_Check(args)) {
		PyErr_SetString(PyExc_TypeError, "release_sem(int)");
		return 0;
	}
	sem = PyInt_AsLong(args);
	Py_BEGIN_ALLOW_THREADS
	status = release_sem(sem);
	Py_END_ALLOW_THREADS
	if (status == B_NO_ERROR) {
		Py_INCREF(Py_None);
		return Py_None;
	} else {
		PyErr_SetObject(PykernelError,
			Py_BuildValue("ls", status, strerror(status)));
		return 0;
	}
}

static PyObject *
Pykernel_system_time(PyObject *module, PyObject *args)
{
	bigtime_t res;

	if (PyTuple_Size(args) > 0) {
		PyErr_SetString(PyExc_TypeError, "system_time()");
		return 0;
	}
	res = system_time();
	return PyLong_FromLongLong(res);
}

static struct PyMethodDef kernel_global_methods[] = {
	{"create_sem", Pykernel_create_sem, 0},
	{"acquire_sem", Pykernel_acquire_sem, 0},
	{"delete_sem", Pykernel_delete_sem, 0},
	{"release_sem", Pykernel_release_sem, 0},
	{"load_image",  Pykernel_load_image, 0},
	{"resume_thread",  Pykernel_resume_thread, 0},
	{"wait_for_thread",  Pykernel_wait_for_thread, 0},
	{"find_thread", Pykernel_find_thread, 0},
	{"set_thread_priority", Pykernel_set_thread_priority, 0},
	{"system_time", Pykernel_system_time, 0},
	{NULL,		NULL}		/* sentinel */
};

#define set(s) PyDict_SetItemString(module_dict, #s, PyInt_FromLong(s))

#define preset(s) stdPyBLooperObject = PyBLooper_new(); \
	SETEXT_BEPTR(stdPyBLooperObject, s); \
	PyDict_SetItemString(module_dict, #s, (PyObject *) stdPyBLooperObject);

extern "C" void
initkernel()
{
	PyObject *mod = Py_InitModule("kernel", kernel_global_methods);
	module_dict = PyModule_GetDict(mod);
	PykernelError = PyErr_NewException("kernel.error", 0, 0);
	PyDict_SetItemString(module_dict, "error", PykernelError);
}
