/* ArpControllerMap.cpp
 */
#include "ArpControllerMap.h"

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "ArpKernel/ArpDebug.h"
#include "ArpLayout/ArpViewWrapper.h"
#include "ArpViewsPublic/ArpIntFormatterI.h"
#include "ArpViews/ArpIntControl.h"
#include "AmPublic/AmControls.h"
#include "AmPublic/AmFilterConfigLayout.h"

ArpMOD();
static AmStaticResources gRes;

static const char*	FROM_NUMBER		= "From Number";
static const char*	TO_NUMBER		= "To Number";

/*****************************************************************************
 * ARP-CONTROLLER-MAP-FILTER
 *****************************************************************************/
ArpControllerMapFilter::ArpControllerMapFilter(	ArpControllerMapAddOn* addon,
											AmFilterHolderI* holder,
											const BMessage* config)
		: AmFilterI(addon),
		  mAddOn(addon), mHolder(holder),
		  mFromNumber(0), mToNumber(0)
{
	if (config) PutConfiguration(config);
}

ArpControllerMapFilter::~ArpControllerMapFilter()
{
}

AmEvent* ArpControllerMapFilter::HandleEvent(AmEvent* event, const am_filter_params* /*params*/)
{
	if (!event) return event;
	ArpVALIDATE(mAddOn != NULL && mHolder != NULL, return event);
	event->SetNextFilter(mHolder->ConnectionAt(0) );
	if (mFromNumber == mToNumber) return event;
	if (event->Type() != event->CONTROLCHANGE_TYPE) return event;
	AmControlChange*		e = dynamic_cast<AmControlChange*>(event);
	if (!e) return event;
	if (e->ControlNumber() != mFromNumber) return event;
	e->SetControlNumber(mToNumber);	
	return event;
}

status_t ArpControllerMapFilter::GetConfiguration(BMessage* values) const
{
	status_t err = AmFilterI::GetConfiguration(values);
	if (err != B_OK) return err;

	if ( (err = values->AddInt32(FROM_NUMBER, mFromNumber)) != B_OK) return err;
	if ( (err = values->AddInt32(TO_NUMBER, mToNumber)) != B_OK) return err;

	return B_OK;
}

status_t ArpControllerMapFilter::PutConfiguration(const BMessage* values)
{
	status_t err = AmFilterI::PutConfiguration(values);
	if (err != B_OK) return err;

	int32		i;
	if (values->FindInt32(FROM_NUMBER, &i) == B_OK) mFromNumber = i;
	if (values->FindInt32(TO_NUMBER, &i) == B_OK) mToNumber = i;

	return B_OK;
}

class ArpControllerMapSettings : public AmFilterConfigLayout
{
public:
	ArpControllerMapSettings(AmFilterHolderI* target,
						  const BMessage& initSettings)
		: AmFilterConfigLayout(target, initSettings)
	{
		try {
			AddLayoutChild((new ArpRunningBar("TopVBar"))
				->SetParams(ArpMessage()
					.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
					.SetFloat(ArpRunningBar::IntraSpaceP, .5)
				)
				->AddLayoutChild((new ArpTextControl(
										SZ_FILTER_LABEL, "Label:","",
										mImpl.AttachTextControl(SZ_FILTER_LABEL)))
					->SetParams(ArpMessage()
						.SetString(ArpTextControl::MinTextStringP, "8")
						.SetString(ArpTextControl::PrefTextStringP, "8888888888")
					)
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,0)
						.SetInt32(ArpRunningBar::FillC,ArpEastWest)
						.SetBool(ArpRunningBar::AlignLabelsC,false)
					)
				)

				->AddLayoutChild((new ArpBox("FromNumberBox", "From Number"))
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,1)
						.SetInt32(ArpRunningBar::FillC,ArpFillAll)
						.SetBool(ArpRunningBar::AlignLabelsC,false)
					)
					->AddLayoutChild((new AmControlChangeListPanel("Control Changes", this, initSettings, FROM_NUMBER))
						->SetParams(ArpMessage()
							.SetBool(ArpScrollArea::ScrollHorizontalP,false)
							.SetBool(ArpScrollArea::ScrollVerticalP,true)
							.SetBool(ArpScrollArea::InsetCornerP,false)
							.SetInt32(ArpScrollArea::BorderStyleP,B_FANCY_BORDER)
						)
						->SetConstraints(ArpMessage()
							.SetFloat(ArpRunningBar::WeightC,1)
							.SetInt32(ArpRunningBar::FillC,ArpWest)
							.SetBool(ArpRunningBar::AlignLabelsC,false)
						)
					)
				)
				->AddLayoutChild((new ArpBox("ToNumberBox", "To Number"))
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,1)
						.SetInt32(ArpRunningBar::FillC,ArpFillAll)
						.SetBool(ArpRunningBar::AlignLabelsC,false)
					)
					->AddLayoutChild((new AmControlChangeListPanel("Control Changes", this, initSettings, TO_NUMBER))
						->SetParams(ArpMessage()
							.SetBool(ArpScrollArea::ScrollHorizontalP,false)
							.SetBool(ArpScrollArea::ScrollVerticalP,true)
							.SetBool(ArpScrollArea::InsetCornerP,false)
							.SetInt32(ArpScrollArea::BorderStyleP,B_FANCY_BORDER)
						)
						->SetConstraints(ArpMessage()
							.SetFloat(ArpRunningBar::WeightC,1)
							.SetInt32(ArpRunningBar::FillC,ArpWest)
							.SetBool(ArpRunningBar::AlignLabelsC,false)
						)
					)
				)
			);
		} catch(...) {
			throw;
		}
		Implementation().RefreshControls(mSettings);
	}

protected:
	typedef AmFilterConfigLayout inherited;
};

status_t ArpControllerMapFilter::Configure(ArpVectorI<BView*>& panels)
{
	BMessage config;
	status_t err = GetConfiguration(&config);
	if (err != B_OK) return err;
	panels.push_back(new ArpControllerMapSettings(mHolder, config));
	return B_OK;
}

/*****************************************************************************
 * ARP-CONTROLLER-MAP-ADDON
 *****************************************************************************/
void ArpControllerMapAddOn::LongDescription(BString& name, BString& str) const
{
	AmFilterAddOn::LongDescription(name, str);
	str << "<p>I take every control change with a number of From Number and
	transform them into a control change with a number of To Number.</p>";
}

void ArpControllerMapAddOn::GetVersion(int32* major, int32* minor) const
{
	*major = 1;
	*minor = 0;
}

BBitmap* ArpControllerMapAddOn::Image(BPoint requestedSize) const
{
	const BBitmap* bm = gRes.Resources().FindBitmap("Class Icon");
	if (bm) return new BBitmap(bm);
	return NULL;
}

AmFilterI* ArpControllerMapAddOn::NewInstance(AmFilterHolderI* holder,
											const BMessage* config)
{
	return new ArpControllerMapFilter(this, holder, config);
}

extern "C" _EXPORT AmFilterAddOn* make_nth_filter(int32 n, image_id /*you*/,
												  const void* cookie, uint32 /*flags*/, ...)
{
	if (n == 0) return new ArpControllerMapAddOn(cookie);
	return NULL;
}
