//
// GammaUtil
//  SiS6326ドライバ用ガンマ値変更ユーティリティ
//

#include	<Application.h>
#include	<Window.h>
#include	<CheckBox.h>
#include	<Slider.h>
#include	<Button.h>
#include	<Screen.h>
#include	<File.h>

#include	<stdio.h>
#include	<unistd.h>
#include	<fcntl.h>
#include	<string.h>
#include	<dirent.h>

#include	"uSEQR.h"
#include	"SiSDriverIF.h"
#include	"IOaccess.h"
#include	"SiSaccelutil.h"
#include	"SiSgamma.h"
#include	"DriverInterface.h"

int	fd;
#define	kDEVDIR	"/dev/graphics/"
#define	kDEVNAME_BASE	"1039_6326_"

#define	kINITVALUE	1.0

#define	kMSG_SLIDER		'mSLD'
#define	kMSG_ENABLEGAMMA	'mENB'
#define	kMSG_EXEC			'EXEC'

class GammaWin : public BWindow {
public:
				GammaWin(void);
	void			MessageReceived(BMessage *argMessage);
	void			ScreenChanged(BRect, color_space);
	void			WorkSpaceChanged(uint32, uint32);
	
private:
	bool			QuitRequested(void);

	void			SetGamma(double argGamma);
	void			DisableGamma(void);

	void			EnableControls(bool argEnable);
	void			SetGammaLabel(void);
	void			SavePref(double);
	void			CheckScreenAndSetControl();
		
	BCheckBox		*fEnable;
	BSlider		*fSlider;
	BButton		*fExec;
};

GammaWin::GammaWin()
	: BWindow(BRect(100, 100, 300, 200)
			, "SiS6326 Gamma"
			, B_TITLED_WINDOW
			, B_NOT_ZOOMABLE | B_NOT_RESIZABLE) {
	bool		theEnable;
	double	theValue;
	
	fEnable = new BCheckBox(BRect(10, 2, 189, 21)
						, "GammaCheck"
						, "Enable Gamma inverse"
						, new BMessage(kMSG_ENABLEGAMMA));
	AddChild(fEnable);
	
	fSlider = new BSlider(BRect(10,22, 189, 71)
					, "GammaSlider"
					, "Gamma = 1.0"
					, new BMessage(kMSG_SLIDER)
					, (int32)(0.5 * 10)
					, (int32)(2.5 * 10));
	fSlider->SetHashMarks(B_HASH_MARKS_BOTTOM);
	fSlider->SetHashMarkCount(5);
	fSlider->SetLimitLabels("0.5", "2.5");
	fSlider->SetValue((int32)kINITVALUE * 10);
	AddChild(fSlider);
	
	fExec = new BButton(BRect(70,72,130,89)
					, "GammaExec"
					, "Set"
					, new BMessage(kMSG_EXEC));
	AddChild(fExec);

	theEnable = GetGammaPref(&theValue);
	
	if (theEnable == true) {
		fEnable->SetValue(B_CONTROL_ON);
		fSlider->SetValue((int)((theValue + 0.05) * 10));
	}else{
		fEnable->SetValue(B_CONTROL_OFF);
		fSlider->SetValue(10);
	}
	SetGammaLabel();		
	CheckScreenAndSetControl();
}

void
GammaWin::MessageReceived(BMessage *argMessage) {
	switch(argMessage->what) {
	case kMSG_ENABLEGAMMA :
		if (fEnable->Value() == B_CONTROL_ON) {
			fSlider->SetEnabled(true);
		}else{
			fSlider->SetEnabled(false);
		}
		break;		
	case kMSG_SLIDER :
		SetGammaLabel();
		break;
	case kMSG_EXEC :
		fprintf(stderr, "kMSG_EXEC received.\n");
		if (fEnable->Value() == B_CONTROL_ON) {
			SetGamma((double)fSlider->Value() / 10);
		}else{
			DisableGamma();
		}
		break;
	default:
		BWindow::MessageReceived(argMessage);
	}
}

void
GammaWin::ScreenChanged(BRect, color_space argColorSpace) {
	if (argColorSpace == B_CMAP8) {
		EnableControls(false);
		fSlider->SetLabel("Can't set inverse on 256-color mode.");
	}else{
		EnableControls(true);
		SetGammaLabel();
	}
}

void
GammaWin::WorkSpaceChanged(uint32, uint32) {
	CheckScreenAndSetControl();
}

void
GammaWin::CheckScreenAndSetControl() {
	BScreen	*theScreen;
	
	theScreen = new BScreen(this);
	if (theScreen->ColorSpace() == B_CMAP8) {
		EnableControls(false);
		fSlider->SetLabel("Can't set inverse on 256-color mode.");
	}else{
		EnableControls(true);
		SetGammaLabel();
	}
}

bool
GammaWin::QuitRequested() {
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

void
GammaWin::SetGamma(double argVal) {
	tIOGammaTbl	theSetTable;
	status_t		theRet;
	
	CalcGamma(&theSetTable, argVal);
	theRet = ioctl(fd, YSIS_SET_GAMMA, &theSetTable); 
	
	SavePref(argVal);
}

void
GammaWin::DisableGamma() {
	tIOGammaTbl	theSetTable;

	theSetTable.fToken = kGAMMA_SET_TOKEN;	
	theSetTable.fEnable = false;
	
	(void)ioctl(fd, YSIS_SET_GAMMA, &theSetTable); 
	
	SavePref(0.0);
}

void
GammaWin::EnableControls(bool argEnable) {
	fEnable->SetEnabled(argEnable);
	if (argEnable == true && fEnable->Value() ==  B_CONTROL_ON) {
		fSlider->SetEnabled(true);
	}else{
		fSlider->SetEnabled(false);
	}
	fExec->SetEnabled(argEnable);
}

void
GammaWin::SetGammaLabel() {
	char	abuf[256];

	sprintf(abuf, "Inverse = %.1f", (double)fSlider->Value() / 10.0);	
	fSlider->SetLabel(abuf);
}

void
GammaWin::SavePref(double argVal) {
	BFile		*thePrefFile;
	char		theBuf[256];
	
	thePrefFile = new BFile(kPREFFILE, B_WRITE_ONLY | B_CREATE_FILE );
	if (thePrefFile->InitCheck() != B_OK) {
		return;
	}
	if (fEnable->Value() == B_CONTROL_ON) {
		sprintf(theBuf, "Gamma %f\n", argVal);
	}else{
		sprintf(theBuf, "Gamma disabled\n");
	}
	thePrefFile->Write(theBuf, strlen(theBuf));
	delete thePrefFile;
}
			
//
// GammaUtil 
//
class GammaUtil : public BApplication {
public:
			GammaUtil();
};

GammaUtil::GammaUtil()
	: BApplication("application/x-vnd.ySiS6326_GammaUtil") {
	GammaWin	*theWin;
	
	theWin = new GammaWin();
	theWin->Show();
}

int
main(int , char *[])
{
	char			theDevName[PATH_MAX + 1];
	char			theDevFullName[PATH_MAX + 1];
	
	DIR			*theDir;
	struct dirent	*theEntry;	
	GammaUtil		theApp;

	strcpy(theDevFullName, kDEVDIR);
	if (theDir = opendir(kDEVDIR)) {
		theDevName[0] = '\0';
		while(theEntry = readdir(theDir)) {
			if (0 == strncmp(kDEVNAME_BASE, theEntry->d_name, strlen(kDEVNAME_BASE))) {
				strcpy(theDevName, theEntry->d_name);
				break;
			}
		}
		if (theDevName[0]) {
			strcat(theDevFullName, theDevName);
			fprintf(stderr, "%s\n", theDevFullName);
			fd = open(theDevFullName, O_RDWR);
			if (fd < 0) {
				perror(theDevFullName);
				exit(1);
			}
			
			theApp.Run();

			close(fd);	
			return 0;
		}else{
			fprintf(stderr, "Cannot find supported device.\n");
			return 1;
		}			
	}else{
		fprintf(stderr, "Cannot find %s.\n", kDEVDIR);
		return 1;
	}
}