//
// SiSaccelutil.c
//	A part of SiS6326 driver for BeOS
//	Copyright (C) 2000 Yasutaka Uematu All rights reserved.
//	yasutaka2000@bemail.org
//

#include	<support/Debug.h>
#include	"GlobalData.h"
#include	"generic.h"
#define	MAIN
#include	"SiSaccelData.h"
#include	"SiSDriverIF.h"
#include	"IOaccess.h"
#include	"SiSaccelutil.h"
#include	"SiSdebugutil.h"

extern unsigned long Get_Card_Mem_Size();	// InitAccelerant.c

void
SetFrameAddress(uint32 argAddr)
{
	gSEQR.s.fSR20.LABaddress2619 = (uint8)((argAddr >> 19) & 0xff);
	gSEQR.s.fSR21.LABaddress3127 = (uint8)((argAddr >> 27) & 0x1f);
	SERIAL_PRINT(("SR20 = %x, SR21 = %x\n", gSEQR.fValue[0x20], gSEQR.fValue[0x21]));
}

void
SeqOut(uint8 argIdx, uint8 argVal)
{
	tIOIdxAccess	theIO;
	
	theIO.fOffset = SEQ_IDX;
	theIO.fIndex = argIdx;
	theIO.fValue = argVal;
	ioctl(fd, YSIS_WRITE_INDEXED, &theIO);
}

void
CrtOut(uint8 argIdx, uint8 argVal)
{
	tIOIdxAccess	theIO;
	
	theIO.fOffset = CRT_IDX;
	theIO.fIndex = argIdx;
	theIO.fValue = argVal;
	ioctl(fd, YSIS_WRITE_INDEXED, &theIO);
}

void
GctOut(uint8 argIdx, uint8 argVal)
{
	tIOIdxAccess	theIO;
	
	theIO.fOffset = GCT_IDX;
	theIO.fIndex = argIdx;
	theIO.fValue = argVal;
	ioctl(fd, YSIS_WRITE_INDEXED, &theIO);
}

void
AtrOut(uint8 argIdx, uint8 argVal)
{
	uint8	theDmy;

	theDmy = inb(IOP_ISTATUS1);
	outb(ATR_IDX, argIdx);
	outb(ATR_WRITE, argVal);
}

//
// レジスタのうち、固定の値をセットしておく
void
SiSPutRes()
{
	uint32	theFBaddr;
	
	theFBaddr = (uint32)(si->framebuffer_pci);
	SERIAL_PRINT(("ySiS accel : SiSPutRes() theFBaddr = %p\n", theFBaddr));

	// SEQR
	gSEQR.s.fSR0.Areset = 1;	// No reset
	gSEQR.s.fSR0.Sreset = 0;	// MiscOutの変更、SR1の変更前には必ず０にすること。
	SeqOut(0x00, gSEQR.fValue[0x00]);

	// Generic
	gMiscOut = 0xdf;
	gFeature = 0x00;			// Cleared 001109
	gVGAEnable = 0x01;
	gSegSel0 = 0x00;
	gSegSel1 = 0x00;

	// Misc, etc
	outb(IOP_MISC_WRITE, gMiscOut);
	outb(IOP_FEATURE_WRITE, gFeature);
	outb(IOP_VGAENABLE, gVGAEnable);
	outb(IOP_SEGSEL0, gSegSel0);
	outb(IOP_SEGSEL1, gSegSel1);

	gSEQR.s.fSR1.DC89 = 1;	// Dot Clock /= 8
	gSEQR.s.fSR1.Soff = 1;	// Display Off
	SeqOut(0x01, gSEQR.fValue[0x01]);

	// MiscOut, SR1の変更が終わったのでSyncResetを1(No reset)にする。
	gSEQR.s.fSR0.Sreset = 1;	// No reset
	SeqOut(0x00, gSEQR.fValue[0x00]);
	
	gSEQR.s.fSR2.P0Wenable = 1;
	gSEQR.s.fSR2.P1Wenable = 1;
	gSEQR.s.fSR2.P2Wenable = 1;
	gSEQR.s.fSR2.P3Wenable = 1;
	SeqOut(0x02, gSEQR.fValue[0x02]);

	gSEQR.s.fSR4.Ememory = 1;	// Select 256k
	gSEQR.s.fSR4.OEMenable = 1;	// Odd/Even Mode Disable
	gSEQR.s.fSR4.C4Menable = 1;	// Chain-4 mode Enable	
	SeqOut(0x04, gSEQR.fValue[0x04]);

	gSEQR.s.fSR6.EGMenable = 1;	// Enhanced graphics mode enable
	gSEQR.s.fSR6.GMLAenable = 1;	// Graphics mode linear addressing enable
	SeqOut(0x06, gSEQR.fValue[0x06]);

//   下の１行。速度向上に寄与するかと思ったが、なんの関係もない様子。0にする。
	gSEQR.s.fSR7.MVLBiCfifo = 0;	// Merge video line buffer into CRT FIFO
	gSEQR.s.fSR7.IRPSmode = 1;	// Internal RAMDAC power saving mode(1=High power mode)
	SeqOut(0x07, gSEQR.fValue[0x07]);
	
	gSEQR.s.fSR9.AAthreshold = 0x0;	// ASCII/Attribute Threshold Bit[3:0]
	SeqOut(0x09, gSEQR.fValue[0x09]);

	gSEQR.s.fSRB.MMIOSselection = 3;	// Memory-mapped I/O Space Selection Bit[1:0]
								// 3 = Select PCI config register 14H
	gSEQR.s.fSRB.IOGenable = 1;	// I/O gating enable while write-buffer is not empty
	SeqOut(0x0b, gSEQR.fValue[0x0b]);
	
	gSEQR.s.fSRC.RACOenable = 1;	// Read-ahead cache operation enable
	gSEQR.s.fSRC.GM32MAenable = 1;	// Graphics mode 32?bit memory access enable
	SeqOut(0x0c, gSEQR.fValue[0x0c]);
	
	// Add 001103
	// ときどき、screen saverで死ぬ場合がある。再起動しか手段はないが、
	// DPMSモードをりせっとしておかないと、再起動しても画面は真っ暗のまま。
	// そのための対策をここでおこなう。
	gSEQR.s.fSR11.FVsuspend = 0;	// Force VGA into suspend mode = Disable
	gSEQR.s.fSR11.FVstandby = 0;	// Force VGA into standby mode = Disable
	SeqOut(0x11, gSEQR.fValue[0x11]);
	// Add 001103 end
	// framebufferのアドレス設定(PCI側のアドレスを設定すること)
	SetFrameAddress(theFBaddr);
	// Linear Addressing Space Aperture Bit[2:0]=4(8MB)
	gSEQR.s.fSR21.LASaperture = 4;
	SeqOut(0x20,  gSEQR.fValue[0x20]);
	SeqOut(0x21,  gSEQR.fValue[0x21]);

	gSEQR.s.fSR23.DCSDcompensation = 1;		// DRAM Control Signal Delay Compensation Bit = 1(5ns)
	gSEQR.s.fSR23.EDOenable = 0;	// EDO DRAM Enable Bit = 0(Disable)
	SeqOut(0x23, gSEQR.fValue[0x23]);

	gSEQR.s.fSR26.PBMenable = 1;	// PCI Burst-Write Mode enable 
	gSEQR.s.fSR26.CMDAenable = 0;	// Continuous Memory Data Access Enable Bit = 0(Disable)
	gSEQR.s.fSR26.SDRPtime = 0;	// Slow DRAM RAS pre-charge time=0(Disable)
	gSEQR.s.fSR26.SFEDR2CTenable = 0;	// Slow FP/EDO DRAM RAS* to CAS* Timing enable=0(Disable)
	SeqOut(0x26, gSEQR.fValue[0x26]);
	
	gSEQR.s.fSR27.TQEenable = 0;	// Turbo Queue Engine disable
	gSEQR.s.fSR27.GERPenable = 1;	// Graphics Engine Register Programming enable
	gSEQR.s.fSR27.LSWaBselect = 2;	// Logical Screen Width and Byte-Per-Pixel Select Bit[1:0]=2
							// (4096 , 256 colors or 2048, 32k/64k colors)
	SeqOut(0x27, gSEQR.fValue[0x27]);

	gSEQR.s.fSR2D.PSselect = 4;	// Page Size Select=4(1 KB at 32-bit mode, 2 KB at 64-bit mode)
	SeqOut(0x2d, gSEQR.fValue[0x2d]);

	gSEQR.s.fSR2F.EFPflip = 1;	// Enable Fast Page Flip (for MOVE_DISPLAY)
	SeqOut(0x2f, gSEQR.fValue[0x2f]);
	
	gSEQR.s.fSR30.FPFSaddress70 = 0;	// Extended Fast Page Flip Starting Address Low Register
	SeqOut(0x30, gSEQR.fValue[0x30]);
	
	gSEQR.s.fSR31.FPFSaddress158 = 0;	// Extended Fast Page Flip Starting Address Middle Register
	SeqOut(0x31, gSEQR.fValue[0x31]);
	
	gSEQR.s.fSR32.FPFSaddress1916 = 0;	// Extended Fast Page Flip Starting Address High Register 
	SeqOut(0x32, gSEQR.fValue[0x32]);
	
	gSEQR.s.fSR33.SVGAIODdisable = 1;
	gSEQR.s.fSR33.EPCEDtiming = 0;	// Enable one cycle EDO DRAM timing=0(Disable)
	gSEQR.s.fSR33.SGlatency = 0;	// Select SGRAM Latency=0(3)
	gSEQR.s.fSR33.EStiming = 1;	// Enable SGRAM timing=1(Enable)
	gSEQR.s.fSR33.ESMWtiming = 0;	// Enable SGRAM Mode Write timing=0(Disable)
	SeqOut(0x33, gSEQR.fValue[0x33]);

	gSEQR.s.fSR34.DCOCWenable = 1;		// DRAM controller one cycle write enable=1(Enable)
	gSEQR.s.fSR34.DCOCRenable = 0;		// DRAM controller one cycle read enable=0(Enable)
	gSEQR.s.fSR34.EDOPLPcomsumption = 0;		// Enable DRAM output PAD low power consumption=0(Disable)
	SeqOut(0x34, gSEQR.fValue[0x34]);

	gSEQR.s.fSR35.EPBWzerowait = 1;		// Enable PCI burst write zero-wait
	gSEQR.s.fSR35.SBTenable = 0;			// SGRAM burst timing enable=0(Enable)
	gSEQR.s.fSR35.MDcompensation = 0;		// MA delay compensation=0(0ns)
	gSEQR.s.fSR35.DCLPWcompensation = 0;	// DRAM CAS LOW period width compensation bit[1:0]=0(0ns)
	gSEQR.s.fSR35.EPBWCretry = 0;			// Write Retry Disable
	gSEQR.s.fSR35.EPBRCretry = 0;			// Read Retry Disable
	SeqOut(0x35, gSEQR.fValue[0x35]);
	
	gSEQR.s.fSR38.DLcompare = 1;	// Line Compare disable
	SeqOut(0x38, gSEQR.fValue[0x38]);

	// set CRTC Values
	gCRTC.s.fCRF.TClocation70 = 0x00;

	gCRTC.s.fCR14.Reserved = 0;
	gCRTC.s.fCR14.DWMenable = 1;		// Double-word mode enable
	gCRTC.s.fCR14.CB4 = 0;			// Count by 4 disable
	gCRTC.s.fCR14.ULlocation = 0;		// Underline LocationBit[4:0] = 0

	gCRTC.s.fCR17.Hreset = 1;		// Enable horizontal and vertical retrace outputs
	gCRTC.s.fCR17.WBAmode = 0;		// Set the memory address mode to word
	gCRTC.s.fCR17.Awrap = 1;			// Enable the full 256k of memory
	gCRTC.s.fCR17.Reserved = 0;
	gCRTC.s.fCR17.CB2 = 0;			// Count by two = 0(Byte refresh)
	gCRTC.s.fCR17.HRselect = 0;		// Horizontal retrace select=0(Normal)
	gCRTC.s.fCR17.RA1rMA14 = 1;		// RA1 replace MA14 Disabled
	gCRTC.s.fCR17.RA0rMA13 = 1;		// RA0 replace MA13 Disabled
	
	gCRTC.s.fCR18.Lcompare70 = 0xff;	// Line Compare[7:0]
	

	CrtOut(0x0f, gCRTC.fValue[0x0f]);

	CrtOut(0x14, gCRTC.fValue[0x14]);

	CrtOut(0x17, gCRTC.fValue[0x17]);
	CrtOut(0x18, gCRTC.fValue[0x18]);

	// GCTR
	gGCTR.s.fGR5.Reserved1 = 0;
	gGCTR.s.fGR5.C256mode = 1;		// 256-color Mode enable
	gGCTR.s.fGR5.SRmode = 0;			// Shift Register Mode=0(EGA compatible)
	gGCTR.s.fGR5.OEAMenable = 0;		// Odd/Even Addressing Mode enable
	gGCTR.s.fGR5.Rmode = 0;			// Read mode=0(Map select read)
	gGCTR.s.fGR5.Reserved2 = 0;
	gGCTR.s.fGR5.Wmode = 0;			// Write mode=0(Direct processor write)
	
	gGCTR.s.fGR6.Reserved = 0;
	gGCTR.s.fGR6.MAselect = 1;		// Memory Address Select=1(A0000 to AFFFF)
	gGCTR.s.fGR6.COEmaps = 0;		// Chain Odd And Even Maps disable
	gGCTR.s.fGR6.GMenable = 1;		// Graphics Mode Enable

	gGCTR.s.fGR7.Reserved = 0;
	gGCTR.s.fGR7.P3Dcare = 1;		// Plane 3 Don't Care=1(Enable color comparison)
	gGCTR.s.fGR7.P2Dcare = 1;		// Plane 2 Don't Care=1(Enable color comparison)
	gGCTR.s.fGR7.P1Dcare = 1;		// Plane 1 Don't Care=1(Enable color comparison)
	gGCTR.s.fGR7.P0Dcare = 1;		// Plane 0 Don't Care=1(Enable color comparison)
	
	gGCTR.s.fGR8.BMenable = 0xff;		// Bit Mask Enable Bit
	
	GctOut(0x05, gGCTR.fValue[0x05]);
	GctOut(0x06, gGCTR.fValue[0x06]);
	GctOut(0x07, gGCTR.fValue[0x07]);
	GctOut(0x08, gGCTR.fValue[0x08]);

	// ATRR
	gATRR.s.fAR10.P4P5Sselect = 0;	// P4, P5 Source Select=0(AR0-F Bit[5:4] are used as the source for the Lookup Table Address Bit[5:4])
	gATRR.s.fAR10.PDCselect = 0;		// The pixels are clocked at every clock cycle
	gATRR.s.fAR10.PPCwLcompare = 0;	// PEL Panning Compatibility with Line Compare Disable
	gATRR.s.fAR10.Reserved =0;
	gATRR.s.fAR10.BIBenable = 0;		// Background Intensity attribute enable
	gATRR.s.fAR10.LGenable = 0;		// The ninth bit of nine-bit-wide character cell will be the same as the background.
	gATRR.s.fAR10.Dtype = 0;			// The contents of the Attribute byte are treated as color attribute.
	gATRR.s.fAR10.GTmode = 1;		// The Attribute Controller will function in graphics mode.
	
	// ATRR
	AtrOut(0x10, gATRR.fValue[0x10]);
}

void
SiSReset()
{
	// Do reset at lease 400ns.
	outb(SEQ_IDX, 0);
	gSEQR.fValue[0x0] = inb(SEQ_DATA);

	gSEQR.s.fSR0.Areset = 0;
	gSEQR.s.fSR0.Sreset = 0;
	outb(SEQ_DATA, gSEQR.fValue[0x0]);

	snooze(1);	// 1us > 400ns

	outb(SEQ_IDX, 0);
	gSEQR.s.fSR0.Areset = 1;
	gSEQR.s.fSR0.Sreset = 1;
	outb(SEQ_DATA, gSEQR.fValue[0x0]);
}

void
readGeneric()
{
	gMiscOut = inb(0xc);
	gFeature = inb(0xa);
	gIStatus0 = inb(0x2);
	gIStatus1 = inb(0x1a);
	gVGAEnable = inb(0x3);
	gSegSel0 = inb(0xd);
	gSegSel1 = inb(0xb);
}

void
readCRTC()
{
	tIOIdxAccess	theCRTC;
	
	theCRTC.fOffset = CRT_IDX;
	for(theCRTC.fIndex = 0;theCRTC.fIndex <= 0x26; theCRTC.fIndex++) {
		(void)ioctl(fd, YSIS_READ_INDEXED, &theCRTC);
		gCRTC.fValue[theCRTC.fIndex] = theCRTC.fValue;
	}
}

void
readSEQR()
{
	tIOIdxAccess	theSEQR;
	
	theSEQR.fOffset = SEQ_IDX;
	for(theSEQR.fIndex = 0;theSEQR.fIndex <= 0x3c; theSEQR.fIndex++) {
		(void)ioctl(fd, YSIS_READ_INDEXED, &theSEQR);
		gSEQR.fValue[theSEQR.fIndex] = theSEQR.fValue;
	}
}

void
readGCTR()
{
	tIOIdxAccess	theGCTR;
	
	theGCTR.fOffset = GCT_IDX;
	for(theGCTR.fIndex = 0;theGCTR.fIndex <= 0x8; theGCTR.fIndex++) {
		(void)ioctl(fd, YSIS_READ_INDEXED, &theGCTR);
		gGCTR.fValue[theGCTR.fIndex] = theGCTR.fValue;
	}
}

void
readATRR()
{
        int     theLoop;
        uchar   theDmy;

        for(theLoop = 0; theLoop <= 0x14; theLoop++) {
                theDmy = inb(0x1a);
                snooze(10L);
                outb(ATR_IDX, (uint8)theLoop);
                snooze(10L);
                gATRR.fValue[theLoop] = inb(ATR_READ);
                snooze(10L);
        }
        theDmy = inb(0x1a);
        snooze(10L);
        outb(ATR_IDX, 0x20);
}

//
// Read out all VGA/SiS Registers
void
SiSReadOut()
{
	readGeneric();
	readCRTC();
	readSEQR();
	readGCTR();
	readATRR();
}

//
// 表示/停止を決める
void
SiSEnableDisplay(bool argEnable)
{
	tIOIdxAccess	theSEQR;

	theSEQR.fOffset = SEQ_IDX;
	theSEQR.fIndex = 0x1;
	(void)ioctl(fd, YSIS_READ_INDEXED, &theSEQR);
	gSEQR.fValue[0x1] = theSEQR.fValue;

	if (argEnable == true) {	// Display On?
		gSEQR.s.fSR1.Soff = 0;
	}else{				// Display Off?
		gSEQR.s.fSR1.Soff = 1;
	}
	outb(SEQ_DATA, gSEQR.fValue[0x1]);
}

//
// 指定されたメモリバンク構成でメモリ周りを初期化する。
void
SiSConfMemBank(int argMconf2, int argMconfigure, int argMconf3, int argSS816)
{
	// メモリバンク構成指定
	gSEQR.s.fSRC.Mconf2 = argMconf2;
	gSEQR.s.fSRC.Mconfiguration = argMconfigure;
	SeqOut(0x0c, gSEQR.fValue[0x0c]);

	gSEQR.s.fSR3D.Mconf3 = argMconf3;
	SeqOut(0x3d, gSEQR.fValue[0x3d]);

	gSEQR.s.fSR3E.SS816 = argSS816;
	SeqOut(0x3e, gSEQR.fValue[0x3e]);

	// MCLK設定(=100MHz)
	gSEQR.s.fSR13.MPscale = 0;
	SeqOut(0x13, gSEQR.fValue[0x13]);

	gSEQR.s.fSR28.Mnumerator = 55;
	gSEQR.s.fSR28.Mdivider = 0;
	SeqOut(0x28, gSEQR.fValue[0x28]);

	gSEQR.s.fSR29.Mdenumrator = 1;
	gSEQR.s.fSR29.MPscale = 3;
	gSEQR.s.fSR29.MVgain = 1;
	SeqOut(0x29, gSEQR.fValue[0x29]);
	
	// Enable SGRAM Mode Write timing=0(Disable)
	gSEQR.s.fSR33.ESMWtiming = 0;
	SeqOut(0x33, gSEQR.fValue[0x33]);

	snooze(10);	// ??? see timing chart

	gSEQR.s.fSR33.ESMWtiming = 1;
	SeqOut(0x33, gSEQR.fValue[0x33]);

	SiSReset();
}

//
// メモリ周りを初期化する。
bool
SiSMemInit()
{
	static int		theBanks[][2] = {
		{ 1, 3 },	// CARDEXpert 8MB
		{ 1, 1 },	// JOYIMAGE 4MB
		{ 1, 2 },
		{ 1, 0 },
		{ 0, 3 },
		{ 0, 2 },
		{ 0, 1 },
		{ 0, 0 }
	};
	int	theLoop, theLoop1, theLoop2;
	unsigned long	theMemSize;
	unsigned long	theBestMemSize;
	int	theBestBank[4];
	
	// メモリの初期化を行う。
	// バンク構成をハードウェアから知ることはできないので、
	// バンク構成のうち、ありえそうな順序から試してみる。
	theBestMemSize = 0;	// 仮に見つからなかったものとする。
	for(theLoop = 0; theLoop < 8; theLoop++) {
		SERIAL_PRINT(("ySiS accel : theLoop = %d\n", theLoop));
		for(theLoop1 = 0; theLoop1 <= 1; theLoop1++) {
			for(theLoop2 = 0; theLoop2 <=1; theLoop2++) {
				SiSConfMemBank(theBanks[theLoop][0], theBanks[theLoop][1], theLoop1 , theLoop2);
				theMemSize = Get_Card_Mem_Size();
				if (theMemSize > 0) {	// メモリテストOk?
					if (theBestMemSize < theMemSize) {	// もっとも望ましい構成?
						theBestBank[0] = theBanks[theLoop][0];
						theBestBank[1] = theBanks[theLoop][1];
						theBestBank[2] = theLoop1;
						theBestBank[3] = theLoop2;
						theBestMemSize = theMemSize;
						SERIAL_PRINT(("ySiS accel : Get_Card_Mem_Size() = %d\n", theMemSize));
					}
				}
			}
		}
	}
	if (theBestMemSize == 0) {	// 結局使えそうな構成は見つからなかった?
		return false;
	}
	SiSConfMemBank(theBestBank[0], theBestBank[1], theBestBank[2], theBestBank[3]);
	SERIAL_PRINT(("ySiS accel : finally , Mconf2 = %d, Mconf = %d, SS816 = %d, Mconf3 %d Mem Size = %dKbytes\n" 	, theBestBank[0] , theBestBank[1] , theBestBank[2] , theBestBank[3] , theBestMemSize / 1024));
	snooze(1000);		// メモリが落ち着くのを待つ
	si->mem_size = theBestMemSize;
	
	return true;
}

// SiS6326の初期化を行う。チップのリセットと、レジスタの
// アンロックは既にカーネルドライバによって行われている
// ため、行う必要はない。
bool
SiS6326Init()
{
	SERIAL_PRINT(("ySiS accel : enter SiS6326Init()\n"));
	// 現時点のレジスタ値を全て取り出す
	SiSReadOut();
	SiSPutRes();
	
	// 表示を停止する。
	SiSEnableDisplay(false);

	// メモリ周りの初期化
	if (SiSMemInit() == false) {
		SERIAL_PRINT(("ySiS accel : leave SiS6326Init() return false\n"));
		return false;
	}
	SiSReset();

	// SiS6326のH/W Cursorのサイズは64x64。
	// もっとも表示バッファを使用するモード(1600x1200,16bpp)でもメモリの使用量は3750KB。
	// そのため、H/WCursorのイメージバッファには4MB直前の領域(0x003fc000)を割り当てる。
	si->cursor.data = (uint8 *)si->framebuffer + 0x3fc000;
	gSEQR.s.fSR38.HClocation = 0xf;
	SeqOut(0x38, gSEQR.fValue[0x38]);

	SERIAL_PRINT(("ySiS accel : leave SiS6326Init() return true\n"));
	return true;
}
