//
// SiSkernutil.c
//
//	A part of SiS6326 driver for BeOS
//	Copyright (C) 2000 Yasutaka Uematu All rights reserved.
//	yasutaka2000@bemail.org
// 

#include <KernelExport.h>
#include <PCI.h>
#include <OS.h>
#include <malloc.h>
#include	<SupportDefs.h>
#include	"DriverInterface.h"
#include	"SiSkernutil.h"
#include	"SiSDriverIF.h"
#include	"uCRTC.h"
#include	"uSEQR.h"

#if DEBUG > 0
#define ddprintf(a)	dprintf a
#else
#define	ddprintf(a)
#endif

// defined in driver.c
extern uint16	gIOBase;
extern pci_module_info	*pci_bus;
extern void	delay(bigtime_t);


// y_SiS : I/O空間へのアクセス

void
SiSReset()
{
	union {
		SR0		s;
		uint8	fValue;
	} theSR0;	

	// Do reset at lease 400ns.
	outb(SEQ_IDX, 0);
	theSR0.fValue = inb(SEQ_DATA);

	theSR0.s.Areset = 0;
	theSR0.s.Sreset = 0;
	outb(SEQ_DATA, theSR0.fValue);

	delay(1);	// 1us > 400ns

	outb(SEQ_IDX, 0);
	theSR0.s.Areset = 1;
	theSR0.s.Sreset = 1;
	outb(SEQ_DATA, theSR0.fValue);
}

//
// 全てのレジスタのロックをはずす
//
bool
SiSUnlockRegs()
{
	union {
		CR11	s;
		uint8	fValue;
	} theCR11;
	uint8	theProtect;

	dprintf("ySiS : enter SiSUnlockRegs()\n");
	dprintf("ySiS : gIOBase = %04x\n", gIOBase);
	
	// Unlock all SRx registers
	outb(SEQ_IDX, 5);
	outb(SEQ_DATA, 0x86);
	theProtect = inb(SEQ_DATA);
	dprintf("ySiS : theProtect = %02x\n", theProtect);
	if (theProtect != (uint8)0xa1) {
		return false;
	}

	// Unlock CRTC
	outb(CRT_IDX, (uint8)0x11);
	theCR11.fValue = inb(CRT_DATA);

	theCR11.s.Wprotect = 0;
	outb(CRT_DATA, theCR11.fValue);

	return true;
}

//
// 0x3b0~0x3dfのポートをアクセス不可とする。
//
void
SiSKillPort()
{
	union {
		SR33	s;
		uint8	fValue;
	} theSR33;

	outb(SEQ_IDX, (uint8)0x33);
	theSR33.fValue = inb(SEQ_DATA);

	theSR33.s.SVGAIODdisable = 1;
	theSR33.s.RVGAIODenable = 0;

	outb(SEQ_IDX, (uint8)0x33);
	outb(SEQ_DATA, theSR33.fValue);
}

//
// 垂直帰線割り込みが発生していて、処理済みになっていない？
bool
SiSIsVintPending()
{
	uint8	theIStatus0;

	theIStatus0 = inb(ISTATUS0);
	if (theIStatus0 & 0x80) {
		return true;	// Interrupt pending
	}else{
		return false;	// Interrupt clear
	}
}

//
// 垂直帰線割り込みクリア
void
SiSClearVint(int argVal)
{
	union {
		CR11	s;
		uint8	fValue;
	} theCR11;

	outb(CRT_IDX, 0x11);
	theCR11.fValue = inb(CRT_DATA);

	theCR11.s.VIclear = argVal;

	outb(CRT_IDX, (uint8)0x11);
	outb(CRT_DATA, theCR11.fValue);
}

//
// 垂直ブランク中?
bool
SiSIsVblank(shared_info *argSi)
{
	union {
		CR1C	s;
		uint8	fValue;
	} theCR1C;
	union {
		CR1D s;
		uint8	fValue;
	} theCR1D;

	{
		static int theOnce = false;
		
		if (theOnce == false) {
			ddprintf(("ySiS : si->dm.virtual_height = %d\n"
						, argSi->dm.virtual_height));
			theOnce = true;
		}
	}
	
	outb(CRT_IDX, (uint8)0x20);
	theCR1C.fValue = inb(CRT_DATA);	// Latch
	
	outb(CRT_IDX, (uint8)0x1c);
	theCR1C.fValue = inb(CRT_DATA);
	outb(CRT_IDX, (uint8)0x1d);
	theCR1D.fValue = inb(CRT_DATA);

	if ((theCR1D.s.CRTVcounter108 << 8) + theCR1C.s.CRTVcounter70
		 > argSi->dm.virtual_height) {	// This means in Virtual Desktop mode, somewhat may corrupt.
		return true;
	}else{
		return false;
	}	
}

//
// 垂直帰線割り込みを許可・禁止する
bool
SiSEnableVint(bool argEnable)
{
	union {
		CR11	s;
		uint8	fValue;
	} theCR11;
	bool	theLastStat;

	outb(CRT_IDX, (uint8)0x11);
	theCR11.fValue = inb(CRT_DATA);

	if (theCR11.s.VIenable == 0) {		// 現在は許可？
		theLastStat = true;
	}else{
		theLastStat = false;
	}

	if (argEnable == true) {			// 許可？
		theCR11.s.VIenable = 0;		// 許可
	}else{
		theCR11.s.VIenable = 1;		// 禁止
	}

	outb(CRT_IDX, (uint8)0x11);
	outb(CRT_DATA, theCR11.fValue);

	return theLastStat;
}

//
//
status_t
SiSReadPort(tIOAccess *argIO)
{
	// I/Oポートオフセットチェック
	if (argIO->fOffset > 0x1f) {
		return B_DEV_INVALID_IOCTL;
	}

	argIO->fValue = inb(gIOBase + argIO->fOffset);
	return B_OK;
}

//
//
status_t
SiSWritePort(tIOAccess *argIO)
{
	// I/Oポートオフセットチェック
	if (argIO->fOffset > 0x1f) {
		return B_DEV_INVALID_IOCTL;
	}

	outb(gIOBase + argIO->fOffset, argIO->fValue);
	return B_OK;
}

//
//
status_t
SiSReadIndexed(tIOIdxAccess *argIO)
{
	int	thePort;
	
	// I/Oポートオフセットチェック
	if (argIO->fOffset > 0x1f) {
		return B_DEV_INVALID_IOCTL;
	}

	thePort = argIO->fOffset;
	outb(gIOBase + thePort, argIO->fIndex);
	argIO->fValue = inb(gIOBase + thePort + 1);
	return B_OK;
}

//
//
status_t
SiSWriteIndexed(tIOIdxAccess *argIO)
{
	int	thePort;

	// I/Oポートオフセットチェック
	if (argIO->fOffset > 0x1f) {
		return B_DEV_INVALID_IOCTL;
	}

	thePort = argIO->fOffset;
	outb(gIOBase + thePort, argIO->fIndex);
	outb(gIOBase + thePort + 1, argIO->fValue);
	return B_OK;
}


//
//
status_t
SiSSetGamma(tIOGammaTbl *argGamma)
{
	union {
		SR7		s;
		uint8	fValue;
	} theSR7;
	
	ddprintf(("ySIS : enter SiSSetGamma()\n"));	// y_SiS
	ddprintf(("ySIS : argGamma->fEnable = %d\n", argGamma->fEnable));	// y_SiS
	
	if (argGamma->fToken != kGAMMA_SET_TOKEN) {
		ddprintf(("ySIS : failed SiSSetGamma()\n"));	// y_SiS
		return B_DEV_INVALID_IOCTL;
	}
	
	outb(SEQ_IDX, 7);
	theSR7.fValue = inb(SEQ_DATA);
	if (argGamma->fEnable == true) {
		theSR7.s.CP24fDCmode = 1;
		outb(SEQ_DATA, theSR7.fValue);

		(void)SiSSetDAC(argGamma);		
	}else{
		theSR7.s.CP24fDCmode = 0;
		outb(SEQ_DATA, theSR7.fValue);
	}
	
	ddprintf(("ySIS : leave SiSSetGamma()\n"));	// y_SiS

	return B_OK;
}
status_t
SiSSetDAC(tIOGammaTbl *argDAC)
{
	int	theLoop;
	cpu_status	theStat = disable_interrupts();
		
	outb(DAC_WRITE_IDX, 0);
	for(theLoop = 0;theLoop < 768; theLoop+=3) {
		delay(1);
		outb(DAC_DATA, argDAC->fDACValue[theLoop]);
		delay(1);
		outb(DAC_DATA, argDAC->fDACValue[theLoop + 1]);
		delay(1);
		outb(DAC_DATA, argDAC->fDACValue[theLoop + 2]);
	}
	restore_interrupts(theStat);
	
	return B_OK;
}
//
// 表示/停止を決める
void
SiSEnableDisplay(bool argEnable)
{
	union {
		SR1		s;
		uint8	fValue;
	} theSR1;	

	outb(gIOBase + SEQ_IDX, 0x1);
	theSR1.fValue = inb(gIOBase + SEQ_DATA);

	if (argEnable == true) {	// Display On?
		theSR1.s.Soff = 0;
	}else{				// Display Off?
		theSR1.s.Soff = 1;
	}
	outb(gIOBase + SEQ_DATA, theSR1.fValue);
}

// y_SiS : SR6～SR3Eを0クリアする
void
SiSSEQClear()
{
	uint8	theSRIDX;
	
	for(theSRIDX = 6; theSRIDX <= 0x7; theSRIDX++) {
		outb(SEQ_IDX, theSRIDX);
		outb(SEQ_DATA, (uint8)0);
	}
	for(theSRIDX = 0x0a; theSRIDX <= 0xb; theSRIDX++) {
		outb(SEQ_IDX, theSRIDX);
		outb(SEQ_DATA, (uint8)0);
	}
	for(theSRIDX = 0x0f; theSRIDX <= 0x10; theSRIDX++) {
		outb(SEQ_IDX, theSRIDX);
		outb(SEQ_DATA, (uint8)0);
	}
	for(theSRIDX = 0x12; theSRIDX <= 0x12; theSRIDX++) {
		outb(SEQ_IDX, theSRIDX);
		outb(SEQ_DATA, (uint8)0);
	}
	for(theSRIDX = 0x14; theSRIDX <= 0x1f; theSRIDX++) {
		outb(SEQ_IDX, theSRIDX);
		outb(SEQ_DATA, (uint8)0);
	}
	for(theSRIDX = 0x24; theSRIDX <= 0x25; theSRIDX++) {
		outb(SEQ_IDX, theSRIDX);
		outb(SEQ_DATA, (uint8)0);
	}
	for(theSRIDX = 0x2e; theSRIDX <= 0x32; theSRIDX++) {
		outb(SEQ_IDX, theSRIDX);
		outb(SEQ_DATA, (uint8)0);
	}
	for(theSRIDX = 0x36; theSRIDX <= 0x3e; theSRIDX++) {
		outb(SEQ_IDX, theSRIDX);
		outb(SEQ_DATA, (uint8)0);
	}
}

// y_SiS : CR0～CR10, CR12～CR18を0クリアする
// CR11は割り込み禁止ビットを立てる。
void
SiSCRTCClear()
{
	uint8	theIDX;
	
	for(theIDX = 0; theIDX <= 0x10; theIDX++) {
		outb(CRT_IDX, theIDX);
		outb(CRT_DATA, (uint8)0);
	}
	
	outb(CRT_IDX, 0x11);
	outb(CRT_DATA, (uint8)0x20);
	
	for(theIDX = 0x12; theIDX <= 0x18; theIDX++) {
		outb(CRT_IDX, theIDX);
		outb(CRT_DATA, (uint8)0);
	}
}

// y_SiS : GR0～GR8を0クリアする
void
SiSGCTRClear()
{
	uint8	theIDX;
	
	for(theIDX = 0; theIDX <= 0x8; theIDX++) {
		outb(GCT_IDX, theIDX);
		outb(GCT_DATA, (uint8)0);
	}
}

void
SiSATRRClear()
{
	uint8	theIdx, theDmy;

	for(theIdx = 0; theIdx <= 0x14; theIdx++) {
		theDmy = inb(ISTATUS1);
		outb(ATR_IDX, theIdx);
		outb(ATR_WRITE, (uint8)0);
	}
}
