/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.

	SiS6326 graphics card kernel driver
	written by Yasutaka Uematu (yasutaka2000@bemail.org)

	レジスタの操作について：
	  SiS6326の多くの重要なレジスタはI/O空間にしか存在しない。アクセレラント
	からはI/O空間はアクセスできないため、アクセレラントからは、ioctl()で
	I/O空間にアクセスする。
	  ただし、2D/3Dアクセラレーション機能については、メモリ空間上にレジスタが
	存在するため、アクセレラント自身で処理を行う。
	BeBoxではI/O空間はメモリマップされているので、実は気にすることはないのだが。
	
*/


/* standard kernel driver stuff */
#include <KernelExport.h>
#include <PCI.h>
#include <OS.h>
#include <malloc.h>

/* this is for the standardized portion of the driver API */
/* currently only one operation is defined: B_GET_ACCELERANT_SIGNATURE */
#include <graphic_driver.h>

/* this is for sprintf() */
#include <stdio.h>

/* this is for string compares */
#include <string.h>

/* The private interface between the accelerant and the kernel driver. */
#include "../include/DriverInterface.h"
#include "../include/SiSDriverIF.h"	// y_SiS
#include	"SiSkernutil.h"

#if DEBUG > 0
#define ddprintf(a)	dprintf a
#else
#define	ddprintf(a)
#endif

#define get_pci(o, s) (*pci_bus->read_pci_config)(pcii->bus, pcii->device, pcii->function, (o), (s))
#define set_pci(o, s, v) (*pci_bus->write_pci_config)(pcii->bus, pcii->device, pcii->function, (o), (s), (v))

#define MAX_DEVICES	8

/* Tell the kernel what revision of the driver API we support */
int32	api_version = 2;

/* these structures are private to the kernel driver */
typedef struct device_info device_info;

#if defined(POST_R4_0)
typedef struct {
	timer		te;				/* timer entry for add_timer() */
	device_info	*di;			/* pointer to the owning device */
	bigtime_t	when_target;	/* when we're supposed to wake up */
} timer_info;
#endif

struct device_info {
	uint32		is_open;			/* a count of how many times the devices has been opened */
	area_id		shared_area;		/* the area shared between the driver and all of the accelerants */
	shared_info	*si;				/* a pointer to the shared area, for convenience */
	vuint32		*regs;				/* kernel's pointer to memory mapped registers */
									// y_SiS : これは2Dレジスタを指す。
	int32		can_interrupt;		/* when we're faking interrupts, let's us know if we should generate one */
#if defined(POST_R4_0)
	timer_info	ti_a;				/* a pool of two timer managment buffers */
	timer_info	ti_b;
	timer_info	*current_timer;		/* the timer buffer that's currently in use */
#else
	thread_id	tid;
#endif
#if DEBUG > 0
	uint32		interrupt_count;	/* if we're debugging, a count of how many times the interrupt handler has
										been called for this device */
#endif
	pci_info	pcii;					/* a convenience copy of the pci info for this device */
	char		name[B_OS_NAME_LENGTH];	/* where we keep the name of the device for publishing and comparing */
};

typedef struct {
#if DEBUG > 0
	uint32		total_interrupts;	/* total number of interrupts seen by our handler */
#endif
	uint32		count;				/* number of devices actually found */
	benaphore	kernel;				/* for serializing opens/closes */
	char		*device_names[MAX_DEVICES+1];	/* device name pointer storage */
	device_info	di[MAX_DEVICES];	/* device specific stuff */
} DeviceData;

// splinlock for critical section
spinlock	gSpinlock = 0;

/* prototypes for our private functions */
static status_t open_hook (const char* name, uint32 flags, void** cookie);
static status_t close_hook (void* dev);
static status_t free_hook (void* dev);
static status_t read_hook (void* dev, off_t pos, void* buf, size_t* len);
static status_t write_hook (void* dev, off_t pos, const void* buf, size_t* len);
static status_t control_hook (void* dev, uint32 msg, void *buf, size_t len);
static status_t map_device(device_info *di);
static void unmap_device(device_info *di);
static void probe_devices(void);
static int32 ySiS_interrupt(void *data);

#if DEBUG > 0
static int ySiSdump(int argc, char **argv);
#endif

static DeviceData		*pd;
pci_module_info	*pci_bus;				// export for SiSkernutil.c
static device_hooks graphics_device_hooks = {
	open_hook,
	close_hook,
	free_hook,
	control_hook,
	read_hook,
	write_hook,
	NULL,
	NULL,
	NULL,
	NULL
};

uint16	gIOBase;	// y_SiS : I/O空間上のレジスタのベースアドレス。
#define VENDOR_ID			0x1039	/* y_SiS : vendor = SiS */

static uint16 ySiS_device_list[] = {	// y_SiS : device = SiS6326
	0x6326,	/* SiS6326 */
	0
};

static struct {
	uint16	vendor;
	uint16	*devices;
} SupportedDevices[] = {
	{VENDOR_ID, ySiS_device_list},
	{0x0000, NULL}
};

// public functions
void	delay(bigtime_t);

/*
 * Beta2:
 * このチップの垂直帰線割り込みを有効にした状態で
 * BDirectDrawを使ったアプリケーションを動かすと、
 * チップがハングアップする問題がある。
 * そのため、デフォルトでは、このチップの垂直帰線
 * 割り込みは使用せず、疑似割り込みを使用することとする。
 */
#ifndef	FORCE_HW_VINT_ENABLE
bool	gHWVint = false;
#else
bool gHWVint = true;
#endif

/*
	init_hardware() - Returns B_OK if one is
	found, otherwise returns B_ERROR so the driver will be unloaded.
*/
status_t
init_hardware(void) {
	long		pci_index = 0;
	pci_info	pcii;
	bool		found_one = FALSE;
	
	ddprintf(("ySIS : enter init_hardware()\n"));	// y_SiS

	/* choke if we can't find the PCI bus */
	if (get_module(B_PCI_MODULE_NAME, (module_info **)&pci_bus) != B_OK)
		return B_ERROR;

	/* while there are more pci devices */
	while ((*pci_bus->get_nth_pci_info)(pci_index, &pcii) == B_NO_ERROR) {
		int vendor = 0;
		
		ddprintf(("ySiS : init_hardware(): checking pci index %ld, device 0x%04x/0x%04x\n", pci_index, pcii.vendor_id, pcii.device_id));
		/* if we match a supported vendor */
		while (SupportedDevices[vendor].vendor) {
			if (SupportedDevices[vendor].vendor == pcii.vendor_id) {
				uint16 *devices = SupportedDevices[vendor].devices;
				/* while there are more supported devices */
				while (*devices) {
					/* if we match a supported device */
					if (*devices == pcii.device_id ) {
						ddprintf(("ySiS : we support this device\n"));
						found_one = TRUE;
						goto done;
					}
					/* next supported device */
					devices++;
				}
			}
			vendor++;
		}
		/* next pci_info struct, please */
		pci_index++;
	}
	ddprintf(("ySiS : init_hardware - no supported devices\n"));

done:
	/* put away the module manager */
	put_module(B_PCI_MODULE_NAME);

	ddprintf(("ySIS : leave init_hardware()\n"));	// y_SiS

	return (found_one ? B_OK : B_ERROR);
}

status_t
init_driver(void) {
#if !(defined(POST_R4_0))
	int	theLoop;					// ySiS
#endif
	ddprintf(("ySiS : enter init_driver()\n"));	// y_SiS

	/* get a handle for the pci bus */
	if (get_module(B_PCI_MODULE_NAME, (module_info **)&pci_bus) != B_OK)
		return B_ERROR;

	/* driver private data */
	pd = (DeviceData *)calloc(1, sizeof(DeviceData));
	if (!pd) {
		put_module(B_PCI_MODULE_NAME);
		return B_ERROR;
	}
	/* initialize the benaphore */
	INIT_BEN(pd->kernel);
#if !(defined(POST_R4_0))
	// ySiS : 割り込み模擬スレッドIDを初期化する。
	for(theLoop = 0; theLoop < MAX_DEVICES; theLoop++) {
		pd->di[theLoop].tid = -1;
	}	
#endif
	/* find all of our supported devices */
	probe_devices();
	
#if DEBUG > 0
	add_debugger_command("ySiSdump", ySiSdump, "dump ySiS kernel driver persistant data");	// ySiS
#endif
	ddprintf(("ySIS : leave init_driver()\n"));	// y_SiS

	return B_OK;
}

const char **
publish_devices(void) {
	/* return the list of supported devices */
	return (const char **)pd->device_names;
}

device_hooks *
find_device(const char *name) {
	int index = 0;

	while (pd->device_names[index]) {
		if (strcmp(name, pd->device_names[index]) == 0)
			return &graphics_device_hooks;
		index++;
	}

	return NULL;

}

void uninit_driver(void) {
#if !(defined(POST_R4_0))
	int	theLoop;
	status_t	theDummy;
#endif
	
	ddprintf(("ySIS : enter uninit_driver()\n"));	// y_SiS

#if DEBUG > 0
	remove_debugger_command("ySiSdump", ySiSdump);	// y_SiS
#endif
#if !(defined(POST_R4_0))
	for(theLoop = 0; theLoop < MAX_DEVICES;theLoop++) {
		// Did I faked for this device?
		if ((pd->di[theLoop].pcii.u.h0.interrupt_pin == 0x00)
		 || (pd->di[theLoop].pcii.u.h0.interrupt_line == 0xff)
		 || gHWVint == false ){
			// Terminate fake thread.
			if ((pd->di[theLoop].tid != -1) && (pd->di[theLoop].si != NULL)) {
				ddprintf(("ySiS : tell fake thread to quit\n"));
				(void)atomic_and(&pd->di[theLoop].si->flags, ~YSIS_HANDLER_INSTALLED);
				ddprintf(("ySiS : waiting for thread body.\n"));
				wait_for_thread(pd->di[theLoop].tid, &theDummy);
			}
		}
	}
#endif
	/* free the driver data */
	DELETE_BEN(pd->kernel);
	free(pd);
	pd = NULL;

	/* put the pci module away */
	put_module(B_PCI_MODULE_NAME);
	ddprintf(("ySIS : leave uninit_driver()\n"));	// y_SiS

}

static status_t map_device(device_info *di) {
	/* default: frame buffer in [0], control regs in [1] */
	// y_SiS : 構成は上記のとおり。ただし[1]に入るのはcontrol regs
	// ではなく、2D/3Dアクセラレーション用レジスタの位置。
	int regs = 1;
	int fb   = 0;
	char buffer[B_OS_NAME_LENGTH];
	shared_info *si = di->si;
	uint32	tmpUlong;
	pci_info *pcii = &(di->pcii);
	uint32	theIO;
	
	ddprintf(("ySIS : enter map_device()\n"));	// y_SiS

	/* enable memory mapped IO, disable VGA I/O */
	tmpUlong = get_pci(PCI_command, (unsigned)4);

	// y_SiS : I/O空間へのアクセスは必要。
	tmpUlong |= 0x00000003;

	set_pci(PCI_command, 4, tmpUlong);

	// y_SiS : I/O空間のベースアドレスにオフセットを加算してI/Oアクセス用ベースアドレスを
	// 作る。
	theIO = get_pci(PCI_base_registers + 0x08, (unsigned)4);
	ddprintf(("y_SiS : theIO = %08x\n", theIO));
	
	gIOBase = (uint16)((get_pci(PCI_base_registers + 0x08, (unsigned)4) + 0x40) & 0x0000fffc);
	
	// y_SiS : 表示を隠す
	SiSEnableDisplay(false);
	
	// y_SiS : リセットする。
	SiSReset();
	// y_SiS : レジスタのロックを全てはずす
	if (SiSUnlockRegs() == false) {
		return B_ERROR;
	}
	// y_SiS : 0x3b0~0x3dfのポートアクセスを不可とする。
	SiSKillPort();

	SiSSEQClear();
	SiSCRTCClear();
	SiSGCTRClear();
	SiSATRRClear();

	// DACのPELマスクレジスタを0xff(全て有効)にする
	outb(DAC_PEL_MASK, 0xff);
	
	// y_SiS : romを読めるようにして何になる？と思うが
	// とりあえず生かしておく。

	/* enable ROM decoding */
	tmpUlong = get_pci(PCI_rom_base, 4);
	tmpUlong |= 0x00000001;
	set_pci(PCI_rom_base, 4, tmpUlong);

	/* map the areas */
	sprintf(buffer, "%04X_%04X_%02X%02X%02X regs",
		di->pcii.vendor_id, di->pcii.device_id,
		di->pcii.bus, di->pcii.device, di->pcii.function);
	si->regs_area = map_physical_memory(
		buffer,
		(void *) di->pcii.u.h0.base_registers[regs],
		di->pcii.u.h0.base_register_sizes[regs],
		B_ANY_KERNEL_ADDRESS,
		0, /* B_READ_AREA + B_WRITE_AREA, */ /* neither read nor write, to hide it from user space apps */
		(void **)&(di->regs));
	/* return the error if there was some problem */
	if (si->regs_area < 0) return si->regs_area;

	sprintf(buffer, "%04X_%04X_%02X%02X%02X rom",
		di->pcii.vendor_id, di->pcii.device_id,
		di->pcii.bus, di->pcii.device, di->pcii.function);
	si->rom_area = map_physical_memory(
		buffer,
		(void *)di->pcii.u.h0.rom_base,
		di->pcii.u.h0.rom_size,
		B_ANY_KERNEL_ADDRESS,
		B_READ_AREA,
		(void **)&(si->rom));
	/* return the error if there was some problem */
	if (si->rom_area < 0) {
		delete_area(si->regs_area);
		si->regs_area = -1;
		return si->rom_area;
	}

	sprintf(buffer, "%04X_%04X_%02X%02X%02X framebuffer",
		di->pcii.vendor_id, di->pcii.device_id,
		di->pcii.bus, di->pcii.device, di->pcii.function);
	si->fb_area = map_physical_memory(
		buffer,
		(void *) di->pcii.u.h0.base_registers[fb],
		di->pcii.u.h0.base_register_sizes[fb],
#if defined(__INTEL__)
#if defined(POST_R4_0)
		B_ANY_KERNEL_BLOCK_ADDRESS | B_MTR_WC,
#else
		B_ANY_KERNEL_ADDRESS | B_MTR_WC,
#endif
#else
		B_ANY_KERNEL_BLOCK_ADDRESS,
#endif
		B_READ_AREA + B_WRITE_AREA,
		&(si->framebuffer));

#if defined(__INTEL__)
	if (si->fb_area < 0) {
		/* try to map this time without write combining */
		/*
		After R4.0 (Intel), map_physical_memory() will try B_ANY_KERNEL_ADDRESS if
		a call with B_ANY_KERNEL_BLOCK_ADDRESS would fail.  It always worked this way
		under PPC.
		*/
		si->fb_area = map_physical_memory(
			buffer,
			(void *) di->pcii.u.h0.base_registers[fb],
			di->pcii.u.h0.base_register_sizes[fb],
#if defined(POST_R4_0)
			B_ANY_KERNEL_BLOCK_ADDRESS,
#else
			B_ANY_KERNEL_ADDRESS,
#endif
			B_READ_AREA + B_WRITE_AREA,
			&(si->framebuffer));
	}
#endif
		
	/* if there was an error, delete our other areas */
	if (si->fb_area < 0) {
		delete_area(si->regs_area);
		si->regs_area = -1;
		delete_area(si->rom_area);
		si->rom_area = -1;
	}
	/* remember the DMA address of the frame buffer for BDirectWindow purposes */
	si->framebuffer_pci = (void *) di->pcii.u.h0.base_registers_pci[fb];
	// アドレス関係は、どのみちBus masteringはBeBoxでは有効ではないので、ここまでとする。
	// ここをセットして置けば、ターゲットデバイスとして動作できるはず。
	
	// shared infoのほかの要素をセットする。
	// y_SiS : pixel clockの最大値を設定する。
	si->pix_clk_max16 = si->pix_clk_max8 = 175000000;
	
	ddprintf(("ySIS : leave map_device()\n"));	// y_SiS

	/* in any case, return the result */
	return si->fb_area;
}

static void unmap_device(device_info *di) {
	shared_info *si = di->si;
	uint32	tmpUlong;
	pci_info *pcii = &(di->pcii);

	ddprintf(("unmap_device(%08lx) begins...\n", (uint32)di));
	ddprintf(("  regs_area: %ld\n  fb_area: %ld\n", si->regs_area, si->fb_area));
	
	/* disable memory mapped IO & I/O space*/
	tmpUlong = get_pci(PCI_command, 4);
	tmpUlong &= 0xfffffffc;
	set_pci(PCI_command, 4, tmpUlong);
	/* disable ROM decoding */
	tmpUlong = get_pci(PCI_rom_base, 4);
	tmpUlong &= 0xfffffffe;
	set_pci(PCI_rom_base, 4, tmpUlong);
	/* delete the areas */
	if (si->rom_area >= 0) delete_area(si->rom_area);
	if (si->regs_area >= 0) delete_area(si->regs_area);
	if (si->fb_area >= 0) delete_area(si->fb_area);
	si->rom_area = si->regs_area = si->fb_area = -1;
	si->framebuffer = NULL;
	di->regs = NULL;
	si->rom = NULL;
	ddprintf(("unmap_device() ends.\n"));
}

static void probe_devices(void) {
//	uint32 pci_index = 0;
	int32 pci_index = 0;
	uint32 count = 0;
	device_info *di = pd->di;

	ddprintf(("ySIS : enter probe_devices()\n"));	// y_SiS

	/* while there are more pci devices */
	while ((count < MAX_DEVICES) && ((*pci_bus->get_nth_pci_info)(pci_index, &(di->pcii)) == B_NO_ERROR)) {
		int vendor = 0;
		
		ddprintf(("ySiS : checking pci index %ld, device 0x%04x/0x%04x\n", pci_index, di->pcii.vendor_id, di->pcii.device_id));
		/* if we match a supported vendor */
		while (SupportedDevices[vendor].vendor) {
			if (SupportedDevices[vendor].vendor == di->pcii.vendor_id) {
				uint16 *devices = SupportedDevices[vendor].devices;
				/* while there are more supported devices */
				while (*devices) {
					/* if we match a supported device */
					if (*devices == di->pcii.device_id ) {
						/* publish the device name */
						sprintf(di->name, "graphics/%04X_%04X_%02X%02X%02X",
							di->pcii.vendor_id, di->pcii.device_id,
							di->pcii.bus, di->pcii.device, di->pcii.function);
						ddprintf(("ySiS : making /dev/%s\n", di->name));
						/* remember the name */
						pd->device_names[count] = di->name;
						/* mark the driver as available for R/W open */
						di->is_open = 0;
						/* mark areas as not yet created */
						di->shared_area = -1;
						/* mark pointer to shared data as invalid */
						di->si = NULL;
						/* inc pointer to device info */
						di++;
						/* inc count */
						count++;
						/* break out of these while loops */
						goto next_device;
					}
					/* next supported device */
					devices++;
				}
			}
			vendor++;
		}
next_device:
		/* next pci_info struct, please */
		pci_index++;
	}
	/* propagate count */
	pd->count = count;
	/* terminate list of device names with a null pointer */
	pd->device_names[pd->count] = NULL;
	ddprintf(("ySiS : probe_devices: %ld supported devices\n", pd->count));
	ddprintf(("ySIS : leave probe_devices()\n"));	// y_SiS

}

static int32 thread_interrupt_work(shared_info *si) {
	int32 handled = B_HANDLED_INTERRUPT;

	/* release the vblank semaphore */
	if (si->vblank >= 0) {
		int32 blocked;
		if ((get_sem_count(si->vblank, &blocked) == B_OK) && (blocked < 0)) {
			release_sem_etc(si->vblank, -blocked, B_DO_NOT_RESCHEDULE);
			handled = B_INVOKE_SCHEDULER;
		}
	}

	return handled;
}

static int32
ySiS_interrupt(void *data)
{
	int32 handled = B_UNHANDLED_INTERRUPT;
	device_info *di = (device_info *)data;
	shared_info *si = di->si;
	int32 *flags = &(si->flags);

#if DEBUG > 0
	pd->total_interrupts++;
#endif
	
	/* is someone already handling an interrupt for this device? */
	if (atomic_or(flags, YSIS_HANDLER_INSTALLED) & YSIS_HANDLER_INSTALLED) {
#if DEBUG > 0
		kprintf("ySiS : Already in handler!\n");
#endif
		goto exit0;
	}


	/* did this card cause an interrupt */
	if (SiSIsVintPending() == true)	{		// y_SiS :
		/* do our stuff */
		handled = thread_interrupt_work(si);
#if DEBUG > 0
		/* increment the counter for this device */
		di->interrupt_count++;
#endif
		/* clear the interrupt status */
		acquire_spinlock(&gSpinlock);
		SiSClearVint(0);
		SiSClearVint(1);
		release_spinlock(&gSpinlock);
	}

	/* note that we're not in the handler any more */
	atomic_and(flags, ~YSIS_HANDLER_INSTALLED);


exit0:
	return handled;				
}

#if defined(POST_R4_0)
static int32 timer_interrupt_func(timer *te, uint32 pc) {
	bigtime_t now = system_time();
	/* get the pointer to the device we're handling this time */
	device_info *di = ((timer_info *)te)->di;
	shared_info *si = di->si;
	int32 *flags = &(si->flags);
	uint32 vbl_status = 0 /* read vertical blank status */;
	int32 result = B_HANDLED_INTERRUPT;

	/* are we suppoesed to handle interrupts still? */
	if (atomic_and(flags, -1) & YSIS_HANDLER_INSTALLED) {
		/* reschedule with same period by default */
		bigtime_t when = si->refresh_period;
		timer *to;

		/* if interrupts are "enabled", do our thing */
		if (di->can_interrupt) {
			/* insert code to sync to interrupts here */
			// y_SIS :擬似割り込みと垂直ブランクを同期させる方法を考える。
			//		やっぱり垂直ブランクになるまで待つような処理しかないか？
			// Beta1: もうすこし考える。現在のように、じわじわあわせていくか？
			// Beta2: 結局の所、しっかり同期を取らせようとすると、凄く遅くなる。
			// 同期はすこしづつ合わせるだけとする。
			acquire_spinlock(&gSpinlock);
			vbl_status = SiSIsVblank(si);		// y_SiS : 垂直ブランク中？
			release_spinlock(&gSpinlock);
			if (!vbl_status) {				// y_SiS : 垂直ブランク中でなければ
				when -= si->blank_period - 4;	// y_SiS : 次回割り込みをすこし速める。
			} 
			/* do the things we do when we notice a vertical retrace */
			result = thread_interrupt_work(si);
		}

		/* pick the "other" timer */
		to = (timer *)&(di->ti_a);
		if (to == te) to = (timer *)&(di->ti_b);
		/* our guess as to when we should be back */
		((timer_info *)to)->when_target = now + when;
		/* reschedule the interrupt */
		add_timer(to, (timer_hook)timer_interrupt_func, ((timer_info *)to)->when_target, B_ONE_SHOT_ABSOLUTE_TIMER);
		/* remember the currently active timer */
		di->current_timer = (timer_info *)to;
	}

	return result;
}
#else	// y_SiS : R4.0のみで使用され、 R4.5以降では決して使用されないコード。
static int32 fake_interrupt_thread_func(void *_di)
{
	device_info *di = (device_info *)_di;
	shared_info *si = di->si;
	int32 *flags = &(si->flags);
	
	bigtime_t this_sync;
	bigtime_t when;
	cpu_status	theStat;
	uint32 vbl_status = 0 /* read vertical blank status */;
	
	// ySiS : POST_R4_0とほぼ同じ処理。
	/* loop until notified */
	while(atomic_and(flags, -1) & YSIS_HANDLER_INSTALLED) {
		/* see if "interrupts" are enabled */
		when = si->refresh_period;
		if((volatile int32)(di->can_interrupt)) {
			/* get the system_time */
			this_sync = system_time();
			theStat = disable_interrupts();	// これは割り込みハンドラではない。そのためSpinlock確保前に割り込みの禁止が必要。
			acquire_spinlock(&gSpinlock);
			vbl_status = SiSIsVblank(si);		// y_SiS : 垂直ブランク中？
			release_spinlock(&gSpinlock);
			restore_interrupts(theStat);
			if (!vbl_status) {
				when -= si->blank_period - 4;	// y_SiS : 次回割り込みをすこし速める。
			}			
			/* do our stuff */
			thread_interrupt_work(si);
		} else {
			/* get the system_time */
			this_sync = system_time();
		}
		
		/* snooze until our next retrace */
		snooze_until(this_sync + when, B_SYSTEM_TIMEBASE);
	}
	
	ddprintf(("fake_interrupt_thread_func ends with flags = 0x%08lx\n", *flags));
	
	/* gotta return something */
	
	return B_OK;
}
#endif

/* delay for some microseconds.  The sample driver doesn't use it, but you might. */
// y_SiS : いくつかのレジスタは書き込んだ後、遅延を必要とする。
void delay(bigtime_t i)
{
	bigtime_t start = system_time();
	while(system_time() - start < i)
		/* do nothing */;
}

#if DEBUG > 0

static void dumpCRTC(void);

static int ySiSdump(int argc , char ** argv) {	// y_SiS
	int i;

	kprintf("ySiS Kernel Driver Persistant Data\n\nThere are %ld card(s)\n", pd->count);	// y_SiS
	kprintf("Driver wide benahpore: %ld/%ld\n", pd->kernel.ben, pd->kernel.sem);

	kprintf("Total seen interrupts: %ld\n", pd->total_interrupts);
	for (i = 0; i < pd->count; i++) {
		device_info *di = &(pd->di[i]);
		uint16 device_id = di->pcii.device_id;
		shared_info *si = di->si;
		
		kprintf("  device_id: 0x%04x\n", device_id);
		kprintf("  interrupt count: %ld\n", di->interrupt_count);
		if (si) {
			kprintf("  cursor: %d,%d\n", si->cursor.x, si->cursor.y);
			kprintf("  flags:");
			if (si->flags & YSIS_HANDLER_INSTALLED) kprintf(" YSIS_HANDLER_INSTALLED");
			if (si->flags & YSIS_MOVE_CURSOR) kprintf(" YSIS_MOVE_CURSOR");
			if (si->flags & YSIS_PROGRAM_CLUT) kprintf(" YSIS_PROGRAM_CLUT");
			if (si->flags & YSIS_SET_START_ADDR) kprintf(" YSIS_SET_START_ADDR");
			kprintf("  vblank semaphore id: %ld\n", si->vblank);
			kprintf("\n");
			dumpCRTC();
		}
	}
	return 1; /* the magic number for success */
}

static void
dumpCRTC(void) {
	int	theLoop;
	uint8	theData;

	kprintf("gIOBase = %04x, CRTC:", gIOBase);
	for(theLoop = 0; theLoop <= 0x18; theLoop++) {
		outb(CRT_IDX, (uint8)theLoop);
		theData = inb(CRT_DATA);			
		kprintf("%02x ", theData);
	}
	kprintf("\n");
}

#endif

static status_t open_hook (const char* name, uint32 flags, void** cookie) {
	int32 index = 0;
	device_info *di;
	shared_info *si;
	thread_id	thid;
	thread_info	thinfo;
	status_t	result = B_OK;
	vuint32		*regs;
	char shared_name[B_OS_NAME_LENGTH];

	ddprintf(("ySiS : open_hook(%s, %ld, 0x%08lx)\n", name, flags, (uint32)cookie));

	/* find the device name in the list of devices */
	/* we're never passed a name we didn't publish */
	while (pd->device_names[index] && (strcmp(name, pd->device_names[index]) != 0)) index++;

	/* for convienience */
	di = &(pd->di[index]);

	/* make sure no one else has write access to the common data */
	AQUIRE_BEN(pd->kernel);

	/* if it's already open for writing */
	if (di->is_open) {
		/* mark it open another time */
		goto mark_as_open;
	}
	/* create the shared area */
	sprintf(shared_name, "%04X_%04X_%02X%02X%02X shared",
		di->pcii.vendor_id, di->pcii.device_id,
		di->pcii.bus, di->pcii.device, di->pcii.function);
	/* create this area with NO user-space read or write permissions, to prevent accidental dammage */
	di->shared_area = create_area(shared_name, (void **)&(di->si), B_ANY_KERNEL_ADDRESS, ((sizeof(shared_info) + (B_PAGE_SIZE - 1)) & ~(B_PAGE_SIZE - 1)), B_FULL_LOCK, 0);
	if (di->shared_area < 0) {
		/* return the error */
		result = di->shared_area;
		goto done;
	}

	/* save a few dereferences */
	si = di->si;

	/* save the vendor and device IDs */
	si->vendor_id = di->pcii.vendor_id;
	si->device_id = di->pcii.device_id;
	si->revision = di->pcii.revision;

	/* map the device */
	result = map_device(di);
	if (result < 0) goto free_shared;
	result = B_OK;

	/* create a semaphore for vertical blank management */
	si->vblank = create_sem(0, di->name);
	if (si->vblank < 0) {
		result = si->vblank;
		goto unmap;
	}

	/* change the owner of the semaphores to the opener's team */
	/* this is required because apps can't acquire kernel semaphores */
	thid = find_thread(NULL);
	get_thread_info(thid, &thinfo);
	set_sem_owner(si->vblank, thinfo.team);

	/* assign local regs pointer for SAMPLExx() macros */
	regs = di->regs;

	/* disable and clear any pending interrupts */
	SiSEnableVint(false);	// y_SiS : 垂直帰線割り込み禁止
	SiSClearVint(0);		// y_SiS : 垂直帰線割り込みクリア

	/* if we're faking interrupts */
	if ((di->pcii.u.h0.interrupt_pin == 0x00) || (di->pcii.u.h0.interrupt_line == 0xff) || gHWVint == false){
		/* fake some kind of interrupt with a timer */
		di->can_interrupt = FALSE;
		si->flags = YSIS_HANDLER_INSTALLED;
		si->refresh_period = 16666; /* fake 60Hz to start */	// y_SiS : fakeというが、この段階ではこうするよりほかはない。accelerantのほうで、正しい値をセットするので心配無用。
		si->blank_period = si->refresh_period / 20;		// y_SiS : これは大嘘。accelerantで正しい値がセットされるので心配無用。
#if defined(POST_R4_0)
		di->ti_a.di = di;	/* refer to ourself */
		di->ti_b.di = di;
		di->current_timer = &(di->ti_a);
		/* program the first timer interrupt, and it will handle the rest */
		result = add_timer((timer *)(di->current_timer), (timer_hook)timer_interrupt_func, si->refresh_period, B_ONE_SHOT_RELATIVE_TIMER);
		/* bail if we can't add the timer */
		if (result != B_OK) goto delete_the_sem;
#else
		/* fake some kind of interrupt with a thread */		
		di->tid = spawn_kernel_thread(fake_interrupt_thread_func, "ySiS fake interrupt", B_REAL_TIME_DISPLAY_PRIORITY, di);
		/* bail if we can't spawn the thread */
		if(di->tid < 0) goto delete_the_sem;
		/* start up the thread */
		resume_thread(di->tid);
		result = B_OK;
#endif
	} else {
		/* otherwise install our interrupt handler */
		result = install_io_interrupt_handler(di->pcii.u.h0.interrupt_line, ySiS_interrupt, (void *)di, 0);
		ddprintf(("ySiS : real interrupt. interrupt line %d handler installed. %d\n", di->pcii.u.h0.interrupt_line, result));	// ySiS
		/* bail if we couldn't install the handler */
		if (result != B_OK) goto delete_the_sem;
		SiSClearVint(1);		// y_SiS : 垂直帰線割り込みクリア 001106
	}

mark_as_open:
	/* mark the device open */
	di->is_open++;

	/* send the cookie to the opener */
	*cookie = di;
	
	goto done;


delete_the_sem:
	delete_sem(si->vblank);

unmap:
	unmap_device(di);

free_shared:
	/* clean up our shared area */
	delete_area(di->shared_area);
	di->shared_area = -1;
	di->si = NULL;

done:
	/* end of critical section */
	RELEASE_BEN(pd->kernel);

	/* all done, return the status */
	ddprintf(("open_hook returning 0x%08lx\n", result));
	return result;
}

/* ----------
	read_hook - does nothing, gracefully
----- */
static status_t
read_hook (void* dev , off_t pos , void* buf , size_t* len)
{
	*len = 0;
	return B_NOT_ALLOWED;
}


/* ----------
	write_hook - does nothing, gracefully
----- */
static status_t
write_hook (void* dev , off_t pos , const void* buf , size_t* len)
{
	*len = 0;
	return B_NOT_ALLOWED;
}

/* ----------
	close_hook - does nothing, gracefully
----- */
static status_t
close_hook (void* dev)
{
	ddprintf(("ySiS : close_hook(%08lx)\n", (uint32)dev));
	/* we don't do anything on close: there might be dup'd fd */
	return B_NO_ERROR;
}

/* -----------
	free_hook - close down the device
----------- */
static status_t
free_hook (void* dev) {
	device_info *di = (device_info *)dev;
	shared_info	*si = di->si;

	ddprintf(("ySiS : free_hook() begins...\n"));
	/* lock the driver */
	AQUIRE_BEN(pd->kernel);

	/* if opened multiple times, decrement the open count and exit */
	if (di->is_open > 1)
		goto unlock_and_exit;

	/* disable and clear any pending interrupts */
	SiSEnableVint(false);	// y_SiS :
	SiSClearVint(0);	// y_SiS :
	
	/* if we were faking the interrupts */
	if ((di->pcii.u.h0.interrupt_pin == 0x00) || (di->pcii.u.h0.interrupt_line == 0xff) || gHWVint == false){
		/* stop our interrupt faking thread */
		si->flags = 0;
		di->can_interrupt = FALSE;
#if defined(POST_R4_0)
		/* cancel the timer */
		/* we don't know which one is current, so cancel them both and ignore any error */
		cancel_timer((timer *)&(di->ti_a));
		cancel_timer((timer *)&(di->ti_b));
#else
		/* we don't do anything here, as the R4 kernel reaps its own threads */
		/* After R4.0 we can do it ourselves, but we'd rather use timers */
#endif
	/* otherwise */
	} else {
		/* remove interrupt handler */
		remove_io_interrupt_handler(di->pcii.u.h0.interrupt_line, ySiS_interrupt, di);
	}

	/* delete the semaphores, ignoring any errors ('cause the owning team may have died on us) */
	delete_sem(si->vblank);
	si->vblank = -1;

	/* free regs and framebuffer areas */
	unmap_device(di);

	/* clean up our shared area */
	delete_area(di->shared_area);
	di->shared_area = -1;
	di->si = NULL;

unlock_and_exit:
	/* mark the device available */
	di->is_open--;
	/* unlock the driver */
	RELEASE_BEN(pd->kernel);
	ddprintf(("ySiS : free_hook() ends.\n"));
	/* all done */
	return B_OK;
}

/* -----------
	control_hook - where the real work is done
	y_SiS : ほとんどの処理が、ここで行われる。
----------- */
static status_t
control_hook (void* dev, uint32 msg, void *buf, size_t len) {
	device_info *di = (device_info *)dev;
	status_t result = B_DEV_INVALID_IOCTL;
	static bool	theLastVintStat = false;	// 割り込み状態 (起動直後は割り込み不可)

	switch (msg) {
		/* the only PUBLIC ioctl */
		case B_GET_ACCELERANT_SIGNATURE: {
			char *sig = (char *)buf;
			ddprintf(("ySiS : B_GET_ACCELERANT_SIGNATURE\n"));	// y_SiS
			strcpy(sig, "SiS6326.accelerant");	// y_SiS :
			result = B_OK;
		} break;
		
		/* PRIVATE ioctl from here on */
		case YSIS_GET_PRIVATE_DATA: {
			sample_get_private_data *gpd = (sample_get_private_data *)buf;
			ddprintf(("ySiS : YSIS_GET_PRIVATE_DATA\n"));	// y_SiS
			if (gpd->magic == YSIS_PRIVATE_DATA_MAGIC) {
				gpd->shared_info_area = di->shared_area;
				result = B_OK;
			}
		} break;
		case YSIS_GET_PCI: {
			sample_get_set_pci *gsp = (sample_get_set_pci *)buf;
			ddprintf(("ySiS : YSIS_GET_PCI\n"));	// y_SiS
			if (gsp->magic == YSIS_PRIVATE_DATA_MAGIC) {
				pci_info *pcii = &(di->pcii);
				gsp->value = get_pci((uchar)gsp->offset, (uchar)gsp->size);
				result = B_OK;
			}
		} break;
		case YSIS_SET_PCI: {
			sample_get_set_pci *gsp = (sample_get_set_pci *)buf;
			ddprintf(("ySiS : YSIS_SET_PCI\n"));	// y_SiS
			if (gsp->magic == YSIS_PRIVATE_DATA_MAGIC) {
				pci_info *pcii = &(di->pcii);
				set_pci((uchar)gsp->offset, (uchar)gsp->size, gsp->value);
				result = B_OK;
			}
		} break;
		case YSIS_DEVICE_NAME: {
			sample_device_name *dn = (sample_device_name *)buf;
			if (dn->magic == YSIS_PRIVATE_DATA_MAGIC) {
				strncpy(dn->name, di->name, MAX_YSIS_DEVICE_NAME_LENGTH);
				result = B_OK;
			}
		} break;
		case YSIS_RUN_INTERRUPTS: {
			sample_set_bool_state *ri = (sample_set_bool_state *)buf;
			ddprintf(("ySiS : YSIS_RUN_INTERRUPTS %d, theLastVintStat = %d\n", ri->do_it, theLastVintStat));	// y_SiS
			if (ri->magic == YSIS_PRIVATE_DATA_MAGIC) {
				/* are we faking interrupts? */
				if ((di->pcii.u.h0.interrupt_pin == 0x00) || (di->pcii.u.h0.interrupt_line == 0xff) || gHWVint == false){
					di->can_interrupt = ri->do_it;
				} else {
					cpu_status	theStat;
					theStat = disable_interrupts();
					acquire_spinlock(&gSpinlock);
					if (ri->do_it) {
						/* resume interrupts */
						theLastVintStat = SiSEnableVint(true);	// y_SiS : 割り込み許可
					} else {
						/* disable interrupts */
						theLastVintStat = SiSEnableVint(false);	// y_SiS : ここで以前の状態を覚えておく
					}
					release_spinlock(&gSpinlock);
					restore_interrupts(theStat);
				}
				result = B_OK;
			}
		} break;
		// y_SiS : この後に、このドライバ固有の各種処理が続く。
		case YSIS_READ_PORT :	// I/Oポート読み込み
			{
				cpu_status	theStat;
				theStat = disable_interrupts();
				acquire_spinlock(&gSpinlock);		// TEST 001107
				result = SiSReadPort((tIOAccess *)buf);
				release_spinlock(&gSpinlock);		// TEST 001107
				restore_interrupts(theStat);
			}
			break;
		case YSIS_WRITE_PORT :	// I/Oポート書き込み
			{
				cpu_status	theStat;
				theStat = disable_interrupts();
				acquire_spinlock(&gSpinlock);
				result = SiSWritePort((tIOAccess *)buf);
				release_spinlock(&gSpinlock);
				restore_interrupts(theStat);
			}
			break;
		case YSIS_READ_INDEXED : // CRTC,SEQRなどのIDX,DATAが連続したポートから読み込み
			{
				cpu_status	theStat;
				theStat = disable_interrupts();
				acquire_spinlock(&gSpinlock);		// TEST 001107
				result = SiSReadIndexed((tIOIdxAccess *)buf);
				release_spinlock(&gSpinlock);		// TEST 001107
				restore_interrupts(theStat);
			}
			break;
		case YSIS_WRITE_INDEXED : // CRTC,SEQRなどのIDX,DATAが連続したポートへの出力
			{
				cpu_status	theStat;
				theStat = disable_interrupts();
				acquire_spinlock(&gSpinlock);		// TEST 001107
				result = SiSWriteIndexed((tIOIdxAccess *)buf);
				release_spinlock(&gSpinlock);
				restore_interrupts(theStat);
			}
			break;
		case YSIS_SET_GAMMA :	// Gamma inverse
			if (atomic_or(&(pd->di[0].si->flags), YSIS_PROGRAM_CLUT) & YSIS_PROGRAM_CLUT) {
#if DEBUG > 0
				kprintf("ySiS : Already in Gamma!\n");
#endif
			}
			result = SiSSetGamma((tIOGammaTbl *)buf);
			atomic_and(&(pd->di[0].si->flags), ~YSIS_PROGRAM_CLUT);
			break;
		case YSIS_SET_DAC :	// Set DAC Palette
			if (atomic_or(&(pd->di[0].si->flags), YSIS_PROGRAM_CLUT) & YSIS_PROGRAM_CLUT) {
#if DEBUG > 0
				kprintf("ySiS : Already in YSIS_SET_DAC!\n");
#endif
			}
			result = SiSSetDAC((tIOGammaTbl *)buf);
			atomic_and(&(pd->di[0].si->flags), ~YSIS_PROGRAM_CLUT);
			break;
	}
	
	return result;
}
