
#include "WizardPad.h"

BInputServerDevice* instantiate_input_device()
{
	return new WizardPadDevice();
}

WizardPadDevice::WizardPadDevice()
:BInputServerDevice(), quit( true ), speed_multiplier( 1 ),
dbl_clk_speed( 250000 ), absolute_coord( true ),
btn_tip( 1 ), btn_barrel( 2 ), btn_ext1( 4 ), btn_ext2( 8 ), btn_eraser( 8 )
{
	input_device_ref dev = { "Genius WizardPad Tablet",
							 B_POINTING_DEVICE,	NULL };
	input_device_ref *dev_list[2] = { &dev, NULL };
	RegisterDevices( dev_list );
}

WizardPadDevice::~WizardPadDevice()
{
}

status_t WizardPadDevice::InitCheck()
{
	if ( BInputServerDevice::InitCheck() != B_OK )
		return B_ERROR;

	serial.SetDataRate( B_9600_BPS );
	serial.SetStopBits( B_STOP_BITS_2 );

	int num_serials = serial.CountDevices();
	for ( int i=0; i<num_serials; i++ ) {
		char dev_name[B_OS_NAME_LENGTH];
		serial.GetDeviceName( i, dev_name );
		if ( serial.Open( dev_name ) <= 0 )
			continue;

		serial.SetTimeout( 100000 ); // 0.1s
		serial.SetBlocking( true );

		serial.Write( ":", 1 );
		snooze( 250000 ); // 0.25s
		serial.Write( "OT", 2 );
		
		char buf[40]; for ( int k=0; k<40; k++ ) buf[k] = 0;
		int num_read, pos = 0;
		do {
			num_read = serial.Read( &buf[pos], 40-pos );
			pos += num_read;
		} while ( num_read );
		buf[pos] = 0;
		num_read = pos, pos = 0;
		if ( num_read == 0 )
			continue;

		char* mtx = "KW2600";
		for ( int m=0; m<6; m++ )
			if ( buf[m] != mtx[m] )
				continue;
		return B_OK;
	}
	return B_ERROR;
}

int32 wizard_thread( void* device )
{
	return ((WizardPadDevice*)device)->thread();
}

status_t WizardPadDevice::Start( const char *device, void *cookie )
{
	quit = false;
	threadid = spawn_thread( wizard_thread, device, B_DISPLAY_PRIORITY, this );
	resume_thread( threadid );
	return B_NO_ERROR;
}

status_t WizardPadDevice::Stop( const char *device, void *cookie )
{
	quit = true;
	status_t err = B_OK;
	wait_for_thread( threadid, &err );
	threadid = B_ERROR;
	return B_NO_ERROR;
}

status_t WizardPadDevice::Control( const char *device, void *cookie,
									uint32 code, BMessage *message )
{
	switch ( code ) {
	case B_CLICK_SPEED_CHANGED:
		get_click_speed( &dbl_clk_speed );
		break;
	case B_MOUSE_SPEED_CHANGED:
		adjust_speed();
		break;
	}
	return B_NO_ERROR;
}

int32 WizardPadDevice::get_byte()
{
	unsigned char buf[1];
	int num_read;
	do {
		if ( quit ) return 0xFF;
		num_read = serial.Read( buf, 1 );
	} while ( ! num_read );
	return buf[0];
}

int32 WizardPadDevice::thread()
{
	// Reset tablet
	serial.Write( "O", 1 );

	adjust_speed();
	get_click_speed( &dbl_clk_speed );
	serial.SetTimeout( 600000 ); // 0.6s
	int old_x = 0, old_y = 0, old_buttons = 0;
	bigtime_t last_click_time = 0, old_time = 0;

	while ( ! quit ) {
		// Get start byte
		if ( get_byte() != 'F' ) continue;

		// Get buttons
		int32 i = get_byte(); if ( i & 0xC0 ) continue;
		int32 buttons = 0;
		switch ( i ) {
		case 2: buttons = btn_barrel; break;	// barrel button
		case 9: buttons = btn_ext1; break;		// extension button 1
		case 10: buttons = btn_ext2; break;		// extension button 2
		case 13: buttons = btn_barrel | btn_ext1; break; // barrel + ext 1
		case 14: buttons = btn_barrel | btn_ext2; break; // barrel + ext 2
		default: buttons = 0;
		}
		bool eraser = ( i & btn_eraser ) != 0;

		// Get X position
		i = get_byte(); if ( i & 0xC0 ) continue;
		int32 x = i;
		i = get_byte(); if ( i & 0xC0 ) continue;
		x += i << 6;
		i = get_byte(); if ( i & 0xFE ) continue;
		x += i << 12;

		// Get Y position
		i = get_byte(); if ( i & 0xC0 ) continue;
		int32 y = i;
		i = get_byte(); if ( i & 0xC0 ) continue;
		y += i << 6;
		i = get_byte(); if ( i ) continue;
		y = 0xED8 - y;

		// Get pressure
		i = get_byte(); if ( i & 0xF0 ) continue;
		int32 pressure = i;
		i = get_byte(); if ( i & 0xF0 ) continue;
		pressure += i << 4;
		if ( pressure ) buttons |= btn_tip; // tip button

		// Choose message to send
		BMessage* event;
		if ( buttons == old_buttons ) event = new BMessage( B_MOUSE_MOVED );
		else if ( buttons > old_buttons ) event = new BMessage( B_MOUSE_DOWN );
		else event = new BMessage( B_MOUSE_UP );

		// Add time information
		bigtime_t cur_time = system_time();
		event->AddInt64( "when", cur_time );

		// Add position information
		float fx = (float)((double)x / 5200.0f), fy = (float)((double)y / 3800.0f);
		if ( absolute_coord ) {
			event->AddFloat( "x", fx );
			event->AddFloat( "y", fy );
		} else {
			static float dx, dy;
			int32 ax = 0, ay = 0;
			if ( cur_time - old_time < 100000 ) { // 0.1s
				dx += (float)(x-old_x) * speed_multiplier;
				dy += (float)(old_y-y) * speed_multiplier;
				if ( dx >= 1 || dx <= -1 ) ax = (int32)dx, dx = 0;
				if ( dy >= 1 || dy <= -1 ) ay = (int32)dy, dy = 0;
			}
			event->AddInt32( "x", ax );
			event->AddInt32( "y", ay );
			old_x = x, old_y = y;
			old_time = cur_time;
		}
		event->AddFloat( "be:tablet_x", fx );
		event->AddFloat( "be:tablet_y", fy );

		// Add pressure information
		event->AddFloat( "be:tablet_pressure", (float)pressure / 255.0f );

		// Add eraser information
		if ( btn_eraser )
			event->AddInt32( "be:tablet_eraser", eraser );

		// Add doubleclick info
		if ( buttons > old_buttons ) {
			static int clicks;
			if ( cur_time - last_click_time > dbl_clk_speed ) clicks = 1;
			else clicks++;
			event->AddInt32( "clicks", clicks );
			event->AddInt32( "modifiers", modifiers() );
			last_click_time = cur_time;
		} else if ( buttons < old_buttons ) {
			event->AddInt32( "clicks", 0 );
			event->AddInt32( "modifiers", modifiers() );
		}

		// Add buttons information
		event->AddInt32( "buttons", buttons );

		// Post the message
		EnqueueMessage( event );

		// Replace buttons status
		old_buttons = buttons;
	}
}
