/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.

	Other authors:
	Rudolf Cornelissen 7/2004.
*/

/* standard kernel driver stuff */
#include <KernelExport.h>
#include <Drivers.h>
#include "AGP.h"

/* Tell the kernel what revision of the driver API we support */
int32	api_version = B_CUR_DRIVER_API_VERSION;

/* we don't export a name so users cannot attempt to use this driver.
 * Note:
 * the driver should be 'placed' in /dev/graphics/ so the app_server loads it. 
 * The whole point of this driver is to be loaded automatically so it can enable
 * AGP mode without a 'real' graphics driver at all.
 * This way AGP FW is enabled on supporting hardware _even_ in VESA mode, and
 * _without_ a graphics driver explicitly supporting it if one is loaded after all. */
static char *devNames[] = {
	NULL
};

/* prototypes for our private functions */
static status_t open_hook (const char* name, uint32 flags, void** cookie);
static status_t close_hook (void* dev);
static status_t free_hook (void* dev);
static status_t read_hook (void* dev, off_t pos, void* buf, size_t* len);
static status_t write_hook (void* dev, off_t pos, const void* buf, size_t* len);
static status_t control_hook (void* dev, uint32 msg, void *buf, size_t len);
static status_t setup_agp(void);


static agp_module_info	*agp_bus;
static device_hooks graphics_device_hooks = {
	open_hook,
	close_hook,
	free_hook,
	control_hook,
	read_hook,
	write_hook,
	NULL,
	NULL,
	NULL,
	NULL
};


#define AGPDRV_DEBUG
#ifdef AGPDRV_DEBUG
#define TRACE dprintf
#else
#define TRACE silent
#endif
void silent(const char *, ... ) {};


/* 
	init_hardware() - Returns B_OK if AGP busmanager is found,
	else returns B_ERROR so the driver will be unloaded.
*/
status_t init_hardware(void)
{
	TRACE("agp_drv V2: init_hardware called\n");

	/* choke if we can't find the AGP bus */
	if (get_module(B_AGP_MODULE_NAME, (module_info **)&agp_bus) != B_OK)
		return B_ERROR;

	/* put away the module manager */
	put_module(B_AGP_MODULE_NAME);

	return B_OK;
}

status_t init_driver(void)
{
	TRACE("agp_drv V2: init_driver called\n");

	/* get a handle for the agp bus */
	if (get_module(B_AGP_MODULE_NAME, (module_info **)&agp_bus) != B_OK)
		return B_ERROR;

	/* enable AGP mode */
	return setup_agp();
}

const char **publish_devices(void)
{
	/* return the list of supported devices (which contains no names) */
	return (const char **)devNames;
}

device_hooks *find_device(const char *name)
{
	return &graphics_device_hooks;
}

void uninit_driver(void)
{
	TRACE("agp_drv V2: uninit_driver called\n");

	/* put the agp module away */
	put_module(B_AGP_MODULE_NAME);
}

static status_t open_hook (const char* name, uint32 flags, void** cookie)
{
	return B_OK;
}

/* read_hook - does nothing, gracefully */
static status_t read_hook (void* dev, off_t pos, void* buf, size_t* len)
{
	*len = 0;
	return B_NOT_ALLOWED;
}

/* write_hook - does nothing, gracefully */
static status_t write_hook (void* dev, off_t pos, const void* buf, size_t* len)
{
	*len = 0;
	return B_NOT_ALLOWED;
}

/* close_hook - does nothing, gracefully */
static status_t close_hook (void* dev)
{
	return B_NO_ERROR;
}

/* free_hook - close down the device */
static status_t
free_hook (void* dev)
{
	return B_OK;
}

/* control_hook - where the real work is done */
static status_t control_hook (void* dev, uint32 msg, void *buf, size_t len)
{
	return B_DEV_INVALID_IOCTL;
}

static status_t setup_agp(void)
{
	agp_info agpi[8];
	uint32 cmd;
	uint8 dev = 0;
	uint8 cnt;

	/* get AGP info from all devices */
	while ((dev < 8) && (((*agp_bus->get_nth_agp_info)(dev, &(agpi[dev]))) == B_NO_ERROR))
	{
		dev++;
	}
	/* abort if no devices found */
	if (!dev) return B_ERROR;

	/* only enable AGP if not already done */
	for (cnt = 0; cnt < dev; cnt++)
		if (!(agpi[cnt].interface.agp_cmd & AGP_enable)) goto cont;

	TRACE("agp_drv V2: AGP already enabled, aborting.\n");
	return B_OK;

cont:
	TRACE("agp_drv V2: enabling AGP\n");

	/* activate AGP mode:
	 * let the AGP busmanager worry about what mode to set.. */
	cmd = (0xfffffff7);
	/* ..but we do need to select the right speed scheme fetched from a device */
	/* note:
	 * because the AGP busmanager currently sets all devices to the same mode, and
	 * it doesn't distinquish different AGP busses yet, this is working OK as it is. */
	if (agpi[0].interface.agp_stat & AGP_rate_rev) cmd |= AGP_rate_rev;
	/* do it */
	(*agp_bus->enable_agp)(&cmd);

	TRACE("agp_drv V2: AGP cmd readback: $%08lx.\n", cmd);

	return B_OK;
}
