//amd-cpu driver
//a driver to enable features of AMD cpus that BeOS does not
//writen by John Johansen, 2001
//This driver and all associated code is placed in the public domain


// Family 5
// k6 model 6 .35 micron
// k6 model 7 .25 micron
// k6-2 model 8[0-7]:
// k6-2 CX core model 8[8-f]: k6-2 + MTRRs, 3DNow, 100 MHz bus
// k6-3 model 9: K6-2 CX + 256K L2 cache, external L3, 3DNow extensions
// k6-2+ model D[4-7] k6-3 + PowerNow, L2 cache only 128K, .18 micron
// k6-3+ model D[0-3] k6-3 + PowerNow, .18 micron

// Family 6
// Athlon, Duron

//AMD K6, K6-2, K6-3 MSR INFO and which are used in this driver.
//
//C000_0080h EFER: Extended Feature Enable Register
//   models 6,7,8[0-7]  SCE flag
//   models 8[8-f] added flags: EWBE, DPE
//   models 9,D added flag: L2D
//C000_0081h STAR: Syscall/Sysret Target Address Register **** NOT USED ****
//   models 8,9,D
//C000_0082h WHCR: Write Handling Control Register
//   models 6,7,8[0-7] supports 504MB Write Allocate
//   models 8[8-f],9,D supports 4GB Write Allocate
//C000_0085h UWCCR: Cacheability Control Register
//    models 8[8-f],9,D 2 uncacheable or write combining ranges
//C000_0086h EPMR: Enhanced Power Management Register
//    models D
//C000_0087h PSOR: Processor State Observabilty Register
//    models 8[8-f] NOL2, STEP, BF
//    models 9 added flag: NOL2
//    models D added flag: VID, PBF
//C000_0088h PFIR: Page Flush Invalidate Register  **** NOT USED ****
//    models 8[8-f]
//C000_0089h LCAAR: Level-2 Cache Array Access Register **** NOT USED ****
//    models 9,D

//AMD Athlon, only the programmable MTRR ranges of the Athlon are supported

#include <drivers/Drivers.h>
#include <drivers/driver_settings.h>
#include <drivers/KernelExport.h>

#include <malloc.h>
#include <string.h>

#include "amdcpu.h"

#define DPRINTF(a) dprintf("amd-cpu: "); dprintf a

#define DEVICE_NAME "misc/amd-cpu"

typedef struct {
  unsigned int flags;
  unsigned int cr4;
  unsigned int cr0;
} save_context;

//setting files unknown value strings
static char *unknownValue = "unknownValue";
static char *noArgValue = "noArgValue";
static char vendorAMD[13] = "AuthenticAMD";

//map the memory types to their intel #'s
static int maptypes[6] = {0,	// off
                          0,	// uc
                          1,	// wc
                          4,	// wt
                          5,	// wp
                          6};	// wb

static int revmaptypes[7] = {1,  // uc
                             2,  // wc
                             0,  // shouldn't happen
                             0,  // shouldn't happen
                             3,  // wt
                             4,  // wp
                             5}; // wb

//the Signature of the processor
static AMDSignature signature;

//the multiplier settings for the k6-x+ cpus, steps 0-7
static int maxMult = -1;
static unsigned int k6mults[8] = {4<<5,5<<5,7<<5,2<<5,0<<5,1<<5,3<<5,6<<5};
static unsigned int k6revmults[8] = {4,5,3,6,0,1,7,2};
static char k6multStrings[4*8] = {"2.0\0003.0\0003.5\0004.0\0004.5\0005.0\0005.5\0006.0"};

// athlon masks start at 4K while k6 at 128k
// to convert to k6 masks chop off high 9 bits
static unsigned int mtrrmasks[25] = {0xffffff,	// 4k,	128K
                                     0xfffffe,	// 8k,	256K
                                     0xfffffc,	// 16k,	512K
                                     0xfffff8,	// 32k,	1M
                                     0xfffff0,	// 64k,	2M                                     
                                     0xffffe0,	// 128k,4M
                                     0xffffc0,	// 256K,8M
                                     0xffff80,	// 512K,16M
                                     0xffff00,	// 1M,	32M
                                     0xfffe00,	// 2M,	64M
                                     0xfffc00,	// 4M,	128M
                                     0xfff800,	// 8M,	256M
                                     0xfff000,	// 16M,	512M
                                     0xffe000,	// 32M,	1G
                                     0xffc000,	// 64M,	2G
                                     0xff8000,	// 128M	4G
                                     0xff0000,	// 256M
                                     0xfe0000,	// 512M
                                     0xfc0000,	// 1G
                                     0xf80000,	// 2G
                                     0xf00000,	// 4G
                                     0xe00000,	// 8G
                                     0xc00000,	// 16G
                                     0x800000, 	// 32G
                                     0x000000};	// 64G
 
inline void get_msr(int reg, AllMSRTypes *data) {
	asm("rdmsr\n"
	    : "=a" (data->raw.a), "=d" (data->raw.b)
	    : "c" (reg)
	    );
}

inline void set_msr(int reg, AllMSRTypes *data) {
	asm("wrmsr\n"
	    :
	    : "c" (reg), "a" (data->raw.a), "d" (data->raw.b)	// input regs
	    );
}


// routines used to prepare memory and restore when playing with mtrrs etc
// even though the k6 doesn't know about global pages we will use the same
// routine.
inline void prepare_mem(save_context *c) {

  asm volatile("
    // turn of interrupts
    cli
    
    movl  %%cr4,%%eax
    movl  %%eax,4(%%ebx)
    andb  $0x7f,%%al
    movl  %%eax,%%cr4
    
    movl  %%cr0,%%eax
    movl  %%eax,8(%%ebx)
    orl   $0x40000000,%%eax
    wbinvd
    movl  %%eax,%%cr0
    wbinvd
    mov   %%cr3,%%eax
    mov   %%eax,%%cr3\n"
    : // no outputs
    : "b" (c)    // inputs
    : "%eax", "memory" //scratch
  );
  
}

inline void finish_mem(save_context *c) {
  asm volatile("
    // flush caches and tlb
    wbinvd
    mov  %%cr3,%%eax
    mov  %%eax,%%cr3
    
    // enable caches
    movl  8(%%ebx),%%eax
    movl  %%eax,%%cr0
    // restore pge
    movl  4(%%ebx),%%eax
    movl  %%eax,%%cr4
    // turn interrupts on
    sti\n"
    : //no outputs
    : "b" (c)  // inputs
    : "%eax", "memory"
  );
  
}


//check to see if we have an AMD processor and which version.
int checkProcessor() {
  char vendor[13];

  vendor[12] = 0;
  
  asm volatile("xor   %%eax,%%eax\n"
      "cpuid\n"
      "mov   %%ebx,(%%esi)\n"
      "mov   %%edx,4(%%esi)\n"
      "mov   %%ecx,8(%%esi)\n"
      "mov   $1,%%eax\n"
      "cpuid\n"
      "mov %%eax,(%%edi)\n"
      : // no output
      : "S" (&vendor), "D" (&signature)
      : "%eax","%ebx","%ecx","%edx" // scratch regs
     );

  //dprintf("check processor: %s, 0x%x\n", vendor, signature);

  if (strcmp(vendor, vendorAMD) != 0) return 0;

  return 1;  
}

/* k6 functions */
int setWA(int size) {
  save_context context;
  AllMSRTypes tmpmsr;

  if (signature.family != 5) return 0;
  
  if (signature.model <= 8 && signature.stepping < 8) {
    get_msr(MSR_UWWHCR, &tmpmsr);
    tmpmsr.whcrold.WAELIMIT = ((size+3)>>22)&0x7f;
  
  } else {
    get_msr(MSR_UWWHCR, &tmpmsr);
    tmpmsr.whcr.WAELIMIT = ((size+3)>>2)&0x3ff;
  }
  
  prepare_mem(&context);
  set_msr(MSR_UWWHCR, &tmpmsr);
  finish_mem(&context);  

  return 1;
}

int setWAHole(int allochole) {
  save_context context;
  AllMSRTypes tmpmsr;

  if (signature.family != 5) return 0;
  
  if (signature.model <= 8 && signature.stepping < 8) {
    get_msr(MSR_UWWHCR, &tmpmsr);
    if (allochole) tmpmsr.whcrold.hole = 1; else tmpmsr.whcrold.hole = 0;
    
  } else {
    get_msr(MSR_UWWHCR, &tmpmsr);
    if (allochole) tmpmsr.whcr.hole = 1; else tmpmsr.whcr.hole = 0;

  }

  prepare_mem(&context);
  set_msr(MSR_UWWHCR, &tmpmsr);
  finish_mem(&context);  
  
  return 1;
}

int setMult(int index) {
  // currently doesn't support setting volatage
  if (signature.family != 5 || signature.model != 0xd) return 0;
  asm volatile("//disable interrupts and setup EPMR
       cli
       movl	$0xC0000086,%%ecx
       xorl	%%edx,%%edx
       movl	$0x0001,%%eax
       wrmsr
        
       // read in the ebf divisor and voltage settings and mask them to
       // our new values then write them back out entering EPM stop grant
       // state for 1 cycle so the transition can take place
       movl	$8,%%edx
       inl	%%dx,%%eax
       andl	$0x1f,%%eax		// keep VIDO
       orl	%%ebx,%%eax     // set the multiplier
       orl	$0x1600,%%eax   // stop grant timer and transition mode
       outl	%%eax,%%dx      // enter stop grant and do transition 
         
       // disable EPMR so everything else will work
       xorl	%%edx,%%edx
       xorl	%%eax,%%eax
       wrmsr
       sti\n"
      :
      : "b" (index)
      : "%eax", "%ecx", "%edx"
     );
  return 1;
}

// set MTRR 0 or 1
// addr - in bytes
// size - in bytes
// mode - k6 MTRR modes, 0=noeffect, 1=uncacheable, 2=writecombining
// k6 - mtrr requires sizes to be in powers of 2 starting at 128K
//    - addresses must be size aligned, so we will take care of it.
int setK6MTRR(int mtrr, unsigned int addr, unsigned int size, int mode) {
  save_context context;
  AllMSRTypes tmpmsr;
  int index;
  unsigned int t;
  
  if (mtrr < 0 || mtrr > 1) return B_ERROR;
  if (mode < 0 || mode > 2) return B_ERROR;
  
  if (mode == 0) {
    size = 0;
    addr = 0;
  }
  
  if (size==0 || size > 0x80000000) {
    index = 15;
  } else {
    t = 1<<17;
    index=0;
    while (size > t) {
      t <<= 1;
      index++;
    }
  }

  get_msr(MSR_UWCCR, &tmpmsr);
  if (mtrr==0) {
    tmpmsr.mtrr.MASK0 = mtrrmasks[index]&0x7fff;
    tmpmsr.mtrr.BASE0 = (addr>>17)&(mtrrmasks[index]&0x7fff); // align to size constraint
    tmpmsr.mtrr.UC0 = mode&1;
    tmpmsr.mtrr.WC0 = mode>>1;
  } else {
    tmpmsr.mtrr.MASK1 = mtrrmasks[index]&0x7fff;
    tmpmsr.mtrr.BASE1 = (addr>>17)&(mtrrmasks[index]&0x7fff); // align to size constraint
    tmpmsr.mtrr.UC1 = mode&1;
    tmpmsr.mtrr.WC1 = mode>>1; 
  }

  prepare_mem(&context);
  set_msr(MSR_UWCCR, &tmpmsr);
  finish_mem(&context);

  return B_OK;
}

// Athlon has 8 variable range MTRRs
int setAthlonMTRR(int mtrr, unsigned int addr, unsigned int size, int mode) {
  save_context context;
  MTRRPhysBase bmtrr;
  MTRRMask mmtrr;
  
  int index;
  unsigned int t;
  
  
  if (mtrr < 0 || mtrr > 7) return B_ERROR;
  mtrr = (mtrr<<1) + ATHLON_MTRR_BASE;

  if (mode < 0 || mode > 5) return B_ERROR;
    
  if (size==0  || size > 0x80000000) {
    index = 24;
  } else {
    // drop the lowest 12 bits
    t = 1<<12;
    index=0;
    while (size > t) {
      t <<= 1;
      index++;
    }
  }

  bmtrr.type = maptypes[mode];
  bmtrr.baselow = ((addr>>12)&mtrrmasks[index])&0xfffff;
  bmtrr.basehigh = ((addr>>12)&mtrrmasks[index])>>20;
  bmtrr.pad0 = 0;
  bmtrr.pad1 = 0;
  if (mode) mmtrr.v = 1; else mmtrr.v =0;

  mmtrr.masklow = mtrrmasks[index]&0xfffff;
  mmtrr.maskhigh = mtrrmasks[index]>>20;
  mmtrr.pad0 = 0;
  mmtrr.pad1 = 0;
  
  prepare_mem(&context);
  asm volatile("
    //disable MTRRs
    mov   $0x2ff,%%ecx	//MTRRdefTypeReg
    rdmsr
    pushl %%eax
    pushl %%edx
    andw  $0xf300,%%ax
    wrmsr
       
    //set the MTRR
    mov  (%%edi),%%eax
    mov  4(%%edi),%%edx
    movl %%ebx,%%ecx
    wrmsr
       
    mov  (%%esi),%%eax
    mov  4(%%esi),%%edx
    movl %%ebx,%%ecx
    add  $1,%%ecx
    wrmsr

    //enable MTRRs
    mov  $0x2ff,%%ecx	  //MTRRdefTypeReg
    popl %%edx
    popl %%eax
    orw  $0x800,%%ax
    wrmsr\n"
    : // no output
    : "b" (mtrr), "D" (&bmtrr), "S" (&mmtrr)	//  input
    : "%eax", "%ecx", "%edx", "memory"	// scratch regs
  );
  finish_mem(&context);
  
  return B_OK;
}


int setMTRR(int mtrr, unsigned int addr, unsigned int size, int mode) {
  if (signature.family == 5 && (signature.model > 8 || 
    (signature.model == 8 && signature.stepping > 7)))
    return setK6MTRR(mtrr, addr, size, mode);
  else if (signature.family == 6)
    return setAthlonMTRR(mtrr, addr, size, mode);
  
  return B_ERROR;
}


int getMTRR(MTRRInterface *m) {
  MTRR mtrr;
  MTRRPhysBase bmtrr;
  MTRRMask mmtrr;
  
  if (signature.family == 5 && (signature.model > 8 || 
    (signature.model == 8 && signature.stepping > 7))) {
      get_msr(MSR_UWCCR, (AllMSRTypes *) &(mtrr));
      if (m->mtrr == 0) {
        m->address = mtrr.BASE0<<17;
        m->size = (((~mtrr.MASK0)&0x7fff)+1)<<17;
        if (m->size == 0 && m->address == 0) {
          m->mode = 0;
        } else {
          m->mode = mtrr.UC0 | (mtrr.WC0<<1);
        }
      } else {
        m->address = mtrr.BASE1<<17;
        m->size = (((~mtrr.MASK1)&0x7fff)+1)<<17;
        if (m->size == 0 &&m->address) {
          m->mode = 0;
        } else {
          m->mode = mtrr.UC1 | (mtrr.WC1<<1);
        }
      }  
  } else if (signature.family == 6) {
    get_msr(ATHLON_MTRR_BASE+(m->mtrr<<1), (AllMSRTypes *) &(bmtrr));
    get_msr(ATHLON_MTRR_BASE+((m->mtrr<<1)+1), (AllMSRTypes *) &(mmtrr));
    m->address = bmtrr.baselow << 12;  // just drop base high we aren't supporting 36 bit addrs
    m->size = (((~(mmtrr.masklow|(mmtrr.maskhigh<<20)))&0xffffff)+1)<<12;
    if (mmtrr.v == 0) {
      m->mode = 0;
    } else {
      m->mode = revmaptypes[bmtrr.type];
    }
  }

}



// routine to map characters to internal #
int memMode(char c) {
  static char memtype[] = {'n', /* no effect */
                           'u', /*uncacheable*/
                           'c', /*write combining */
                           't', /*write through - athlon only*/
                           'p', /*write protect -athlon only*/
                           'b', /*write back - athlon only*/
                           '\000'
                          };
  int i;
  for (i=0; memtype[i] != 0; i++) {
    if (c == memtype[i]) return i;
  }
  return 0;
}

int parseMTRRline(const char *s, int *mtrr, unsigned int *base, unsigned int *size, int *mode) {
  char *rest;
  unsigned int high;
  *mtrr = strtol(s, &rest, 0);
  *base = strtoul(&(rest[1]), &rest, 0);
  high = strtoul(&(rest[1]), &rest, 0);
  *mode = memMode(rest[1]);
  if (high <= *base) return 0;
  *size = high-*base+1;
  return 1;
}

/**************** Device interface functions *****************/
static status_t amd_open(const char *name, uint32 mode, void **cookie) {
  return B_OK;
}

static status_t amd_close(void *cookie) {
  return B_OK;
}

static status_t amd_free(void *cookie) {
  return B_OK;
}

static status_t amd_ioctl(void *cookie, uint32 op, void *data, size_t len) {
  status_t	result = B_OK;
  uint32	CReg;
  AllMSRTypes tmpmsr;
  BVC bcv;
  uint32 address;
  MTRRInterface *mtrri;
  
  unsigned int temp;
  int tmp;
  
  MSRs *msrs = (MSRs *) data;
//set_dprintf_enabled(true);
//dprintf("examining op %d\n", op);
//set_dprintf_enabled(false);

  switch (op) {
	case AMD_GET_SIGNATURE:
	  // return the signature
	  *((AMDSignature *) data) = signature;
	  break;
    case AMD_GET_K6MSRS:
      if (signature.family!=5) return B_ERROR;
      get_msr(MSR_EFER, (AllMSRTypes *) &(msrs->efer));
      get_msr(MSR_UWWHCR, (AllMSRTypes *) &(msrs->whcr));
      if (signature.model > 7) get_msr(MSR_UWCCR, (AllMSRTypes *) &(msrs->mtrr));
      if (signature.model == 0xd) get_msr(MSR_EPMR, (AllMSRTypes *) &(msrs->epmr));
      if (signature.model > 7) get_msr(MSR_PSOR, (AllMSRTypes *) &(msrs->psor));
      break;
    case AMD_GET_WRITE_ALLOCATE:
      // put # of megs write allocated in *data
      if (signature.family != 5) return B_ERROR;
      get_msr(MSR_UWWHCR, &tmpmsr);
      *((int *)data)= tmpmsr.whcr.WAELIMIT<<2;
      break;
    case AMD_SET_WRITE_ALLOCATE:
      result = setWA(*((int *)data));
      break;
    case AMD_GET_WRITE_ALLOC_HOLE:
      // put true or false in *data
      if (signature.family != 5) return B_ERROR;
  
      get_msr(MSR_UWWHCR, &tmpmsr);
      if (signature.model <= 8 && signature.stepping < 8) {
        *((int *)data) = tmpmsr.whcrold.hole;    
      } else {
        *((int *)data) = tmpmsr.whcr.hole;    
      }
      break;
    case AMD_SET_WRITE_ALLOC_HOLE:
      // *data true if to write allocate 15-16M hole else false
      result = setWAHole(*(int *) data);
      break;
    case AMD_GET_MTRR:
      mtrri = (MTRRInterface *) data;
      result = getMTRR(mtrri);
      break;
    case AMD_SET_MTRR:
      mtrri = (MTRRInterface *) data;
      result = setMTRR(mtrri->mtrr, mtrri->address, mtrri->size, mtrri->mode);
      break;
    case AMD_GET_WRITE_ORDERING:
      // ?????
      break;
    case AMD_SET_WRITE_ORDERING:
      // ?????
      break;
    case AMD_GET_MULTIPLIER_STRINGS:
      memcpy((char *) data, k6multStrings, 4*8);
      break;
    case AMD_GET_MULTIPLIER:
      // put the multiplier index 0-7 in *data
      if (signature.family != 5) return B_ERROR;
      get_msr(MSR_PSOR, &tmpmsr);
      (*((int *) data)) = k6revmults[tmpmsr.psor.EBF];
      break;
    case AMD_SET_MULTIPLIER:	    
      // pass multiplier index - 0 - 7 in *data field
      result = setMult(k6mults[(*((int *)data))&7]);
      break;
    case AMD_GET_MAX_MULTIPLIER:
      *(int *)data = maxMult;
      break;
    case AMD_SET_MAX_MULTIPLIER:
      maxMult = *(int *)data;
      if ((maxMult < 0) || (maxMult > 7)) maxMult = -1;
      break;
    case AMD_SET_TO_MAX_MULTIPLIER:	    
      // pass multiplier index - 0 - 7 in *data field
      if (maxMult > -1) result = setMult(k6mults[maxMult]);
      break;
    case AMD_INC_MULTIPLIER:
      if (signature.family != 5 || signature.model != 0xd) return B_ERROR;
      get_msr(MSR_PSOR, &tmpmsr);
      tmp = k6revmults[tmpmsr.psor.EBF];
      tmp++;
      if(maxMult == -1) {
        if (tmp <= 7) setMult(k6mults[tmp]);
      } else {
        if (tmp<=maxMult) setMult(k6mults[tmp]);
      }
      break;
    case AMD_DEC_MULTIPLIER:
      if (signature.family != 5 || signature.model != 0xd) return B_ERROR;
      get_msr(MSR_PSOR, &tmpmsr);
      tmp = k6revmults[tmpmsr.psor.EBF];
      tmp--;
      if (tmp>=0) setMult(k6mults[tmp]);
      break;
    default:
      return B_ERROR;
  }

  return result;
}

static status_t amd_read(void *cookie, off_t pos, void *buffer, size_t *len) {
	return B_OK;
}

static status_t amd_write(void *cookie, off_t pos, const void *buffer, size_t *len) {
	return B_OK;
}

/***************************/
// mtrr0 start-end,mode={n,u,w}
// mtrr1 start-end,mode
// writealloc size
// writeallochole true/false
// maxmult value (eg 0-7 or boot)
// setmult value

status_t init_hardware() {
  const char *param;
  unsigned int low, size;
  int mtrr, mode;
  void *handle;
  AllMSRTypes tmpmsr;
  
  
//set_dprintf_enabled(true);
dprintf("init hardware\n");
  if (!checkProcessor()) return B_ERROR;

  handle = load_driver_settings("amd-cpu");
  
  param = get_driver_parameter(handle, "writealloc", unknownValue, noArgValue);
  if ((param != unknownValue) && (param != noArgValue)) {
    setWA(atoi(param));
  }
  param = get_driver_parameter(handle, "writeallochole", unknownValue, noArgValue);
  if ((param != unknownValue) && (param != noArgValue)) {
    setWAHole(strcmp(param, "true")==0);
  }
  param = get_driver_parameter(handle, "mtrr", unknownValue, noArgValue);
  if ((param != unknownValue) && (param != noArgValue)) {
    dprintf("setMTRR%d: 0x%x,0x%x,%d\n", low, size, mode);
    if (parseMTRRline(param,&mtrr,&low,&size,&mode)) setMTRR(mtrr,low,size,mode);
  }
  param = get_driver_parameter(handle, "setmult", unknownValue, noArgValue);
  if ((param != unknownValue) && (param != noArgValue)) {
    setMult(k6mults[atoi(param)&7]);
  }
  param = get_driver_parameter(handle, "maxmult", unknownValue, noArgValue);
  if ((param != unknownValue) && (param != noArgValue)) {

    if (strcmp(param, "boot")==0 && signature.family == 5) {
      get_msr(MSR_PSOR, &tmpmsr);
      maxMult = k6revmults[tmpmsr.psor.EBF];
    } else {
      maxMult = atoi(param);
      if ((maxMult < 0) || (maxMult > 7)) maxMult = -1;
    }

  }
  
//set_dprintf_enabled(false);

  unload_driver_settings(handle);
  return B_OK;
}

const char **publish_devices() {
  static const char *devices[] = {
    DEVICE_NAME, NULL
  };

  return devices;
}

device_hooks *find_device(const char *name) {
  static device_hooks hooks = {
    &amd_open,
    &amd_close,
    &amd_free,
    &amd_ioctl,
    &amd_read,
    &amd_write,
    NULL,
    NULL,
    NULL,
    NULL
  };

  if (!strcmp(name, DEVICE_NAME)) return &hooks;

  return NULL;
}

status_t init_driver() {
  const char *param;
  void *handle;
  
set_dprintf_enabled(true);
  // get the processor signature
  if (!checkProcessor()) return B_ERROR;

dprintf("passed processor check\n");

  handle = load_driver_settings("amd-cpu");

  
  param = get_driver_parameter(handle, "maxmult", unknownValue, noArgValue);
  if ((param != unknownValue) && (param != noArgValue)) {
    maxMult = atoi(param);
    if ((maxMult < 0) || (maxMult > 7)) maxMult = -1;
  }
  
  unload_driver_settings(handle);  
set_dprintf_enabled(false);

  return B_OK;
}

void uninit_driver() {
}
