/*
	IDE controller module skeleton for BeOS R4.5 x86
*/

#include <drivers/KernelExport.h>
#include <drivers/bus_manager.h>
#include <drivers/PCI.h>
#include "IDE.h"
#include "ioport.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <inttypes.h>
#include "cpufunc.h"

#include "pcmcia.h"
#include "ss_cs.h"
#include "config.h"
#include "resource.h"
#include "enum.h"
#include "parse.h"

#ifndef	DRIVER_NAME
#define	DRIVER_NAME	"ide_skel"
#endif

struct resource_list	io_list, mem_list;
struct resource_mask	irq_mask;

/* task file mapping types */
enum
{
	IDE_TYPE_IO_STD,		/* standard (separate) I/O mapped (primary or secondary IDE) */
	IDE_TYPE_IO_CONT_16,	/* contiguous I/O mapped (PC Card ATA) */
	IDE_TYPE_IO_CONT_8,
	IDE_TYPE_MEM,			/* memory mapped (PC Card ATA) */
};

typedef struct cookie_t cookie_t;
struct cookie_t
{
	cookie_t	*next;

	uint16	cmd_blk_regs_base;	/* 0x1f0, 0x170, ... */
	uint16	ctrl_blk_regs_base;	/* 0x3f6, 0x376, ... */
	int		irq;
	int		bus, abs_bus;
	int		type;		/* IDE_TYPE_* */
	int		flags;

	int		bus_lock_flag;
	int		intgate;
	sem_id	sem_mutex;
	sem_id	sem_intr;
};

pci_module_info *pci = NULL;
static int bus_count = 0;
static cookie_t *first_cookie = NULL;
static area_id cis_area = -1;

/* invoke kernel debugger if "enable console debugging" is on */

void	debug_pause	(const char *s)
{
	bool	flag;

	flag = set_dprintf_enabled (false);
	if (flag)
	{
		set_dprintf_enabled (true);
		kernel_debugger (s);
	}
}

static int32 inthand (void *cookie)
{
	if ((cookie_t *)cookie == NULL)
		return B_UNHANDLED_INTERRUPT;
#if DEBUG
	dprintf (DRIVER_NAME ": inthand(%d)\n", ((cookie_t *)cookie)->bus);
#endif

	/* XXX: interrupt sharing not taken into account */

	if (((cookie_t *)cookie)->bus_lock_flag == 0)
	{
		dprintf (DRIVER_NAME ": inthand(): bus not locked\n");
		return B_UNHANDLED_INTERRUPT;
	}

	/* read status reg; clear interrupt */
	read_io_8 (((cookie_t *)cookie)->cmd_blk_regs_base + 7);

	if (((cookie_t *)cookie)->intgate == 0)
	{
		dprintf (DRIVER_NAME ": inthand(): int gate 0\n");
		return B_UNHANDLED_INTERRUPT;
	}

	((cookie_t *)cookie)->intgate = 0;
	if (release_sem_etc (((cookie_t *)cookie)->sem_intr, 1, B_DO_NOT_RESCHEDULE) < 0)
		dprintf (DRIVER_NAME ": inthand(): failed to release sem\n");

	return B_INVOKE_SCHEDULER;
}

static status_t	rescan (void)
{
#if DEBUG
	dprintf (DRIVER_NAME ": rescan()\n");
#endif

	return B_OK;
}

static uint32 get_nth_cookie (uint32 bus)
{
	cookie_t *cookie;

#if DEBUG
	dprintf (DRIVER_NAME ": get_nth_cookie(%d)\n", bus);
#endif
	if (bus >= bus_count)
		return (uint32) NULL;

	for (cookie = first_cookie ; bus > 0 && cookie != NULL; --bus)
		cookie = cookie->next;
	return (uint32) cookie;
}

static uint32 get_bus_count (void)
{
#if DEBUG
	dprintf (DRIVER_NAME ": get_bus_count() = %d\n", bus_count);
#endif

	return bus_count;
}

static int32 get_abs_bus_num (uint32 cookie)
{
	if ((cookie_t *)cookie == NULL)
		return B_IDE_BUS_OTHER;
#if DEBUG
	dprintf (DRIVER_NAME ": get_abs_bus_num(%d)\n", ((cookie_t *)cookie)->bus);
#endif

	return ((cookie_t *)cookie)->abs_bus;
}

static status_t acquire_bus (uint32 cookie)
{
	status_t	result;

	if ((cookie_t *)cookie == NULL)
		return B_ERROR;
#if DEBUG
	dprintf (DRIVER_NAME ": acquire_bus(%d)\n", ((cookie_t *)cookie)->bus);
#endif

#if 1
	result = acquire_sem_etc
		(((cookie_t *) cookie)->sem_mutex, 1, B_CAN_INTERRUPT, 0);
#else
	result = acquire_sem (((cookie_t *) cookie)->sem_mutex);
#endif
	if (result == B_OK)
		((cookie_t *)cookie)->bus_lock_flag = 1;
	else
		dprintf (DRIVER_NAME ": failed to acquire bus\n");
	return result;
}

static status_t release_bus (uint32 cookie)
{
	status_t	result;

	if ((cookie_t *)cookie == NULL)
		return B_ERROR;
#if DEBUG
	dprintf (DRIVER_NAME ": release_bus(%d)\n", ((cookie_t *)cookie)->bus);
#endif

	result = release_sem_etc
		(((cookie_t *) cookie)->sem_mutex, 1, B_DO_NOT_RESCHEDULE);
	if (result == B_OK)
		((cookie_t *)cookie)->bus_lock_flag = 0;
	else
		dprintf (DRIVER_NAME ": failed to release bus\n");
	return result;
}

static status_t write_command_block_regs
	(uint32 cookie, ide_task_file *tf, ide_reg_mask mask)
{
	uint32	port;
	char	buf [256], buf2 [16];

	if ((cookie_t *)cookie == NULL)
		return B_ERROR;
#if DEBUG
	dprintf (DRIVER_NAME ": write_command_block_regs(%d)\n", ((cookie_t *)cookie)->bus);
#endif

	if (((cookie_t *)cookie)->bus_lock_flag == 0)
		return B_ERROR;

	port = ((cookie_t *)cookie)->cmd_blk_regs_base + 1;
#if DEBUG
	strcpy (buf, DRIVER_NAME ": write_command_block_regs:");
#endif
	while (mask != 0)
	{
		if (mask & 1)
		{
			write_io_8 (port, *tf);
#if DEBUG
			sprintf (buf2, " 0x%02x", *tf);
#endif
		}
#if DEBUG
		else
			strcpy (buf2, " --");
		strcat (buf, buf2);
#endif
		tf++;
		port++;
		mask >>= 1;
	}
#if DEBUG
	strcat (buf, "\n");
	dprintf (buf);
#endif
	return B_OK;
}

static status_t read_command_block_regs
	(uint32 cookie, ide_task_file *tf, ide_reg_mask mask)
{
	uint32	port;

	if ((cookie_t *)cookie == NULL)
		return B_ERROR;
#if DEBUG
	dprintf (DRIVER_NAME ": read_command_block_regs(%d)\n", ((cookie_t *)cookie)->bus);
#endif

	if (((cookie_t *)cookie)->bus_lock_flag == 0)
		return B_ERROR;

	port = ((cookie_t *)cookie)->cmd_blk_regs_base + 1;
	while (mask != 0)
	{
		if (mask & 1)
			*tf = read_io_8 (port);
		tf++;
		port++;
		mask >>= 1;
	}
	return B_OK;
}

static uint8 get_altstatus (uint32 cookie)
{
	uint8	val;

	if ((cookie_t *)cookie == NULL)
		return 0;

	if (((cookie_t *)cookie)->bus_lock_flag == 0)
		return B_ERROR;

	val = read_io_8 (((cookie_t *)cookie)->ctrl_blk_regs_base);
#if DEBUG
	dprintf (DRIVER_NAME ": get_altstatus(%d) = 0x%02x\n",
		((cookie_t *)cookie)->bus, val);
#endif

	return val;
}

static void write_device_control (uint32 cookie, uint8 val)
{
	if ((cookie_t *)cookie == NULL)
		return;
#if DEBUG
	dprintf (DRIVER_NAME ": write_device_control(%d, 0x%02x)\n",
		((cookie_t *)cookie)->bus, val);
#endif

	if (((cookie_t *)cookie)->bus_lock_flag == 0)
		return;

	write_io_8 (((cookie_t *)cookie)->ctrl_blk_regs_base, val);
}

static void write_pio_16
	(uint32 cookie, uint16 *data, uint16 count)
{
	uint32	dataport;
	int	type;

	if ((cookie_t *)cookie == NULL || data == NULL)
		return;
#if DEBUG
	dprintf (DRIVER_NAME ": write_pio_16(%d)\n", ((cookie_t *)cookie)->bus);
#endif

	if (((cookie_t *)cookie)->bus_lock_flag == 0)
		return;

	dataport = ((cookie_t *)cookie)->cmd_blk_regs_base;
	type = ((cookie_t *)cookie)->type;
	if (type == IDE_TYPE_IO_CONT_16 || type == IDE_TYPE_IO_CONT_8)
		dataport += 8;
	if (type == IDE_TYPE_IO_CONT_8)
		for ( ; count > 0; --count)
		{
			write_io_8 (dataport    , *(uint8 *)data++);
			write_io_8 (dataport + 1, *(uint8 *)data++);
		}
	else
#if USE_STRING_IO
		outsw (dataport, data, count);
#else
		for ( ; count > 0; --count)
			write_io_16 (dataport, *data++);
#endif
}

static void read_pio_16
	(uint32 cookie, uint16 *data, uint16 count)
{
	uint32	dataport;
	int	type;

	if ((cookie_t *)cookie == NULL || data == NULL)
		return;
#if DEBUG
	dprintf (DRIVER_NAME ": read_pio_16(%d)\n", ((cookie_t *)cookie)->bus);
#endif

	if (((cookie_t *)cookie)->bus_lock_flag == 0)
		return;

	dataport = ((cookie_t *)cookie)->cmd_blk_regs_base;
	type = ((cookie_t *)cookie)->type;
	if (type == IDE_TYPE_IO_CONT_16 || type == IDE_TYPE_IO_CONT_8)
		dataport += 8;	
	if (type == IDE_TYPE_IO_CONT_8)
		for ( ; count > 0; --count)
		{
			*(uint8 *)data++ = read_io_8 (dataport   );
			*(uint8 *)data++ = read_io_8 (dataport + 1);
		}
	else
#if USE_STRING_IO
		insw (dataport, data, count);
#else
		for ( ; count > 0; --count)
			*data++ = read_io_16 (dataport);
#endif
}

static status_t intwait (uint32 cookie, bigtime_t timeout)
{
	status_t	result;

	if ((cookie_t *)cookie == NULL)
		return B_ERROR;
#if DEBUG
	dprintf (DRIVER_NAME ": intwait(%d)\n", ((cookie_t *)cookie)->bus);
#endif

	if (((cookie_t *)cookie)->bus_lock_flag == 0)
		return B_ERROR;

	result = acquire_sem_etc (((cookie_t *) cookie)->sem_intr, 1,
		B_RELATIVE_TIMEOUT, timeout);
	if (result == B_OK)
		((cookie_t *)cookie)->intgate = 1;
	/* else B_TIMED_OUT or B_WOULD_BLOCK */
	return result;
}

/* DMA functions: not implemented */

static status_t prepare_dma
	(uint32 cookie, void *buffer, size_t *size, bool to_device)
{
	return B_NOT_ALLOWED;
}

static status_t start_dma (uint32 cookie)
{
	return B_NOT_ALLOWED;
}

static status_t finish_dma (uint32 cookie)
{
	return B_NOT_ALLOWED;
}

static int get_bad_alignment_mask (uint32 cookie)
{
	return B_NOT_ALLOWED;
}

static status_t get_dma_mode (uint32 cookie, uint8 x, uint32 y)
{
	return B_NOT_ALLOWED;
}

static status_t set_dma_mode (uint32 cookie, ...)
{
	return B_NOT_ALLOWED;
}

/* new_isa_device() */
static status_t register_ide_bus
(uint32 cmd_blk_regs_base, uint32 ctrl_blk_regs_base, int irq, int abs_bus, int type, int flags)
{
	cookie_t	*cookie;
	sem_id		result;

#if DEBUG
	dprintf (DRIVER_NAME ": register_ide_bus (0x%X, 0x%X, %d, %d)\n",
		cmd_blk_regs_base, ctrl_blk_regs_base, irq, flags);
#endif

	/* allocate memory, semaphores */
	if ((cookie = malloc (sizeof (cookie_t))) == NULL)
		return B_NO_MEMORY;

	cookie->cmd_blk_regs_base = cmd_blk_regs_base;
	cookie->ctrl_blk_regs_base = ctrl_blk_regs_base;
	cookie->irq = irq;
	cookie->flags = flags;
	cookie->bus = bus_count;
	cookie->type = type;
	cookie->abs_bus = abs_bus;

	result = create_sem (1, "ide_bus_mutex");
	if (result >= 0)
	{
		cookie->sem_mutex = result;

		/* initial thread count = 0; causes next acquire_sem() to block */
		result = create_sem (0, "ide_int_sem");
		if (result >= 0)
		{
			cookie->sem_intr = result;
			cookie->bus_lock_flag = 1;
			write_io_8 (ctrl_blk_regs_base, 0x0a);	/* interrupt disable */
			read_io_8 (cmd_blk_regs_base + 7);
			cookie->intgate = 1;
			result = install_io_interrupt_handler (irq, inthand, cookie, 0);
			cookie->bus_lock_flag = 0;
			if (result == B_OK)
			{
				cookie->bus_lock_flag = 1;
				write_io_8 (ctrl_blk_regs_base, 0x08);	/* interrupt enable */
				cookie->bus_lock_flag = 0;
				cookie->next = NULL;

				/* append at end of chain */
				if (first_cookie == NULL)
					first_cookie = cookie;
				else
				{
					cookie_t	*cookie2;

					for (cookie2 = first_cookie; ; cookie2 = cookie2->next)
						if (cookie2->next == NULL)
						{
							cookie2->next = cookie;
							break;
						}
				}

				dprintf (DRIVER_NAME ": register_ide_bus(%d)\n", bus_count);
				bus_count++;
				return B_OK;
			}
			delete_sem (cookie->sem_intr);
		}
		delete_sem (cookie->sem_mutex);
	}
	free (cookie);
#if DEBUG
	dprintf (DRIVER_NAME ": register_ide_bus (): error\n");
#endif
	return (status_t) result;
}

#include "enable.c"

static status_t find_devices (void)
{
	status_t	result;
	int socket;
	static cookie_t	cookie [SOCKETS];
	static bool	first = true;

#if DEBUG
	dprintf (DRIVER_NAME ": find_devices()\n");
#endif

	if (first)
	{
		build_resource_table ();
		debug_pause ("");
	}

	if (!CheckPCIC ())
	{
		dprintf (DRIVER_NAME ": no PCIC\n");
		debug_pause ("");
		return B_ERROR;
	}

	for (socket = 0; socket < SOCKETS; socket++)
	{
		if (first)
			cookie [socket].flags = 0;
		if (cookie [socket].flags == 0)
		{
			ResetSocket (socket);
			if (!CheckCard (socket))
			{
				dprintf (DRIVER_NAME ": no card in socket %d\n", socket);
				continue;
			}

			result = enable (socket, &cookie [socket]);
			debug_pause ("");
			if (result != B_OK)
				continue;
			cookie [socket].flags = 1;
		}
		/* must call register_ide_bus() every time B_MODULE_INIT is called */
		register_ide_bus (
			cookie [socket].cmd_blk_regs_base,
			cookie [socket].ctrl_blk_regs_base,
			cookie [socket].irq, ATA_ABS_BUS,
			cookie [socket].type, 0);
	}
	first = false;
	return B_OK;
}

/* uninit() */

static void unregister_ide_buses (void)
{
	cookie_t	*cookie;

#if DEBUG
	dprintf (DRIVER_NAME ": unregister_ide_buses()\n");
#endif

	while (first_cookie != NULL)
	{
		cookie = first_cookie;
		first_cookie = cookie->next;

		/* stop controller */
		write_io_8 (cookie->ctrl_blk_regs_base, 0x0a);	/* interrupt disable */

		/* remove interrupt handler */
		remove_io_interrupt_handler (cookie->irq, inthand, cookie);

		/* release memory, semaphores */
		delete_sem (cookie->sem_intr);
		delete_sem (cookie->sem_mutex);
		free (cookie);
	}
}

static status_t std_ops(int32 op, ...)
{
	switch(op) {
	case B_MODULE_INIT:
		dprintf (DRIVER_NAME ": B_MODULE_INIT\n");
#if 0
		bus_count = 0;
		if (first_cookie != NULL)
			unregister_ide_buses ();
		if (cis_area != -1)
			delete_area (cis_area);
		if (pci != NULL)
			put_module (B_PCI_MODULE_NAME);
#endif

		if ((cis_area = CreateCISArea ()) >= 0)
		{
			if (get_module (B_PCI_MODULE_NAME, (module_info **) &pci) == B_OK)
			{
#if 0
				if (register_ide_bus (0x1f0, 0x3f6, 14, B_IDE_BUS_PRIMARY,
					IDE_TYPE_IO_STD, 0)) == B_OK)
					return B_OK;
#else
				find_devices ();
				if (bus_count > 0)
					return B_OK;
#endif
				put_module (B_PCI_MODULE_NAME);
				pci = NULL;
			}
			delete_area (cis_area);
			cis_area = -1;
		}
		return B_ERROR;

	case B_MODULE_UNINIT:
		dprintf (DRIVER_NAME ": B_MODULE_UNINIT\n");
		unregister_ide_buses ();
		bus_count = 0;
		put_module (B_PCI_MODULE_NAME);
		pci = NULL;
		delete_area (cis_area);
		cis_area = -1;
		return B_OK;
	
	default:
		dprintf (DRIVER_NAME ": unknown op %d\n", op);
		return B_ERROR;
	}
}

static ide_bus_info ide_skel_info =
{
	/* bus_manager_info binfo = */
	{
		/* module_info minfo = */
		{
			"busses/ide/" DRIVER_NAME "/v0.6", 0, std_ops
		},
		rescan
	},

	/* bus management functions */
	get_nth_cookie,
	get_bus_count,
	get_abs_bus_num,
	acquire_bus,
	release_bus,

	/* task file access functions */
	write_command_block_regs,
	read_command_block_regs,
	get_altstatus,
	write_device_control,

	/* PIO transfer functions */
	write_pio_16,
	read_pio_16,

	/* synchronization functions */
	intwait,

	/* DMA functions */
	prepare_dma,
	start_dma,
	finish_dma,
	get_bad_alignment_mask,
	get_dma_mode,
	set_dma_mode
};

_EXPORT module_info *modules [] =
{
	(module_info *) &ide_skel_info, NULL
};

