#include <drivers/KernelExport.h>
#include <drivers/config_manager.h>
#include <drivers/PCI.h>
#include <drivers/isapnp.h>
#include <drivers/ISA.h>
#include <stdlib.h>
#include <stdio.h>
#include "resource.h"
#include "enum.h"

static struct config_manager_for_driver_module_info *module;

static void	register_resources (struct device_configuration *config)
{
	int i, num;
	resource_descriptor r;
	struct range	range;

	num = module->count_resource_descriptors_of_type(config, B_IRQ_RESOURCE);
	if (num)
	{
		for (i = 0; i < num; i++)
		{
			module->get_nth_resource_descriptor_of_type
				(config, i, B_IRQ_RESOURCE, &r, sizeof(resource_descriptor));
#if DEBUG
			dprintf ("irq: %04X\n", r.d.m.mask);
#endif
			add_resource_mask (&irq_mask, r.d.m.mask);
		}
	}

	num = module->count_resource_descriptors_of_type(config, B_IO_PORT_RESOURCE);
	if (num)
	{
		for (i = 0; i < num; i++)
		{
			module->get_nth_resource_descriptor_of_type
				(config, i, B_IO_PORT_RESOURCE, &r, sizeof(resource_descriptor));
#if DEBUG
			dprintf("i/o: %04x - %04x\n",
					r.d.r.minbase, r.d.r.minbase + r.d.r.len - 1);
#endif
			range.start = r.d.r.minbase;
			range.end = r.d.r.minbase + r.d.r.len - 1;
			add_resource_list (&io_list, &range);
		}
	}

	num = module->count_resource_descriptors_of_type(config, B_MEMORY_RESOURCE);
	if (num)
	{
		for (i = 0; i < num; i++)
		{
			module->get_nth_resource_descriptor_of_type
				(config, i, B_MEMORY_RESOURCE, &r, sizeof(resource_descriptor));
#if DEBUG
			dprintf("mem: %08x - %08x\n",
					r.d.r.minbase, r.d.r.minbase + r.d.r.len - 1);
#endif
			range.start = r.d.r.minbase;
			range.end = r.d.r.minbase + r.d.r.len - 1;
			add_resource_list (&mem_list, &range);
		}
	}
}

static bool	callback	(struct device_info *dinfo, struct device_configuration *config)
{
	register_resources (config);
	return false;
}

static void	enumerate_bus	(bus_type bus,
	bool (*callback)(struct device_info *, struct device_configuration *))
{
	uint64	cookie;
	struct device_info	info;
	bool	breakf = false;

	cookie = 0;
	while (!breakf && module->get_next_device_info
		(bus, &cookie, &info, sizeof (struct device_info)) == B_OK)
	{
		status_t result;
		struct device_info *dinfo;
		struct device_configuration *config;

		if (info.config_status == B_OK)
		{
			dinfo = malloc (info.size);
			if (dinfo != NULL)
			{
				module->get_device_info_for (cookie, dinfo, info.size);
				result = module->get_size_of_current_configuration_for (cookie);
				if (result >= 0)
				{
					config = malloc (result);
					if (config != NULL)
					{
						module->get_current_configuration_for (cookie, config, result);
						breakf = callback (dinfo, config);
						free (config);
					}
				}
				free (dinfo);
			}
		}
	}
}

status_t build_resource_table (void)
{
	init_resource_mask (&irq_mask);
	init_resource_list (&io_list);
	init_resource_list (&mem_list);

	if (get_module (B_CONFIG_MANAGER_FOR_DRIVER_MODULE_NAME,
		(module_info **) &module) < 0)
	{
		/* configuration manager not present: boot from floppy */
		reserve_known_resources ();
		probe_pci_devices ();
		probe_isa_devices ();
	}
	else
	{
		/* configuration manager present: boot from hard disk */
		enumerate_bus (B_PCI_BUS, callback);
		enumerate_bus (B_ISA_BUS, callback);
		put_module (B_CONFIG_MANAGER_FOR_DRIVER_MODULE_NAME);
	}

	dump_resource_list (&io_list);
	dump_resource_mask (&irq_mask);

	return B_OK;
}

