#include <drivers/KernelExport.h>
#include <drivers/PCI.h>
#include <drivers/isapnp.h>
#include <drivers/ISA.h>
#include <stdlib.h>
#include <stdio.h>
#include "resource.h"
#include "enum.h"

extern pci_module_info *pci;

/* heuristics for ISA devices */

status_t	check_free_io	(struct range *r)
{
	uint16 port = r->start, end = r->end;

	for ( ; port <= end; port++)
		if (read_io_8 (port) != 0xff)
			return B_ERROR;
	return B_OK;
}

void	probe_isa_devices	(void)
{
	area_id	area;
	uint8	*vaddr;
	uint16	*baseports;
	uint32	mask = 0;
	struct range	r;

	area = map_physical_memory ("bios_work",
		(void *) 0x0, B_PAGE_SIZE,
		B_ANY_KERNEL_ADDRESS, B_READ_AREA, 
		(void **)&vaddr);

	if (area >= 0)
	{
		baseports = (uint16 *)(vaddr + 0x400);
		dprintf (DRIVER_NAME ": COM1=0x%x, COM2=0x%x, LPT1=0x%x\n",
			baseports [0], baseports [1], baseports [4]);
		if (baseports [0] != 0)	/* COM1 */
		{
			r.start = baseports [0];
			r.end = r.start + 7;
			add_resource_list (&io_list, &r);
			mask |= 1 << 4;
		}
		if (baseports [1] != 0)	/* COM2 */
		{
			r.start = baseports [1];
			r.end = r.start + 7;
			add_resource_list (&io_list, &r);
			mask |= 1 << 3;
		}
		if (baseports [4] != 0)	/* LPT1 */
		{
			r.start = baseports [4];
			r.end = r.start + 7;
			add_resource_list (&io_list, &r);
			mask |= 1 << 7;
		}
		delete_area (area);
	}
	else
	{
		/* COM1 */
		r.start = 0x3f8;
		r.end = 0x3ff;
		if (check_free_io (&r) == B_ERROR)
		{
			add_resource_list (&io_list, &r);
			mask |= 1 << 4;
		}

		/* COM2 */
		r.start = 0x2f8;
		r.end = 0x2ff;
		if (check_free_io (&r) == B_ERROR)
		{
			add_resource_list (&io_list, &r);
			mask |= 1 << 3;
		}

		/* LPT1 */
		r.start = 0x378;
		r.end = 0x37f;
		if (check_free_io (&r) == B_ERROR)
		{
			add_resource_list (&io_list, &r);
			mask |= 1 << 7;
		}

		/* LPT1 (MDA) */
		r.start = 0x3bc;
		r.end = 0x3bf;
		if (check_free_io (&r) == B_ERROR)
		{
			add_resource_list (&io_list, &r);
			mask |= 1 << 7;
		}
	}

	/* Sound Blaster */
	r.start = 0x220;
	r.end = 0x22f;
	if (check_free_io (&r) == B_ERROR)
	{
		add_resource_list (&io_list, &r);
		mask |= 1 << 5;
	}

	add_resource_mask (&irq_mask, mask);
}

void	reserve_known_resources (void)
{
	int	i;
	static struct range reserved_io [] =
	{
		{ 0x00, 0xff },		/* system devices */
		{ 0x1f0, 0x1f7 },	/* primary IDE */
		{ 0x3f2, 0x3f7 },	/* FDC and primary IDE */
		{ 0x170, 0x177 },	/* secondary IDE */
		{ 0x376, 0x377 },	/* secondary IDE */
		{ 0x200, 0x20f },	/* game port */
		{ 0x278, 0x27f },	/* LPT2 & ISA PnP enumeration */
		{ 0x330, 0x331 },	/* MPU-401 compatible MIDI port */
		{ 0x388, 0x38b },	/* AdLib compatible FM synth */
		{ 0x3b0, 0x3bb },	/* MDA */
		{ 0x3c0, 0x3df },	/* VGA */
		{ 0x3e0, 0x3e7 },	/* i82365 compatible PCIC */
		{ 0, 0 }
	};

	/* IRQ15,14,13,12,8,6,2,1,0 */
	add_resource_mask (&irq_mask, 0xf147);

	/* register reserved I/O ranges */
	for (i = 0; ; i++)
	{
		if (reserved_io [i].end == 0)
			break;
		add_resource_list (&io_list, &reserved_io [i]);
	}
}

void	probe_pci_devices	(void)
{
	int	i, j, irq;
	struct range	r;
	pci_info	pinfo;
	uint32	mask = 0;

	/* fill PCI resources */
	for (i = 0; ; i++)
	{
		if (pci->get_nth_pci_info (i, &pinfo) != B_OK)
			break;

		irq = pinfo.u.h0.interrupt_line;
		if (1 <= irq && irq <= 15)
			mask |= 1 << irq;
		switch (pinfo.header_type & 0x7f)
		{
		case 0:		/* generic PCI devices */
			for (j = 0; j < 6; j++)
			{
				if ((pinfo.u.h0.base_register_flags [i] & 1) == 1 &&
					pinfo.u.h0.base_register_sizes [i] <= 256)
				{
					r.start = pinfo.u.h0.base_registers [i];
					r.end = r.start + pinfo.u.h0.base_register_sizes [i] - 1;
					add_resource_list (&io_list, &r);
				}
			}
			break;
		case 1:		/* PCI-PCI bridges */
#if 0
			for (j = 0; j < 2; j++)
			{
				if (pinfo.u.h1.base_register_flags [i] & 1)
				{
					r.start = pinfo.u.h1.base_registers [i];
					r.end = r.start + pinfo.u.h1.base_register_sizes [i];
					add_resource_list (&io_list, &r);
				}
			}
#endif
			break;
		case 2:		/* PCI-CardBus bridges */
			break;
		}
	}

	add_resource_mask (&irq_mask, mask);
}
