#include <drivers/KernelExport.h>
#include <kernel/image.h>
#include <stdio.h>
#include "ioport.h"

#include "config.h"
#include "pcmcia.h"
#include "i82365reg.h"
#include "ss_cs.h"

#ifndef	DRIVER_NAME
#define	DRIVER_NAME	"atacard"
#endif

#define	CIS_MEM_WIN_ID	4
#define	CIS_AREA_NAME	"cis_area"

static void WritePCICReg (int socket, int index, int data)
{
	write_io_8 (PCIC_PORT_BASE, index | (socket << 6));
	spin (10);
	write_io_8 (PCIC_PORT_BASE + 1, data);
	spin (10);
}

static void WritePCICRegPair (int socket, int index, int data)
{
	WritePCICReg (socket, index,     data & 0xFF);
	WritePCICReg (socket, index + 1, data >> 8);
}

static int ReadPCICReg (int socket, int index)
{
	int	data;

	write_io_8 (PCIC_PORT_BASE, index | (socket << 6));
	spin (10);
	data = read_io_8 (PCIC_PORT_BASE + 1);
	spin (10);
	return data;
}

static int ReadPCICRegPair (int socket, int index)
{
	return
		 ReadPCICReg (socket, index    ) |
		(ReadPCICReg (socket, index + 1) << 8);
}

void DumpPCICRegs (int socket)
{
	char	buf [256];

	sprintf (buf, "%02X %02X %02X %02X %02X %02X %02X %02X\n",
		ReadPCICReg (socket, PCIC_ID_REV),
		ReadPCICReg (socket, PCIC_STATUS),
		ReadPCICReg (socket, PCIC_POWER),
		ReadPCICReg (socket, PCIC_INT_GEN),
		ReadPCICReg (socket, PCIC_STAT_CHG),
		ReadPCICReg (socket, PCIC_STAT_INT),
		ReadPCICReg (socket, PCIC_ADDRWINE),
		ReadPCICReg (socket, PCIC_IOCTL));
	dprintf (buf);
	sprintf (buf, "I/O: %04X-%04X %04X-%04X\n",
		ReadPCICRegPair (socket, PCIC_IO0    ),
		ReadPCICRegPair (socket, PCIC_IO0 + 2),
		ReadPCICRegPair (socket, PCIC_IO1    ),
		ReadPCICRegPair (socket, PCIC_IO1 + 2));
	dprintf (buf);
	sprintf (buf, "mem: %04X-%04X %04X-%04X %04X-%04X %04X-%04X %04X-%04X\n",
		ReadPCICRegPair (socket, PCIC_MEMBASE0    ),
		ReadPCICRegPair (socket, PCIC_MEMBASE0 + 2),
		ReadPCICRegPair (socket, PCIC_MEMBASE1    ),
		ReadPCICRegPair (socket, PCIC_MEMBASE1 + 2),
		ReadPCICRegPair (socket, PCIC_MEMBASE2    ),
		ReadPCICRegPair (socket, PCIC_MEMBASE2 + 2),
		ReadPCICRegPair (socket, PCIC_MEMBASE3    ),
		ReadPCICRegPair (socket, PCIC_MEMBASE3 + 2),
		ReadPCICRegPair (socket, PCIC_MEMBASE4    ),
		ReadPCICRegPair (socket, PCIC_MEMBASE4 + 2));
	dprintf (buf);
}

bool CheckPCIC (void)
{
	return (ReadPCICReg (0, PCIC_ID_REV) & 0xF0) == 0x80;
}

bool CheckCard (int socket)
{
	if ((ReadPCICReg (socket, PCIC_STATUS) & PCIC_CD) == PCIC_CD)
		return true;
	return false;
}

bool CheckCardChange (int socket)
{
	if ((ReadPCICReg (socket, PCIC_STAT_CHG) & PCIC_CDTCH) == PCIC_CDTCH)
		return true;
	return false;
}

void WriteScratchPad (int socket, int block, int value)
{
	int	index, data;

	index = PCIC_MEMBASE0 + 1 + block * 8;
	data = ReadPCICReg (socket, index);
	data = (data & 0xCF) | ((value & 3) << 4);
	WritePCICReg (socket, index, data);
}

int ReadScratchPad (int socket, int block)
{
	int	index, data;

	index = PCIC_MEMBASE0 + 1 + block * 8;
	data = ReadPCICReg (socket, index);
	return (data & 0x30) >> 4;
}

void ResetSocket (int socket)
{
	int	i;

	WritePCICReg (socket, PCIC_POWER, PCIC_DISRST /*| PCIC_APSENA*/ | PCIC_PCPWRE | PCIC_VCC_5V | PCIC_VPP_5V);
	/* memory card mode may cause hang-up */
	WritePCICReg (socket, PCIC_INT_GEN, PCIC_IOCARD);
	WritePCICReg (socket, PCIC_POWER, PCIC_OUTENA | PCIC_DISRST /*| PCIC_APSENA*/ | PCIC_PCPWRE | PCIC_VCC_5V | PCIC_VPP_5V);
	spin (100*1000);
	WritePCICReg (socket, PCIC_INT_GEN, PCIC_IOCARD | PCIC_CARDRESET);
	spin (100*1000);
	WritePCICReg (socket, PCIC_ADDRWINE, 0);
	WritePCICReg (socket, PCIC_STAT_INT, 0x0f);
	ReadPCICReg (socket, PCIC_STAT_CHG);
	if (!CheckCard (socket)) return;
	for (i = 0; i < 20; i++)
	{
		if (ReadPCICReg (socket, PCIC_STATUS) & PCIC_READY)
			break;
		dprintf (DRIVER_NAME ": ResetSocket(): waiting for card ready\n");
		spin (100*1000);
	}
}

void CloseMemory (int socket, int win)
{
	int	x;

	x = ReadPCICReg (socket, PCIC_ADDRWINE);
	WritePCICReg (socket, PCIC_ADDRWINE, x & ~(1 << win));
}

void OpenMemory (int socket, int win, int type, uint32 base_addr, uint32 len, uint32 card_addr)
{
	int	x;
	uint32	offset = ((card_addr - base_addr) >> 12) & 0x3FFF;

	/* close window first to avoid anomaly */
	x = ReadPCICReg (socket, PCIC_ADDRWINE);
	WritePCICReg (socket, PCIC_ADDRWINE, x & ~(1 << win));

	WritePCICRegPair (socket, PCIC_MEMBASE0 + 8 * win,
		(base_addr >> 12) | 0x8000);
	WritePCICRegPair (socket, PCIC_MEMBASE0 + 2 + 8 * win, 
		((base_addr + len - 1) >> 12) | 0xc000);
	WritePCICRegPair (socket, PCIC_MEMBASE0 + 4 + 8 * win, 
		offset | (type << 14));

	WritePCICReg (socket, PCIC_ADDRWINE, x | (1 << win) | PCIC_MEMCS16);
}

area_id CreateCISArea (void)
{
	area_id	area;
	void	*vaddr;

	area = map_physical_memory (CIS_AREA_NAME,
		(void *)CIS_MEM_WIN_BASE, CIS_MEM_WIN_SIZE,
		B_ANY_KERNEL_ADDRESS, B_WRITE_AREA | B_READ_AREA, 
		(void **)&vaddr);
	return area;
}

status_t CopyCIS (int socket, uint8 *buf)
{
	area_id	aid;
	area_info	ainfo;
	uint8	*vaddr;
	int	i;

	aid = find_area (CIS_AREA_NAME);
	if (aid < 0)
		return B_ERROR;
	get_area_info (aid, &ainfo);
	vaddr = ainfo.address;
	OpenMemory (socket, CIS_MEM_WIN_ID, ATTR_MEM, CIS_MEM_WIN_BASE, 
		CIS_MEM_WIN_SIZE, 0x00000000);
	clear_caches (vaddr, CIS_MEM_WIN_SIZE, B_INVALIDATE_DCACHE);
	/* memcpy() may fail to read IBM Ethernet II */
	for (i = 0; i < CIS_MEM_WIN_SIZE / 2; i++)
		buf [i] = vaddr [i * 2];
	CloseMemory (socket, CIS_MEM_WIN_ID);
	return B_OK;
}

void RequestIO (int socket, uint16 IOBasePort1, int NumPorts1, uint16 IOBasePort2, int NumPorts2)
{
	int	x;

	/* close windows first to avoid anomaly */
	x = ReadPCICReg (socket, PCIC_ADDRWINE) & ~(PCIC_IO0_EN | PCIC_IO1_EN);
	WritePCICReg (socket, PCIC_ADDRWINE, x);

	WritePCICRegPair (socket, PCIC_IO0, IOBasePort1);
	WritePCICRegPair (socket, PCIC_IO0 + 2, IOBasePort1 + NumPorts1 - 1);
	x |= PCIC_IO0_EN;
	if (NumPorts2 > 0)
	{
		WritePCICRegPair (socket, PCIC_IO1, IOBasePort2);
		WritePCICRegPair (socket, PCIC_IO1 + 2, IOBasePort2 + NumPorts2 - 1);
		x |= PCIC_IO1_EN;
	}

	WritePCICReg (socket, PCIC_IOCTL,
		  PCIC_IO_CS16 | PCIC_IO_16BIT | PCIC_IO_WS |
		((PCIC_IO_CS16 | PCIC_IO_16BIT | PCIC_IO_WS) << 4));
	WritePCICReg (socket, PCIC_ADDRWINE, x);
}

void RequestIRQ (int socket, int IRQ)
{
	int	x;

	x = ReadPCICReg (socket, PCIC_INT_GEN);
	WritePCICReg (socket, PCIC_INT_GEN, (x & 0xF0) | IRQ);
}

status_t RequestConfiguration (int socket, uint32 ConfigBase, int ConfigIndex, int CSR_Value)
{
	int	offset = ConfigBase & (CIS_MEM_WIN_SIZE - 1);
	area_id	aid;
	area_info	ainfo;
	uint8	*vaddr;

	aid = find_area (CIS_AREA_NAME);
	if (aid < 0)
		return B_ERROR;
	get_area_info (aid, &ainfo);
	vaddr = ainfo.address;
	OpenMemory (socket, CIS_MEM_WIN_ID, ATTR_MEM, CIS_MEM_WIN_BASE, 
		CIS_MEM_WIN_SIZE, ConfigBase);
		/* implicitly aligned to page boundary */
	clear_caches (vaddr, CIS_MEM_WIN_SIZE, B_INVALIDATE_DCACHE);
	vaddr [offset] = ConfigIndex | 0x40;
	vaddr [offset + 2] = CSR_Value;
	CloseMemory (socket, CIS_MEM_WIN_ID);
	return B_OK;
}

/*
	limitations:
	long link tuple and no link tuple are not processed
	default long link does not occur
	does not proceed to next page
	TupleBufLen ignored
	multiple instances of same type not supported
*/
status_t GetSingleTuple (int socket, int TupleID, int nth, uint8 *TupleBuf, int TupleBufLen)
{
	int	i, id, link, offset = 0, count = 0, factor = 2, nullcnt = 0;
	bool	readable = false;
	area_id	aid;
	area_info	ainfo;
	uint8	*vaddr;

	dprintf (DRIVER_NAME ": GetSingleTuple(%X,%X,%d)\n", TupleID, TupleBuf, TupleBufLen);
	aid = find_area (CIS_AREA_NAME);
	if (aid < 0)
		return B_DEV_NO_MEMORY;
	get_area_info (aid, &ainfo);
	vaddr = ainfo.address;
	OpenMemory (socket, CIS_MEM_WIN_ID, ATTR_MEM, CIS_MEM_WIN_BASE, 
		CIS_MEM_WIN_SIZE, 0x00000000);
	clear_caches (vaddr, CIS_MEM_WIN_SIZE, B_INVALIDATE_DCACHE);
#if 0
	dprintf (DRIVER_NAME ": CIS dump: ");
#endif
	id = vaddr [0];
	for (i = 0; i < 16; i++)
	{
#if 0
		dprintf ("%02X ", vaddr [i * factor]);
#endif
		if (vaddr [i * factor] != id)
			readable = true;
	}
#if 0
	dprintf ("\n");
#endif
	if (id != CISTPL_NULL && id != CISTPL_DEVICE && id != CISTPL_END)
	{
		dprintf (DRIVER_NAME ": GetSingleTuple(): invalid first tuple\n");
		CloseMemory (socket, CIS_MEM_WIN_ID);
		return B_DEV_UNREADABLE;
	}
	if (!readable)
	{
		dprintf (DRIVER_NAME ": GetSingleTuple(): CIS unreadable or not exist\n");
		CloseMemory (socket, CIS_MEM_WIN_ID);
		return B_DEV_UNREADABLE;
	}

#if 0
	dprintf (DRIVER_NAME ": ");
#endif
	while (count < 2048)
	{
		id = vaddr [offset];
		if (id == CISTPL_END)
		{
#if 0
			dprintf ("CISTPL_END ");
#endif
			break;
		}
		if (id == CISTPL_NULL)
		{
#if 0
			dprintf ("CISTPL_NULL, ");
#endif
			offset += factor;
			count++;
			if (++nullcnt == 8)
			{
				dprintf ("too many contiguous CISTPL_NULL");
				break;
			}
		}
		else
		{
			nullcnt = 0;
			link = vaddr [offset + factor];
#if 0
			dprintf ("tuple %X link %X, ", id, link);
#endif
			if (TupleID == id && --nth <= 0)
			{
				TupleBuf [0] = id;
				TupleBuf [1] = link;
				for (i = 0; i < link; i++)
					TupleBuf [2 + i] = vaddr [offset + (2 + i) * factor];
				break;
			}
			offset += (2 + link) * factor;
			count += 2 + link;
		}
	}
#if 0
	dprintf ("\n" DRIVER_NAME ": GetSingleTuple(): exit\n");
#endif
	CloseMemory (socket, CIS_MEM_WIN_ID);
	return TupleID == id ? B_OK : B_ENTRY_NOT_FOUND;
}


