//
// BeOS bits for bebeeb
// Sets up a BeOS front end, then spawns the
//  emulator as a thread, a la UAE.
// 
// J. Belson	1998.07.05
//


#include <time.h>

#include "Beos.h"


//
// Instances of our classes...
//
static bbc_audio *audio;
static BitmapView *bitmap_view;
static bebeeb_window *bitmap_window;
static bebeeb *the_app;


static int the_argc;
static char **the_argv;

//
// New main entry point
//
int main(int argc, char **argv)
{
	the_argc = argc;
	the_argv = new char *[argc];
	for (int i=0; i<argc; i++) {
		the_argv[i] = /*strdup*/ (argv[i]);
	}

	request_disk_change = false;
	restart_thread = false;

	the_app = new bebeeb;
	the_app->Run();
	
	delete the_app;
	return 0;
}


// Initialise audio...
int bbc_audio::init(void)
{
	// Initialise sound...
	subscriber = new BSubscriber("BBC sound emulation");
	stream = new BDACStream();
	if (B_OK != (subscriber->Subscribe(stream))) {
		puts("Error initialising audio");
	}
	status_t s = subscriber->EnterStream(NULL,
										true,		// Head of list 
										this,		// User data
										stream_func,	// Stream function
										NULL,		// Completion function
										true);		// Run in background
	if (s != B_OK) {
		puts("EnterStream() returned an error");
	}

	// Switch off all channels
	for (int i=0; i<NUM_CHANNELS; i++) {
		channel_info[i].playing = false;
	}

	stream->SetSamplingRate(SAMPLE_RATE);
}


void bbc_audio::play_note(int channel, unsigned int pitch, int volume)
{
	//printf("channel = %d, pitch = %d, volume = %d\n", channel, pitch, volume);

	if (channel < NUM_CHANNELS) {
		channel_info[channel].playing	= true;
		channel_info[channel].count		= 0;
		channel_info[channel].volume	= volume;
		channel_info[channel].state		= 1;
		channel_info[channel].step		= int (float (SAMPLE_RATE) / float (pitch))/2;
	} else {
		puts("channel too high");
	}

}

void bbc_audio::stop_note(int channel)
{
	if (channel < NUM_CHANNELS) {
		channel_info[channel].playing = false;
	} else {
		puts("channel too high");
	}
}


bool bbc_audio::play_next(char *buffer, long count)
{
	standard_frame *sound_data;
	int32  frame_count;

	for (int channel=0; channel<NUM_CHANNELS; channel++) {
		str_channel_info *ch = &channel_info[channel];
		if (ch->playing) {
			frame_count = count/sizeof(standard_frame);
			sound_data = (standard_frame *) buffer;
			while (frame_count-- >= 0) {
				int32 val;
				if (channel%2) {	// Weedy attempt at stereo...
					val = sound_data->left;
				} else {
					val = sound_data->right;
				}

				// Next state...
				val += ((ch->state) ? ch->volume * 1024 : -ch->volume * 1024);

				// Clip to amplitude limit...
				if (val > 0x7fff) {
					val = 0x7fff;
				} else if (val < -0x7fff) {
					val = -0x7fff;
				}
				if (channel%2) {
					sound_data->left = val;
				} else {
					sound_data->right = val;
				}

				// Switch state of wave
				if (channel) {
					if (!(ch->count % ch->step)) {
						ch->state = 1 - ch->state;
					}
				} else {
					if (!(ch->count % ch->step)) {
						ch->state = (rand()%2) ? 0 : 1;
					}
				}
				ch->count++;
				sound_data++;
			}
		}
	}

	return true;
}


// Static member function used as callback...
bool bbc_audio::stream_func(void *user_data, char *buffer, size_t count, void *header)
{
	bbc_audio *snd = (bbc_audio *) user_data;
	return snd->play_next(buffer, count);
}


bbc_audio::~bbc_audio()
{
	delete stream;
	delete subscriber;
}


bebeeb::bebeeb() : BApplication('beeb')
{
	app_info info;
	BPath path;
	GetAppInfo(&info);
	BEntry entry(&info.ref);
	entry.GetPath(&path);
	path.GetParent(&path);
	chdir(path.Path());

	// Check for existance of required roms...
	FILE *fp;
	bool no_error = true;
	if (NULL==(fp = fopen("roms/BASIC2.rom", "r"))) {
		char buffer[256];
		sprintf(buffer, "\n         I can't find the file\n"
						"\n            'BASIC2.rom'\n"
						"\n         in the roms directory.\n");
		BAlert *the_alert = new BAlert("", buffer, "Aww!");
		the_alert->Go();
		no_error = false;
	} else {
		fclose(fp);
	}
	if (NULL==(fp = fopen("roms/OS1.2p1.rom", "r"))) {
		char buffer[256];
		sprintf(buffer, "\n          I can't find the file\n"
						"\n             'OS1.2p1.rom'\n"
						"\n          in the roms directory.\n");
		BAlert *the_alert = new BAlert("", buffer, "Aww!");
		the_alert->Go();
		no_error = false;
	} else {
		fclose(fp);
	}


	// Open the main window...
	if (no_error) {
		bitmap_window = new bebeeb_window(BRect(0, 0, WIDTH, HEIGHT));
	}
	
	// Instantiate audio emulation...
	audio = new bbc_audio;
	audio->init();
}


// Spawn the emulation thread...
void bebeeb::ReadyToRun(void)
{
	the_thread = spawn_thread(thread_func, "BeBeeb", B_NORMAL_PRIORITY, this);
	resume_thread(the_thread);
}

void bebeeb::RestartThread(void)
{
	// Tell emulator thread to quit itself...
	QuitEmulator = 1;

	// ..wait for it to finish..
	status_t l;
	wait_for_thread(the_thread, &l);

	// ..then restart it
	bitmap_view->ClearScreen();
	the_thread = spawn_thread(thread_func, "BeBeeb", B_NORMAL_PRIORITY, this);
	resume_thread(the_thread);
}


bool bebeeb::QuitRequested(void)
{
	// Tell emulator thread to quit itself...
	QuitEmulator = 1;

	status_t l;
	wait_for_thread(the_thread, &l);
	return true;
}


long bebeeb::thread_func(void *obj)
{
	real_main(the_argc, the_argv);
	return 0;
}


bebeeb_window::bebeeb_window(BRect frame) :
		BWindow(frame, "bebeeb", B_TITLED_WINDOW,
		B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
	Lock();
	MoveTo(80, 80);

	// Set up menus
	BMenuBar *menubar = new BMenuBar(Bounds(), "MB");
	BMenu *a_menu = new BMenu("BeBeeb");
	a_menu->AddItem(new BMenuItem("About BeBeeb", new BMessage(MSG_ABOUT)));
	a_menu->AddItem(new BSeparatorItem);
	a_menu->AddItem(new BMenuItem("Preferences...", new BMessage(MSG_PREFS)));
	a_menu->AddItem(new BSeparatorItem);
	a_menu->AddItem(new BMenuItem("Mount disk...", new BMessage(MSG_MOUNT)));
	a_menu->AddItem(new BMenuItem("Reset machine", new BMessage(MSG_RESET)));
	a_menu->AddItem(new BSeparatorItem);
	a_menu->AddItem(new BMenuItem("Quit", new BMessage(MSG_QUIT)));
	menubar->AddItem(a_menu);
	AddChild(menubar);

	// Make room for menu bar...
	float mb_height = menubar->Bounds().Height() + 1;
//	ResizeBy(0, mb_height);

	// Create bitmap view beneath menubar...
	BRect BitmapRect = frame;
	BitmapRect.top += mb_height;

	// Create our bitmap class
	bitmap_view = new BitmapView(BitmapRect);
	AddChild(bitmap_view);
	bitmap_view->MakeFocus();

	// Allocate palette
	BScreen scr(bitmap_window);
	for (int i=0; i<16; i++) {
		Cells[i] = scr.IndexForColor(rgb_values[i][0], rgb_values[i][1], rgb_values[i][2]);
	}
	
	// Show window
	Show();
	Unlock();
}


void bebeeb_window::MessageReceived(BMessage *msg)
{
	BMessage *msg2;

	switch (msg->what) {
		case MSG_RESET:
			restart_thread = false;
			the_app->RestartThread();
			break;

		case MSG_QUIT:
			be_app->PostMessage(B_QUIT_REQUESTED);
			break;

		case MSG_PREFS:
			char str[256];
			sprintf(str, "Sorry - Preferences not implemented yet\n");
			BAlert *the_alert = new BAlert("", str, "OK");
			the_alert->Go();
			break;

		case MSG_ABOUT: {
			for (int i=0; i<4; i++) BeStopNote(i);
			char str[256];
			sprintf(str, "     BeBeeb - a BBC Micro emulator\n"
				     	"             by James Fidell       \n\n"
				     	"      BeOS port by Jonathan Belson\n");
			BAlert *the_alert = new BAlert("", str, "OK");
			the_alert->Go();
			break;
		}

		case MSG_MOUNT:
			request_disk_change = true;
			break;

		case MSG_DISK_CHANGED:
			{
			entry_ref ref;
			BEntry entry;
			entry_ref open_dir;
			BPath my_path;

			if (msg->HasRef("refs")) {
				char dir_name[512];
				msg->FindRef("refs", &ref);
				entry.SetTo(&ref);				// Get filename..
				entry.GetParent(&entry);
				entry.GetRef(&open_dir);
				entry.GetPath(&my_path);		// Get path..
				if ((strlen(my_path.Path()) + 1 + strlen(ref.name)) < sizeof(dir_name)) {
					strcpy(dir_name, my_path.Path());
					strcat(dir_name, "/");
					strcat(dir_name, ref.name);
					// Sent directory path to EFS emulation...
					ChangeDiskDirectory((char *) dir_name);
				} else {
					puts("Error - dirpath name too long");
				}
			}
			}
			break;

		default:
			//puts("Unknown message");
			BWindow::MessageReceived(msg);
			break;
	}
}


bool bebeeb_window::QuitRequested(void)
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return false;
}



BitmapView::BitmapView(BRect frame) : BView(frame, "", B_FOLLOW_NONE, B_WILL_DRAW)
{
	// Make bitmaps of required size...
	the_bitmap[0] = new BBitmap(BRect(0, 0, WIDTH, HEIGHT), B_COLOR_8_BIT);
	the_bitmap[1] = new BBitmap(BRect(0, 0, WIDTH, HEIGHT), B_COLOR_8_BIT);

	// Clear bitmap...
	char *bits = (char *) the_bitmap[0]->Bits();
	memset(bits, Cells[0], WIDTH*HEIGHT);

	bytes_per_row = the_bitmap[0]->BytesPerRow();
	bitmap_data = (uchar *) the_bitmap[0]->Bits();
}


//
// Draw from BitmapView bitmap to deestination view...
//
void BitmapView::Draw(BRect update)
{
	DrawBitmap(the_bitmap[current], update, update);
}

// Render string into our bitmap...
void BitmapView::DrawImageString(int x, int y, unsigned char *text, int len)
{
	// Centralise mode 7 display
	x += OFF_X;
	y += OFF_Y;
	unsigned char *line_start = bitmap_data + x + (y*bytes_per_row);

	bitmap_window->Lock();
	for (int i=0; i<len; i++) {					// For each character...
		unsigned char *char_start = line_start + (i*CHAR_WIDTH);
		for (int height=0; height<CHAR_HEIGHT; height++) {
			unsigned short datum = current_font[( (text[i] - 32)*19L ) + height];
			for (int width=0; width<CHAR_WIDTH; width++) {
				unsigned char val = 0x0;
				if (datum&0x8000) {
					val = ttext_foreground;
				} else {
					val = ttext_background;
				}
				*(char_start + width + (height*bytes_per_row)) = val;
				datum <<= 1;		// Next bit
			}
		}
	}
	bitmap_view->Draw(BRect(x, y, x+len*CHAR_WIDTH, y + 19));

	bitmap_window->Unlock();
}

void BitmapView::ClearScreen(void)
{
	bitmap_window->Lock();
	memset(bitmap_data, Cells[0], WIDTH*HEIGHT);
	bitmap_view->Draw(BRect(0, 0, WIDTH, HEIGHT));
	bitmap_window->Unlock();
}


// Draw mode 7 cursor...
void BitmapView::FillRect(int x, int y, int width, int depth)
{
	x += OFF_X;
	y += (OFF_Y + CHAR_HEIGHT);
	unsigned char *cursor_start = bitmap_data + x + (y*bytes_per_row);
	for (int i=0; i < width; i++) {
		*(cursor_start + i) = ttext_foreground;
	}

	if (bitmap_window->Lock()) {
		bitmap_view->Draw(BRect(x, y, x + width, y + depth));
		bitmap_window->Unlock();
	} else {
		// Currently quitting, so Lock() has failed...
	}
}

void BitmapView::PutImage(int Xsize, int Ysize)
{
	bitmap_window->Lock();
	bitmap_view->Draw(BRect(0, 0, WIDTH, HEIGHT));
	bitmap_window->Unlock();
}


//
// Handle key events...
//
void BitmapView::KeyDown(const char *bytes, int32 numBytes)
{
	if (numBytes == 1) {
		unsigned char key = switch_key(bytes);
		if (key != -1) {
			KeyboardMatrixUpdate(key, KEY_PRESSED);
		} else {
			//puts("Unknown key pressed");
		}
	}
}

void BitmapView::KeyUp(const char *bytes, int32 numBytes)
{
	if (numBytes == 1) {
		unsigned char key = switch_key(bytes);
		if (key != -1) {
			KeyboardMatrixUpdate(key, KEY_RELEASED);
		} else {
			//puts("Unknown key released");
		}
	}
}

unsigned char BitmapView::switch_key(const char *bytes)
{
	unsigned char key;

	switch (bytes[0]) {
		case B_ENTER:		key = KEY_RETURN;		break;
		case B_TAB:			key = KEY_TAB;			break;
		case B_ESCAPE:		key = KEY_ESCAPE;		break;
		case B_DELETE:
		case B_BACKSPACE:	key = KEY_DELETE;		break;
		case B_INSERT:		key = KEY_COPY;			break;
		case ';':
		case ':':			key = KEY_SEMICOLON;	break;
		case '\'':
		case '@':			key = KEY_COLON;		break;
		case ' ':			key = KEY_SPACE;		break;
		case '<':
		case ',':			key = KEY_COMMA;		break;
		case '>':
		case '.':			key = KEY_PERIOD;		break;
		case '\\':			key = KEY_BACKSLASH;	break;
		case '?':
		case '/':			key = KEY_SLASH;		break;
		case '-':
		case '_':			key = KEY_MINUS;		break;
		case '=':			key = KEY_HAT;			break;

		// Function keys...
		case B_FUNCTION_KEY: {
			BMessage *m = Window()->CurrentMessage();
			if (m) {
				int32 k;
				m->FindInt32("key", &k);
				switch (k) {
					case B_F12_KEY:
						BreakKeypress = 1 - BreakKeypress;
						break;

					// Functions keys...
					case B_F1_KEY:	key = KEY_F0;		break;
					case B_F2_KEY:	key = KEY_F1;		break;
					case B_F3_KEY:	key = KEY_F2;		break;
					case B_F4_KEY:	key = KEY_F3;		break;
					case B_F5_KEY:	key = KEY_F4;		break;
					case B_F6_KEY:	key = KEY_F5;		break;
					case B_F7_KEY:	key = KEY_F6;		break;
					case B_F8_KEY:	key = KEY_F7;		break;
					case B_F9_KEY:	key = KEY_F8;		break;
					case B_F10_KEY:	key = KEY_F9;		break;
					default:
						key =-1;
				}
			}
			break;
		}

		// Editing keys...
		case B_LEFT_ARROW:	key = KEY_LEFT;	break;
		case B_RIGHT_ARROW:	key = KEY_RIGHT;break;
		case B_UP_ARROW:	key = KEY_UP;	break;
		case B_DOWN_ARROW:	key = KEY_DOWN;	break;

		// Alpha...
		case 'a': case 'A':	key = KEY_A;	break;
		case 'b': case 'B':	key = KEY_B;	break;
		case 'c': case 'C':	key = KEY_C;	break;
		case 'd': case 'D':	key = KEY_D;	break;
		case 'e': case 'E':	key = KEY_E;	break;
		case 'f': case 'F':	key = KEY_F;	break;
		case 'g': case 'G':	key = KEY_G;	break;
		case 'h': case 'H':	key = KEY_H;	break;
		case 'i': case 'I':	key = KEY_I;	break;
		case 'j': case 'J':	key = KEY_J;	break;
		case 'k': case 'K':	key = KEY_K;	break;
		case 'l': case 'L':	key = KEY_L;	break;
		case 'm': case 'M':	key = KEY_M;	break;
		case 'n': case 'N':	key = KEY_N;	break;
		case 'o': case 'O':	key = KEY_O;	break;
		case 'p': case 'P':	key = KEY_P;	break;
		case 'q': case 'Q':	key = KEY_Q;	break;
		case 'r': case 'R':	key = KEY_R;	break;
		case 's': case 'S':	key = KEY_S;	break;
		case 't': case 'T':	key = KEY_T;	break;
		case 'u': case 'U':	key = KEY_U;	break;
		case 'v': case 'V':	key = KEY_V;	break;
		case 'w': case 'W':	key = KEY_W;	break;
		case 'x': case 'X':	key = KEY_X;	break;
		case 'y': case 'Y':	key = KEY_Y;	break;
		case 'z': case 'Z':	key = KEY_Z;	break;

		// Numeric...
		case '0':
		case ')':	key = KEY_0;	break;
		case '1':
		case '!':	key = KEY_1;	break;
		case '2':
		case '"':	key = KEY_2;	break;
		case '3':	key = KEY_3;	break;
		case '4':
		case '$':	key = KEY_4;	break;
		case '5':
		case '%':	key = KEY_5;	break;
		case '6':
		case '^':	key = KEY_6;	break;
		case '7':
		case '&':	key = KEY_7;	break;
		case '8':
		case '*':	key = KEY_8;	break;
		case '9':
		case '(':	key = KEY_9;	break;

		default:
			key = -1;
	}
	return key;
}


// Check for modifier key presses, requests for disk image
//  changes etc..
void BitmapView::CheckEvents(void)
{
	if (request_disk_change) {
		request_disk_change = false;
		if (disk_dir) {
			disk_dir->Show();
		} else {
			disk_dir = new BFilePanel(B_OPEN_PANEL,
								NULL,				// target
								NULL,
								B_DIRECTORY_NODE,	// node_flavours
								false,				// multiple selection
								new BMessage(MSG_DISK_CHANGED), NULL,	// message, filter
								true,				// modal
								true);				// hide-when-done
			disk_dir->Window()->SetTitle("Select disk directory...");
			disk_dir->SetButtonLabel(B_DEFAULT_BUTTON, "Select as disk");
			disk_dir->SetPanelDirectory(".");
			disk_dir->SetTarget(bitmap_window);
			disk_dir->Show();
		}
	}

	// Read modifier keys...
	uint32 mod = modifiers();

	// Current states of modifier keys..
	static bool shift;
	if (mod & B_SHIFT_KEY) {
		if (!shift) {
			KeyboardMatrixUpdate(KEY_SHIFT, KEY_PRESSED);
			shift = true;
		}
	} else {
		if (shift) {
			shift = false;
			KeyboardMatrixUpdate(KEY_SHIFT, KEY_RELEASED);
		}
	}

	static bool ctrl;
	if (mod & B_CONTROL_KEY) {
		KeyboardMatrixUpdate(KEY_CTRL, KEY_PRESSED);
	} else {
		KeyboardMatrixUpdate(KEY_CTRL, KEY_RELEASED);
	}

	static bool caps_on = false;
	if (mod & B_CAPS_LOCK) {
		if (!caps_on) {
			KeyboardMatrixUpdate(KEY_CAPSLOCK, KEY_PRESSED);
			caps_on = true;
		}
	} else {
		if (caps_on) {
			KeyboardMatrixUpdate(KEY_CAPSLOCK, KEY_RELEASED);
			caps_on = false;
		}
	}
}


// Start and stop channel generation.....
void BePlayNote(int channel, unsigned int pitch, int volume)
{
	audio->play_note(channel, pitch, volume);
}
void BeStopNote(int channel)
{
	audio->stop_note(channel);
}



//
// Handle teletext screen update
//
// On entry		: x, y is pixel position of text origin
//				  text is line to be displayed
//				  len is number of characters to be printed
void BeDrawImageString(int x, int y, unsigned char *text, int len) {
	bitmap_view->DrawImageString(x, y, text, len);
}

void BeSetForeground(int colour)	{ ttext_foreground = colour;	}
void BeSetBackground(int colour)	{ ttext_background = colour;	}

void BeFillRectangle(int x, int y, int width, int height) {
	bitmap_view->FillRect(x, y, width, height);
}


// Switch between teletext fonts
void BeSetFontStd(void)		{ current_font = ttext_std;		}
void BeSetFontStdDblL(void)	{ current_font = ttext_std_ldh;	}
void BeSetFontStdDblU(void)	{ current_font = ttext_std_udh;	}
void BeSetFontGrc(void)		{ current_font = ttext_grc;		}
void BeSetFontGrcDblL(void)	{ current_font = ttext_grc_ldh;	}
void BeSetFontGrcDblU(void)	{ current_font = ttext_grc_udh;	}
void BeSetFontGrs(void)		{ current_font = ttext_grs;		}
void BeSetFontGrsDblL(void)	{ current_font = ttext_grs_ldh;	}
void BeSetFontGrsDblU(void)	{ current_font = ttext_grs_udh;	}


// Access functions for Bitmap
unsigned char *BeImageData(void)
{
	return bitmap_view->GetBitmapData();
}

int BeGetBytesPerRow(void)
{
	return bitmap_view->GetBytesPerRow();
}

// Lazy version of Xlib function
void BePutImage(int Xmin1, int Ymin1, int Xmin2, int Ymin2, int Xsize, int Ysize)
{
	bitmap_view->PutImage(Xsize, Ysize);
}

// Clear rendering bitmap
void BeClearScreen(void)
{
	bitmap_view->ClearScreen();
}

// Check for shift, ctrl etc...
void BeCheckEvents(void)
{
	bitmap_view->CheckEvents();
}







