// Canon BubbleJet BJC series driver Class
//
// 2001, Philippe Houdoin
// Based on:
// - gdevbjc.c from Aladdin GhostScript
//
// Printers supported
//   bj30, bj100, bj200, bj200e, bj200ex, bj230, bjc70, bjc210,
//   bjc600, bjc600e, bjc800, bjc4000, bjc4100, etc
// 
// Todo list:
// - color support
// - add landscape
// - gamma (global or by color) correction?

#define DEBUG	1

#include <Debug.h>

#include <stdio.h>
#include <string.h>			// for memset()

#include "BJC.h"

// Constructor & destructor
// ------------------------

BJC::BJC()
	: PrinterDriver()
{
	// FIXME: set this from setup_msg / default = max from settings file
	SetResolution(360);
}

BJC::~BJC()
{
	if ( Transport() )
		{
		ExitPrinter();
		CloseTransport();
		};
}


// Methods
// -------

// --------------------------------------------------
status_t BJC::PrintPage
	(
	int32		page_number,
	int32		page_count
	)
{
	uint32	vskip;
	uint32	w;
	void *	scanline;
	uint8 *	buffer;
	int 	count;
		
	// Fire renderer thread
	StartPageRenderer(B_GRAY1);

	// BEWARE HERE: scanline info (ScanLineWidth(), etc)
	// aren't available yet, not until first GetScanLine() returns.

	if ( page_number == 1 )
		// Init printer before first page
		InitPrinter();

	BeginPage();
	
	buffer = NULL;
	
	vskip = 0;
	w = 0;
	while( GetScanLine(&scanline) == B_OK )
		{
		if ( IsEmptyScanLine(scanline) )
			vskip++;
		else
			{
			if ( ! buffer )
				{
				// first time: allocate the temporary plane buffer...
				w = ScanLineWidth();
				// nb bits by row to nb bytes per row
				if (w % 8)
					w = (w / 8) + 1;
				else
					w = w / 8;

				// w + (w % 128) + 1 is worst case scanline compression
				buffer = (uint8 *) malloc(w + (w % 128) + 1 + 16); // +16 = margin
				if ( buffer == NULL )
					break;
				};

			count = CompressPlane((uint8 *) scanline, w, buffer);

			if ( vskip )
				AdvancePaper(vskip);
			vskip = 1;

			// Raster Plane start
			WriteCommand('A', count + 1);	// +1 = Ink code, first byte of data
			Transport()->Write("K", 1);	// Black ink
			Transport()->Write(buffer, count);
			Transport()->Write("\x0d", 1);		// Raster Plane end
			};
		
		NextScanLine();
		}; // for each scanline

	if ( buffer )
		free(buffer);
	
	EndPage();

	StopPageRenderer();

	return B_OK;
}


// --------------------------------------------------
int BJC::CompressPlane
	(
	const uint8 *	row,
	uint32 			line_length,
	uint8 *			compressed
	)
{
  const uint8 * end_row = row;
  register const uint8 * exam = row;
  register uint8 * cptr = compressed; /* output pointer into compressed bytes */
  
  end_row += line_length;

  while ( exam < end_row ) {
    /* Search ahead in the input looking for a run */
    /* of at least 4 identical bytes. */
    const uint8 *compr = exam;
    const uint8 *end_dis;
    const uint8 *next;
    register uint8 test, test2;

    test = *exam;
    while ( exam < end_row ) {
      test2 = *++exam;
      if ( test == test2 )
	  break;
      test = test2;
    }
    

    /* Find out how long the run is */
    end_dis = exam - 1;
    if ( exam == end_row ) { /* no run */
      next = --end_row;
    } else { 

      next = exam + 1;
      while ( next < end_row && *next == test ) next++;
    }
    

    /* Now [compr..end_dis) should be encoded as dissimilar, */
    /* and [end_dis..next) should be encoded as similar. */
    /* Note that either of these ranges may be empty. */
    

    for ( ; ; ) {	/* Encode up to 128 dissimilar bytes */
      uint count = end_dis - compr; /* uint for faster switch */
      switch ( count ) { /* Use memcpy only if it's worthwhile. */
      case 6: cptr[6] = compr[5];
      case 5: cptr[5] = compr[4];
      case 4: cptr[4] = compr[3];
      case 3: cptr[3] = compr[2];
      case 2: cptr[2] = compr[1];
      case 1: cptr[1] = compr[0];
	*cptr = count - 1;
	cptr += count + 1;
      case 0: /* all done */
	break;
      default:
	if ( count > 128 ) count = 128;
	*cptr++ = count - 1;
	memcpy(cptr, compr, count);
	cptr += count, compr += count;
	continue;
      }
      break;
    }
    

    {	/* Encode up to 128 similar bytes. */
      /* Note that count may be <0 at end of row. */
      int count = next - end_dis;
      if (next < end_row || test != 0)
	while ( count > 0 ) { 

	  int this_one = (count > 128 ? 128 : count);
	  *cptr++ = 257 - this_one;
	  *cptr++ = (uint8) test;
	  count -= this_one;
	}
      exam = next;
    }
  }
  return (int)(cptr - compressed);
}

			
// --------------------------------------------------
status_t BJC::InitPrinter()
{
	// Reset to default setting...
	Transport()->Write("\x1b[K\x02\x00\x00\x0f", 7);
	
	// ???  ESC ( a <count lo> <count hi> ...
	WriteCommand('a', 1, "\x01");
	
	// Set media supply method ESC ( l <count lo> <count hi> <param1> <param2>
	WriteCommand('l', 1, "\x14");	// 14 = ????
	
	// Set page margins ESC ( g <count lo> <count hi> ...
	WriteCommand('g', 3, "\x75\x01\x72"); // = ????
	
	// Set raster resolution ESC ( d <count lo> <count hi> <y_res> [<x_res>]
	if ( XResolution() == YResolution() )
		WriteCommand('d', 2);
	else
		{
		WriteCommand('d', 4);
		WriteHiLo(YResolution());
		}
	WriteHiLo(XResolution());
	return B_OK;
}


// --------------------------------------------------
status_t BJC::ExitPrinter()
{
	if (! Transport())
		return B_ERROR;
		
	// ???  ESC ( a <count lo> <count hi> ...
	WriteCommand('a', 1, "\x00");
	
	// Set data compression off ESC ( b 0x01 0x00 <bool>
	WriteCommand('b', 1, "\x00");
	
	// Initial condition ESC @
	return Transport()->Write("\x1b@", 2);
}


// --------------------------------------------------
status_t BJC::BeginPage()
{
	// Set data compression on ESC ( b 0x01 0x00 <bool>
	WriteCommand('b', 1, "\x01");

	// Carriage Return...
	return Transport()->Write("\x0d", 1);
}


// --------------------------------------------------
status_t BJC::EndPage()
{
	// Form Feed
	return Transport()->Write("\x0c", 1);
}


// --------------------------------------------------
status_t BJC::AdvancePaper
	(
	int		count
	)
{
	WriteCommand('e', 2);
	WriteHiLo(count);

	return B_OK;
}


// --------------------------------------------------
status_t BJC::AdvanceHead
	(
	int		count
	)
{
/*
	int 	nb;
	uint8	byte;

	// print head has only 120 dpi step precision

	nb	   = 120 * count / x_resolution;
	count -= (x_resolution * nb / 120); 

	// first, quick horizontal skip, at 120 dpi resolution
	transport->Write("\033d", 2);
	WriteLoHi(nb);
	
	if ( count )
		{
		// second, print nothing up to x_resolution * count dots

		int i;

		count *= head_nb_bytes;	// 6*8 bits on print head (48 dots)

		transport->Write("\033[g", 3);
		WriteLoHi(count);
		byte = m_graphic_mode;
		transport->Write(&byte, 1);
		
		for ( i = 0; i < count ; i++)
			transport->Write("\000\000\000\000\000\000", 6);
		};
*/		
	return B_OK;
}


// --------------------------------------------------
status_t BJC::WriteCommand
	(
	uint8 			command,
	int				length = 0,
	const void *	data = NULL
	)
{
	uint8	tmp[3] = { 0x1b, '(', ' ' };
	
	tmp[2] = command;
	
	Transport()->Write(tmp, 3);
	WriteLoHi(length);
	
	if ( data != NULL && length > 0 )
		Transport()->Write(data, length);
		
	return B_OK;
}


// --------------------------------------------------
status_t BJC::WriteLoHi(int value)
{
	uint8 tmp[2];
	
	tmp[0] = (value & 0xff);
	tmp[1] = (value & 0xffff) >> 8;
	
	return Transport()->Write(tmp, 2);
}

	 
// --------------------------------------------------
status_t BJC::WriteHiLo(int value)
{
	uint8 tmp[2];
	
	tmp[0] = (value & 0xffff) >> 8;
	tmp[1] = (value & 0xff);
	
	return Transport()->Write(tmp, 2);
}
 
