// Printer driver generic class
//
// 2001, Philippe Houdoin
//

#define DEBUG	1

#include <Debug.h>

#include <stdio.h>
#include <string.h>			// for memset()

#include <StorageKit.h>

#include "PrinterDriver.h"

#include "PrinterSetupWindow.h"
#include "PageSetupWindow.h"
#include "JobSetupWindow.h"

#include "BitmapWindow.h"

// Private prototypes
// ------------------

#ifdef CODEWARRIOR
	#pragma mark [Constructor & destructor]
#endif

// Constructor & destructor
// ------------------------

PrinterDriver::PrinterDriver()
{
	m_job_file				= NULL;
	m_printer_node			= NULL;
	m_job_msg				= NULL;

	m_transport				= NULL;
	m_transport_add_on		= 0;
	m_transport_init_proc	= NULL;
	m_transport_exit_proc	= NULL;

	m_renderer_thread_id	= -1;
	
	m_buffer	= NULL;
	m_scanlines	= NULL;
}

PrinterDriver::~PrinterDriver()
{
	StopPageRenderer();
	
	if ( m_scanlines )
		free(m_scanlines);

	if ( m_buffer )
		free(m_buffer);
}

#ifdef CODEWARRIOR
	#pragma mark [Public methods]
#endif

// Public methods
// --------------

status_t PrinterDriver::PrintJob
	(
	BFile *		job_file,		// spool file
	BNode * 	printer_node,	// printer node, used by OpenTransport() to find & load transport add-on
	BMessage * 	job_msg			// job message
	)
{
	print_file_header	pfh;
	status_t			status;
	BMessage * 			msg;
	int32 				page;
	uint32				copy;
	uint32				copies;

	m_job_file		= job_file;
	m_printer_node	= printer_node;
	m_job_msg		= job_msg;

	if ( ! m_job_file || 
		 ! m_printer_node )
		return B_ERROR;

	// open transport
	if ( OpenTransport() != B_OK )
		return B_ERROR;

	// read print file header	
	m_job_file->Seek(0, SEEK_SET);
	m_job_file->Read(&pfh, sizeof(pfh));
	
	// read job message
	msg = new BMessage();
	msg->Unflatten(m_job_file);
	
	if ( msg->HasInt32("copies") )
		copies = msg->FindInt32("copies");
	else
		copies = 1;
	
	status = B_OK;
	for ( copy = 0; copy < copies; copy++ )
		{	
		for ( page = 1; page <= pfh.page_count && status == B_OK; page++ )
			status = PrintPage(page, pfh.page_count);

		if ( status != B_OK )
			break;

		// re-read job message for next page
		m_job_file->Seek(sizeof(pfh), SEEK_SET);
		msg->Unflatten(m_job_file);
		};
		
	CloseTransport();

	return status;
}


status_t PrinterDriver::PrintPage
	(
	int32		page_number,
	int32		page_count
	)
{
	char 		text[128];

	sprintf(text, "Faking print of page %ld/%ld...", page_number, page_count);
	BAlert *	alert = new BAlert("PrinterDriver::PrintPage()", text, "Hmm?");
	alert->Go();
	return B_OK;
}


status_t PrinterDriver::StartPageRenderer
	(
	color_space		space = B_RGB32,
	uint32			min_scanlines = 128
	)
{
	if ( m_scanlines )		
		free(m_scanlines);
	m_scanlines = NULL;

	if ( m_buffer )
		free(m_buffer);
	m_buffer = NULL;

	m_buffer_size			= ((min_scanlines < 128) ? 128 : min_scanlines);
	m_color_space			= space;
	
	m_head 					= 0;
	m_tail 					= 0;
	
	m_reader_sem			= create_sem(0, "reader_sem");
	m_writer_sem			= create_sem(m_buffer_size, "write_sem");
	m_lock_sem				= create_sem(1, "lock_sem");

	m_exit_renderer 		= false;

	m_renderer_thread_id = spawn_thread(_PageRendererThread, "renderer", 
										B_NORMAL_PRIORITY, this);

	return resume_thread(m_renderer_thread_id);
}



status_t PrinterDriver::StopPageRenderer()
{
	status_t	exit_status;

	if ( m_renderer_thread_id < 0 )
		return B_OK;

	m_exit_renderer = true;
	
	delete_sem(m_writer_sem);
	delete_sem(m_reader_sem);
	delete_sem(m_lock_sem);

	wait_for_thread(m_renderer_thread_id, &exit_status);
	m_renderer_thread_id = -1;

	return exit_status;
}


int32 PrinterDriver::PageRendererThread()
{
	status_t	status;
	BBitmap *	raster;
	BView * 	view;
	BRect		r;
	uint32 		picture_count;
	BRect 		print_rect;
	BRect * 	pic_rects;
	BPoint * 	pic_points;
	BRegion * 	pic_region;
	BPicture **	pictures;
	uint32		i;
	float		scanline;
	float		nb_scanlines;
	void *		convert_data;
	
	print_rect = m_job_msg->FindRect("printable_rect");
	
	m_job_file->Read(&picture_count, sizeof(uint32));

	pictures = (BPicture **) 	malloc(picture_count * sizeof(BPicture *));
	pic_rects = (BRect *)		malloc(picture_count * sizeof(BRect));
	pic_points = (BPoint *)		malloc(picture_count * sizeof(BPoint));
	pic_region = new BRegion();
	
	for (i = 0; i < picture_count; i++)
		{
		m_job_file->Seek(40 + sizeof(off_t), SEEK_CUR);
		m_job_file->Read(&pic_points[i], sizeof(BPoint));
		m_job_file->Read(&pic_rects[i], sizeof(BRect));
		pictures[i] = new BPicture();
		pictures[i]->Unflatten(m_job_file);
		pic_region->Include(pic_rects[i]);
		};
	
	r  = pic_region->Frame();
	delete pic_region;

	float scale = XResolution() / 72.0;	// FIXME: use constante here

	BRect raster_rect(0, 0, r.Width() * scale, m_buffer_size - 1);
						
	m_width 		= raster_rect.IntegerWidth();
	nb_scanlines 	= r.Height() * scale;

	raster 	= new BBitmap(raster_rect, B_RGB32, true, false); // accept BView, no contiguous

	view 	= new BView(raster->Bounds(), "rendering_view", B_FOLLOW_ALL, B_WILL_DRAW);
	raster->AddChild(view);
	
	view->Window()->Lock();

	// Okay, now allocate scanlines buffer
	m_buffer 	= malloc(m_buffer_size * (ScanLineLength() + 16));
	m_scanlines	= (void **) malloc(m_buffer_size * sizeof(void *));

	uint8 *	ptr;
	ptr = (uint8 *) m_buffer;
	for ( i = 0; i < m_buffer_size; i++)
		{
		m_scanlines[i] = ptr;
		ptr += (ScanLineLength() + 16);
		};
		
	convert_data = NULL;
		
	// Okay, start the real rendering, by multiple rendition, 
	scanline = 0.0;
	while ( scanline < nb_scanlines )
		{
		if ( m_exit_renderer )
			break;

		// render_scanlines

		// Okay, white background...
		view->SetHighColor(255,255,255);
		view->FillRect(view->Bounds());

		view->SetScale(scale);

		for (i = 0; i < picture_count; i++)
			{
			// if ( m_orientation == LANDSCAPE )
			// 	view->MovePenTo(scanline - pic_region->Frame().Width(), 0);
			// else // Portrait
				view->MovePenTo(0, -scanline/scale );
			
			view->DrawPicture(pictures[i]);
			};

		view->Sync();

		// now convert (if needed) each scanline rendered and copy them in
		// m_scanlines FIFO buffers
		
		uint32 max;
		
		max = raster->Bounds().IntegerHeight();
		if ( scanline + max > nb_scanlines )
			max = (uint32) (nb_scanlines - scanline);

		uint32 y;
		
		for ( y = 0; y < max; y++ )
			{
			// find an empty scanline entry
			status = acquire_sem(m_writer_sem);
			if (status != B_OK )
				break;

			switch (m_color_space)
				{
				case B_GRAY1:
					{

					errors_rows * 	er;
					uint32			ew;
					
					ew = m_width + 8; // 8 = dummy margins at left and right

					if ( convert_data == NULL )
						{
						// first time, alloc errors_rows buffer
						er = (errors_rows *) malloc(sizeof(errors_rows) +  2 * (sizeof(int8) * ew) );

						er->first 	= (int8 *) (er + 1);
						memset(er->first, 0, (sizeof(int8) * ew));
						er->second = er->first + ew;
						// skip some values for easy clipped access
						er->first += 4;
						er->second += 4;

						convert_data = er;
						}
					else
						er = (errors_rows *) convert_data;

					memset(er->second, 0, (sizeof(int8) * ew));
					rgb32_to_gray1(raster, m_scanlines[m_head], y, m_orientation, er);
					break;
					};
					
				case B_CMY24:
					// rgb32_to_cmy24(raster, m_scanlines[m_head], y, m_orientation);
					break;
				
				case B_CMYK32:
					rgb32_to_cmyk32(raster, m_scanlines[m_head], y, m_orientation);
					break;
			
				default:
					{
					if ( m_orientation == LANDSCAPE_ORIENTATION )
						{
						// TODO...;
					 	}
					else
						{
						ptr = (uint8 *) raster->Bits();
						ptr += (y * raster->BytesPerRow());
						// portrait orientation: raster bitmap is already in wanted orientation...
						memcpy(m_scanlines[m_head], ptr, raster->BytesPerRow());
						};
					break;
					};
				};	// switch (m_color_space)

			acquire_sem(m_lock_sem);
			m_head++;
			m_head %= m_buffer_size;
			release_sem(m_lock_sem);

			release_sem(m_reader_sem);
			}; // for y

		// next rendition pass please!
		scanline += raster->Bounds().Height();
		};

	view->RemoveSelf();

	delete raster;
	
	free(pic_points);
	free(pic_rects);
	
	for (i = 0; i < picture_count; i++)
		delete pictures[i];
	free(pictures);

	if ( convert_data )
		free(convert_data);

	m_exit_renderer = true;

	delete_sem(m_writer_sem);
	delete_sem(m_reader_sem);
	delete_sem(m_lock_sem);

	return B_OK;
}


status_t PrinterDriver::GetScanLine
	(
	void ** 	scanline,
	bigtime_t	timeout = B_INFINITE_TIMEOUT
	)
{
	status_t	status;
	
	if (scanline == NULL)
		return B_BAD_VALUE;
	
	status = acquire_sem_etc(m_reader_sem, 1, B_RELATIVE_TIMEOUT, timeout);
	if ( status != B_OK && ! m_exit_renderer )
		return status;
		
	acquire_sem(m_lock_sem);
	if ( m_exit_renderer && m_tail == m_head )
		status = NO_MORE_DATA;
	else
		{
		*scanline = m_scanlines[m_tail];
		status = B_OK;
		};
	release_sem(m_lock_sem);

	return status;
}


status_t PrinterDriver::NextScanLine()
{
	acquire_sem(m_lock_sem);
	m_tail++;
	m_tail %= m_buffer_size;
	release_sem(m_lock_sem);
		
	return release_sem(m_writer_sem);
}


bool PrinterDriver::IsEmptyScanLine
	(
	void *		scanline,
	rgb_color *	background_color = NULL
	)
{
	if (scanline == NULL)
		return B_BAD_VALUE;
	
	switch(m_color_space)
		{
		case B_GRAY1:
			{
			uint32 	x, w;
			uint8 *	ptr;
			
			ptr = (uint8 *) scanline;
			w = ScanLineLength();
			for ( x = 0; x < w; x++ )
				{
				if ( *ptr )
					return false;
				ptr++;
				};
			return true;
			};
			
		default:
			return false;
		};

	return false;
}


uint32 PrinterDriver::ScanLineLength()
{
	switch(m_color_space)
		{
		case B_RGB32:
		case B_RGBA32:
		case B_RGB32_BIG:
		case B_RGBA32_BIG:
		case B_CMY32:
		case B_CMYA32:
		case B_CMYK32:
			return 4 * m_width;
			
		case B_RGB24:
		case B_CMY24:
			return 3 * m_width;
			
		case B_GRAY8:
			return m_width;
			
		case B_GRAY1:
			{
			uint32 w;
			
			w = m_width / 8;
			if ( m_width % 8 )
				w++;
			return w;
			};
			
		default:
			return 0;
		};
}


status_t PrinterDriver::PrinterSetup
	(
	char * 	printer_name	// name of printer, to attach printer settings
	)
{
	PrinterSetupWindow * psw;

	psw = new PrinterSetupWindow(printer_name);
	return psw->Go();
}


status_t PrinterDriver::PageSetup
	(
	BMessage *	msg			// page setup message
	)
{
	BRect paper_rect;
	BRect print_rect;

#define SCREEN 			(72.0f)
#define A4_WIDTH		(8.3f)
#define A4_HEIGHT	 	(12.0f)
#define LETTER_WIDTH 	(8.5f)
#define LETTER_HEIGHT 	(11.0f)

	if (msg->HasInt64("xres"))
		msg->ReplaceInt64("xres", XResolution());
	else
		msg->AddInt64("xres", XResolution());
		
	if (msg->HasInt64("yres"))
		msg->ReplaceInt64("yres", YResolution());
	else
		msg->AddInt64("yres", YResolution());
				
	if (msg->HasInt32("orientation"))
		msg->ReplaceInt32("orientation", 1);
	else
		msg->AddInt32("orientation", 1);
			
	paper_rect = BRect(0, 0, LETTER_WIDTH * SCREEN, LETTER_HEIGHT * SCREEN);
	print_rect = BRect(0, 0, LETTER_WIDTH * SCREEN, LETTER_HEIGHT * SCREEN);
 	
	if (msg->HasRect("paper_rect"))
		msg->ReplaceRect("paper_rect", paper_rect);
	else
		msg->AddRect("paper_rect", paper_rect);

	if (msg->HasRect("printable_rect"))
		msg->ReplaceRect("printable_rect", print_rect);
	else
		msg->AddRect("printable_rect", print_rect);
	 		
	if (msg->HasFloat("scaling"))
		msg->ReplaceFloat("scaling", SCREEN);
	else
		msg->AddFloat("scaling", SCREEN);

	PageSetupWindow * psw;
	
	psw = new PageSetupWindow(msg);
	return psw->Go();
}



status_t PrinterDriver::JobSetup
	(
	BMessage *	msg				// job setup message
	)
{
	if (msg->HasInt32("copies"))
		msg->ReplaceInt32("copies", 1);
	else
		msg->AddInt32("copies", 1);

	if (msg->HasInt32("first_page"))
		msg->ReplaceInt32("first_page", 1);
	else
		msg->AddInt32("first_page", 1);
		
	if (msg->HasInt32("last_page"))
		msg->ReplaceInt32("last_page", 0xffff);
	else
		msg->AddInt32("last_page", 0xffff);

	JobSetupWindow * jsw;

	jsw = new JobSetupWindow(msg);
	return jsw->Go();
}


status_t PrinterDriver::OpenTransport()
{
	char 		buffer[512];
	BPath * 	path;
	

	if ( ! m_printer_node )
		return B_ERROR;

	// first, find & load transport add-on
	path = new BPath();
	
	// find name of this printer transport add-on 
	m_printer_node->ReadAttr("transport", B_STRING_TYPE, 0, buffer, sizeof(buffer));
	
	// try first on user add-ons directory
	find_directory(B_USER_ADDONS_DIRECTORY, path);
	path->Append("Print/transport");
	path->Append(buffer);
	m_transport_add_on = load_add_on(path->Path());
	
	if (m_transport_add_on < 0)
		{
		// add-on not in user add-ons directory. try system one
		find_directory(B_BEOS_ADDONS_DIRECTORY, path);
		path->Append("Print/transport");
		path->Append(buffer);
		m_transport_add_on = load_add_on(path->Path());
		};
	
	if (m_transport_add_on < 0)
		{
		BAlert * alert = new BAlert("Uh oh!", "Couldn't find transport add-on.", "OK");
		alert->Go();
		return B_ERROR;
		};
	
	// get init & exit proc
	get_image_symbol(m_transport_add_on, "init_transport", B_SYMBOL_TYPE_TEXT, (void **) &m_transport_init_proc);
	get_image_symbol(m_transport_add_on, "exit_transport", B_SYMBOL_TYPE_TEXT, (void **) &m_transport_exit_proc);
	
	if ( ! m_transport_init_proc ||
		 ! m_transport_exit_proc )
		{
		BAlert * alert = new BAlert("Uh oh!", "Couldn't resolve transport symbols.", "OK");
		alert->Go();
		return B_ERROR;
		};
	
	delete path;
	
	// now, init transport add-on
	node_ref 	ref;
	BDirectory 	dir;

	m_printer_node->GetNodeRef(&ref);
	dir.SetTo(&ref);
	
	path = new BPath(&dir, NULL);
	strcpy(buffer, path->Path());
	
	// create BMessage for init_transport()
	BMessage *msg = new BMessage('TRIN');
	msg->AddString("printer_file", buffer);
	
	m_transport = (*m_transport_init_proc)(msg);
	
	delete msg;
	delete path;
	
	if (m_transport == 0)
		{
		BAlert * alert = new BAlert("Uh oh!", "Couldn't open transport.", "OK");
		alert->Go();
		return B_ERROR;
		};
	
	return B_OK;
}


status_t PrinterDriver::CloseTransport()
{
	if ( ! m_transport_add_on )
		return B_ERROR;

	if ( m_transport_exit_proc )
		(*m_transport_exit_proc)();

	unload_add_on(m_transport_add_on);
	m_transport_add_on = 0;
	m_transport = NULL;
	
	return B_OK;
}

#ifdef CODEWARRIOR
	#pragma mark [Privates routines]
#endif

// Private routines
// ----------------

status_t PrinterDriver::rgb32_to_gray1
	(
	BBitmap * 		raster,
	void *			scanline,
	uint32			y,
	Orientation 	ori,
	errors_rows *	errors
	)
{
	int8 *		row_error;
	int8 *		next_row_error;
	int32		x;
	uint8 *		in;
	uint8 *		out;
	rgb_color 	color;
	int 		gray;
	int8		error;
	uint8		byte;
	int8 		bit;

	row_error 		= errors->first;
	next_row_error	= errors->second;
	
	out = (uint8 *) scanline;
	
	bit = 7;
	byte = 0;
	
	in = (uint8 *) raster->Bits();
	in += (y * raster->BytesPerRow());
		
	for ( x = 0; x < raster->Bounds().IntegerWidth(); x++ )
		{
		// For each pixel
		color.blue 	= *in;
		color.green	= *(in+1);
		color.red	= *(in+2);

		// compute gray level
 		gray = ((color.red * 77 + color.green * 151 + color.blue * 28) >> 8);

		// TODO: gamma correction here
		
		// diffuse error
		gray += *row_error;
		
		if ( gray > 127 )
			{
			// mostly white
			error = gray - 255;
			}
		else
			{
			// mostly black
			byte |= ( 1 << bit);
			error = gray;
			};

		in += 4;	// next pixel
		bit--;
		if ( bit < 0 )
			{
			*out = byte;
			out++;
			byte = 0;
			bit = 7;
			};

		// diffuse this gray error to sibblings pixels
		// Burkes dithering matrix :
		//
		//     x 8 4
		// 2 4 8 4 2

		*(row_error + 1) 			+= (8 * error) / 32;
		*(row_error + 2) 			+= (4 * error) / 32;

		*(next_row_error + x - 2) 	+= (2 * error) / 32;
		*(next_row_error + x - 1) 	+= (4 * error) / 32;
		*(next_row_error + x) 		+= (8 * error) / 32;
		*(next_row_error + x + 1) 	+= (4 * error) / 32;
		*(next_row_error + x + 2) 	+= (2 * error) / 32;

		row_error++;
		};
			
	if ( bit >= 0 )
		{
		// write last byte...
		*out = byte;
		out++;
		};
		

	// swap diffusion errors rows
	int8 *	first;
	
	first 			= errors->first;
	errors->first 	= errors->second;
	errors->second	= first;

	return B_OK;
}



status_t PrinterDriver::rgb32_to_cmyk32
	(
	BBitmap * 		raster,
	void *			scanline,
	uint32			y,
	Orientation 	ori
	)
{
	int32		x;
	uint8 *		in;
	uint8 *		out;
	rgb_color 	color;
	uint8 cyan, magenta, yellow, black;

	out = (uint8 *) scanline;
	
	in = (uint8 *) raster->Bits();
	in += (y * raster->BytesPerRow());
		
	for ( x = 0; x < raster->Bounds().IntegerWidth(); x++ )
		{
		// For each pixel
		color.blue 	= *in;
		color.green	= *(in+1);
		color.red	= *(in+2);
		
		cyan 	= 255 - color.red;
    	magenta = 255 - color.green;
    	yellow	= 255 - color.blue;

		black = ((cyan < magenta) ? ((cyan < yellow) ? cyan : yellow) : ((magenta < yellow) ? magenta : yellow));
	
		cyan 	-= black;
		magenta	-= black;
		yellow	-= black;

		*out 		= cyan;
		*(out+1) 	= magenta;
		*(out+2)	= yellow;
		*(out+3)	= black;

		in += 4;	// next pixel
		out	+= 4;
		};
			
	return B_OK;
}

