#ifndef PRINTERDRIVER_H
#define PRINTERDRIVER_H

#include <AppKit.h>
#include <InterfaceKit.h>

#ifndef ROUND_UP
	#define ROUND_UP(x, y) (((x) + (y) - 1) & ~((y) - 1))
#endif

// transport add-on calls definition 
extern "C"
	{
	typedef BDataIO * 	(*init_transport_proc)(BMessage *);
	typedef void 		(*exit_transport_proc)(void);
	};	


class PrinterDriver
	{
	public:
		// constructors / destructor
								PrinterDriver();
		virtual					~PrinterDriver();
		
		virtual status_t 		PrintJob(BFile * job_file, BNode * printer_node, BMessage * job_msg);
		virtual status_t		PrintPage(int32 page_number, int32 page_count);

		// scanline rendition methods
		status_t				StartPageRenderer(color_space = B_RGB32, uint32 min_nb_scanlines = 128 );
		status_t				StopPageRenderer();
		virtual int32			PageRendererThread();
		
		status_t				GetScanLine(void ** buffer, bigtime_t timeout = B_INFINITE_TIMEOUT);			
		status_t 				NextScanLine();
		virtual bool			IsEmptyScanLine(void * scanline, rgb_color * background_color = NULL);
		virtual uint32			ScanLineLength();
		inline uint32			ScanLineWidth()	{ return m_width; }

		// configuration default methods
		virtual status_t 		PrinterSetup(char * printer_name);
		virtual status_t 		PageSetup(BMessage * msg);
		virtual status_t 		JobSetup(BMessage * msg);
		
		// settings methods
		virtual status_t		SetResolution(uint32 xres, uint32 yres = 0) { m_x_resolution = xres; m_y_resolution = (yres ? yres : xres); return B_OK; }  
		
		// transport-related methods
		status_t				OpenTransport();
		status_t				CloseTransport();

		// accessors
		inline BFile *			JobFile()		{ return m_job_file; }
		inline BNode *			PrinterNode()	{ return m_printer_node; }
		inline BMessage *		JobMsg()		{ return m_job_msg; }
		inline BDataIO *		Transport()		{ return m_transport; }
		inline uint32			XResolution()	{ return m_x_resolution; }				
		inline uint32			YResolution()	{ return m_y_resolution; }
		
		// publics status code
		enum {
			NO_MORE_DATA		= 100
		};			

		typedef enum {
			PORTRAIT_ORIENTATION,
			LANDSCAPE_ORIENTATION
		} Orientation;

	private:
		BFile *					m_job_file;
		BNode *					m_printer_node;
		BMessage *				m_job_msg;

		uint32					m_x_resolution;		// in dpi
		uint32					m_y_resolution;		// in dpi

		// page renderer 
		static	int32			_PageRendererThread(void * p) { return ((PrinterDriver *) p)->PageRendererThread(); }
		thread_id				m_renderer_thread_id;
		sem_id					m_lock_sem;
		sem_id					m_reader_sem;
		sem_id					m_writer_sem;
		volatile bool			m_exit_renderer;
		void *					m_buffer;
		uint32					m_buffer_size;
		void **					m_scanlines;
		volatile uint32			m_head;
		volatile uint32			m_tail;
		color_space				m_color_space;
		uint32					m_width;
		Orientation				m_orientation;
		
		typedef struct {
			int8 *	first;
			int8 *	second;
		} errors_rows;

		status_t 				rgb32_to_gray1(BBitmap * raster, void * scanline, uint32 y, Orientation ori, errors_rows * errors);
		// status_t 				rgb32_to_cmy24(BBitmap * raster, void * scanline, uint32 y, Orientation ori);
		status_t 				rgb32_to_cmyk32(BBitmap * raster, void * scanline, uint32 y, Orientation ori);

		// transport-related 
		BDataIO *				m_transport;
		image_id				m_transport_add_on;
		init_transport_proc		m_transport_init_proc;
		exit_transport_proc		m_transport_exit_proc;
	};

#endif // #ifndef PRINTERDRIVER_H

