/* 
	es1370.c - Device driver for the Ensoniq AudioPCI ES1370 device.

    Copyright (C) 1998 HockYiung Huang (leclec@pacific.net.sg)
    Port to R4 (C) 1999 Marc Schefer (mschefer@iprolink.ch)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.h"
#include "es1370.h"
#include "hardware.h"


/*****************************************************************************
 * FORWARD DECLARATIONS
 */
static status_t		es1370_open(const char *name, uint32 flags, void **cookie);
static status_t		es1370_close(void *cookie);
static status_t		es1370_free(void *cookie);
static status_t		es1370_control(void *cookie, uint32 op, void *data, size_t len);
static status_t		es1370_read(void *cookie, off_t pos, void *data, size_t *len);
static status_t		es1370_write(void *cookie, off_t pos, const void *data, size_t *len);

static int32			es1370_inth(void *data);

static status_t		es1370_sound_get_params(dev_info *d, sound_setup *ss);
static status_t		es1370_sound_set_params(dev_info *d, sound_setup *ss);
static status_t		es1370_sound_set_playback_completion_sem(dev_info *d, sem_id *sem);
static status_t		es1370_sound_set_capture_completion_sem(dev_info *d, sem_id *sem);
static status_t		es1370_sound_write_buffer(dev_info *d, audio_buffer_header *header);
static status_t		es1370_sound_read_buffer(dev_info *d, audio_buffer_header *header);
static status_t		es1370_sound_lock_for_dma(dev_info *d, audio_buffer_header *header);

/*****************************************************************************
 * BeOS driver API version number 
 */

int32 api_version = B_CUR_DRIVER_API_VERSION;

/*****************************************************************************
 * device_hooks structure - has function pointers to the various entry points 
 * for device operations
 */
static device_hooks es1370_device_hooks = {
	&es1370_open,
	&es1370_close,
	&es1370_free,
	&es1370_control,
	&es1370_read,
	&es1370_write
};

/*****************************************************************************
 * list of device names to be returned by publish_devices()
 */
static char		*name_list[] = {
	"audio/old/es1370",
	0
};

/*****************************************************************************
 * Default sound setup
 */
static sound_setup es1370_sound_setup = {
	// left channel
	{ 	
		cd,		// adc_source 
		20, 	// adc_gain
		0, 		// mic_gain_enable
		30, 	// cd_mix_gain
		0, 		// cd_mix_mute
		20, 	// aux2_mix_gain
		0, 		// aux2_mix_mute
		20, 	// line_mix_gain
		0, 		// line_mix_mute
		10, 	// dac_attn
		0, 		// dac_mute
		0,0
	},	
	// right channel
	{ 	
		cd,		// adc_source 
		20, 	// adc_gain
		0, 		// mic_gain_enable
		30, 	// cd_mix_gain
		0, 		// cd_mix_mute
		20, 	// aux2_mix_gain
		0, 		// aux2_mix_mute
		20, 	// line_mix_gain
		0, 		// line_mix_mute
		10, 	// dac_attn
		0, 		// dac_mute
		0,0
	},	
	kHz_44_1,	// sample_rate
	16,			// playback_format (ignored, always 16bit-linear)
	16,			// capture_format (ignored, always 16bit-linear)
	0,			// dither_enable 
	10,			// mic_attn
	1,			// mic_enable
	0,  		// output_boost (ignored, always on)
	0,			// highpass_enable (ignored, always on)
	30,			// mono_gain
	0			// mono_mute
};

/*****************************************************************************
 * globals for this driver.  The values are NOT preserved when the driver is 
 * unloaded and later reloaded.  The init_driver() call should set these up 
 * every time...
 */
static char		*reg_base;	/* base address of reg access area */
static int32	is_open;	/* bitfield of open logical devices */
int32	portbase;
pci_module_info *pci_bus;


/*****************************************************************************
 * lookup_pci_device()
 *****************************************************************************
 * Locates the pci_info record for a particular pci device.
 */
static bool
lookup_pci_device (short vendor, short device, pci_info *returned_info)
{
	int	i;
	for (i = 0; ; i++) {
		if (pci_bus->get_nth_pci_info (i, returned_info) != B_OK)
			return false;
		if (returned_info->vendor_id == vendor &&
		    returned_info->device_id == device)
		    	break;
	}
	return true;
}

/*****************************************************************************
 * lookup_device_name()
 *****************************************************************************
 * Looks up device name, returns id
 */
static int
lookup_device_name (const char *name)
{
	int i;
	
	for (i = 0; name_list[i]; i++)
		if (!strcmp (name_list[i], name))
			return i;
	return -1; /* Invalid device name */
}


/*****************************************************************************
 * DRIVER ENTRY POINTS
 *****************************************************************************
 */

/*****************************************************************************
 * init_hardware()
 *****************************************************************************
 * Called once by the kernel during system startup to allow the driver to
 * reset the hardware to a known default state. If your device doesn't need
 * initialization, you don't have to provide this function.
 */
status_t
init_hardware(void)
{
   // Find out hardware configuration(ie i/o ports, interrupt, etc).
   // Reset the hardware to a known default state.
	pci_info	info;
	long		command;
	
	dbgprintf("es1370: init_hardware()\n");

	if (get_module(B_PCI_MODULE_NAME, (module_info **) &pci_bus) < 0)
		return ENODEV;

	if (!lookup_pci_device (ES1370_VENDOR_ID, 
	                        ES1370_DEVICE_ID, &info))
		return ENODEV;
	
	command = pci_bus->read_pci_config(info.bus,info.device,info.function,0x04,2);
	// turn on bus master support
	command |= 4;
	pci_bus->write_pci_config(info.bus,info.device,info.function,0x04,1,command);
		
	portbase = info.u.h0.base_registers[0];

	hw_init();
	
	put_module( B_PCI_MODULE_NAME );
			
	return B_OK;
}

/*****************************************************************************
 * init_driver()
 *****************************************************************************
 * Called whenever the driver is loaded into memory. This allows the driver
 * to set up any kernel resources needed, as well as to initialize any global
 * variables it uses. If your driver doesn't need to be initialized, you
 * don't have to provide this function.
 */
status_t
init_driver(void)
{
    // Allocate and initialize the driver object. Note this is not the
    // same as the device object(cookie) created in the device hook's open().
    // Driver object contains the information about the piece of hardware
    // that this driver is running on.
	pci_info	info;
	int32		base;

	dbgprintf("es1370: init_driver()\n");
	
	if (get_module(B_PCI_MODULE_NAME, (module_info **) &pci_bus) < 0)
		return ENODEV;

	if (!lookup_pci_device (ES1370_VENDOR_ID, 
	                        ES1370_DEVICE_ID, &info))
		return ENODEV;
	
	base = info.u.h0.base_registers[0];
		
	is_open = 0;		/* no devices open yet */
	
	return B_OK;
}

/*****************************************************************************
 * uninit_driver()
 *****************************************************************************
 * Called whenever the driver is unloaded from memory. This can be used to
 * release any system resources allocated by init_driver(). If your driver
 * doesn't need to be uninitialized, you don't have to provide this function.
 */
void
uninit_driver(void)
{
	dbgprintf("es1370: uninit_driver()\n");
	put_module( B_PCI_MODULE_NAME );
    // Release the driver object
}

/*****************************************************************************
 * publish_devices()
 *****************************************************************************
 * Called by the kernel to obtain a list of device names supported by the
 * driver.
 */
const char **
publish_devices(void)
{
	//dbgprintf("es1370: publish_devices()\n");

    return (const char ** )name_list;
}

/*****************************************************************************
 * find_device()
 *****************************************************************************
 * Called by the kernel when it needs to obtain a list of the hook functions
 * for a particular device.
 */
device_hooks *
find_device(const char *name)
{
	//dbgprintf("es1370: find_device()\n");
   
    if (lookup_device_name(name) >= 0)
        return &es1370_device_hooks;   /* Return hooks list */

    return NULL;    /* Device not found */
}


/*****************************************************************************
 * DEVICE HOOK FUNCTIONS
 *****************************************************************************
 */

/*****************************************************************************
 * es1370_open()
 *****************************************************************************
 * Handles the Posix open() function.  This function should prepare a device
 * for reading or writing.  We pass a pointer to the pci_info record for our 
 * device back to the kernel.  This will be the 'cookie' which we will use to 
 * get at the device in all the other device operations.
 */
static status_t	
es1370_open(const char *name, uint32 flags, void **cookie)
{
	dev_info	*d = NULL;
	int			id;
	int32		mask;
	status_t	err;
	char		sem_name [B_OS_NAME_LENGTH];
	
	dbgprintf(("es1370: open()\n"));
   
	/* figure out which sub-device, by looking at the name */
	id = lookup_device_name (name);
	if (id < 0)
		return EINVAL;

	/* only one client at a time */
	mask = 1 << id;
	if (atomic_or (&is_open, mask) & mask)
		return B_BUSY;

    // Create a device object and returned  back as cookie.
	/* allocate a per-logical-device record in the kernel
	   heap.  This will be our 'cookie'. Note that the kernel 
	   heap is always locked, so this will always be resident */
	  
	d = (dev_info *) malloc (sizeof (dev_info));
	if (!d)
		return B_NO_MEMORY;
	
	/* find the device */
	err = ENODEV;
	if (!lookup_pci_device (ES1370_VENDOR_ID,
	                        ES1370_DEVICE_ID, &d->pci))
		goto err1;
	
	portbase = d->pci.u.h0.base_registers[0];
   
	sprintf (sem_name, "playback hw lock %d", id);
	d->playback.hw_lock = err = create_sem (8, sem_name);
	if (err < 0)
		goto err1;
	set_sem_owner(err, B_SYSTEM_TEAM);

	sprintf (sem_name, "capture hw lock %d", id);
	d->capture.hw_lock = err = create_sem (0, sem_name);
	if (err < 0)
		goto err2;
	set_sem_owner(err, B_SYSTEM_TEAM);
 
	//d->playback.time = 0;
	//d->playback.playback_total = 0;

	/* setup device record */
	d->id = id;
	d->ss = (sound_setup *) malloc(sizeof (sound_setup));
	if (!d->ss)
		goto err3;
	
	/* initialize sound setup */
	es1370_sound_set_params(d, &es1370_sound_setup);	
        
    /* initialize the semaphores */
    d->playback.completion_sem = 0;
    d->capture.completion_sem = 0;
    
    /* initialize the states */
    d->playback.state = STATE_PAUSE;
    d->capture.state = STATE_PAUSE;
	
	d->playback.transfer_area = create_area ("es1370 playback transfer",
									&d->playback.data,
									B_ANY_ADDRESS,
									B_PAGE_SIZE * 4,
									B_FULL_LOCK | B_CONTIGUOUS,
									B_READ_AREA | B_WRITE_AREA );
	memset(d->playback.data, 0, B_PAGE_SIZE * 4);								
	get_memory_map (d->playback.data, B_PAGE_SIZE * 4, d->playback.scatter, 1);
	d->playback.offset = 0;	
	d->write_count = 0;
	
	d->capture.transfer_area = create_area ("es1370 capture transfer",
									&d->capture.data,
									B_ANY_ADDRESS,
									B_PAGE_SIZE * 4,
									B_FULL_LOCK | B_CONTIGUOUS,
									B_READ_AREA | B_WRITE_AREA );
	memset(d->capture.data, 0, B_PAGE_SIZE * 4);								
	get_memory_map (d->capture.data, B_PAGE_SIZE * 4, d->capture.scatter, 1);
	d->capture.offset = 0;	
	d->read_count = 0;
	
	d->playback.sample_rate = d->capture.sample_rate = kHz_44_1;
	
	*cookie = d;
	
	/* install interrupt handler */
	install_io_interrupt_handler (
		d->pci.u.h0.interrupt_line,		/* interrupt number */
		es1370_inth,					/* -> interrupt handler */
		d,								/* pass this to handler */
		0
	);

	return B_OK;

err3:
	delete_sem(d->capture.hw_lock);

err2:
	delete_sem(d->playback.hw_lock);

err1:
	free (d);

	return err;
}


/*****************************************************************************
 * es1370_close()
 *****************************************************************************
 * Handles the Posix close() function. This function is called by a client
 * program when it has finished using the device.
 */
static status_t
es1370_close(void *cookie)
{
	dbgprintf("es1370: close()\n");
    // Should not deallocate anything here. Wait for the free().
    return B_OK;
}

/*****************************************************************************
 * es1370_free()
 *****************************************************************************
 * The free hook releases a device after all I/O transactions have been
 * completed. It does not directly correspond to any Posix or other C
 * function.
 * Called after close, and after all read/write calls have returned.  It 
 * sounds strange that a read/write call could still be	pending after the 
 * device is closed, but it is certainly possible in a multi-threaded world.
 */
static status_t
es1370_free(void *cookie)
{
	dev_info *d = (dev_info *) cookie;  
    
	dbgprintf(("es1370: free()\n"));

    // free all resources that is grabbed during open().
	hw_stop_playback();
	hw_stop_capture();
	
	remove_io_interrupt_handler (
		d->pci.u.h0.interrupt_line,		/* interrupt number */
		es1370_inth,				/* -> interrupt handler */
		d
	);

	delete_area(d->playback.transfer_area);
	delete_area(d->capture.transfer_area);
	
	atomic_and (&is_open, ~(1 << d->id));
	
	delete_sem (d->playback.hw_lock);
	delete_sem (d->capture.hw_lock);

    free(d->ss);
    free(d);
	   
    return B_OK;
}

/*****************************************************************************
 * es1370_control()
 *****************************************************************************
 * Handles the ioctl() function. This function provides the mechanism by
 * which the system, and client programs, can control how the device
 * functions.
 */
static status_t
es1370_control(void *cookie, uint32 op, void *data, size_t len)
{
	dev_info *d = (dev_info *) cookie;  
    status_t err = EINVAL;
    
	//dbgprintf("es1370: control()\n");

    switch (op)
    {
        case SOUND_GET_PARAMS:
			err = es1370_sound_get_params(d, (sound_setup *)data);
            break;
        case SOUND_SET_PARAMS:
			err = es1370_sound_set_params(d, (sound_setup *)data);
            break;
        case SOUND_SET_PLAYBACK_COMPLETION_SEM:
			err = es1370_sound_set_playback_completion_sem(d, (sem_id *)data);
            break;
        case SOUND_SET_CAPTURE_COMPLETION_SEM:
			err = es1370_sound_set_capture_completion_sem(d, (sem_id *)data);
			break;
        case SOUND_RESERVED_1:
            break;
        case SOUND_RESERVED_2:
            break;
        case SOUND_DEBUG_ON:
            break;
        case SOUND_DEBUG_OFF:
            break;
        case SOUND_WRITE_BUFFER:
 			err = es1370_sound_write_buffer(d, (audio_buffer_header *)data);
            break;
        case SOUND_READ_BUFFER:
 			err = es1370_sound_read_buffer(d, (audio_buffer_header *)data);
            break;
        case SOUND_LOCK_FOR_DMA:
 			err = es1370_sound_lock_for_dma(d, (audio_buffer_header *)data);
            break;
        default:
        	break;
    }

    return err;
}

/*****************************************************************************
 * es1370_read()
 *****************************************************************************
 * Handles the Posix read() function. This function reads data from the
 * device into system memory.
 */
static status_t
es1370_read(void *cookie, off_t pos, void *data, size_t *len)
{
	dev_info *d = (dev_info *) cookie;  

	//dbgprintf("es1370: read()\n");

    return B_OK;
}

/*****************************************************************************
 * es1370_write()
 *****************************************************************************
 * Handles the Posix write() function. This function writes data from system
 * memory to the device.
 */
static status_t
es1370_write(void *cookie, off_t pos, const void *data, size_t *len)
{
	dev_info *d = (dev_info *) cookie;  

	//dbgprintf("es1370: write()\n");

    return B_OK;
}

/*****************************************************************************
 * es1370_inth()
 *****************************************************************************
 * The interrupt handler
 */
static int32
es1370_inth(void *data)
{
	bigtime_t st = system_time();

	dev_info	*d = (dev_info *)data;

    uint32 interrupt_status = dio_read_register_dword(ICS_STATUS_REGISTER);

    if (interrupt_status & B_ICS_STATUS_INTR)
    {
        // UART interrupt
        if (interrupt_status & B_ICS_STATUS_UART)
        {
			uint8 status, control;
            
            status = dio_read_register_byte(UART_STATUS_REGISTER);

            // If there is outstanding work to do and there is a port-driver for
            // the uart miniport...
            if (!(status & B_UART_STATUS_RXRDY))
            {
                // ...notify the MPU port driver to do work.
            }

            control = dio_read_register_byte(UART_CONTROL_REGISTER);

            // Ack the interrupt
            if (status & B_UART_STATUS_RXINT)
            {
                control &= ~B_UART_CONTROL_RXINTEN;
                dio_write_register_byte(UART_CONTROL_REGISTER,control);
                control |= B_UART_CONTROL_RXINTEN;
                dio_write_register_byte(UART_CONTROL_REGISTER,control);
            }
            else
            {
                control &= ~B_UART_CONTROL_TXINTEN;
                dio_write_register_byte(UART_CONTROL_REGISTER,control);
                control |= B_UART_CONTROL_TXRDYINTEN;
                dio_write_register_byte(UART_CONTROL_REGISTER,control);
            }
        }

        //  ADC/DAC2 Interrupt
        if (interrupt_status & (B_ICS_STATUS_DAC2 | B_ICS_STATUS_ADC));
        {
            uint32 control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);

            // Ack the interrupt
            if (interrupt_status & B_ICS_STATUS_DAC2)
            {
				//kprintf("P");
				spinlock lock = 0;
   				cpu_status former = disable_interrupts();
   				acquire_spinlock(&lock);

				d->write_count--;
				if (d->write_count <= 0)
				{ 
					// no more buffers to play...
					d->playback.offset = 0;
					d->write_count = 0;
					hw_stop_playback();
					d->playback.state = STATE_PAUSE;
				}

				d->playback.time = st;
				if( d->write_count <= 3 )
					d->playback.sample_clock += (2080LL * 2500LL / 441LL);
				else if( d->write_count <= 6 )
					d->playback.sample_clock += (2064LL * 2500LL / 441LL);
				else
					d->playback.sample_clock += (2048LL * 2500LL / 441LL);


				release_spinlock(&lock);
				restore_interrupts(former);
				   
               	// Ack DAC2 channel
               	control &= ~B_SERIAL_INTF_CONTROL_P2_INTR_EN;
               	dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
               	control |= B_SERIAL_INTF_CONTROL_P2_INTR_EN;
               	dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
				// release hardware for next playback
 				release_sem_etc (d->playback.hw_lock, 1, B_DO_NOT_RESCHEDULE);
            }
            else
            {
				spinlock lock = 0;
				acquire_spinlock(&lock);
				d->read_count--;
				if (d->read_count < 0)
				{ 
					// no more buffers to play...
					d->read_count = 0;
					d->capture.offset = 0;
					hw_stop_capture();
					d->capture.state = STATE_PAUSE;
				}
				release_spinlock(&lock);
                // Ack ADC channel
                control &= ~B_SERIAL_INTF_CONTROL_R1_INT_EN;
                dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
                control |= B_SERIAL_INTF_CONTROL_R1_INT_EN;
                dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
				if (d->capture.state != STATE_PAUSE)
					// release hardware for next capture
 					release_sem_etc (d->capture.hw_lock, 1, B_DO_NOT_RESCHEDULE);
            }
        }

        // DAC1 interrupt
        if (interrupt_status & B_ICS_STATUS_DAC1)
        {
            // just ack it...
            uint32 control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);

            control &= ~B_SERIAL_INTF_CONTROL_P1_INTR_EN;
            dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
            control |= B_SERIAL_INTF_CONTROL_P1_INTR_EN;
            dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
        }

        // CCB interrupt
        if (interrupt_status & B_ICS_STATUS_MCCB)
        {
            // just ack it...
            uint32 control = dio_read_register_dword(ICS_CONTROL_REGISTER);

            control &= ~B_ICS_CONTROL_CCB_INTRM;
            dio_write_register_dword(ICS_CONTROL_REGISTER,control);
            control |= B_ICS_CONTROL_CCB_INTRM;
            dio_write_register_dword(ICS_CONTROL_REGISTER,control);
        }
    }

	return false;
}

/*****************************************************************************
 * IMPLEMENTATIONS FOR CONTROL CODES
 */

/*****************************************************************************
 * es1370_sound_get_params()
 *****************************************************************************
 * Read a set of parameters which correspond to the settings in the sound 
 * preferences panel.
 */
static status_t
es1370_sound_get_params(dev_info *d, sound_setup *ss)
{
	*ss = *d->ss;

	//dbgprintf("es1370: sound_get_params()\n");
		
	return B_OK;
}

/*****************************************************************************
 * es1370_sound_set_params()
 *****************************************************************************
 * Write a set of parameters which correspond to the settings in the sound 
 * preferences panel.
 */
static status_t
es1370_sound_set_params(dev_info *d, sound_setup *ss)
{	
	*d->ss = *ss;	
	
	// adc source
	hw_write_bits_to_codec(LCH_INPUT_MIXER_SW1_REGISTER ,1,MIXBIT_INPUT_SW1_LINEL,(ss->left.adc_source == line)?1:0);
	hw_write_bits_to_codec(RCH_INPUT_MIXER_SW1_REGISTER ,1,MIXBIT_INPUT_SW1_LINER,(ss->left.adc_source == line)?1:0);
	hw_write_bits_to_codec(LCH_INPUT_MIXER_SW1_REGISTER ,1,MIXBIT_INPUT_SW1_CDL,(ss->left.adc_source == cd)?1:0);
	hw_write_bits_to_codec(RCH_INPUT_MIXER_SW1_REGISTER ,1,MIXBIT_INPUT_SW1_CDR,(ss->left.adc_source == cd)?1:0);
	hw_write_bits_to_codec(LCH_INPUT_MIXER_SW1_REGISTER ,1,MIXBIT_INPUT_SW1_MIC,(ss->left.adc_source == mic)?1:0);
	hw_write_bits_to_codec(LCH_INPUT_MIXER_SW2_REGISTER ,1,MIXBIT_INPUT_SW2_VOICE,(ss->left.adc_source == loopback)?1:0);
	hw_write_bits_to_codec(RCH_INPUT_MIXER_SW2_REGISTER ,1,MIXBIT_INPUT_SW2_VOICE,(ss->left.adc_source == loopback)?1:0);
	// adc gain
	hw_write_bits_to_codec(AD_INPUT_SELECT_REGISTER,1,MIXBIT_AD_INPUT_SELECT_ADSEL,ss->left.adc_gain ? 0 : 1);
	hw_write_bits_to_codec(AD_INPUT_SELECT_REGISTER,1,MIXBIT_AD_INPUT_SELECT_ADSEL,ss->right.adc_gain ? 0 : 1);
	
	// mic gain enable
	hw_write_bits_to_codec(MIC_AMP_GAIN_REGISTER,1,MIXBIT_MIC_AMP_GAIN_MGAIN,ss->left.mic_gain_enable ? 1 : 0);
	
	// cd volume
    hw_codec_reg_write(CD_AUDIO_VOLUME_LCH_REGISTER,(uint8)((ss->left.cd_mix_gain)));
    hw_codec_reg_write(CD_AUDIO_VOLUME_RCH_REGISTER,(uint8)((ss->right.cd_mix_gain)));
 	// cd mute    
	hw_write_bits_to_codec(OUTPUT_MIXER_SW1_REGISTER,1, MIXBIT_OUTPUT_SW1_CDL,ss->left.cd_mix_mute ? 0 : 1);
	hw_write_bits_to_codec(OUTPUT_MIXER_SW1_REGISTER,1, MIXBIT_OUTPUT_SW1_CDR,ss->right.cd_mix_mute ? 0 : 1);

	// line volume
    hw_codec_reg_write(LINE_VOLUME_LCH_REGISTER,(uint8)((ss->left.line_mix_gain)));
    hw_codec_reg_write(LINE_VOLUME_RCH_REGISTER,(uint8)((ss->right.line_mix_gain)));
 	// line mute    
	hw_write_bits_to_codec(OUTPUT_MIXER_SW1_REGISTER,1,MIXBIT_OUTPUT_SW1_LINEL,ss->left.line_mix_mute ? 0 : 1);
	hw_write_bits_to_codec(OUTPUT_MIXER_SW1_REGISTER,1,MIXBIT_OUTPUT_SW1_LINER,ss->right.line_mix_mute ? 0 : 1);

	// dac attn
    hw_codec_reg_write(VOICE_VOLUME_LCH_REGISTER,(uint8)((ss->left.dac_attn>>1)));
    hw_codec_reg_write(VOICE_VOLUME_RCH_REGISTER,(uint8)((ss->right.dac_attn>>1)));
 	// dac mute    
	hw_write_bits_to_codec(OUTPUT_MIXER_SW2_REGISTER,1,MIXBIT_OUTPUT_SW2_VOICEL,ss->left.dac_mute ? 0 : 1);
	hw_write_bits_to_codec(OUTPUT_MIXER_SW2_REGISTER,1,MIXBIT_OUTPUT_SW2_VOICER,ss->right.dac_mute ? 0 : 1);

	// sample rate
	//if (ss->sample_rate <= kHz_6_62) hw_set_sample_rate(sample_rate_table[ss->sample_rate]);
	
	// playback format(ignored, always 16bit-linear)
	//hw_set_playback_format(FORMAT_STEREO, ss->dither_enable ? FORMAT_8_BIT : FORMAT_16_BIT);
	
	// capture format(ignored, always 16bit-linear)
	//hw_set_capture_format(FORMAT_STEREO, ss->dither_enable ? FORMAT_8_BIT : FORMAT_16_BIT);
	
	// dither enable

	// mic attn
    hw_codec_reg_write(MIC_VOLUME_REGISTER,(uint8)((ss->mic_attn>>1)));
	// mic enable
	hw_write_bits_to_codec(OUTPUT_MIXER_SW1_REGISTER,1,MIXBIT_OUTPUT_SW1_MIC,ss->mic_enable ? 1 : 0);
	
	// output boost(ignored, always on)
	// highpass enable(ignored, always on)
	
	// mono_gain
	// mono_mute
    hw_codec_reg_write(MONO_OUT_VOLUME_REGISTER,(uint8)(ss->mono_gain>>3) | 
    				   (ss->mono_mute ? (1<<MIXBIT_MONO_OUT_VOLUME_MUTE) : 0));

	return B_OK;
}

/*****************************************************************************
 * es1370_sound_set_playback_completion_sem()
 *****************************************************************************
 * Takes a (sem_id*) argument which points to a semaphore that must be 
 * released once for each buffer written. The semaphore should be released 
 * when the data in the buffer is no longer needed.
 */
static status_t
es1370_sound_set_playback_completion_sem(dev_info *d, sem_id *sem)
{
	//dbgprintf(("es1370_sound_set_playback_completion_sem\n"));

	d->playback.completion_sem = *sem;
	
	return B_OK;
}

/*****************************************************************************
 * es1370_sound_set_capture_completion_sem()
 *****************************************************************************
 * Takes a (sem_id*) argument which points to a semaphore that must be 
 * released once for each buffer read. The semaphore should be released when 
 * the data in the buffer is valid.
 */
static status_t
es1370_sound_set_capture_completion_sem(dev_info *d, sem_id *sem)
{
	//dbgprintf(("es1370_sound_set_capture_completion_sem\n"));

	d->capture.completion_sem = *sem;

	return B_OK;
}

/*****************************************************************************
 * es1370_sound_write_buffer()
 *****************************************************************************
 * Takes an (audio_buffer_header*) argument which is defined in MediaDefs.h.
 * Call is allowed to return before the data in the buffer has been consumed 
 * but the playback completion semaphore must be released when the buffer can 
 * be recycled.
 */
static status_t
es1370_sound_write_buffer(dev_info *d, audio_buffer_header *header)
{
	size_t size = header->reserved_1 - sizeof(*header);
	void *data = (void *) (header + 1);

	acquire_sem( d->playback.hw_lock ); // prevent buffer overflow
	
	// Begin critical section
	spinlock lock = 0;
	cpu_status former = disable_interrupts();
	acquire_spinlock(&lock);

	if (d->playback.state == STATE_PAUSE )
		d->playback.time = d->playback.sample_clock = system_time();

	d->write_count++;
	header->time = d->playback.time;
	header->sample_clock = d->playback.sample_clock;

	release_spinlock(&lock);
	restore_interrupts(former);
	
	if ((d->playback.offset+size) > (d->playback.scatter[0].size))
	{
		memcpy((void *)((uint32)d->playback.data+d->playback.offset),data,d->playback.scatter[0].size-d->playback.offset);
		memcpy((void *)d->playback.data,(void *)((uint32)data+(d->playback.scatter[0].size-d->playback.offset)),size-(d->playback.scatter[0].size-d->playback.offset));
	}
	else
	{
		memcpy((void *)((uint32)d->playback.data+d->playback.offset),data,size);
	}

	d->playback.offset += size;
	d->playback.offset %= d->playback.scatter[0].size;

	if (d->playback.state == STATE_PAUSE )
	{
		d->playback.sample_rate = d->ss->sample_rate;
		hw_set_playback_buffer((int32) pci_bus->ram_address (d->playback.scatter[0].address),d->playback.scatter[0].size);
		hw_set_playback_sample_count((1024/(d->ss->dither_enable ? 1 : 2)) - 1); // 16-bit samples, o-based count
		hw_set_sample_rate(sample_rate_table[d->playback.sample_rate]);
		hw_set_playback_format(FORMAT_STEREO, d->ss->dither_enable ? FORMAT_8_BIT : FORMAT_16_BIT);
		hw_start_playback();
		d->playback.state = STATE_RUN;
	}

   	release_sem(d->playback.completion_sem);
	
   	return B_OK;
}

/*****************************************************************************
 * es1370_sound_read_buffer()
 *****************************************************************************
 * Takes an (audio_buffer_header*) argument. Call is allowed to return before 
 * the buffer is full but the capture completion semaphore must be released 
 * as soon as the buffer is full.
 */
static status_t
es1370_sound_read_buffer(dev_info *d, audio_buffer_header *header)
{
	size_t size = header->reserved_1 - sizeof(*header);
	void *data = (void *) (header + 1);

	if (size)
	{
		d->read_count++;

		if (d->capture.state == STATE_PAUSE)
		{
			d->capture.sample_rate = d->ss->sample_rate;
			hw_set_capture_buffer((int32) pci_bus->ram_address (d->capture.scatter[0].address),d->capture.scatter[0].size);
			hw_set_capture_sample_count((1024/(d->ss->dither_enable ? 1 : 2)) - 1); // 16-bit samples, o-based count
			hw_set_sample_rate(sample_rate_table[d->capture.sample_rate]);
			hw_set_capture_format(FORMAT_STEREO, d->ss->dither_enable ? FORMAT_8_BIT : FORMAT_16_BIT);
			hw_start_capture();
			d->capture.state = STATE_RUN;
		}
	}
	
	header->time = system_time();

	if (size)
	{
		acquire_sem_etc(d->capture.hw_lock,1,B_CAN_INTERRUPT,0);

		if ((d->capture.offset+size) > (d->capture.scatter[0].size))
		{
			memcpy(data,(void *)((uint32)d->capture.data+d->capture.offset),
				   d->capture.scatter[0].size-d->capture.offset);
			memcpy((void *)((uint32)data+(d->capture.scatter[0].size-d->capture.offset)),
				   (void *)d->capture.data,size-(d->capture.scatter[0].size-d->capture.offset));
		}
		else
		{
			memcpy(data,(void *)((uint32)d->capture.data+d->capture.offset),size);
		}
	
		d->capture.offset += size;
		d->capture.offset %= d->capture.scatter[0].size;				
	}	
   	
   	release_sem(d->capture.completion_sem);

	return B_OK;
}

/*****************************************************************************
 * es1370_sound_lock_for_dma()
 *****************************************************************************
 * Can be ignored except on Macintosh. A Macintosh driver should call 
 * lock_memory() on the audio buffer with the B_DMA_IO flag.
 */
static status_t
es1370_sound_lock_for_dma(dev_info *d, audio_buffer_header *header)
{
	size_t size = header->reserved_1 - sizeof(*header);
	void *data = (void *) (header + 1);

	//dbgprintf(("es1370_sound_lock_for_dma\n"));
	
	return B_OK;
}
