/* 
	hardware.c - Hardware access for the Ensoniq AudioPCI ES1370 device.

    Copyright (C) 1998 HockYiung Huang (leclec@pacific.net.sg)
    Port to R4 (C) 1999 Marc Schefer (mschefer@iprolink.ch)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.h"
#include "hardware.h"


static
uint8 es1370_register_defaults[] =
{//    0,      1,      2,      3,      4,      5,      6,      7,
    0x80,   0x80,   0x06,   0x06,   0x06,   0x06,   0x06,   0x06,
    0x06,   0x06,   0x06,   0x06,   0x06,   0x06,   0x06,   0x80,
    0x00,   0x00,   0x00,   0x00,   0x00,   0x00,   0x03,   0x03,
    0x00,   0x00
};

static
uint8 es1370_register[] =
{//    0,      1,      2,      3,      4,      5,      6,      7,
    0x80,   0x80,   0x06,   0x06,   0x06,   0x06,   0x06,   0x06,
    0x06,   0x06,   0x06,   0x06,   0x06,   0x06,   0x06,   0x80,
    0x00,   0x00,   0x00,   0x00,   0x00,   0x00,   0x03,   0x03,
    0x00,   0x00
};

void
hw_init(void)
{
	uint32 control;

	// Enable GP bit ??
	control = B_ICS_CONTROL_XCTL1;
	
	// Enable the CODEC here, since both the topology & wave miniports
    // need to access it anyway.
    control |= B_ICS_CONTROL_CDC_EN;
	
	// Enable SERR
    control &= ~B_ICS_CONTROL_SERR_DISABLE;

    // Write it !
    dio_write_register_dword(ICS_CONTROL_REGISTER,control);
    spin(10);
    
    // reset the codec
    hw_codec_reset();
    
    // unmute master 
    hw_codec_reg_write(MASTER_VOLUME_LCH_REGISTER,0x00);
    hw_codec_reg_write(MASTER_VOLUME_RCH_REGISTER,0x00);
	
	// there are more to do here !!
	// need to initialize the DAC/ADC to a known state.
    // Read ICS control register
    control = dio_read_register_dword(ICS_CONTROL_REGISTER);

    // Select the programmable clock generator as source for the
    // CODEC DAC(DAC2).
    control &= ~B_ICS_CONTROL_MSBB;

    // Both CODEC not run in SYNC.
    control &= ~B_ICS_CONTROL_DAC_SYNC;

    // Disable DAC2 & ADC
    control &= ~B_ICS_CONTROL_DAC1_EN;
    control &= ~B_ICS_CONTROL_DAC2_EN;
    control &= ~B_ICS_CONTROL_ADC_EN;

    // Write settings back.
    dio_write_register_dword(ICS_CONTROL_REGISTER,control);

    // Selects the clocks for CODEC-ADC(PLL) & CODEC-DAC(PLL).
    dio_write_register_word(CODEC_WRITE_REGISTER,((CLOCK_SELECT_REGISTER<<8)|0));
    // wait...
    spin(CODEC_WRITE_DELAY);
	
	hw_set_sample_rate(11025/16);
}

/*****************************************************************************
 * Reset the codec and init the default register values cache, which
 * correspond to the default hardware values.
 */
void
hw_codec_reset(void)
{
    uint32 counter;
    uint32 status;
    uint32 i;
    
    // #RST = 0, #PD & #RST will init to 1
    dio_write_register_word(CODEC_WRITE_REGISTER,((RESET_POWER_DOWN_REGISTER<<8)|
                                                  B_RESET_POWER_DOWN_PD));
    // Some delay required
    spin(CODEC_RESET_DELAY);
    
    dio_write_register_word(CODEC_WRITE_REGISTER,((RESET_POWER_DOWN_REGISTER<<8)|
                                                  B_RESET_POWER_DOWN_PD|B_RESET_POWER_DOWN_RST));
    // Some delay required
    spin(CODEC_RESET_DELAY);

    // Poll the CSTAT bit, which tells if the CODEC is ready.
    counter = 0xffff;
    status = dio_read_register_dword(ICS_STATUS_REGISTER);
    while ((counter!=0) && (status & B_ICS_STATUS_CSTAT))
    {
        status = dio_read_register_dword(ICS_STATUS_REGISTER);
        counter--;
    }

    // Initialize the registers
    for (i=0;i<sizeof(es1370_register_defaults);i++)
    {
        es1370_register[i] = es1370_register_defaults[i];
    }
}

uint8
hw_codec_reg_read(uint8 reg)
{
    return es1370_register[reg];
}

void
hw_codec_reg_write(uint8 reg, uint8 value)
{
    // write it out
    dio_write_register_word(CODEC_WRITE_REGISTER,((reg<<8)|value));
    // wait for a bit while...
    spin(CODEC_WRITE_DELAY);
    // store it
    es1370_register[reg] = value;
}

inline
uint8 this_many_ones(uint8 ones)
{
    return ~(((uint8)0xff) << ones);
}

uint8
hw_read_bits_from_codec(uint8 reg, uint bits, uint8 shift)
{
    uint8 data = hw_codec_reg_read(reg);

    return(data >> shift) & this_many_ones(bits);
}

void
hw_write_bits_to_codec(uint8 reg, uint8 bits, uint8 shift, uint8 value)
{
    uint8 mask = this_many_ones(bits) << shift;
    uint8 data = hw_codec_reg_read(reg);

    hw_codec_reg_write(reg, (data & ~mask) | ( (value << shift) & mask));
}

void
hw_set_sample_rate(uint32 sample_rate)
{
    uint32 control = dio_read_register_dword(ICS_CONTROL_REGISTER);

    // Programmable clock freq = Master clock freq/ratio
    uint32 ratio = ((MASTER_CLOCK_FREQUENCY/16)/sample_rate)-2;

    // Set the programmable clock divide ratio.
    control &= ~B_ICS_CONTROL_PCLKDIV;
    control |= (ratio<<16);

    // Write it.
    dio_write_register_dword(ICS_CONTROL_REGISTER,control);
}

/*****************************************************************************
 * hw_set_playback_buffer()
 *****************************************************************************
 * Set the pci transfer address & size for the playback stream.
 */
void
hw_set_playback_buffer(uint32 physical_address, uint32 size)
{
    // Memory page register = 1100b.
    dio_write_register_dword(MEMORY_PAGE_REGISTER,0x0000000C);
    // Program the DAC2 frame register 1
    dio_write_register_dword(DAC2_FRAME_REGISTER_1,physical_address);
    // Program the DAC2 frame register 2
    size /= 4;
    dio_write_register_dword(DAC2_FRAME_REGISTER_2,size-1);
}

/*****************************************************************************
 * hw_set_capture_buffer()
 *****************************************************************************
 * Set the pci transfer address & size for the capture stream.
 */
void
hw_set_capture_buffer(uint32 physical_address, uint32 size)
{
	// Memory page register = 1101b.
    dio_write_register_dword(MEMORY_PAGE_REGISTER,0x0000000D);
    // Program the ADC frame register 1
    dio_write_register_dword(ADC_FRAME_REGISTER_1,physical_address);
    // Program the ADC frame register 2
    size /= 4;
    dio_write_register_dword(ADC_FRAME_REGISTER_2,size-1);
}

/*****************************************************************************
 * hw_set_playback_format()
 *****************************************************************************
 * Set the playback stream sample format.
 */
void
hw_set_playback_format(bool format_stereo, bool format_16_bit)
{
	uint32 mask;
	uint8 bits;
	uint32 control;
	
    mask = B_STEREO|B_SIXTEEN_BIT;
    mask <<= 2;
    bits = (format_stereo) ? B_STEREO : 0;
    bits |= (format_16_bit) ? B_SIXTEEN_BIT : 0;
    bits <<= 2;

    control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
    control &= ~mask;

    dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER, (control|bits));
}

/*****************************************************************************
 * hw_set_capture_format()
 *****************************************************************************
 * Set the capture stream sample format.
 */
void
hw_set_capture_format(bool format_stereo, bool format_16_bit)
{
	uint32 mask;
	uint8 bits;
	uint32 control;
	
    mask = B_STEREO|B_SIXTEEN_BIT;
    mask <<= 4;
    bits = (format_stereo) ? B_STEREO : 0;
    bits |= (format_16_bit) ? B_SIXTEEN_BIT : 0;
    bits <<= 4;

    control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
    control &= ~mask;

    dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER, (control|bits));
}


/*****************************************************************************
 * hw_set_playback_sample_count()
 *****************************************************************************
 * Set the playback stream sample(not frame!!) count for interrupt. This is 
 * number of samples (0-based !!) that the DAC will transfer before issuing an
 * interrupt.
 */
void
hw_set_playback_sample_count(uint32 sample_count)
{
    // Program DAC2 channel sample count register
    dio_write_register_dword(DAC2_CHANNEL_SAMPLE_COUNT_REGISTER,sample_count);
}

/*****************************************************************************
 * hw_set_capture_sample_count()
 *****************************************************************************
 * Set the capture stream sample(not frame!!) count for interrupt. This is 
 * number of samples (0-based !!) that the ADC will transfer before issuing an
 * interrupt.
 */
void
hw_set_capture_sample_count(uint32 sample_count)
{
	// Program ADC channel sample count register
    dio_write_register_dword(ADC_CHANNEL_SAMPLE_COUNT_REGISTER,sample_count);
}

/*****************************************************************************
 * hw_start_playback()
 *****************************************************************************
 * Start the playback...
 */
void 
hw_start_playback(void)
{
    uint32 control;

    // P2_END_INC ?? huh... where is next buffer 8bit : +1; 16bit : +2;
    control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
    control &= ~B_SERIAL_INTF_CONTROL_P2_END_INC;
    // need frame alignment ???
    control |= (control & 0x08) ? (2<<19) : (1<<19);
    //control |= (FORMAT_16_BIT) ? (2<<19) : (1<<19);
    dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
    // need frame alignment ???
    // P2_ST_INC ?? zero lah...
    control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
    control &= ~B_SERIAL_INTF_CONTROL_P2_ST_INC;
    dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
    // DAC2 in loop mode
    // DAC2 in play mode
    // DAC2 playback zeros when disabled
    control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
    control &= ~B_SERIAL_INTF_CONTROL_P2_LOOP_SEL; // loop mode
    control &= ~B_SERIAL_INTF_CONTROL_P2_PAUSE;
    control &= ~B_SERIAL_INTF_CONTROL_P2_DAC_SEN;
    dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
    // clear interrupt
    control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
    control &= ~B_SERIAL_INTF_CONTROL_P2_INTR_EN;
    dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
    control |= B_SERIAL_INTF_CONTROL_P2_INTR_EN;
    dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
    // enable DAC2
    control = dio_read_register_dword(ICS_CONTROL_REGISTER);
    control &= ~B_ICS_CONTROL_DAC2_EN;
    dio_write_register_dword(ICS_CONTROL_REGISTER,control);
    control |= B_ICS_CONTROL_DAC2_EN;
    dio_write_register_dword(ICS_CONTROL_REGISTER,control);
}

/*****************************************************************************
 * hw_start_capture()
 *****************************************************************************
 * Start the capture...
 */
void 
hw_start_capture(void)
{
    uint32 control;

	// ADC in loop mode
	control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
	control &= ~B_SERIAL_INTF_CONTROL_R1_LOOP_SEL;
	dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
	// clear interrupt
	control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
	control &= ~B_SERIAL_INTF_CONTROL_R1_INT_EN;
	dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
	control |= B_SERIAL_INTF_CONTROL_R1_INT_EN;
	dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);
	// enable ADC
	control = dio_read_register_dword(ICS_CONTROL_REGISTER);
	control &= ~B_ICS_CONTROL_ADC_EN;
	dio_write_register_dword(ICS_CONTROL_REGISTER,control);
	control |= B_ICS_CONTROL_ADC_EN;
	dio_write_register_dword(ICS_CONTROL_REGISTER,control);
}

/*****************************************************************************
 * hw_stop_playback()
 *****************************************************************************
 * Stop the playback...
 */
void 
hw_stop_playback(void)
{
    uint32 control;

    // Pause...
    control = dio_read_register_dword(SERIAL_INTF_CONTROL_REGISTER);
    control |= B_SERIAL_INTF_CONTROL_P2_PAUSE;
    dio_write_register_dword(SERIAL_INTF_CONTROL_REGISTER,control);

    // Stop by disabling DAC2
    control = dio_read_register_dword(ICS_CONTROL_REGISTER);
    control &= ~B_ICS_CONTROL_DAC2_EN;
    dio_write_register_dword(ICS_CONTROL_REGISTER,control);
}

/*****************************************************************************
 * hw_stop_capture()
 *****************************************************************************
 * Stop the capture...
 */
void 
hw_stop_capture(void)
{
    uint32 control;

    // stop...
    control = dio_read_register_dword(ICS_CONTROL_REGISTER);
    control &= ~B_ICS_CONTROL_ADC_EN;
    dio_write_register_dword(ICS_CONTROL_REGISTER,control);
}
