/* 
	hardware.h - Hardware definitions for the Ensoniq AudioPCI ES1370 device.

    Copyright (C) 1998 HockYiung Huang (leclec@pacific.net.sg)
    Port to R4 (C) 1999 Marc Schefer (mschefer@iprolink.ch)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _HARDWARE_H_
#define _HARDWARE_H_

/*****************************************************************************
* ENSONIQ ES1370 SPECIFICATIONS
*****************************************************************************/

/*****************************************************************************
* Interrupt/Chip Select(ICS)
*/
#define ICS_CONTROL_REGISTER        0x00
#define ICS_STATUS_REGISTER         0x04

#define B_ICS_CONTROL_ADC_STOP      0x80000000
#define B_ICS_CONTROL_XCTL1         0x40000000
#define B_ICS_CONTROL_NOTUSED_29    0x20000000
#define B_ICS_CONTROL_PCLKDIV       0x1FFF0000
#define B_ICS_CONTROL_MSFMTSEL      0x00008000
#define B_ICS_CONTROL_MSBB          0x00004000
#define B_ICS_CONTROL_WTSRSEL       0x00003000
#define B_ICS_CONTROL_DAC_SYNC      0x00000800
#define B_ICS_CONTROL_CCB_INTRM     0x00000400
#define B_ICS_CONTROL_M_CB          0x00000200
#define B_ICS_CONTROL_XCTL0         0x00000100
#define B_ICS_CONTROL_BREQ          0x00000080
#define B_ICS_CONTROL_DAC1_EN       0x00000040
#define B_ICS_CONTROL_DAC2_EN       0x00000020
#define B_ICS_CONTROL_ADC_EN        0x00000010
#define B_ICS_CONTROL_UART_EN       0x00000008
#define B_ICS_CONTROL_JYSTK_EN      0x00000004
#define B_ICS_CONTROL_CDC_EN        0x00000002
#define B_ICS_CONTROL_SERR_DISABLE  0x00000001

#define B_ICS_STATUS_INTR           0x80000000
#define B_ICS_STATUS_NOTUSED_3011   0x7FFFF800
#define B_ICS_STATUS_CSTAT          0x00000400
#define B_ICS_STATUS_CBUSY          0x00000200
#define B_ICS_STATUS_CWRIP          0x00000100
#define B_ICS_STATUS_NOTUSED_07     0x00000080
#define B_ICS_STATUS_VC             0x00000060
#define B_ICS_STATUS_MCCB           0x00000010
#define B_ICS_STATUS_UART           0x00000008
#define B_ICS_STATUS_DAC1           0x00000004
#define B_ICS_STATUS_DAC2           0x00000002
#define B_ICS_STATUS_ADC            0x00000001

#define MASTER_CLOCK_FREQUENCY      22579200
#define MIN_SAMPLING_RATE           5000
#define MAX_SAMPLING_RATE           48000
#define MIN_BITS_PER_SAMPLE         8
#define MAX_BITS_PER_SAMPLE         16
#define MIN_CHANNELS                1
#define MAX_CHANNELS                2

/*****************************************************************************
* UART
*/
#define UART_DATA_REGISTER              0x08
#define UART_STATUS_REGISTER            0x09
#define UART_CONTROL_REGISTER           0x09
#define UART_RESERVED_REGISTER          0x0A

#define B_UART_DATA_DATA                0x00FF

#define B_UART_STATUS_RXINT             0x80
#define B_UART_STATUS_NOTUSED_1411      0x78
#define B_UART_STATUS_TXINT             0x04
#define B_UART_STATUS_TXRDY             0x02
#define B_UART_STATUS_RXRDY             0x01

#define B_UART_CONTROL_RXINTEN          0x80
#define B_UART_CONTROL_TXINTEN          0x60
#define B_UART_CONTROL_TXRDYINTEN       0x20
#define B_UART_CONTROL_NOTUSED_1210     0x1C
#define B_UART_CONTROL_CNTRL            0x03

#define B_UART_RESERVERD_NOTUSED_0711   0xFE
#define B_UART_RESERVED_TESTMODE        0x01

/*****************************************************************************
* Host Interface - Memory Page
*/
#define MEMORY_PAGE_REGISTER            0x0C

#define B_MEMORY_PAGE_NOTUSED_3104      0xFFFFFFF0
#define B_MEMORY_PAGE_MEMORY_PAGE       0x00000003

/*****************************************************************************
* CODEC Write Register
*/
#define CODEC_WRITE_REGISTER            0x10

#define B_CODEC_WRITE_NOTUSED_3116      0xFFFF0000
#define B_CODEC_WRITE_ADDRESS           0x0000FF00
#define B_CODEC_WRITE_DATA              0x000000FF

/*****************************************************************************
* Serial Interface
*/
#define SERIAL_INTF_CONTROL_REGISTER            0x20
#define DAC1_CHANNEL_SAMPLE_COUNT_REGISTER      0x24
#define DAC2_CHANNEL_SAMPLE_COUNT_REGISTER      0x28
#define ADC_CHANNEL_SAMPLE_COUNT_REGISTER       0x2C

#define B_SERIAL_INTF_CONTROL_NOTUSED_3122      0xFFC00000
#define B_SERIAL_INTF_CONTROL_P2_END_INC        0x00380000
#define B_SERIAL_INTF_CONTROL_P2_ST_INC         0x00070000
#define B_SERIAL_INTF_CONTROL_R1_LOOP_SEL       0x00008000
#define B_SERIAL_INTF_CONTROL_P2_LOOP_SEL       0x00004000
#define B_SERIAL_INTF_CONTROL_P1_LOOP_SEL       0x00002000
#define B_SERIAL_INTF_CONTROL_P2_PAUSE          0x00001000
#define B_SERIAL_INTF_CONTROL_P1_PAUSE          0x00000800
#define B_SERIAL_INTF_CONTROL_R1_INT_EN         0x00000400
#define B_SERIAL_INTF_CONTROL_P2_INTR_EN        0x00000200
#define B_SERIAL_INTF_CONTROL_P1_INTR_EN        0x00000100
#define B_SERIAL_INTF_CONTROL_P1_SCT_RLD        0x00000080
#define B_SERIAL_INTF_CONTROL_P2_DAC_SEN        0x00000040
#define B_SERIAL_INTF_CONTROL_R1_S_EB           0x00000020
#define B_SERIAL_INTF_CONTROL_R1_S_MB           0x00000010
#define B_SERIAL_INTF_CONTROL_P2_S_EB           0x00000008
#define B_SERIAL_INTF_CONTROL_P2_S_MB           0x00000004
#define B_SERIAL_INTF_CONTROL_P1_S_EB           0x00000002
#define B_SERIAL_INTF_CONTROL_P1_S_MB           0x00000001

#define B_STEREO        0x01
#define B_SIXTEEN_BIT   0x02

#define B_DAC1_CHANNEL_SAMPLE_COUNT_CURR_SAMP_CT    0xFFFF0000
#define B_DAC1_CHANNEL_SAMPLE_COUNT_SAMP_CT         0x0000FFFF

#define B_DAC2_CHANNEL_SAMPLE_COUNT_CURR_SAMP_CT    0xFFFF0000
#define B_DAC2_CHANNEL_SAMPLE_COUNT_SAMP_CT         0x0000FFFF

#define B_ADC_CHANNEL_SAMPLE_COUNT_CURR_SAMP_CT     0xFFFF0000
#define B_ADC_CHANNEL_SAMPLE_COUNT_SAMP_CT          0x0000FFFF

/*****************************************************************************
* Host Interface - Memory
*/
#define DAC1_FRAME_REGISTER_1       0x30
#define DAC1_FRAME_REGISTER_2       0x34
#define DAC2_FRAME_REGISTER_1       0x38
#define DAC2_FRAME_REGISTER_2       0x3C
#define ADC_FRAME_REGISTER_1        0x30
#define ADC_FRAME_REGISTER_2        0x34
#define UART_FIFO_REGISTER          0x30

#define B_DAC1_FRAME_PCI_ADDRESS    0xFFFFFFFF
#define B_DAC1_FRAME_CURRENT_COUNT  0xFFFF0000
#define B_DAC1_FRAME_BUFFER_SIZE    0x0000FFFF

#define B_DAC2_FRAME_PCI_ADDRESS    0xFFFFFFFF
#define B_DAC2_FRAME_CURRENT_COUNT  0xFFFF0000
#define B_DAC2_FRAME_BUFFER_SIZE    0x0000FFFF

#define B_ADC_FRAME_PCI_ADDRESS     0xFFFFFFFF
#define B_ADC_FRAME_CURRENT_COUNT   0xFFFF0000
#define B_ADC_FRAME_BUFFER_SIZE     0x0000FFFF

#define B_UART_FIFO_NOTUSED_3109    0xFFFFFE00
#define B_UART_FIFO_BYTE_VALID      0x00000100
#define B_UART_FIFO_BYTE            0x000000FF

/*****************************************************************************
* AKM AK4531 CODEC Control Register Map
*/
#define MASTER_VOLUME_LCH_REGISTER      0x00
#define MASTER_VOLUME_RCH_REGISTER      0x01
#define VOICE_VOLUME_LCH_REGISTER       0x02
#define VOICE_VOLUME_RCH_REGISTER       0x03
#define FM_VOLUME_LCH_REGISTER          0x04
#define FM_VOLUME_RCH_REGISTER          0x05
#define CD_AUDIO_VOLUME_LCH_REGISTER    0x06
#define CD_AUDIO_VOLUME_RCH_REGISTER    0x07
#define LINE_VOLUME_LCH_REGISTER        0x08
#define LINE_VOLUME_RCH_REGISTER        0x09
#define AUX_VOLUME_LCH_REGISTER         0x0A
#define AUX_VOLUME_RCH_REGISTER         0x0B
#define MONO1_VOLUME_REGISTER           0x0C
#define MONO2_VOLUME_REGISTER           0x0D
#define MIC_VOLUME_REGISTER             0x0E
#define MONO_OUT_VOLUME_REGISTER        0x0F
#define OUTPUT_MIXER_SW1_REGISTER       0x10
#define OUTPUT_MIXER_SW2_REGISTER       0x11
#define LCH_INPUT_MIXER_SW1_REGISTER    0x12
#define RCH_INPUT_MIXER_SW1_REGISTER    0x13
#define LCH_INPUT_MIXER_SW2_REGISTER    0x14
#define RCH_INPUT_MIXER_SW2_REGISTER    0x15
#define RESET_POWER_DOWN_REGISTER       0x16
#define CLOCK_SELECT_REGISTER           0x17
#define AD_INPUT_SELECT_REGISTER        0x18
#define MIC_AMP_GAIN_REGISTER           0x19

// the bits for volume att/gain
#define BITS_MASTER_VOLUME_ATT          0x1F
#define BITS_VOICE_VOLUME_GAIN          0x1F
#define BITS_FM_VOLUME_GAIN             0x1F
#define BITS_CD_AUDIO__VOLUME_GAIN      0x1F
#define BITS_LINE_VOLUME_GAIN           0x1F
#define BITS_AUX_VOLUME_GAIN            0x1F
#define BITS_MONO1_VOLUME_GAIN          0x1F
#define BITS_MONO2_VOLUME_GAIN          0x1F
#define BITS_MIC_VOLUME_GAIN            0x1F
#define BITS_MONO_OUT_VOLUME_ATT        0x07

// mix bit position(0-7)
#define MIXBIT_MASTER_VOLUME_MUTE       7
#define MIXBIT_VOICE_VOLUME_MUTE        7
#define MIXBIT_FM_VOLUME_MUTE           7
#define MIXBIT_CD_AUDIO_VOLUME_MUTE     7
#define MIXBIT_LINE_VOLUME_MUTE         7
#define MIXBIT_AUX_VOLUME_MUTE          7
#define MIXBIT_MONO1_VOLUME_MUTE        7
#define MIXBIT_MONO2_VOLUME_MUTE        7
#define MIXBIT_MIC_VOLUME_MUTE          7
#define MIXBIT_MONO_OUT_VOLUME_MUTE     7

// mix bit position(0-7)
#define MIXBIT_OUTPUT_SW1_FML           6
#define MIXBIT_OUTPUT_SW1_FMR           5
#define MIXBIT_OUTPUT_SW1_LINEL         4
#define MIXBIT_OUTPUT_SW1_LINER         3
#define MIXBIT_OUTPUT_SW1_CDL           2
#define MIXBIT_OUTPUT_SW1_CDR           1
#define MIXBIT_OUTPUT_SW1_MIC           0
#define MIXBIT_OUTPUT_SW2_AUXL          5
#define MIXBIT_OUTPUT_SW2_AUXR          4
#define MIXBIT_OUTPUT_SW2_VOICEL        3
#define MIXBIT_OUTPUT_SW2_VOICER        2
#define MIXBIT_OUTPUT_SW2_MONO2         1
#define MIXBIT_OUTPUT_SW2_MONO1         0

// mix bit position(0-7)
#define MIXBIT_INPUT_SW1_FML            6
#define MIXBIT_INPUT_SW1_FMR            5
#define MIXBIT_INPUT_SW1_LINEL          4
#define MIXBIT_INPUT_SW1_LINER          3
#define MIXBIT_INPUT_SW1_CDL            2
#define MIXBIT_INPUT_SW1_CDR            1
#define MIXBIT_INPUT_SW1_MIC            0
#define MIXBIT_INPUT_SW2_TMIC           7
#define MIXBIT_INPUT_SW2_TMONO1         6
#define MIXBIT_INPUT_SW2_TMONO2         5
#define MIXBIT_INPUT_SW2_AUXL           4
#define MIXBIT_INPUT_SW2_AUXR           3
#define MIXBIT_INPUT_SW2_VOICE          2
#define MIXBIT_INPUT_SW2_MONO2          1
#define MIXBIT_INPUT_SW2_MONO1          0

// bit to power down & reset
#define B_RESET_POWER_DOWN_PD           0x02
#define B_RESET_POWER_DOWN_RST          0x01

// bit to select the clocks
#define B_CLOCK_SELECT_CSEL2            0x02
#define B_CLOCK_SELECT_CSEL1            0x01

// mix bit position(0-7)
#define MIXBIT_AD_INPUT_SELECT_ADSEL    0

// mix bit position(0-7)
#define MIXBIT_MIC_AMP_GAIN_MGAIN       0

// Write/Reset/Power Down Timing in uS
// Need to double check specs to get exact required timing... this is just
// an arbitray value for now...
#define CODEC_RESET_DELAY       20
#define CODEC_WRITE_DELAY       20


/*****************************************************************************
 * Hardware access routines.
 */
void hw_init(void);
void hw_codec_reset(void);
void hw_codec_reg_write(uint8 reg, uint8 value);
void hw_write_bits_to_codec(uint8 reg, uint8 bits, uint8 shift, uint8 value);
void hw_set_sample_rate(uint32 sample_rate);
void hw_set_playback_buffer(uint32 physical_address, uint32 size);
void hw_set_capture_buffer(uint32 physical_address, uint32 size);
void hw_set_playback_format(bool format_stereo, bool format_16_bit);
void hw_set_capture_format(bool format_stereo, bool format_16_bit);
void hw_set_playback_sample_count(uint32 sample_count);
void hw_set_capture_sample_count(uint32 sample_count);
void hw_start_playback(void);
void hw_start_capture(void);
void hw_stop_playback(void);
void hw_stop_capture(void);

extern int32 portbase;
extern pci_module_info *pci_bus;

/*****************************************************************************
 * Hardware access definitions.
 */
#define dio_write_register_byte(reg,data)   pci_bus->write_io_8(portbase+reg,(uint8)data)
#define dio_write_register_word(reg,data)   pci_bus->write_io_16(portbase+reg,(uint16)data)
#define dio_write_register_dword(reg,data)  pci_bus->write_io_32(portbase+reg,(uint32)data)
#define dio_read_register_byte(reg)         pci_bus->read_io_8(portbase+reg)
#define dio_read_register_word(reg)         pci_bus->read_io_16(portbase+reg)
#define dio_read_register_dword(reg)        pci_bus->read_io_32(portbase+reg)

#define FORMAT_STEREO 	true
#define FORMAT_MONO		false
#define FORMAT_16_BIT	true
#define FORMAT_8_BIT	false

#endif // _HARDWARE_H_