//  inflating: ifsega/ifsega.c
//
//	device driver for I/O data Co. IF-SEGA/ISA and IF-SEGA/PCI/2
//
//  Install directory      : /boot/home/config/add-ons/kernel/drivers/bin
//  Create Link file under : /boot/home/config/add-ons/kernel/drivers/dev/joystick

//  Copyright (C) 2000 Jun Suzuki & Takayuki Ito

#include <KernelExport.h>
#include <Drivers.h>
#include <Errors.h>
#include <PCI.h>
#include <ISA.h>
#include <isapnp.h>
#include <config_manager.h>
#include "ifsega.h"
#include "joystick_driver.h"

#define	IFSEGA_PCI

int type,dev,num,pad,ax,ay,al,ar,data;
struct ifsega_status status;
static const char *ifsega_name[];

//yuki
#define	DUMMY_PORT	0

#ifdef	IFSEGA_PCI
#define	AREA_NAME	"IF-SEGA"
static volatile uint32 *vaddr = NULL;
#endif
static uint32	baseport = 0x1000;	/* ISA */

#ifdef	USE_GENERIC_GAMEPORT_MODULE
static void* storage;
static char gameport_name [] = "generic/gameport/v1";
static generic_gameport_module *gameport;
#endif

static joystick_module_info joy_mod_info;
static joystick_module  *joy_mod;

#define	NUM_PORTS	2
static bool enhanced [NUM_PORTS];
static bool openport [NUM_PORTS];

//yuki

bool xinv = false;
bool yinv = false;
int active = 0;

/* -----
	null-terminated array of device names supported by this driver
----- */

static const char *ifsega_name[] = {
	"joystick/ifsega/0",
	"joystick/ifsega/1",
	NULL
};

/* ----------
	init_hardware - called once the first time the driver is loaded
----- */
_EXPORT status_t
init_hardware (void)
{
#ifdef	IFSEGA_PCI
	status_t err = ENODEV;
	pci_module_info *pci;
	int i;
	pci_info info;
	uint32	physical_address;
	area_id	area;
#endif

	dprintf ("ifsega: init_hardware()\n");

#ifdef	IFSEGA_PCI
	if (get_module (B_PCI_MODULE_NAME, (module_info **)&pci))
		return ENOSYS;

	for (i = 0; (*pci->get_nth_pci_info)(i, &info) == B_OK; i++)
	{
		if (info.vendor_id == 0x1033 && info.device_id == 0x0046)
		{
			err = B_OK;
			break;
		}
	}
	put_module (B_PCI_MODULE_NAME);

	if (err == B_OK)
	{
		/* IF-SEGA/PCI found */
		physical_address = info.u.h0.base_registers [0];
		dprintf ("ifsega: PCI found at 0x%x\n", physical_address);
		area = find_area (AREA_NAME);
		if (area < 0)
			area = map_physical_memory 
				(AREA_NAME, (void *) physical_address, B_PAGE_SIZE,
				B_ANY_KERNEL_ADDRESS, B_WRITE_AREA | B_READ_AREA, 
				(void **) &vaddr);
		if (area < 0)
			return area;
	}
#endif
	return B_OK;
}


/* ----------
	init_driver - optional function - called every time the driver
	is loaded.
----- */
_EXPORT status_t
init_driver (void)
{
	int port;
#ifdef	IFSEGA_PCI
	area_id	area;
	area_info	ainfo;
#endif

	dprintf ("ifsega: init_driver()\n");
#ifdef	IFSEGA_PCI
	vaddr = NULL;
	if ((area = find_area (AREA_NAME)) >= 0)
	{
		/* IF-SEGA/PCI */
		get_area_info (area, &ainfo);
		vaddr = (uint32 *)((uint8 *)ainfo.address + 0x58);
	}
	else
#endif
	{
		/* IF-SEGA/ISA */
		if (probe_isapnp(&baseport) != B_OK) 
			dprintf ("ifsega: IF-SEGA/ISA Seriese not found\n");
//		baseport = 0x1000;
	}

#ifdef	USE_GENERIC_GAMEPORT_MODULE
	if (get_module (gameport_name, (module_info **)&gameport))
	{
		dprintf ("ifsega: failed to get gameport module\n");
		return ENOSYS;
	}
	(*gameport->create_device)(0, &storage);
#endif

	for (port = 0; port < NUM_PORTS; port++)
		openport [port] = false;
	return B_OK;
}


/* ----------
	uninit_driver - optional function - called every time the driver
	is unloaded
----- */
_EXPORT void
uninit_driver (void)
{
}

/* ----------
	publish_devices - return a null-terminated array of devices
	supported by this driver.
----- */

_EXPORT const char**
publish_devices()
{
	return ifsega_name;
}

/* ----------
	open - handle open() calls
----- */

static status_t
ifsega_open (const char *name, uint32 flags, void** cookie)
{
	int i;
	const char *s;

	dprintf ("ifsega: open(%s)\n", name);
	for (i = 0; (s = ifsega_name[i]) != NULL; i++)
	{	if (strcmp (name, s) == 0)
			break;
	}
	if (s == NULL)
	{
		dprintf ("ifsega: no such device\n");
		return ENODEV;
	}
	if (openport [i])
	{
		/* FIXME: only one app can open a port */
		return B_ERROR;
	}

	openport [i] = true;
	enhanced [i] = false;
	*cookie = (void *) i;
#ifdef	USE_GENERIC_GAMEPORT_MODULE
	(*gameport->open_hook)(storage, flags, cookie);
#endif
	return B_OK;
}


/* ----------
	read - handle read() calls
----- */

static status_t
ifsega_read (void* cookie, off_t position, void *buf, size_t* num_bytes)
{
	uint32 op;
	struct ifsega_status io;
	int i;
	int port = (int)cookie;

	ifsega_get_status(port);
				
			io.num = status.num;
			for (i = 0; i < 6; i++) {
			    io.stick[i].type = status.stick[i].type;
				io.stick[i].dev = status.stick[i].dev;
				io.stick[i].pad = status.stick[i].pad;
				io.stick[i].ax = status.stick[i].ax;
				io.stick[i].ay = status.stick[i].ay;
				io.stick[i].al = status.stick[i].al;
				io.stick[i].ar = status.stick[i].ar;
			}

	if (enhanced [port])
	{
		extended_joystick joy_data;
		(*joy_mod->read)(&io, DUMMY_PORT,&joy_data, sizeof (extended_joystick));
		memcpy (buf, &joy_data, sizeof (extended_joystick));
		*num_bytes = sizeof (extended_joystick);
	}
	else
	{
		joystick joy_data;
		joy_data.timestamp = system_time ();
		joy_data.horizontal = ((status.stick[0].pad & 0x8000) ? false:true) * 32767 + ((status.stick[0].pad & 0x4000) ? false:true * -32767);
		joy_data.vertical   = ((status.stick[0].pad & 0x2000) ? false:true) * 32767 + ((status.stick[0].pad & 0x1000) ? false:true * -32767);
		/* false if pressed */
		joy_data.button1 = (status.stick[0].pad & 0x400) ? true : false;
		joy_data.button2 = (status.stick[0].pad & 0x100) ? true : false;
		memcpy (buf, &joy_data, sizeof (joystick));
		*num_bytes = sizeof (joystick);
	}

#ifdef	USE_GENERIC_GAMEPORT_MODULE
	(*gameport->read_hook)(cookie, position, buf, num_bytes);
#endif


	return B_OK;
}


/* ----------
	write - handle write() calls
----- */

static status_t
ifsega_write (void* cookie, off_t position, const void* buffer, size_t* num_bytes)
{
	return B_IO_ERROR;
}


/* ----------
	control - handle ioctl calls
----- */

static status_t
ifsega_control (void* cookie, uint32 op, void* arg, size_t len)
{
	status_t err = B_ERROR;
	int i;
	struct ifsega_status *io = (struct ifsega_status *)arg;
	int port = (int)cookie;

//	dprintf ("ifsega: entering control op=%d arg=%x len=%d\n",op,io->data,len);

	switch (op) {

	case B_JOYSTICK_GET_SPEED_COMPENSATION:
		break;

	case B_JOYSTICK_SET_SPEED_COMPENSATION:
		break;

	case B_JOYSTICK_GET_MAX_LATENCY:
		break;

	case B_JOYSTICK_SET_MAX_LATENCY:
		break;

	// called by BJoystick, Joysticks pref
	case B_JOYSTICK_SET_DEVICE_MODULE:
		// XXX
		if (strncmp (((joystick_module_info *)arg)->module_name, 
			"media/joy/saturn/", 17) == 0)
		{
			joy_mod_info = *(joystick_module_info *)arg;
			if (get_module (joy_mod_info.module_name, 
				(module_info **)&joy_mod) == B_OK)
			{
				dprintf ("ifsega: ioctl(B_JOYSTICK_SET_DEVICE_MODULE): get_module() OK\n");
				enhanced [port] = true;
#if 0
				(*joy_mod->configure)(DUMMY_PORT, &joy_mod_info, 
					sizeof (joystick_module_info), NULL);
#endif
				err = B_OK;
			}
		}
		break;

	case B_JOYSTICK_GET_DEVICE_MODULE:
		if (enhanced [port])
			*(joystick_module_info *)arg = joy_mod_info;
		err = B_OK;
		break;

	case B_JOYSTICK_SET_RAW_MODE:
		break;

	case SELECT_DEVICE:
		active = io->active;
		break;

	case AXIS_INVERSE:
		xinv = io->xinv;
		yinv = io->yinv;
		break;

	case GET_PORTDATA:
			for (i = 1; i < 6; i++) {
			    status.stick[i].type = 0;
				status.stick[i].dev = 0;
				status.stick[i].pad = 0;
				status.stick[i].ax = 0x80;
				status.stick[i].ay = 0x80;
				status.stick[i].al = 0;
				status.stick[i].ar = 0;
			}
			ifsega_get_status(port);
			io->num = status.num;
			io->active = active;
			io->xinv = xinv;
			io->yinv = yinv;
			for (i = 0; i < 6; i++) {
			    io->stick[i].type = status.stick[i].type;
				io->stick[i].dev = status.stick[i].dev;
				io->stick[i].pad = status.stick[i].pad;
				io->stick[i].ax = status.stick[i].ax;
				io->stick[i].ay = status.stick[i].ay;
				io->stick[i].al = status.stick[i].al;
				io->stick[i].ar = status.stick[i].ar;
			}
		break;

	default:
		break;


}

#ifdef	USE_GENERIC_GAMEPORT_MODULE
	(*gameport->control_hook)(cookie, op, arg, len);
#endif
	return err;
}


/* ----------
	close - handle close() calls
----- */

static status_t
ifsega_close (void* cookie)
{
	dprintf ("ifsega: close()\n");
	openport [(int)cookie] = false;
	return B_OK;
}


/* -----
	free - called after the last device is closed, and after
	all i/o is complete.
----- */
static status_t
ifsega_free (void* cookie)
{
	return B_OK;
}


/* -----
	function pointers for the device hooks entry points
----- */

device_hooks ifsega_hooks = {
	ifsega_open, 			/* -> open entry point */
	ifsega_close, 			/* -> close entry point */
	ifsega_free,			/* -> free cookie */
	ifsega_control, 		/* -> control entry point */
	ifsega_read,			/* -> read entry point */
	ifsega_write			/* -> write entry point */
};

/* ----------
	find_device - return ptr to device hooks structure for a
	given device name
----- */

_EXPORT device_hooks*
find_device(const char* name)
{
	return &ifsega_hooks;
}

/*
	IF-SEGA low level functions
*/

#define	CTRL_REG	0
#define	DATA_REG	1

/*
	port = 0 or 1
	reg = CTRL_REG or DATA_REG
*/
static uint8 my_read_reg (int port, int reg)
{
	int idx = ((port & 1) << 1) | (reg & 1);
	uint8 value;

#ifdef	IFSEGA_PCI
	if (vaddr != NULL)
	{
		/* IF-SEGA2/PCI */
		uint32 data;
		data = 0xff003000 | (idx << 8);
		*vaddr = data | 0xC000;
		*vaddr = data | 0x8000;
		data = *vaddr;
		*vaddr = data | 0xC000;
		value = data;
	}
	else
#endif
		/* IF-SEGA/ISA */
		value = read_io_8 (baseport + (idx << 1) + 1);
	return value;
}

static void my_write_reg (int port, int reg, uint8 value)
{
	int idx = ((port & 1) << 1) | (reg & 1);

#ifdef	IFSEGA_PCI
	if (vaddr != NULL)
	{
		/* IF-SEGA2/PCI */
		uint32	data;

		data = 0xffff3000 | (idx << 8) | value;
		*vaddr = data | 0xC000;
		*vaddr = data | 0x4000;
		*vaddr = data | 0xC000;
	}
	else
#endif
		/* IF-SEGA/ISA */
		write_io_8 (baseport + (idx << 1) + 1, value);
}

int ifsega_devicetype(int port)
{
  int i= 0x40;
  int a=0;
  int result = 0;
 
  my_write_reg(port, CTRL_REG, 0x1F);
  while(i--)
  {
      my_write_reg(port, DATA_REG, 0x60);
      a = my_read_reg(port, DATA_REG);

      if ((a & 0x07) == 0x01 || (a & 0x07) == 0x04)
      	break;
  }
      if (a & 0x0C) result |= 0x08;
      if (a & 0x03) result |= 0x04;
      my_write_reg(port, DATA_REG, 0x20);
      a = my_read_reg(port, DATA_REG) & 0x0f;
      if (a & 0x0C) result |= 0x02;
      if (a & 0x03) result |= 0x01;

	type = result;

	return type;
}


// analog polling sub routine
int ifsega_apollsub(int port)
{
  int i = 0x100;
  int a = 0;
  int b = 0;

      a = my_read_reg(port, DATA_REG);
      my_write_reg(port, DATA_REG, a^0x20);
	  while(i--)
      {
         b = my_read_reg(port, DATA_REG);
         if ((a & 0x10) != (b & 0x10)) break;
      }
	return (b & 0x0f);
}

// port polling routine
int ifsega_get_status(int port)
{


  int a,b, i, j, multi;
	multi = 0;

  	num = 0;

  	pad = 0;
	ax = 0x80;
	ay = 0x80;
	al = 0;
	ar = 0;
	switch (ifsega_devicetype(port))
	{
		/* digital device */
		case 0x0b:
			/* normal pad & twin stick & codeless pad */
			ifsega_digitaldevice(port);
			break;
		case 5:
  		/* analog device */
  			dev = ifsega_apollsub(port);
			switch (dev)
			{
	  			/* multi controller & racing controller */
				case 1:
		  			/* multi controller analog mode or racing controller */
				case 0:
		  			/* multi controller digital mode */
					ifsega_poll_analogdevice(port, ifsega_apollsub(port));
					break;
	  			/* multi terminal 6 */
				case 4:
					ifsega_apollsub(port);
					ifsega_apollsub(port);
					ifsega_apollsub(port);
					
					for (i = 0; i < 6; i++)
					{
					  	pad = 0;
						ax = 0x80;
						ay = 0x80;
						al = 0;
						ar = 0;
						a = ifsega_apollsub(port);
						b = ifsega_apollsub(port);
						switch (a)
						{
							/* no pad */
							case 0x0f:
									dev = 4;
								break;
							case 0x0e:
								/* shuttle mouse */
								if (b == 3)
								{	
									ifsega_poll_mouse(port);
									dev = 3;
								}
								break;
							case 0x00:
								/* digital device  */
								ifsega_poll_analogdevice(port, b);
								dev = 0;
								break;
							case 0x01:
								/* analog device */
								ifsega_poll_analogdevice(port, b);
								dev = 1;
								break;
							case 0x03:
								/* sega saturn keyboad */
								dev = 3;
								num++;
								break;
							default:
						}
    					status.stick[i].type = type;
						status.stick[i].dev = dev;
						status.stick[i].pad = pad;
						status.stick[i].ax = ax;
						status.stick[i].ay = ay;
						status.stick[i].al = al;
						status.stick[i].ar = ar;
					}
					multi = 1;
					break;
				case 3:
		  			/* sega saturn keyboad */
		  			num++;
					ax = 0x80;
					ay = 0x80;
					al = 0;
					ar = 0;
					break;
			}
			break;
		case 0x0a:
			/* virtua gun */
  			num++;
			ax = 0x80;
			ay = 0x80;
			al = 0;
			ar = 0;
			dev = 0x0a;
			break;
		case 3:
			/* shuttle mouse */
			dev = ifsega_apollsub(port);
			ifsega_apollsub(port);
			ifsega_poll_mouse(port);
			break;
		case 0x0f:
			/* no device */
			num = 0;
			ax = 0x80;
			ay = 0x80;
			al = 0;
			ar = 0;
			dev = 0x0f;
			break;
		default:
			/* unknown device */
			num ++;
			ax = 0x80;
			ay = 0x80;
			al = 0;
			ar = 0;
			dev = 0x0f;
			break;
    }
    
	status.num = num;
	if (!multi) {
	    status.stick[0].type = type;
		status.stick[0].dev = dev;
		status.stick[0].pad = pad;
		status.stick[0].ax = ax;
		status.stick[0].ay = ay;
		status.stick[0].al = al;
		status.stick[0].ar = ar;
	}
    my_write_reg(port, DATA_REG, 0x60);
	return;
}


// multi_controller & racing controller sub routine
int ifsega_poll_analogdevice(int port, int count)
{
  int a,b,c,d;
  int i;

	a = ifsega_apollsub(port) & 0x0F;  //dir
	b = ifsega_apollsub(port) & 0x0F;    //sacb
	c = ifsega_apollsub(port) & 0x0F;    //rxyz
	d = ifsega_apollsub(port) & 0x0F;    //l
    pad = a << 12 | b << 8 | c << 4 | d;

	num++;

	if (count-=2)
		ax = (ifsega_apollsub(port) << 4 | ifsega_apollsub(port));
	else
		return;
	if (count-=1)
		ay = (ifsega_apollsub(port) << 4 | ifsega_apollsub(port));
	else
		return;
	if (count-=1)
		ar = (ifsega_apollsub(port) << 4 | ifsega_apollsub(port));
	else
		return;
	if (count-=1)
		al = (ifsega_apollsub(port) << 4 | ifsega_apollsub(port));
	return;
}

// shuttle_mouse sub routine
int ifsega_poll_mouse(int port)
{
  int a;
  int b,c,d,e;

	ifsega_apollsub(port);
	a = ifsega_apollsub(port);
    pad = 0xffff - (a << 8);

	a = ifsega_apollsub(port);
	b = ifsega_apollsub(port);
	c = ifsega_apollsub(port);
	d = ifsega_apollsub(port);

	ax = a << 4 | b;
	ay = c << 4 | d;

	num++;
	return;
}

// digital_pad sub routine
int ifsega_digitaldevice(int port)
{
  int a,b,c,d;
      my_write_reg(port, DATA_REG, 0xBF);
      a = my_read_reg(port, DATA_REG) & 0x0F; //rldu
      my_write_reg(port, DATA_REG, 0xDF);
      b = my_read_reg(port, DATA_REG) & 0x0F; //sacb 
      my_write_reg(port, DATA_REG, 0x9F);
      c = my_read_reg(port, DATA_REG) & 0x0F; //rxyz;
      my_write_reg(port, DATA_REG, 0xFF);
      d = my_read_reg(port, DATA_REG) & 0x0F; //l
      pad = a << 12 | b << 8 | c << 4 | d;
	  num++;
	  return;
}

// probe ISA PnP

static status_t probe_isapnp (uint32 *baseport)
{
	uint64 cookie;
	bool found = false;
	status_t ret = B_ERROR;
	int cfg_size;
	struct config_manager_for_driver_module_info *cm;
	struct device_info dummy_info, *info;
	struct device_configuration *config;
	resource_descriptor res;
	EISA_PRODUCT_ID ifsega_id;

	if (get_module (B_CONFIG_MANAGER_FOR_DRIVER_MODULE_NAME, 
		(module_info **) &cm) != B_OK)
	return B_ERROR;

	/* IF-SEGA/ISA product ID */
	MAKE_EISA_PRODUCT_ID (&ifsega_id, 'I', 'O', 'D', 0x800, 1);

	cookie = 0;
	while (!found && 
		cm->get_next_device_info (B_ISA_BUS, &cookie, 
		&dummy_info, sizeof (struct device_info)) == B_OK)
	{
		if (dummy_info.config_status == B_OK &&
			(info = malloc (dummy_info.size)) != NULL)
		{
			if (cm->get_device_info_for (cookie, info, dummy_info.size) == B_OK)
			{
				struct isa_info *isa = (struct isa_info *)
					((uchar *) info + info->bus_dependent_info_offset);

				#ifdef DEBUG
					dprintf (DRIVER_NAME ": EISA ID 0x%08x\n", isa->vendor_id);
				#endif
				/* device detection */
				isa->vendor_id &= 0xffffff7f; /* bit 7 must be zero */
				if (EQUAL_EISA_PRODUCT_ID (ifsega_id, *(EISA_PRODUCT_ID *)&isa->vendor_id))
				found = true;
			}
			free (info);
		}
	}

	if (found &&
		(cfg_size = cm->get_size_of_current_configuration_for (cookie)) > 0 &&
		(config = malloc (cfg_size)) != NULL)
	{
		/* obtain current configuration */
		if (cm->get_current_configuration_for 
			(cookie, config, cfg_size) == B_OK &&
			cm->get_nth_resource_descriptor_of_type
			(config, 0, B_IO_PORT_RESOURCE, &res, 
			sizeof (resource_descriptor)) == B_OK)
		{
			*baseport = res.d.r.minbase;
			ret = B_OK;
		}
		free (config);
	}

	put_module (B_CONFIG_MANAGER_FOR_DRIVER_MODULE_NAME);
	return ret;
}