#include "KPrefs.h"

#include <stdio.h>
#include <Alert.h>
#include <Directory.h>
#include <Entry.h>
#include <File.h>
#include <FindDirectory.h>
#include <Path.h>
#include <Font.h>

KPrefs::KPrefs(const char* name, const char* dir_name)
	: BLocker()
{
	fPrefsEntry = NULL;
	fSettingMessage = NULL;
	fDefaultMessage = new BMessage();
	
	this->MakeDefault();
	
	BPath		path;
	BDirectory	dir;
	BEntry*		entry = new BEntry();
	
	::find_directory(B_USER_SETTINGS_DIRECTORY, &path, true);
	dir.SetTo(path.Path());
	
	if(dir_name != NULL){
		BEntry	entry;
		if(dir.FindEntry(dir_name, &entry) == B_NO_ERROR){
			BDirectory	bufDir(&entry);
			dir = bufDir;
		}else{
			dir.CreateDirectory(dir_name, &dir);
		}
	}
	
	if(dir.FindEntry(name, entry) == B_NO_ERROR){
		fPrefsEntry = entry;
		this->LoadPrefs();
	}else{
		delete entry;
		BFile	file;
		if(dir.CreateFile(name, &file) == B_NO_ERROR){
			fPrefsEntry = new BEntry();
			dir.FindEntry(name, fPrefsEntry);
			fSettingMessage = new BMessage();
			fSettingMessage = fDefaultMessage;
		}else{
			
		}
	}
}


KPrefs::~KPrefs()
{
	this->StorePrefs();
	
	delete	fPrefsEntry;
	delete	fSettingMessage;
}

void
KPrefs::LoadPrefs()
{
	BFile	file(fPrefsEntry, B_READ_ONLY);
	if(file.InitCheck() == B_NO_ERROR){
		delete fSettingMessage;
		fSettingMessage = new BMessage();
		fSettingMessage->Unflatten(&file);
	}else{
		(new BAlert(B_EMPTY_STRING,
				"Error...\n"
				"The setting file is not found.",
				"OK"))->Go();
	}
}


void
KPrefs::StorePrefs()
{
	BFile	file(fPrefsEntry, B_WRITE_ONLY);
	if(file.InitCheck() == B_NO_ERROR){
		fSettingMessage->Flatten(&file);
	}else{
		(new BAlert(B_EMPTY_STRING,
				"Error...\n"
				"The setting file is not found.",
				"OK"))->Go();
	}
}


void
KPrefs::MakeDefault()
{
	fDefaultMessage->AddBool("word_wrap", true);
	fDefaultMessage->AddBool("auto_indent", true);
	
	BFont	font(be_plain_font);
	font_family	family;
	font_style	style;
	font.GetFamilyAndStyle(&family, &style);
	
	rgb_color	aBGColor		= { 0xff, 0xff, 0xff };
	rgb_color	aCharColor 		= { 0x00, 0x00, 0x00 };
	rgb_color	aTagColor		= { 0x64, 0x64, 0xc8 };
	rgb_color	aCommentColor	= { 0xaa, 0x00, 0x00 };
	
	
	fDefaultMessage->AddData("background_color", B_RGB_COLOR_TYPE, &aBGColor, sizeof(rgb_color));
	fDefaultMessage->AddData("char_color", B_RGB_COLOR_TYPE, &aCharColor, sizeof(rgb_color));
	fDefaultMessage->AddData("tag_color", B_RGB_COLOR_TYPE, &aTagColor, sizeof(rgb_color));
	fDefaultMessage->AddData("comment_color", B_RGB_COLOR_TYPE, &aCommentColor, sizeof(rgb_color));
	
	
	fDefaultMessage->AddString("font_family", family);
	fDefaultMessage->AddString("font_style", style);
	fDefaultMessage->AddString("text_prefix", ">");
	fDefaultMessage->AddString("email_prefix", "> ");
	
	fDefaultMessage->AddFloat("font_size", font.Size());
	fDefaultMessage->AddFloat("tab_width", 28.0);
	
	fDefaultMessage->AddInt32("key_assign", 0);//0 = KEdit Original.
	fDefaultMessage->AddInt32("multi_button", 0);//0 = show popup menu.
	fDefaultMessage->AddInt32("line_width", 60);
	
	fDefaultMessage->AddBool("auto_detect_filetype", true);
	fDefaultMessage->AddBool("memorize_window_rect", true);
	fDefaultMessage->AddBool("reset_filetype_and_preferredapp", true);
	fDefaultMessage->AddBool("browser_always_makes_new_window", false);
	
}


void
KPrefs::GetData(const char* name, const void** data)
{
	ssize_t		numBytes;
	
	this->GetData(name, data, &numBytes);
}


void
KPrefs::GetData(const char* name, const void** data, ssize_t* numBytes)
{
	status_t	anError = B_NO_ERROR;
	
	anError = fSettingMessage->FindData(name, B_ANY_TYPE, data, numBytes);
	if(anError != B_NO_ERROR){
		anError = fDefaultMessage->FindData(name, B_ANY_TYPE, data, numBytes);
	}
}


void
KPrefs::GetData(const char* name, BRect* rect)
{
	status_t	anError = B_NO_ERROR;
	
	anError = fSettingMessage->FindRect(name, rect);
	if(anError != B_NO_ERROR){
		fDefaultMessage->FindRect(name, rect);
	}
}


void
KPrefs::GetData(const char* name, BPoint* point)
{
	status_t	anError = B_NO_ERROR;
	
	anError = fSettingMessage->FindPoint(name, point);
	if(anError != B_NO_ERROR){
		fDefaultMessage->FindPoint(name, point);
	}
}


void
KPrefs::GetData(const char* name, const char** text)
{
	status_t	anError = B_NO_ERROR;
	
	anError = fSettingMessage->FindString(name, text);
	if(anError != B_NO_ERROR){
		fDefaultMessage->FindString(name, text);
	}
}


void
KPrefs::GetData(const char* name, int32* num)
{
	status_t	anError = B_NO_ERROR;
	
	anError = fSettingMessage->FindInt32(name, num);
	if(anError != B_NO_ERROR){
		fDefaultMessage->FindInt32(name, num);
	}
}


void
KPrefs::GetData(const char* name, bool* status)
{
	status_t	anError = B_NO_ERROR;
	
	anError = fSettingMessage->FindBool(name, status);
	if(anError != B_NO_ERROR){
		fDefaultMessage->FindBool(name, status);
	}
}


void
KPrefs::GetData(const char* name, float* num)
{
	status_t	anError = B_NO_ERROR;
	
	anError = fSettingMessage->FindFloat(name, num);
	if(anError != B_NO_ERROR){
		fDefaultMessage->FindFloat(name, num);
	}
}


void
KPrefs::GetData(const char* name, entry_ref* ref)
{
	status_t	anError = B_NO_ERROR;
	
	anError = fSettingMessage->FindRef(name, ref);
	if(anError != B_NO_ERROR){
		fDefaultMessage->FindRef(name, ref);
	}
}


void
KPrefs::GetData(const char* name, rgb_color* color)
{
	status_t	anError = B_NO_ERROR;
	rgb_color*	aColor;
	ssize_t		aBytes;
	
	anError = fSettingMessage->FindData(name, B_RGB_COLOR_TYPE,
		(const void**) &aColor, &aBytes);
	if(anError != B_NO_ERROR){
		fDefaultMessage->FindData(name, B_RGB_COLOR_TYPE,
			(const void **) &aColor, &aBytes);
	}
	*color = *aColor;
}


void
KPrefs::SetData(const char* name, const void* data, ssize_t numBytes)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddData(name, B_ANY_TYPE, data, numBytes);
}

void
KPrefs::SetData(const char* name, BRect rect)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddRect(name, rect);
}


void
KPrefs::SetData(const char* name, BPoint point)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddPoint(name, point);
}


void
KPrefs::SetData(const char* name, const char* text)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddString(name, text);
}


void
KPrefs::SetData(const char* name, int32 num)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddInt32(name, num);
}


void
KPrefs::SetData(const char* name, bool status)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddBool(name, status);
}


void
KPrefs::SetData(const char* name, float num)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddFloat(name, num);
}


void
KPrefs::SetData(const char* name, entry_ref *ref)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddRef(name, ref);
}


void
KPrefs::SetData(const char* name, rgb_color* color)
{
	fSettingMessage->RemoveData(name);
	fSettingMessage->AddData(name, B_RGB_COLOR_TYPE, color, sizeof(rgb_color));
}


void
KPrefs::SetDefaultData(const char* name, const void* data, ssize_t numBytes)
{
	fDefaultMessage->AddData(name, B_ANY_TYPE, data, numBytes);
}


void
KPrefs::PrintToStream()
{
	::printf("\nDefault Settings\n");
	fDefaultMessage->PrintToStream();
	
	::printf("\nUser Settings\n");
	fSettingMessage->PrintToStream();
}



























