#include "KString.h"

#include <stdio.h>
#include <string.h>

#include "List.h"

KString::KString(const char* text, int32 len, bool will_reallocate_memory)
	: fReallocMem(will_reallocate_memory)
{
	fCurrentVer = 1;
	fWordVer = fParagVer = fTextItemVer = 0;
	fString = NULL;
	
	if(fReallocMem){
		int32	aLen = len;
		if(len == K_NO_STRING_LIMIT){ aLen = ::strlen(text); }
		fString = new char[aLen + 1];
		*fString = '\0';
		::strncat(fString, text, aLen);
	}else{
		fString = (char*)text;
	}
	
	fDelimiter = new char[2];
	::sprintf(fDelimiter, "\t");
}


KString::KString()
{
	fCurrentVer = 1;
	fWordVer = fParagVer = fTextItemVer = 0;
	fString = NULL;
	fReallocMem = true;
	
	fString = new char[1];
	*fString = '\0';
	
	fDelimiter = new char[2];
	::sprintf(fDelimiter, "\t");
}


KString::KString(const KString& string, int32 len, bool will_reallocate_memory)
	: fReallocMem(will_reallocate_memory)
{
	fCurrentVer = 1;
	fWordVer = fParagVer = fTextItemVer = 0;
	fString = NULL;

	if(fReallocMem){
		int32	aLen = len;
		if(len == K_NO_STRING_LIMIT){ aLen = string.Length(); }
		fString = new char[aLen + 1];
		*fString = '\0';
		::strncat(fString, string.Text(), aLen);
	}else{
		fString = (char*)string.Text();
	}
	
	fDelimiter = new char[2];
	::sprintf(fDelimiter, "\t");
}


KString::~KString()
{
	if(fReallocMem){
		delete[]	fString;
	}
	
	/* remove word list */
	int32	aNum = fWordList.CountItems();
	for(int32 i = 0; i < aNum; i++){
		delete	(text_item*)fWordList.ItemAt(i);
	}
	
	/* remove paragraph list */
	aNum = fParagraphList.CountItems();
	for(int32 i = 0; i < aNum; i++){
		delete	(text_item*)fParagraphList.ItemAt(i);
	}
	
	delete[]	fDelimiter;
	
	/* remove text item list */
	aNum = fTextItemList.CountItems();
	for(int32 i = 0; i < aNum; i++){
		delete	(text_item*)fParagraphList.ItemAt(i);
	}
}


KString&
KString::operator=(const char* text)
{
	if(fReallocMem){
		delete[]	fString;
		fString = new char[::strlen(text) + 1];
		::strcpy(fString, text);
	}else{
		fString = (char*)text;
	}
	
	this->Modify();
	
	return *this;
}


KString&
KString::SetTo(const char* text, int32 len, bool will_reallocate_memory)
{
	if(fReallocMem){
		delete[]	fString;
	}
	
	fReallocMem = will_reallocate_memory;
	
	if(will_reallocate_memory){
		int32	aLen = len;
		if(len == K_NO_STRING_LIMIT){ aLen = ::strlen(text); }
	
		fString = new char[aLen + 1];
		*(fString + aLen) = '\0';
		::strncpy(fString, text, (int)aLen);
	}else{
		fString = (char*)text;
	}
	
	this->Modify();
	
	return *this;
}


KString&
KString::Insert(const char* text, int32 position, int32 len)
{
	if(!fReallocMem){
		KString("[Caution] Insert() function is forbidden.").PrintToStream();
		return *this;
	}
	
	int32	aTextLen = len;
	if(len == K_NO_STRING_LIMIT){ aTextLen = ::strlen(text); }
	int32	aLen = this->Length() + aTextLen;
	
	
	char*	aBuffer = new char[aLen + 1];
	*aBuffer = '\0';
	::strncat(aBuffer, fString, position);
	::strncat(aBuffer, text, len);
	::strcat(aBuffer, fString + position);
	delete[]	fString;
	fString = aBuffer;
	
	this->Modify();
	
	return *this;
}


KString&
KString::Delete(int32 from, int32 to)
{
	if(!fReallocMem){
		KString("[Caution] Delete() function is forbidden.").PrintToStream();
		return *this;
	}
	
	const int32	aTextLen = this->Length();
	if(to == K_NO_STRING_LIMIT){ to = aTextLen; }
	if(from > to || from < 0 || from > aTextLen || to > aTextLen){ return *this; }
	
	for(int32 i = from; i <= aTextLen; i++){
		*(fString + i) = *(fString + i + to - from);
	}
	
	this->Modify();
	
	return *this;
}


bool
KString::Contains(const char* text, bool ignore_case
					, int32* from, int32* to) const
{
	const int32	aLen = this->Length();
	const int32	aTextLen = ::strlen(text);
	
	for(int32 i = 0; i < aLen; i++){
		if(this->Compare(fString + i, text, ignore_case, aTextLen) == 0){
			if(from != NULL && to != NULL){
				*from = i;
				*to = i + ::strlen(text);
			}
			return true;
		}
	}
	return false;
}



void
KString::AnalyseParagraph()
{
	if(fParagVer >= fCurrentVer){ return; }
	
	int32	aListNum = fParagraphList.CountItems();
	for(int32 i = 0; i < aListNum; i++){
		delete	(text_item*)fParagraphList.ItemAt(i);
	}
	fParagraphList.MakeEmpty();
	
	//Paragraph Analysis...
	int32	cur_pos, prev_pos;
	cur_pos = prev_pos = 0;

	while(*(fString + cur_pos) != '\0'){
		if(cur_pos != 0 && *(fString + cur_pos) == kLF){
			
			text_item*	anItem = new text_item();
			anItem->from = prev_pos;
			anItem->length = cur_pos - prev_pos;
			
			fParagraphList.AddItem(anItem);
			prev_pos = cur_pos + 1;
			
		}
		cur_pos++;
	}
	
	
	text_item*	anItem = new text_item();
	anItem->from = prev_pos;
	anItem->length = cur_pos - prev_pos;
	fParagraphList.AddItem(anItem);
	
	fParagVer = fCurrentVer;
}


void
KString::AnalyseWord()
{
	if(fWordVer >= fCurrentVer){ return; }
	
	const int32	aListNum = fWordList.CountItems();
	for(int32 i = 0; i < aListNum; i++){
		delete	(text_item*)fWordList.ItemAt(i);
	}
	fWordList.MakeEmpty();
	
	const int32	aTextLen = this->Length();
	char	aChar;
	int32	aPrevPos = 0;
	for(int32 i = 0; i <= aTextLen; i++){
		aChar = *(fString + i);
		if(aChar == ' ' || aChar == '\n' || aChar == '\t'
				|| aChar == '(' || aChar == ')' || aChar == '<' || aChar == '>'
				|| aChar == '/' || aChar == ':' || aChar == ';' || aChar == '\''
				|| aChar == '"'
				|| aChar == ',' || aChar == '.' || aChar == '\\'|| aChar == '\0'){
			if(i - aPrevPos != 0){
				text_item*	aItem = new text_item();
				aItem->from = aPrevPos;
				aItem->length = i - aPrevPos;
				fWordList.AddItem(aItem);
			}
			
			aPrevPos = i + 1;
		}
	}
	
	fWordVer = fCurrentVer;
	
	return;
}


void
KString::AnalyseTextItem()
{
	if(fTextItemVer >= fCurrentVer){ return; }
	
	int32	prev_pos = 0;
	
	int32	aListNum = fTextItemList.CountItems();
	for(int32 i = 0; i < aListNum; i++){
		delete	(text_item*)fTextItemList.ItemAt(i);
	}
	fTextItemList.MakeEmpty();
	
	int32	aTextLen = ::strlen(fString);
	int32	aDelimLen = ::strlen(fDelimiter);
	for(int32 i = 0; i <= aTextLen; i++){
		if(::strncmp(fString + i, fDelimiter, aDelimLen) == 0
				|| i == aTextLen){
			text_item*	aItem = new text_item();
			aItem->from = prev_pos;
			aItem->length = i - prev_pos;
			fTextItemList.AddItem(aItem);
			
			i += aDelimLen - 1;
			prev_pos = i + 1;
		}
	}
	
	fTextItemVer = fCurrentVer;
}


KString
KString::Character(int32 index)
{
	const int32	aTextLen = this->Length();
	int32		aCharNum = 0;
	
	if(index < 0 || index > aTextLen){ return KString(); }
	
	for(int32 i = 0; i < aTextLen; i++){
		if(this->IsTopOfChar(fString + i)){
			if(aCharNum == index){
				if(this->IsASCII(fString + i)){
					return KString(fString + i, 1);
				}else if((*(fString + i) & 0xE0) == 0xC0){
					return KString(fString + i, 2);
				}else if((*(fString + i) & 0xF0) == 0xE0){
					return KString(fString + i, 3);
				}else{
					return KString();
				}
			}
			aCharNum++;
		}
	}
	return KString();
}


KString
KString::Characters(int32 from, int32 to)
{
	const int32	aTextLen = this->Length();
	int32		aCurrentCharNum = 0, aFromCharNum = 0;
	
	if(from < 0 || from > to || to > aTextLen){ return KString(); }
	
	for(int32 i = 0; i < aTextLen; i++){
		if(this->IsTopOfChar(fString + i)){
			if(aCurrentCharNum == from){ aFromCharNum = i; }
			if(aCurrentCharNum == to){
				if(this->IsASCII(fString + i)){
					return KString(fString + aFromCharNum, i - aFromCharNum + 1);
				}else if((*(fString + i) & 0xE0) == 0xC0){
					return KString(fString + aFromCharNum, i - aFromCharNum + 2);
				}else if((*(fString + i) & 0xF0) == 0xE0){
					return KString(fString + aFromCharNum, i - aFromCharNum + 3);
				}else{
					return KString();
				}
			}
			aCurrentCharNum++;
		}
	}
	return KString();
}


KString
KString::Word(int32 index)
{
	this->AnalyseWord();
	
	if(fWordList.CountItems() >= index){
		text_item*	aItem = (text_item*)fWordList.ItemAt(index);
		return	KString(fString + aItem->from, aItem->length);
	}
	return KString();
}


KString
KString::Words(int32 from, int32 to)
{
	this->AnalyseWord();
	
	if(from <= to && to <= fWordList.CountItems()){
		text_item*	aFrom = (text_item*)fWordList.ItemAt(from);
		text_item*	aTo = (text_item*)fWordList.ItemAt(to);
		return	KString(fString + aFrom->from, aTo->from + aTo->length - aFrom->from);
	}
	return KString();
}


KString
KString::Paragraph(int32 index)
{
	this->AnalyseParagraph();
	
	if(fParagraphList.CountItems() >= index){
		text_item*	aItem = (text_item*)fParagraphList.ItemAt(index);
		return KString(fString + aItem->from, aItem->length);
	}
	return KString();
}


KString
KString::Paragraphs(int32 from, int32 to)
{
	this->AnalyseParagraph();
	
	if(from <= to && to <= fTextItemList.CountItems()){
		text_item*	aFrom = (text_item*)fParagraphList.ItemAt(from);
		text_item*	aTo = (text_item*)fParagraphList.ItemAt(to);
		return	KString(fString + aFrom->from, aTo->from + aTo->length - aFrom->from);
	}
	return KString();
}


KString
KString::TextItem(int32 index)
{
	this->AnalyseTextItem();
	
	if(fTextItemList.CountItems() >= index){
		text_item*	aItem = (text_item*)fTextItemList.ItemAt(index);
		KString	aString(fString + aItem->from, aItem->length);
		aString.SetTextItemDelimiter(this->TextItemDelimiter());
		return aString;
	}
	return KString();
}


KString
KString::TextItems(int32 from, int32 to)
{
	this->AnalyseTextItem();
	
	if(from <= to && to <= fTextItemList.CountItems()){
		text_item*	aFrom = (text_item*)fTextItemList.ItemAt(from);
		text_item*	aTo = (text_item*)fTextItemList.ItemAt(to);
		KString		aString(fString + aFrom->from, aTo->from + aTo->length - aFrom->from);
		aString.SetTextItemDelimiter(this->TextItemDelimiter());
		return aString;
	}
	return KString();
}


int32
KString::CountWords()
{
	this->AnalyseWord();
	
	return fWordList.CountItems();
}


int32
KString::CountParagraphs()
{
	this->AnalyseParagraph();
	
	return fParagraphList.CountItems();
}


int32
KString::CountTextItems()
{
	this->AnalyseTextItem();
	
	return fTextItemList.CountItems();
}


void
KString::SetTextItemDelimiter(const char* text)
{
	delete[]	fDelimiter;
	int32	aLen = ::strlen(text);
	fDelimiter = new char[aLen + 1];
	::strcpy(fDelimiter, text);
	this->Modify();
}


KString&
KString::ReplaceAll(const char* search_text, const char* replace_text, bool ignore_case, int32* count)
{
	if(!fReallocMem){
		KString("[Cautioin] ReplaceAll() function is forbidden.").PrintToStream();
		return *this;
	}
	
	const int32	aTotalLen = this->Length();
	const int32	aSearchLen = ::strlen(search_text);
	const int32	aReplaceLen = ::strlen(replace_text);
	int32		aCount = 0;
	KString		aStringCopy, aSearchCopy;
	
	if(ignore_case){
		aStringCopy.SetTo(fString).ToLower();
		aSearchCopy.SetTo(search_text).ToLower();
	}else{
		aStringCopy.SetTo(fString, K_NO_STRING_LIMIT, false);
		aSearchCopy.SetTo(search_text, K_NO_STRING_LIMIT, false);
	}
	
	for(int32 i = 0; i < aTotalLen; i++){
		if(::strncmp(aStringCopy.Text() + i, aSearchCopy.Text(), aSearchLen) == 0){
			aCount++;
			i += aSearchLen - 1;
		}
	}
	const int32	aReturnLen = aTotalLen + aCount * (aReplaceLen - aSearchLen);
	char*	aBuffer = new char[aReturnLen + 1];
	*(aBuffer + aReturnLen) = '\0';
	
	for(int32 i = 0, j = 0; i < aTotalLen; i++, j++){
		if(::strncmp(aStringCopy.Text() + i, aSearchCopy.Text(), aSearchLen) == 0){
			::strcpy(aBuffer + j, replace_text);
			i += aSearchLen - 1;
			j += aReplaceLen - 1;
		}else{
			*(aBuffer + j) = *(fString + i);
		}
	}
	delete[]	fString;
	fString = aBuffer;
	if(count != NULL){ *count = aCount; }
	
	this->Modify();
	
	
	return *this;
}


KString&
KString::ToUpper()
{
	for(int32 i = 0; *(fString + i) != '\0'; i++){
		if(*(fString + i) >= 'a' && *(fString + i) <= 'z'){ *(fString + i) -= 'a' - 'A'; }
	}
	return *this;
}


KString&
KString::ToLower()
{
	for(int32 i = 0; *(fString + i) != '\0'; i++){
		if(*(fString + i) >= 'A' && *(fString + i) <= 'Z'){ *(fString + i) += 'a' - 'A'; }
	}
	return *this;
}




KString&
KString::operator<<(uint8 var)
{
	char	aBuffer[256];
	::sprintf(aBuffer, "%d", var);
	return this->Append(aBuffer);
}

KString&
KString::operator<<(uint32 var)
{
	char	aBuffer[256];
	::sprintf(aBuffer, "%d", (int8)var);
	return this->Append(aBuffer);
}

KString&
KString::operator<<(int32 var)
{
	char	aBuffer[256];
	::sprintf(aBuffer, "%d", (int)var);
	return this->Append(aBuffer);
}

KString&
KString::operator<<(float var)
{
	char	aBuffer[256];
	::sprintf(aBuffer, "%.2f", var);
	return this->Append(aBuffer);
}

/*
KString&	KString::operator<<(uint64 var)
KString&	KString::operator<<(int64 var)
*/






void
KString::PrintToStream()
{
	if(!this->IsEmpty()){
		::printf("KString : %s\n", fString);
	}else{
		::printf("KString : <empty>\n");
	}
}



BList	KString::WordList(){ this->AnalyseWord(); return fWordList; }
BList	KString::ParagraphList(){ this->AnalyseParagraph(); return fParagraphList; }
BList	KString::TextItemList(){ this->AnalyseTextItem(); return fTextItemList; }
int32	KString::OffsetOfWord(int32 index){ this->AnalyseWord(); return ((text_item*)fWordList.ItemAt(1))->from; }

/*data handling*/
KString&	KString::SetTo(const KString& string, int32 len, bool will_reallocate_memory){ return this->SetTo(string.Text(), len, will_reallocate_memory); }

KString&	KString::Append(const char* text, int32 len){ return this->Insert(text, this->Length(), len); }
KString&	KString::Append(const KString& string, int32 len){ return this->Insert(string.Text(), this->Length(), len); }

KString&	KString::Prepend(const char* text, int32 len){ return this->Insert(text, 0, len); }
KString&	KString::Prepend(const KString& string, int32 len){ return this->Insert(string.Text(), 0, len); }

KString&	KString::Insert(const KString& string, int32 position, int32 len){ return this->Insert(string.Text(), position, len); }
KString&	KString::Delete(){ return this->Delete(0, this->Length()); }

KString&	KString::Copy(const char* text, int32 offset, int32 len){ return this->SetTo(text + offset, len); }
KString&	KString::Copy(const KString& string, int32 offset, int32 len){ return this->SetTo(string.Text() + offset, len); }


/* data comparing */
int32	KString::Compare(const KString& string, bool ignore_case, int32 len) const { return this->Compare(string.Text(), ignore_case, len); }
int32	KString::Compare(const KString& stringA, const KString& stringB, bool ignore_case, int32 len) { return KString::Compare(stringA.Text(), stringB.Text(), ignore_case, len); }
int32	KString::Compare(const char* text, bool ignore_case, int32 len) const { return this->Compare(fString, text, ignore_case, len); }

bool	KString::Is(const char* text, bool ignore_case) const { return (this->Compare(text, ignore_case) == 0); }
bool	KString::Is(const KString& string, bool ignore_case) const { return (this->Compare(string.Text(), ignore_case) == 0); }
bool	KString::StartsWith(const char* text, bool ignore_case) const { return (this->Compare(text, ignore_case, ::strlen(text)) == 0); }
bool	KString::StartsWith(const KString& string, bool ignore_case) const { return (this->Compare(string.Text(), ignore_case, string.Length()) == 0); }
bool	KString::EndsWith(const char* text, bool ignore_case) const { return (this->Compare(fString + this->Length() - ::strlen(text), text, ignore_case) == 0); }
bool	KString::EndsWith(const KString& string, bool ignore_case) const { return this->EndsWith(string.Text(), ignore_case); }
bool	KString::Contains(const KString& string, bool ignore_case, int32* from, int32* to) const { return this->Contains(string.Text(), ignore_case, from, to); }


/*operators*/
KString&	KString::operator=(const KString& string){ return this->operator=(string.Text()); }
KString&	KString::operator+(const char* text){ return this->Append(text); }
KString&	KString::operator+(const KString& string){ return this->Append(string.Text()); }
KString&	KString::operator+=(const char* text){ return this->Append(text); }
KString&	KString::operator+=(const KString& string){ return this->Append(string.Text()); }

KString&	KString::operator<<(const char* text){ return this->Append(text); }
KString&	KString::operator<<(const KString& string){ return this->Append(string.Text()); }






















