/*  This file is part of the KDE project
    Copyright (C) 2000 Simon Hausmann <hausmann@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "scheduler.h"

#include "jobclasses.h"
#include "slavebase.h"
#include "slave.h"
#include "kprotocolmanager.h"

#include <dcopclient.h>
#include <dcopobject.h>
#include <kdebug.h>

#include <stdlib.h>
#include <assert.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>

using namespace KIO;

Scheduler::ProtocolInfo::ProtocolInfo()
{
    for(int i = 0; i < SAM_MAX_TASKS; i++) {
	m_slaves[i] = 0;
	m_running[i] = false;
    }
    m_running_cnt = 0;
}
Scheduler::ProtocolInfo::~ProtocolInfo()
{
}

Scheduler *Scheduler::s_self = 0;

Scheduler::Scheduler()
{
    connect( &m_timer, SIGNAL( timeout() ),
             this, SLOT( slotStep() ) );
}

Scheduler::~Scheduler()
{
}

bool Scheduler::doJob( SimpleJob *job )
{
    KURL url = job->url();

    if ( !SlaveBase::knownProtocol( url.protocol() ) )
        return false;

    ProtocolInfo *nfo = m_schedule[ url.protocol() ];

    if ( !nfo )
    {
        nfo = new ProtocolInfo;
        m_schedule.insert( url.protocol(), nfo );
    }

    if ( nfo->m_jobs.findRef( job ) != -1 )
        return false;

    nfo->m_jobs.append( job );

    m_timer.start( 0, true );

    return true;
}

void Scheduler::releaseJob( SimpleJob *job, bool _killSlave )
{
    kdDebug() << "releaseJob " << job << endl;

    // the job's url might have changed or something, so let's iterate
    // over all entries
    QDictIterator<ProtocolInfo> it( m_schedule );
    for (; it.current(); ++it )
    {
        kdDebug() << it.currentKey() << " : current " << it.current()->m_jobs.current() << endl;
	for(QListIterator <SimpleJob>job_it(it.current()->m_jobs); job_it.current(); ++job_it) {
	    if(job_it.current() == job) {
		int task = job->taskNumber();
		if(task == -1) {
		    kdDebug() << "sam: shouldn't have happend" << endl;
		} else {
		    kdDebug() << "sam: dropping " << task << endl;
		    it.current()->m_running[task] = false;
		    it.current()->m_running_cnt--;
		}
		if ( _killSlave )
		    killSlave( it.current(), task );
	    }
	}
	it.current()->m_jobs.removeRef( job );
    }
    job->setTaskNumber( -1 );
    m_timer.start( 0, true );
}

void Scheduler::slotStep()
{
    kdDebug() << "slot step" << endl;
    QDictIterator<ProtocolInfo> it( m_schedule );
    for (; it.current(); ++it )
        doStep( it.current() );

    it.toFirst();
    while ( it.current() )
    {
        if ( it.current()->m_jobs.count() == 0 && !it.current()->m_running_cnt )
            delete m_schedule.take( it.currentKey() );
        else
            ++it;
    }
}

void Scheduler::doStep( ProtocolInfo *nfo )
{
    if(nfo->m_running_cnt >= SAM_MAX_TASKS)
	return;

    int task = -1;
    for(int i = 0; i < SAM_MAX_TASKS; i++) {
	if(!nfo->m_slaves[i] || !nfo->m_running[i]) {
	    task = i;
	    break;
	}
    }
    if(task == -1)
	return;

    if ( nfo->m_jobs.count() == 0 )
    {
        //work done.
        killSlave( nfo );
        return;
    }

    for(QListIterator <SimpleJob>it(nfo->m_jobs); it.current(); ++it) {
	SimpleJob *job = it.current();
	if(job->taskNumber() != -1)
	    continue;

	nfo->m_running_cnt++;
	nfo->m_running[task]=true;
	if ( !nfo->m_slaves[task] ) {
	    kdDebug() << "creating slave for " << job->url().prettyURL() << endl;
	    nfo->m_slaves[task] = createSlave( job->url().protocol() );
	}
	kdDebug() << "sam: starting task " << task << " " << job->url().prettyURL() << endl;
	job->setTaskNumber(task);
  
    Slave *slave = nfo->m_slaves[task];
	MetaData configData = KProtocolManager::protocolConfig( job->url().protocol() );
    slave->setConfig( configData );
	job->start( slave );

	task = -1;
	for(int i = 0; i < SAM_MAX_TASKS; i++) {
	    if(!nfo->m_slaves[i] || !nfo->m_running[i]) {
		task = i;
		break;
	    }
	}
	if(task == -1)
	    break;
    }
}

void Scheduler::killSlave( ProtocolInfo *nfo, int task )
{
    for(int i = 0; i < SAM_MAX_TASKS; i++) {
	if ( nfo->m_slaves[i] &&
             ( i == task || task == -1 ) ) {
	    kdDebug() << "sam: killing " << i << endl;
	    nfo->m_slaves[i]->kill();
	    delete nfo->m_slaves[i];
	    nfo->m_slaves[i] = 0;
	    nfo->m_running[i]=false;
	}
    }
    nfo->m_running_cnt = 0;
}

Scheduler *Scheduler::self()
{
    if ( !s_self )
        s_self = new Scheduler;
    return s_self;
}

Slave *Scheduler::createSlave( const QString &prot )
{
    int app2slave[ 2 ]; // app -> slave
    int slave2app[ 2 ]; // slave -> app
    int dcopApp2Slave[ 2 ]; // dcop: app -> slave
    int dcopSlave2App[ 2 ]; // dcop: slave -> app

    if ( pipe( app2slave ) == -1 )
    {
        perror( "FATAL: can't create pipe for app->slave communication" );
        ::exit( 0 );
    }

    if ( pipe( slave2app ) == -1 )
    {
        perror( "FATAL: can't create pipe for slave->app communcation" );
        ::exit( 0 );
    }

    if ( pipe( dcopApp2Slave ) == -1 )
    {
        perror( "FATAL: can't create pipe for dcop app->slave communcation" );
        ::exit( 0 );
    }

    if ( pipe( dcopSlave2App ) == -1 )
    {
        perror( "FATAL: can't create pipe for dcop slave->app communication" );
        ::exit( 0 );
    }

    pid_t pid = fork();

    if ( pid == 0 )
    { // child

#ifdef __linux__
        // no core dumps for those children. when debugging crashes in the
        // parent I don't want to end up with the child aborting a few
        // seconds later and overwriting the interesting core :)
        struct rlimit rlim;
        rlim.rlim_cur = rlim.rlim_max = 0;
        setrlimit( RLIMIT_CORE, &rlim );
#endif

        // close unused fds
        ::close( app2slave[ 1 ] );
        ::close( slave2app[ 0 ] );

        ::close( dcopApp2Slave[ 1 ] );
        ::close( dcopSlave2App[ 0 ] );

        Connection *connection = new Connection;
        connection->init( app2slave[ 0 ], slave2app[ 1 ] );

        SlaveBase *slave = SlaveBase::createSlave( prot );

        assert( slave );

        slave->setConnection( connection );

        connection = new Connection;
        connection->init( dcopApp2Slave[ 0 ], dcopSlave2App[ 1 ] );

        DCOPClient::setGlobalConnection( connection );

        slave->dispatchLoop();

        // never reached

        ::exit( 0 );
    }

    assert( pid != -1 );

    // parent

    ::close( app2slave[ 0 ] );
    ::close( slave2app[ 1 ] );

    ::close( dcopApp2Slave[ 0 ] );
    ::close( dcopSlave2App[ 1 ] );

    Connection *connection = new Connection;
    connection->init( slave2app[ 0 ], app2slave[ 1 ] );

  kdDebug() << "about to create new Slave" << endl;
    Slave *res = new Slave( connection, pid );

    connection = new Connection;
    connection->init( dcopSlave2App[ 0 ], dcopApp2Slave[ 1 ] );

    DCOPDispatcher::self()->addClient( connection, res );

    return res;
}

#if (OSTYPE==beos)
void Scheduler::scheduleJob( SimpleJob* job )
{ }
#endif

#include "scheduler.moc"
