#include <config.h>

#include "bookmarks.h"
#include "xmltree.h"

#include <qaction.h>
#include <qlist.h>
#include <qpopupmenu.h>
#include <qregexp.h>
#include <qstring.h>
#include <qwidget.h>

BookmarkAction::BookmarkAction( QString title, QString url, QObject* parent ):
  QAction( parent ), m_title( title ), m_url( url )
{
  setText( m_title );
  setStatusTip( m_url );
  connect( this, SIGNAL( activated() ), SLOT( selfActivated() ) );
}

void BookmarkAction::selfActivated()
{
  emit activated( m_url );
}

bool Bookmark::isFolder() const
{
  if ( !element )
    return false;
  QString tag = element->tagName();
  return ( tag == "xbel" || tag == "folder" );
}

bool Bookmark::isNull() const
{
  return element == 0;
}

bool Bookmark::isSeparator() const
{
  if ( !element )
    return false;
  QString tag = element->tagName();
  return ( tag == "separator" );
}

void Bookmark::setTitle( const QString &title )
{
    if ( !element || isSeparator() || isFolder() )
        return;

    XMLElement *titleElement = element->namedItem( "title" );
    if ( !titleElement )
    {
        titleElement = new XMLElement;
        titleElement->setTagName( "title" );
        element->appendChild( titleElement );
    }

    if ( !titleElement->firstChild() )
        titleElement->appendChild( new XMLElement );

    titleElement->firstChild()->setValue( title );
}

QString Bookmark::title() const
{
  if ( !element )
    return QString::null;

  if ( isSeparator() )
      return QString::fromLatin1( "-" );

  XMLElement *titleXMLElement = element->namedItem( "title" );
  if ( !titleXMLElement || !titleXMLElement->firstChild() )
    return QString::null;

  return titleXMLElement->firstChild()->value();
}

QString Bookmark::menuTitle() const
{
  const uint maxlen = 32;
  QString res = title();
  if( res.length() > maxlen )
    res = res.left( maxlen ).append( "..." );
  res.replace( QRegExp( "&" ), "&&" );
  return res;
}

void Bookmark::setURL( const QString &url )
{
    if ( !element )
        return;

    element->attributes()[ "href" ] = url;
}

QString Bookmark::url() const
{
  if ( !element )
    return QString::null;
  return element->attributes()[ "href" ];
}

BookmarkFolder Bookmark::toFolder() const
{
  return BookmarkFolder( element );
}

Bookmark BookmarkFolder::first() const
{
  if ( !element )
    return Bookmark();

  return Bookmark( nextXMLElement( element->firstChild() ) );
}

Bookmark BookmarkFolder::previous( const Bookmark& current ) const
{
  if ( !current.element )
    return Bookmark();

  return Bookmark( prevXMLElement( current.element->prevChild() ) );
}

Bookmark BookmarkFolder::next( const Bookmark& current ) const
{
  if ( !current.element )
    return Bookmark();

  return Bookmark( nextXMLElement( current.element->nextChild() ) );
}

XMLElement *BookmarkFolder::nextXMLElement( XMLElement *start ) const
{
  XMLElement *elem = start;
  while( elem )
  {
    QString tag = elem->tagName();
    if( tag == "folder" || tag == "bookmark" || tag == "separator" )
      break;
    elem = elem->nextChild();
  } 
  return elem;
}

XMLElement *BookmarkFolder::prevXMLElement( XMLElement *start ) const
{
  XMLElement *elem = start;
  while( !elem )
  {
    QString tag = elem->tagName();
    if( tag == "folder" || tag == "bookmark" || tag == "separator" )
      break;
    elem = elem->prevChild();
  }
  return elem;
}

BookmarkMenu::BookmarkMenu( const BookmarkFolder &top )
{
  m_actions.setAutoDelete( TRUE );
  m_subMenus.setAutoDelete( TRUE );

  refill( top );
}

BookmarkMenu::~BookmarkMenu()
{
  clearMenu();
}

void BookmarkMenu::clearMenu()
{
  QListIterator<BookmarkAction> it( m_actions );
  for (; it.current(); ++it )
    it.current()->removeFrom( this );
  m_actions.clear();
  m_subMenus.clear();
}

void BookmarkMenu::refill( const BookmarkFolder &top )
{
  clearMenu();
  fillBookmarkMenu( top );
}

void BookmarkMenu::fillBookmarkMenu( const BookmarkFolder &top )
{
  m_top = top;

  for( Bookmark bm = m_top.first(); !bm.isNull(); bm = m_top.next( bm ) )
  {
    if( !bm.isFolder() )
	{
	  if( !bm.isSeparator() )
      {
        BookmarkAction *action = 
		new BookmarkAction( bm.menuTitle(), bm.url() );
        action->addTo( this );
        m_actions.append( action );
        connect( action, SIGNAL( activated( const QString& ) ), 
                 SLOT( slotActivated( const QString& ) ) );
	  }
	  else insertSeparator();
	}
    else
	{
// let's skip sub-folders for small resolutions (Simon)
#if !defined(KONQ_GUI_QPE)
	  BookmarkMenu *b = new BookmarkMenu( bm.toFolder() );
	  insertItem( bm.menuTitle(), b );
	  connect( b, SIGNAL( activatedBookmark( const QString& ) ),
               SLOT( slotActivated( const QString& ) ) );
	  m_subMenus.append( b );
#endif
	}
  }

}

void BookmarkMenu::slotActivated( const QString& url )
{
  emit activatedBookmark( url );
}


#include "bookmarks.moc"




