/*
	BeOS socketpair replacement, borrowed from BeWine
    Copyright (C) 1998,99 Kazuho Okui. ALL RIGHT RESERVED
*/

#include <bone/sys/socket.h>
#include <posix/errno.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <iovec.h>
#include <stdlib.h>

#include <kdebug.h>
#include <kio/connection.h>

#include "socketpair.h"

#define O_NDELAY O_NONBLOCK

int socketpair(int af, int type, int protocol, int fd[])
{
	int     listen_socket;
	struct sockaddr_in sin[2];
	int     len;
	
	/* The following is only valid if type == SOCK_STREAM */
	if (type != SOCK_STREAM)
	{
		perror("socketpair-only SOCK_STREAM is supported");
		return -1;
	}
	/* Create a temporary listen socket; temporary, so any port is good */
	listen_socket = socket(af, type, protocol);
	if (listen_socket < 0)
	{
		perror("socketpair-cannot creating listen_socket");
		return -1;
	}
	
	sin[0].sin_family = af;
	sin[0].sin_port = 0; /* Use any port number */
	sin[0].sin_addr.s_addr = inet_addr("127.0.0.1");//inet_makeaddr(INADDR_ANY, 0);
	if (bind(listen_socket, (sockaddr *)&sin[0], sizeof(sin[0])) < 0)
	{
		perror("socketpair-bind");
		return -1;
	}
	len = sizeof(sin[0]);
	
	/* Read the port number we got, so that our client can connect to it */
	if (getsockname(listen_socket, (sockaddr *)&sin[0], &len) < 0)
	{
		perror("socketpair-getsockname");
		return -1;
	}
	
	/* Put the listen socket in listening mode */
	if (listen(listen_socket, 5) < 0)
	{
		perror("socketpair-listen");
		return -1;
	}
	
	/* Create the client socket */
	fd[1] = socket(af, type, protocol);
	if (fd[1] < 0)
	{
		perror("socketpair-creating client_socket");
		return -1;
	}
	
	/* Put the client socket in non-blocking connecting mode */
	fcntl(fd[1], F_SETFL, fcntl(fd[1], F_GETFL, 0) | O_NDELAY);
	if (connect(fd[1], (sockaddr *)&sin[0], sizeof(sin[0])) < 0)
	{
		perror("socketpair-connect");
		return -1;
	}
	
	/* At the listen-side, accept the incoming connection we generated */
	len = sizeof(sin[1]);
	if ((fd[0] = accept(listen_socket, (sockaddr *)&sin[1], &len)) < 0)
	{
		perror("socketpair-accept");
		return -1;
	}
	
	/* Reset the client socket to blocking mode */
	fcntl(fd[1], F_SETFL, fcntl(fd[1], F_GETFL, 0) & ~O_NDELAY);
	close(listen_socket);
	perror("SocketPair success");
	return 0;
}

//====================
ssize_t recvmsg(int s, struct msghdr *msg, int flags)
{
	if (s<0)
		return -1;
	ssize_t ret, nb;
	size_t tot = 0;
	int i, temp_len;
	char *buf, *p;
	struct iovec *iov = msg->msg_iov;
		
	for(i = 0; i < msg->msg_iovlen; ++i)
		tot += iov[i].iov_len;
	//buf = (char *)malloc(tot);
	if (tot != 0 && buf == NULL) {
		perror("recvmsg-not enough buffer memory");
		errno = ENOMEM;
		return -1;
	}
	nb = ret = recvfrom (s, buf, tot, flags, (struct sockaddr *) msg->msg_name, (int *)&msg->msg_namelen);
	perror("recmsg-after recv");
	if (ret < 0) {
		perror("recmsg-ret<0");
		return -1;
	}	
	p = buf;
	if (p==NULL)
		perror("recvmsg-p=NULL");
	perror("success");
	while (nb > 0) {
		ssize_t cnt = (nb < iov->iov_len ? nb : iov->iov_len);
		memcpy (iov->iov_base, p, cnt);
		p += cnt;
		nb -= cnt;
		++iov;
	}
	perror("recvmsg-before free");
	free(buf);
	return ret;
}

//====================
ssize_t sendmsg(int s, const struct msghdr *msg, int flags)
{
    ssize_t ret;
    size_t tot = 0;
    int i;
    char *buf, *p;
    struct iovec *iov = msg->msg_iov;

    for(i = 0; i < msg->msg_iovlen; ++i)
        tot += iov[i].iov_len;
    buf = (char *)malloc(tot);
    if (tot != 0 && buf == NULL) {
        perror("sendmsg-not enough memory");
        errno = ENOMEM;
        return -1;
    }
    p = buf;
    for (i = 0; i < msg->msg_iovlen; ++i) {
        memcpy (p, iov[i].iov_base, iov[i].iov_len);
        p += iov[i].iov_len;
    }
    perror("recvmsg-before sendto");
    ret = sendto (s, buf, tot, flags, (sockaddr *) msg->msg_name, msg->msg_namelen);
    perror("recvmsg-after sendto");
    free (buf);
    return ret;
}