/* This file is part of the KDE libraries
   Copyright (C) 2001 George Staikos <staikos@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

// Simon: modified/stripped down for Konqueror/Embedded

#ifndef __KOPENSSLPROXY_H
#define __KOPENSSLPROXY_H

#define KOSSL KOpenSSLProxy
class KOpenSSLProxyPrivate;

#include <config.h>

#ifdef HAVE_SSL
#define crypt _openssl_crypt
#include <openssl/ssl.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/pem.h>
#include <openssl/rand.h>
#include <openssl/asn1.h>
#include <openssl/pkcs7.h>
#include <openssl/pkcs12.h>
#undef crypt
#endif

class KOpenSSLProxy {
public:

   /**
    * Return an instance of class KOpenSSLProxy *
    * You cannot delete this object.  It is a singleton class.
    */
   static KOpenSSLProxy *self();

   // Here are the symbols that we need.
#ifdef HAVE_SSL

   /*
    *   SSL_connect - initiate the TLS/SSL handshake with an TLS/SSL server
    */
    int SSL_connect(SSL *ssl) { return ::SSL_connect( ssl ); }

   /*
    *   SSL_get_error - get the error code
    */
   int SSL_get_error(SSL *ssl, int rc)
   { return ::SSL_get_error( ssl, rc ); }

   /*
    *   SSL_read - read bytes from a TLS/SSL connection.
    */
    int SSL_read(SSL *ssl, char *buf, int num) { return ::SSL_read( ssl, buf, num ); }

   /*
    *   SSL_write - write bytes to a TLS/SSL connection.
    */
    int SSL_write(SSL *ssl, const char *buf, int num) { return ::SSL_write( ssl, buf, num ); }

   /*
    *   SSL_new - create a new SSL structure for a connection
    */
    SSL *SSL_new(SSL_CTX *ctx) { return ::SSL_new( ctx ); }

   /*
    *   SSL_free - free an allocated SSL structure
    */
    void SSL_free(SSL *ssl) { ::SSL_free( ssl ); }

   /*
    *   SSL_shutdown - shutdown an allocated SSL connection
    */
    int SSL_shutdown(SSL *ssl) { return ::SSL_shutdown( ssl ); }

   /*
    *   SSL_CTX_new - create a new SSL_CTX object as framework for TLS/SSL enabled functions
    */
    SSL_CTX *SSL_CTX_new(SSL_METHOD *method) { return ::SSL_CTX_new( method ); }

   /*
    *   SSL_CTX_free - free an allocated SSL_CTX object
    */
    void SSL_CTX_free(SSL_CTX *ctx) { ::SSL_CTX_free( ctx ); }

   /*
    *   SSL_set_fd - connect the SSL object with a file descriptor
    */
    int SSL_set_fd(SSL *ssl, int fd) { return ::SSL_set_fd( ssl, fd ); }

   /*
    *   SSL_pending - obtain number of readable bytes buffered in an SSL object
    */
    int SSL_pending(SSL *ssl) { return ::SSL_pending( ssl ); }

   /*
    *   SSL_CTX_set_cipher_list - choose list of available SSL_CIPHERs
    */
    int SSL_CTX_set_cipher_list(SSL_CTX *ctx, const char *str)
        { return ::SSL_CTX_set_cipher_list( ctx, str ); }

   /*
    *   SSL_CTX_set_verify - set peer certificate verification parameters
    */
   void SSL_CTX_set_verify(SSL_CTX *ctx, int mode,
                         int (*verify_callback)(int, X509_STORE_CTX *))
        { ::SSL_CTX_set_verify( ctx, mode, verify_callback ); }

   /*
    *   SSL_CTX_use_certificate - load certificate and key data
    */
    int SSL_CTX_use_certificate(SSL_CTX *ctx, X509 *x)
        { return ::SSL_CTX_use_certificate( ctx, x ); }

   /*
    *   SSL_get_current_cipher - get SSL_CIPHER of a connection
    */
   SSL_CIPHER *SSL_get_current_cipher(SSL *ssl)
        { return ::SSL_get_current_cipher( ssl ); }

   /*
    *   SSL_set_options - manipulate SSL engine options
    *   Note: These are all mapped to SSL_ctrl so call them as the comment
    *         specifies but know that the use SSL_ctrl
    */
   /* long SSL_set_options(SSL *ssl, long options); */
   long    SSL_ctrl(SSL *ssl,int cmd, long larg, char *parg)
        { return ::SSL_ctrl( ssl, cmd, larg, parg ); }

   /*
    *   RAND_egd - set the path to the EGD
    */
   int RAND_egd(const char *path)
        { return ::RAND_egd( path ); }


   /*
    *   TLSv1_client_method - return a TLSv1 client method object
    */
   SSL_METHOD *TLSv1_client_method()
        { return ::TLSv1_client_method(); }


   /*
    *   SSLv2_client_method - return a SSLv2 client method object
    */
   SSL_METHOD *SSLv2_client_method()
        { return ::SSLv2_client_method(); }


   /*
    *   SSLv3_client_method - return a SSLv3 client method object
    */
   SSL_METHOD *SSLv3_client_method()
        { return ::SSLv3_client_method(); }


   /*
    *   SSLv23_client_method - return a SSLv23 client method object
    */
   SSL_METHOD *SSLv23_client_method()
        { return ::SSLv23_client_method(); }


   /*
    *   SSL_get_peer_certificate - return the peer's certificate
    */
   X509 *SSL_get_peer_certificate(SSL *s)
        { return::SSL_get_peer_certificate( s ); }


   /*
    *   SSL_CIPHER_get_bits - get the number of bits in this cipher
    */
   int SSL_CIPHER_get_bits(SSL_CIPHER *c,int *alg_bits)
        { return ::SSL_CIPHER_get_bits( c, alg_bits ); }


   /*
    *   SSL_CIPHER_get_version - get the version of this cipher
    */
   char *SSL_CIPHER_get_version(SSL_CIPHER *c)
        { return ::SSL_CIPHER_get_version( c ); }


   /*
    *   SSL_CIPHER_get_name - get the name of this cipher
    */
   const char *SSL_CIPHER_get_name(SSL_CIPHER *c)
        { return ::SSL_CIPHER_get_name( c ); }


   /*
    *   SSL_CIPHER_description - get the description of this cipher
    */
   char *SSL_CIPHER_description(SSL_CIPHER *c,char *buf,int size)
        { return ::SSL_CIPHER_description( c, buf, size ); }


   /*
    *   d2i_X509 - Covert a text representation of X509 to an X509 object
    */
   X509 * d2i_X509(X509 **a,unsigned char **pp,long length)
        { return ::d2i_X509( a, pp, length ); }


   /*
    *   i2d_X509 - Covert an X509 object into a text representation
    */
   int i2d_X509(X509 *a,unsigned char **pp)
        { return ::i2d_X509( a, pp ); }


   /*
    *   X509_cmp - compare two X509 objects
    */
   int X509_cmp(X509 *a, X509 *b)
        { return ::X509_cmp( a, b ); }


   /*
    *   X509_dup - duplicate an X509 object
    */
   X509 *X509_dup(X509 *x509)
        { return ::X509_dup( x509 ); }


   /*
    *   X509_STORE_CTX_new - create an X509 store context
    */
   X509_STORE_CTX *X509_STORE_CTX_new(void)
        { return ::X509_STORE_CTX_new(); }


   /*
    *   X509_STORE_CTX_free - free up an X509 store context
    */
   void X509_STORE_CTX_free(X509_STORE_CTX *v)
        { ::X509_STORE_CTX_free( v ); }


   /*
    *   X509_verify_cert - verify the certificate
    */
   int X509_verify_cert(X509_STORE_CTX *v)
        { return ::X509_verify_cert( v ); }


   /*
    *   X509_STORE_new - create an X509 store
    */
   X509_STORE *X509_STORE_new(void)
        { return ::X509_STORE_new(); }


   /*
    *   X509_STORE_free - free up an X509 store
    */
   void X509_STORE_free(X509_STORE *v)
        { ::X509_STORE_free( v ); }


   /*
    *   X509_free - free up an X509
    */
   void X509_free(X509 *v)
        { ::X509_free( v ); }


   /*
    *   X509_NAME_oneline - return the X509 data in a string
    */
   char *X509_NAME_oneline(X509_NAME *a, char *buf, int size)
        { return ::X509_NAME_oneline( a, buf, size ); }


   /*
    *   X509_get_subject_name - return the X509_NAME for the subject field
    */
   X509_NAME *X509_get_subject_name(X509 *a)
        { return ::X509_get_subject_name( a ); }


   /*
    *   X509_get_issuer_name - return the X509_NAME for the issuer field
    */
   X509_NAME *X509_get_issuer_name(X509 *a)
        { return ::X509_get_issuer_name( a ); }


   /*
    *   X509_STORE_add_lookup - add a lookup file/method to an X509 store
    */
   X509_LOOKUP *X509_STORE_add_lookup(X509_STORE *v, X509_LOOKUP_METHOD *m)
        { return ::X509_STORE_add_lookup( v, m ); }


   /*
    *   X509_LOOKUP_file - Definition of the LOOKUP_file method
    */
   X509_LOOKUP_METHOD *X509_LOOKUP_file(void)
        { return ::X509_LOOKUP_file(); }


   /*
    *   X509_LOOKUP_ctrl - This is not normally called directly (use macros)
    */
   int X509_LOOKUP_ctrl(X509_LOOKUP *ctx, int cmd, const char *argc, long argl, char **ret)
        { return ::X509_LOOKUP_ctrl( ctx, cmd, argc, argl, ret ); }


   /*
    *   X509_STORE_CTX_init - initialize an X509 STORE context
    */
   void X509_STORE_CTX_init(X509_STORE_CTX *ctx, X509_STORE *store, X509 *x509, STACK_OF(X509) *chain)
        { ::X509_STORE_CTX_init( ctx, store, x509, chain ); }


   /*
    *   CRYPTO_free - free up an internally allocated object
    */
   void CRYPTO_free(void *x)
        { ::CRYPTO_free( x ); }

   /*
    *   X509_asn1_meth - used for netscape output
    */
   ASN1_METHOD *X509_asn1_meth()
       { return ::X509_asn1_meth(); }

   /*
    *   ASN1_i2d_fp - used for netscape output
    */
   int ASN1_i2d_fp(FILE *out, unsigned char *x) 
       { return ::ASN1_i2d_fp((int (*)())::i2d_ASN1_HEADER, out, x); }

   
   /*
    *   SSL_CTX_use_PrivateKey - set the private key for the session.
    *                          - for use with client certificates
    */
   int SSL_CTX_use_PrivateKey(SSL_CTX *ctx, EVP_PKEY *pkey)
       { return ::SSL_CTX_use_PrivateKey(ctx, pkey); }
 
   /*
    *   X509_print - print the text form of an X509
    */
   int X509_print(FILE *fp, X509 *x)
       { return ::X509_print_fp(fp, x); }

   /*
    *   Write a PKCS#12 to FILE*
    */
   int i2d_PKCS12_fp(FILE *fp, PKCS12 *p12)
       { return ::i2d_PKCS12_fp(fp, p12); }

   /*
    *   Read a PKCS#12 cert from fp
    */
   PKCS12 *d2i_PKCS12_fp(FILE *fp, PKCS12 **p12)
       { return ::d2i_PKCS12_fp(fp, p12); }
   
   
   /*
    *   Create a new PKCS#12 object
    */
   PKCS12 *PKCS12_new(void)
       { return ::PKCS12_new(); }

   /*
    *   Destroy that PKCS#12 that you created!
    */
   void PKCS12_free(PKCS12 *a) 
       { ::PKCS12_free(a); }
   
   /*
    *   Change the password on a PKCS#12 cert
    */
   int PKCS12_newpass(PKCS12 *p12, char *oldpass, char *newpass)
       { return ::PKCS12_newpass(p12, oldpass, newpass); }

   /* 
    *   Free the Private Key
    */
   void EVP_PKEY_free(EVP_PKEY *x)
       { return ::EVP_PKEY_free(x); }

   /* 
    *   Parse the PKCS#12 
    */
   int PKCS12_parse(PKCS12 *p12, const char *pass, EVP_PKEY **pkey,
                    X509 **cert, STACK_OF(X509) **ca)
       { return ::PKCS12_parse(p12, pass, pkey, cert, ca); }

#endif

private:
    KOpenSSLProxy();

   static KOpenSSLProxy *_me;
};

#endif

