/*
	LMDriver.cpp
	12 October 2000
	This code is part of Ryan Lockhart's Lexmark 5xxx and 7xxx driver for BeOS.
	It is distributed with no warranty of any kind, in hope that it may be useful
	to someone.
*/

#include "LMDriver.h"
#include <malloc.h>
#include <stdio.h>
#include <StorageKit.h>
#include "LMInit.h"
#include "LMPageEject.h"
#include "LMAdvance.h"
#include "LMCommand.h"
#include "LMPageLoad.h"
#include "LMPageSetup.h"
#include "LMDither8.h"
#include "LMGraphUtils.h"

LMDriver::LMDriver()
{
}
	
	
LMDriver::~LMDriver()
{
}


status_t
LMDriver::Print(BFile * spool, BNode * printer, BMessage * msg)
{
	print_file_header h;
	BMessage * spool_msg = new BMessage();
	LMInit * init  = new LMInit();
	LMPageLoad * load = new LMPageLoad();
	LMPageEject * eject = new LMPageEject();
	LMAdvance * advance = new LMAdvance(SWIPE_HEIGHT * 2);
	int page, copies, copy;
	
	if (GetTransport(printer) == B_ERROR)
	{
		CloseTransport();
		return B_ERROR;
	}
	
	spool->Seek(0, SEEK_SET);
	spool->Read(&h, sizeof(print_file_header));
	
	spool_msg->Unflatten(spool);
	
	if (spool_msg->HasInt32("copies"))
		copies = spool_msg->FindInt32("copies");
	else
		copies = 1;
	
	for (copy = 0; copy < copies; copy++)
	{	
		for (page = 0; page < h.page_count; page++)
		{
			if (page == 0 && copy == 0)
				init->Write(transport);
			else
				load->Write(transport);
		
			advance->Write(transport);
		
			if (spool_msg->FindInt32("orientation") == PORTRAIT)
				PrintPage(spool, spool_msg);
			else
				PrintPageL(spool, spool_msg);
				
			eject->Write(transport);
		}
		spool->Seek(sizeof(print_file_header), SEEK_SET);
		spool_msg->Unflatten(spool);
	}

	CloseTransport();
	
	delete spool_msg;
	delete init;
	delete advance;
	delete load;
	delete eject;
	return B_OK;
}



void
LMDriver::PrintPage(BFile * spool, BMessage * msg)
{
	LMAdvance * advance = new LMAdvance();
	LMCommand * command = new LMCommand();
	long picture_count;
	BRect print_rect;
	BRect * pic_rects;
	BPoint * pic_points;
	BRegion * pic_region;
	BPicture ** pictures;
	int32 pic;
	int32 swipes;
	int32 swipe;
	int32 advcount;
	byte * bits;
	
	print_rect = msg->FindRect("printable_rect");
	advance->SetOffset((int)(print_rect.top * 2 * SCALE));
	advance->Write(transport);
	
	spool->Read(&picture_count, sizeof(long));
		
	pictures = (BPicture **)malloc(picture_count * sizeof(BPicture *));
	pic_rects = (BRect *)malloc(picture_count * sizeof(BRect));
	pic_points = (BPoint *)malloc(picture_count * sizeof(BPoint));
	pic_region = new BRegion();
	
	for (pic = 0; pic < picture_count; pic++)
	{
		spool->Seek(40 + sizeof(off_t), SEEK_CUR);
		spool->Read(&pic_points[pic], sizeof(BPoint));
		spool->Read(&pic_rects[pic], sizeof(BRect));
		pictures[pic] = new BPicture();
		pictures[pic]->Unflatten(spool);
		pic_region->Include(pic_rects[pic]);
	}
	
	bits = (byte *)malloc((int)ceil(SWIPE_HEIGHT * (pic_region->Frame().right - pic_region->Frame().left) * SCALE));
	swipes = (int)ceil((pic_region->Frame().bottom - pic_region->Frame().top) * SCALE / SWIPE_HEIGHT);
	
	advance->SetOffset(SWIPE_HEIGHT * 2 - 2);
	
	view = new BView(BRect(0, 0, (pic_region->Frame().right - pic_region->Frame().left) * SCALE,
							SWIPE_HEIGHT - 1), "", B_FOLLOW_ALL, B_WILL_DRAW);
							
	bitmap = new BBitmap(BRect(0, 0, (pic_region->Frame().right - pic_region->Frame().left) * SCALE,
								SWIPE_HEIGHT - 1), B_CMAP8, true);
	 
	bitmap->AddChild(view);
	
	view->Window()->Lock();
	view->SetScale(SCALE);
	
	advcount = 0;
	for (swipe = 0; swipe < swipes; swipe++)
	{
		view->SetHighColor(255,255,255);
		view->FillRect(view->Bounds());
		view->MovePenTo(0,0);
		
		for (pic = 0; pic < picture_count; pic++)
		{
			view->MovePenBy(0, -SWIPE_HEIGHT/SCALE * swipe);
			view->DrawPicture(pictures[pic]);
		}
		
		view->Sync();
		
		dither((byte *)bitmap->Bits(), bitmap->BytesPerRow(), SWIPE_HEIGHT, bits);
		
		if (command->SetData(bits, bitmap->BytesPerRow(), (int16)(print_rect.left * SCALE)) == LMC_OK)
		{
			advance->SetOffset((SWIPE_HEIGHT * 2 - 2) * advcount);
			advance->Write(transport);
			advcount = 1;
			command->Write(transport);
		}
		else
			advcount++;
	}
	
	view->RemoveSelf();
	
	free(bits);
	delete view;
	delete bitmap;
	delete advance;
	delete command;
	delete pic_points;
	delete pic_rects;
	for (pic = 0; pic < picture_count; pic++)
		delete pictures[pic];
	delete pictures;
	delete pic_region;
}


void
LMDriver::PrintPageL(BFile * spool, BMessage * msg)
{
	LMAdvance * advance = new LMAdvance();
	LMCommand * command = new LMCommand();
	long picture_count;
	BRect print_rect;
	BRect paper_rect;
	BRect * pic_rects;
	BPoint * pic_points;
	BRegion * pic_region;
	BPicture ** pictures;
	int32 pic;
	int32 swipes;
	int32 swipe;
	int32 advcount;
	byte * bits;
	byte * rotated;
	float temp;
	
	print_rect = msg->FindRect("printable_rect");
	paper_rect = msg->FindRect("paper_rect");
	
	temp = paper_rect.bottom - print_rect.top;
	print_rect.top = print_rect.left;
	print_rect.left = paper_rect.bottom - print_rect.bottom;
	print_rect.bottom = print_rect.right;
	print_rect.right = temp;
	
	advance->SetOffset((int)(print_rect.top * 2 * SCALE));
	advance->Write(transport);
	
	spool->Read(&picture_count, sizeof(long));
		
	pictures = (BPicture **)malloc(picture_count * sizeof(BPicture *));
	pic_rects = (BRect *)malloc(picture_count * sizeof(BRect));
	pic_points = (BPoint *)malloc(picture_count * sizeof(BPoint));
	pic_region = new BRegion();
	
	for (pic = 0; pic < picture_count; pic++)
	{
		spool->Seek(40 + sizeof(off_t), SEEK_CUR);
		spool->Read(&pic_points[pic], sizeof(BPoint));
		spool->Read(&pic_rects[pic], sizeof(BRect));
		pictures[pic] = new BPicture();
		pictures[pic]->Unflatten(spool);
		pic_region->Include(pic_rects[pic]);
	}
	
	bits = (byte *)malloc((int)ceil(SWIPE_HEIGHT * (pic_region->Frame().right - pic_region->Frame().left) * SCALE));
	rotated = (byte *)malloc((int)ceil(SWIPE_HEIGHT * (pic_region->Frame().right - pic_region->Frame().left) * SCALE));

	swipes = (int)ceil((pic_region->Frame().right - pic_region->Frame().left) * SCALE / SWIPE_HEIGHT);
	
	advance->SetOffset(SWIPE_HEIGHT * 2 - 2);
	
	view = new BView(BRect(0, 0, SWIPE_HEIGHT - 1,
					(pic_region->Frame().bottom - pic_region->Frame().top) * SCALE),
					"", B_FOLLOW_ALL, B_WILL_DRAW);
							
	bitmap = new BBitmap(BRect(0, 0, SWIPE_HEIGHT - 1,
					(pic_region->Frame().bottom - pic_region->Frame().top) * SCALE),
					B_CMAP8, true);
	 
	bitmap->AddChild(view);
	
	view->Window()->Lock();
	view->SetScale(SCALE);
	
	advcount = 0;
	for (swipe = 0; swipe < swipes; swipe++)
	{
		view->SetHighColor(255,255,255);
		view->FillRect(view->Bounds());
		view->MovePenTo(0,0);
		
		for (pic = 0; pic < picture_count; pic++)
		{
			view->MovePenBy(-SWIPE_HEIGHT/SCALE * swipe, 0);
			view->DrawPicture(pictures[pic]);
		}
		
		view->Sync();
		
		rotate((byte *)bitmap->Bits(), bitmap->BytesPerRow(), int32((pic_region->Frame().bottom - pic_region->Frame().top) * SCALE), rotated);
		dither(rotated, int16((pic_region->Frame().bottom - pic_region->Frame().top) * SCALE), SWIPE_HEIGHT, bits);
		
		if (command->SetData(bits, (pic_region->Frame().bottom - pic_region->Frame().top) * SCALE, (int16)(print_rect.left * SCALE)) == LMC_OK)
		{
			advance->SetOffset((SWIPE_HEIGHT * 2 - 2) * advcount);
			advance->Write(transport);
			advcount = 1;
			command->Write(transport);
		}
		else
			advcount++;
	}
	
	view->RemoveSelf();
	
	free(bits);
	free(rotated);
	delete view;
	delete bitmap;
	delete advance;
	delete command;
	delete pic_points;
	delete pic_rects;
	for (pic = 0; pic < picture_count; pic++)
		delete pictures[pic];
	delete pictures;
	delete pic_region;
}

status_t
LMDriver::GetTransport(BNode * printer)
{
	BPath * path = new BPath();
	char buffer[512];
	node_ref nref;
	BDirectory dir;
	
	printer->ReadAttr("transport", B_STRING_TYPE, 0, buffer, sizeof(buffer));
	
	find_directory(B_USER_ADDONS_DIRECTORY, path);
	path->Append("Print/transport");
	path->Append(buffer);
	transport_add_on = load_add_on(path->Path());
	
	if (transport_add_on < 0)
	{
		find_directory(B_BEOS_ADDONS_DIRECTORY, path);
		path->Append("Print/transport");
		path->Append(buffer);
		transport_add_on = load_add_on(path->Path());
	}
	
	if (transport_add_on < 0)
	{
		BAlert * a = new BAlert("Uh oh!", "Couldn't find transport add-on.", "OK");
		a->Go();
		return B_ERROR;
	}
	
	get_image_symbol(transport_add_on, "init_transport", B_SYMBOL_TYPE_TEXT, (void **)&init_transport);
	get_image_symbol(transport_add_on, "exit_transport", B_SYMBOL_TYPE_TEXT, (void **)&exit_transport);
	
	if (!init_transport || !exit_transport)
	{
		BAlert * a = new BAlert("Uh oh!", "Couldn't resolve transport symbols.", "OK");
		a->Go();
		return B_ERROR;
	}
	
	delete path;
	
	printer->GetNodeRef(&nref);
	dir.SetTo(&nref);
	path = new BPath(&dir, NULL);
	strcpy(buffer, path->Path());
	BMessage *msg = new BMessage('TRIN');
	msg->AddString("printer_file", buffer);
	transport = (*init_transport)(msg);
	delete msg;
	delete path;
	
	if (transport == 0)
	{
		BAlert * a = new BAlert("Uh oh!", "Couldn't open transport.", "OK");
		a->Go();
		return B_ERROR;
	}
	
	return B_OK;
}

void
LMDriver::CloseTransport()
{
	if (exit_transport)
		(*exit_transport)();
	if (transport_add_on)
		unload_add_on(transport_add_on);
}

